/*++

Copyright (c) Microsoft Corporation.  All rights reserved.

Module Name:

    controller.cpp

Abstract:

    This module contains implementation of the SpbController class

Environment:

    kernel-mode only

Revision History:

--*/

#include "internal.h"
#include "controller.tmh"

VOID
CScxController::_DeviceInitSetDefaultCallbacks(
    __in  PWDFCXDEVICE_INIT  CxDeviceInit
    )
/*++

  Routine Description:

    This routine sets up the in-caller context and file callbacks for
    the controller object

  Arguments:

    CxDeviceInit - information about the PDO that we are loading on

  Return Value:

    None

--*/
{
    //
    // Set I/O preprocessor callback.
    //

    WdfCxDeviceInitSetIoInCallerContextCallback(
        CxDeviceInit,
        CScxController::_OnIoInCallerContext
        );

    //
    // Assign a preprocess callback for power IRPs.
    //

    {
        NTSTATUS status;

        status = WdfCxDeviceInitAssignWdmIrpPreprocessCallback(
            CxDeviceInit,
            _OnIrpPreprocess,
            IRP_MJ_POWER,
            NULL,
            0);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Failed to assign a preprocess callback for IRP_MJ_POWER"
                " to CxDeviceInit %p - "
                "%!STATUS!",
                CxDeviceInit,
                status
                );
        }

        status = WdfCxDeviceInitAssignWdmIrpPreprocessCallback(
            CxDeviceInit,
            _OnIrpPreprocess,
            IRP_MJ_PNP,
            NULL,
            0);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Failed to assign a preprocess callback for IRP_MJ_PNP"
                " to CxDeviceInit %p - "
                "%!STATUS!",
                CxDeviceInit,
                status
                );
        }
    }

    //
    // Prepare for file object handling.
    //

    {
        WDFCX_FILEOBJECT_CONFIG fileObjectConfig;

        WDFCX_FILEOBJECT_CONFIG_INIT(&fileObjectConfig,
                                     _OnFileCreate,
                                     _OnFileClose,
                                     _OnFileCleanup);

        //
        // ACPI PDO will forward requests it can't handle to its parent stack.
        // Hence we need to be prepared to receive requests that may not
        // have a valid WDFFILEOBJECT associated with them.
        //

        fileObjectConfig.FileObjectClass =
            (WDF_FILEOBJECT_CLASS) (fileObjectConfig.FileObjectClass | WdfFileObjectCanBeOptional);

        WDF_OBJECT_ATTRIBUTES fileObjectAttributes;

        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&fileObjectAttributes,
                                                CScxTarget);
        CScxTarget::_SetObjectAttributes(&fileObjectAttributes);

        WdfCxDeviceInitSetFileObjectConfig(CxDeviceInit,
                                           &fileObjectConfig,
                                           &fileObjectAttributes);
    }

    //
    // Add the CScxRequest as the default context on requests sent to the
    // framework extension
    //

    {
        WDF_OBJECT_ATTRIBUTES requestAttributes;

        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&requestAttributes,
                                                CScxRequest);
        CScxRequest::_SetObjectAttributes(&requestAttributes);

        WdfCxDeviceInitSetRequestAttributes(CxDeviceInit, &requestAttributes);
    }
}

NTSTATUS
CScxController::_DeviceInitSetDefaultDacl(
    __in  PWDFDEVICE_INIT  DeviceInit
    )
/*++

  Routine Description:

    This routine sets up the default DACL for the device

  Arguments:

    DeviceInit - information about the PDO that we are loading on

  Return Value:

    Status.

--*/
{
    NTSTATUS status;

    //
    // Auto-generate the device object name and set the
    // security descriptor to allow UMDF driver access.
    // This can be overridden by the controller driver.
    //

    WdfDeviceInitSetCharacteristics(
        DeviceInit,
        FILE_AUTOGENERATED_DEVICE_NAME,
        TRUE);

    status = WdfDeviceInitAssignSDDLString(
       DeviceInit,
       &SDDL_DEVOBJ_SPBCX_SYS_ALL_ADM_ALL_UMDF_ALL
       );

    if (!NT_SUCCESS(status))
    {
        TraceMessage(
            TRACE_LEVEL_ERROR,
            TRACE_FLAG_CONTROLLER,
            "Failed to assign SDDL string %wZ to DeviceInit %p - "
            "%!STATUS!",
            &SDDL_DEVOBJ_SPBCX_SYS_ALL_ADM_ALL_UMDF_ALL,
            DeviceInit,
            status
            );
    }

    return status;
}

NTSTATUS
CScxController::_CreateAndInitialize(
    __in     WDFDEVICE               FxDevice,
    __in     PSPB_CONTROLLER_CONFIG  Config
    )
{
    NTSTATUS status;
    PCScxController controller = NULL;

    //
    // Allocate the context for the controller
    //

    {
        WDF_OBJECT_ATTRIBUTES controllerAttributes;
        PVOID context;

        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&controllerAttributes,
                                                CScxController);

        CScxController::_SetObjectAttributes(&controllerAttributes);

        status = WdfObjectAllocateContext(FxDevice,
                                          &controllerAttributes,
                                          &context);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error allocating controller object context - %!STATUS!",
                status);
            goto exit;
        }

        controller = new(context) CScxController(FxDevice);

        if (controller == NULL)
        {
            NT_ASSERTMSG("Controller object context shouldn't be NULL", controller != NULL);

            status = STATUS_INSUFFICIENT_RESOURCES;
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error constructing controller object context - %!STATUS!",
                status);
            goto exit;
        }
    }

    status = controller->Initialize(Config);

    if (!NT_SUCCESS(status))
    {
        goto exit;
    }

exit:

    return status;
}

NTSTATUS
CScxController::Initialize(
    __in PSPB_CONTROLLER_CONFIG Config
    )
{
    NTSTATUS status;

    //
    // Register a dispatch callback to dynamically intercept
    // requests and forward to the appropriate target queue.
    // The same callback is used for each IRP major function.
    //

    {
        status = WdfDeviceConfigureWdmIrpDispatchCallback(
            m_FxDevice,
            WdfGetDriver(),
            IRP_MJ_WRITE,
            _OnIrpDispatch,
            NULL);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error registering dispatch callback for "
                "IRP_MJ_WRITE for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        status = WdfDeviceConfigureWdmIrpDispatchCallback(
            m_FxDevice,
            WdfGetDriver(),
            IRP_MJ_READ,
            _OnIrpDispatch,
            NULL);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error registering dispatch callback for "
                "IRP_MJ_READ for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        status = WdfDeviceConfigureWdmIrpDispatchCallback(
            m_FxDevice,
            WdfGetDriver(),
            IRP_MJ_DEVICE_CONTROL,
            _OnIrpDispatch,
            NULL);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error registering dispatch callback for "
                "IRP_MJ_DEVICE_CONTROL for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        status = WdfDeviceConfigureWdmIrpDispatchCallback(
            m_FxDevice,
            WdfGetDriver(),
            IRP_MJ_INTERNAL_DEVICE_CONTROL,
            _OnIrpDispatch,
            NULL);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error registering dispatch callback for "
                "IRP_MJ_INTERNAL_DEVICE_CONTROL for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }
    }

    //
    // Create controller queues
    //

    {
        WDF_OBJECT_ATTRIBUTES queueAttributes;
        WDF_IO_QUEUE_CONFIG queueConfig;

        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&queueAttributes,
                                                PCScxController);
        queueAttributes.ParentObject = GetFxObject();

        //
        // Create the controller level queue.  This one follows the dispatch type
        // and power manager setting specified by the client.
        //

        WDF_IO_QUEUE_CONFIG_INIT(
            &queueConfig,
            Config->ControllerDispatchType
            );

        queueConfig.PowerManaged = Config->PowerManaged;
        queueConfig.Driver = WdfGetDriver();
        queueConfig.EvtIoDefault = _OnControllerIoDefault;

        status = WdfIoQueueCreate(m_FxDevice,
                                  &queueConfig,
                                  &queueAttributes,
                                  &m_ControllerQueue);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error creating controller queue for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        *(_GetControllerFromFxQueue(m_ControllerQueue)) = this;

        //
        // Create the lock queue.  This is the manual-dispatch queue used for
        // coordinating LOCK requests with I/O at the controller level.
        //

        WDF_IO_QUEUE_CONFIG_INIT(&queueConfig,
                                 WdfIoQueueDispatchManual);

        queueConfig.PowerManaged = Config->PowerManaged;
        queueConfig.Driver = WdfGetDriver();

        status = WdfIoQueueCreate(m_FxDevice,
                                  &queueConfig,
                                  &queueAttributes,
                                  &m_LockQueue);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error creating lock queue for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        *(_GetControllerFromFxQueue(m_LockQueue)) = this;

        //
        // Register for notifications when the lock queue becomes ready (e.g.
        // when the queue count goes from 0 -> !0).
        //

#pragma prefast(suppress:__WARNING_RETVAL_IGNORED_FUNC_COULD_FAIL, "The queue is new and not in a state which could cause this to fail")
        WdfIoQueueReadyNotify(m_LockQueue,
                              _OnLockQueueReady,
                              this);
    }

    //
    // Create a synchronization event to protect locked IO
    // execution from a power transition.
    //

    KeInitializeEvent(
        &m_LockedIoSynchronizationEvent,
        SynchronizationEvent,
        TRUE);

    //
    // Create the locked IO watchdog timer.
    //

    {
        WDF_TIMER_CONFIG timerConfig;
        WDF_OBJECT_ATTRIBUTES timerAttributes;

        WDF_TIMER_CONFIG_INIT(&timerConfig, _OnLockedIoWatchdogExpired);

        WDF_OBJECT_ATTRIBUTES_INIT(&timerAttributes);
        timerAttributes.ParentObject = GetFxObject();

        status = WdfTimerCreate(
            &timerConfig,
            &timerAttributes,
            &m_LockedIoWatchdog);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error creating watchdog timer for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }
    }

    //
    // Allocate the wait-lock we use for protecting the list of targets.
    //

    {
        WDF_OBJECT_ATTRIBUTES lockAttributes;

        WDF_OBJECT_ATTRIBUTES_INIT(&lockAttributes);
        lockAttributes.ParentObject = GetFxObject();

        status = WdfWaitLockCreate(&lockAttributes, &m_TargetListLock);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error creating target list lock for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }
    }

    //
    // Create the IoTarget to the Resource Hub
    //

    {
        WDF_OBJECT_ATTRIBUTES ioTargetAttributes;

        WDF_OBJECT_ATTRIBUTES_INIT(&ioTargetAttributes);

        status = WdfIoTargetCreate(m_FxDevice,
                                   &ioTargetAttributes,
                                   &m_ResourceHubTarget);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error creating Resource Hub IoTarget for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }

        WDF_IO_TARGET_OPEN_PARAMS ioTargetParameters;
        UNICODE_STRING resourceHubName;

        RtlInitUnicodeString(&resourceHubName, RESOURCE_HUB_DEVICE_NAME);
        WDF_IO_TARGET_OPEN_PARAMS_INIT_OPEN_BY_NAME(&ioTargetParameters,
                                                    &resourceHubName,
                                                    STANDARD_RIGHTS_ALL);

        ioTargetParameters.EvtIoTargetRemoveComplete = _OnIoTargetRemoveComplete;

        //
        // Open the remote target to the resource hub
        //

        status = WdfIoTargetOpen(m_ResourceHubTarget, &ioTargetParameters);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error opening Resource Hub IoTarget for WDFDEVICE %p - %!STATUS!",
                GetFxObject(),
                status
                );
            goto exit;
        }
    }

    //
    // Save the config in the controller so we can reference the
    // callbacks.
    //

    {
        m_DriverConfig = *Config;

        //
        // Setup the array of invoke methods based on the callbacks the
        // driver has setup.
        //

        if (DriverSupportsLockUnlock())
        {
            m_InvokeIoCallbacks[SpbRequestTypeLockController] =
                &CScxController::InvokeSpbControllerLock;

            m_InvokeIoCallbacks[SpbRequestTypeUnlockController] =
                &CScxController::InvokeSpbControllerUnlock;
        }

        m_InvokeIoCallbacks[SpbRequestTypeRead] =
            &CScxController::InvokeSpbIoRead;

        m_InvokeIoCallbacks[SpbRequestTypeWrite] =
            &CScxController::InvokeSpbIoWrite;

        m_InvokeIoCallbacks[SpbRequestTypeSequence] =
            &CScxController::InvokeSpbIoSequence;

        m_InvokeIoCallbacks[SpbRequestTypeLockConnection] =
            &CScxController::InvokeSpbConnectionLock;

        m_InvokeIoCallbacks[SpbRequestTypeUnlockConnection] =
            &CScxController::InvokeSpbConnectionUnlock;
    }

    //
    // Apply the security descriptor now
    // that the PDO is attached to the device object.
    //

    {
        PDEVICE_OBJECT deviceObject;
        PSECURITY_DESCRIPTOR securityDescriptor;
        SECURITY_INFORMATION securityInformation;
        BOOLEAN allocated;

        deviceObject = WdfDeviceWdmGetDeviceObject(m_FxDevice);
        NT_ASSERTMSG("Device object shouldn't be NULL", deviceObject != NULL);

        status = ObGetObjectSecurity(
            deviceObject,
            &securityDescriptor,
            &allocated);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_CONTROLLER,
                "Error getting SECURITY_DESCRIPTOR for "
                "DEVICE_OBJECT %p - %!STATUS!",
                deviceObject,
                status);

            goto exit;
        }

        if (securityDescriptor != NULL)
        {
            securityInformation = OWNER_SECURITY_INFORMATION |
                                  GROUP_SECURITY_INFORMATION |
                                  DACL_SECURITY_INFORMATION  |
                                  SACL_SECURITY_INFORMATION  |
                                  LABEL_SECURITY_INFORMATION;

            status = ObSetSecurityObjectByPointer(
                deviceObject,
                securityInformation,
                securityDescriptor);

            if (!NT_SUCCESS(status))
            {
                TraceMessage(
                    TRACE_LEVEL_ERROR,
                    TRACE_FLAG_CONTROLLER,
                    "Error setting SECURITY_DESCRIPTOR %p for "
                    "DEVICE_OBJECT %p - %!STATUS!",
                    securityDescriptor,
                    deviceObject,
                    status);
            }
        }

        ObReleaseObjectSecurity(securityDescriptor, allocated);
    }

exit:

    if (!NT_SUCCESS(status))
        if (m_ResourceHubTarget != NULL)
            WdfObjectDelete(m_ResourceHubTarget);

    return status;
}


BOOLEAN
CScxController::_OnFileCreate(
    __in WDFDEVICE     FxDevice,
    __in WDFREQUEST    FxRequest,
    __in WDFFILEOBJECT FxFileObject
    )
/*++

  Routine Description:

    This routine handles create requests to SPBCx.  It gets the ConnectionID from
    the file name and then if no target with that ConnectionID already exists,
    creates the target.  It then completes the request.

  Arguments:

    FxDevice - the device/controller being opened

    FxRequest - the create request

    FxFileObject - the file object/target being opened

  Return Value:

    None

--*/
{
    PCScxController controller = GetControllerFromFxDevice(FxDevice);
    NTSTATUS status;
    LARGE_INTEGER connectionid;
    LPWSTR scopeTag;
    BOOLEAN createHandled = FALSE;
    BOOLEAN refCounted = FALSE;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER spbConnection = NULL;
    PPNP_SERIAL_BUS_DESCRIPTOR spbDescriptor;
    PVOID spbTypeSpecificData;

    //
    // Parse the file object and see if it ends with an appropriate
    // connection Id and scope tag.
    //

    status = CScxTarget::_ParseFileObjectName(FxFileObject,
                                              &connectionid,
                                              &scopeTag);

    if (NT_SUCCESS(status))
    {
        //
        // Filename recognized. The class extension will handle the
        // create and not hand it off to the controller.
        //

        createHandled = TRUE;
    }
    else
    {
        //
        // If we don't recognize the filename, we should either complete
        // the request or hand it off to the controller depending on status.
        //

        switch(status)
        {
            //
            // STATUS_NO_SUCH_FILE - there is no filename. The class
            //     extension treats this error as a query and completes
            //     the request with STATUS_SUCCESS without creating a target.
            //

            case STATUS_NO_SUCH_FILE:

                TraceMessage(
                    TRACE_LEVEL_INFORMATION,
                    TRACE_FLAG_CONTROLLER,
                    "CREATE on controller %p target %wZ no filename, "
                    "treat as query and complete - %!STATUS!",
                    FxDevice,
                    WdfFileObjectGetFileName(FxFileObject),
                    status
                    );
                status = STATUS_SUCCESS;
                createHandled = TRUE;
                goto exit;

            //
            // STATUS_OBJECT_NAME_INVALID - the filename format is recognized
            //     but invalid. The class extension will fail the create
            //     and not hand it off to the controller.
            //

            case STATUS_OBJECT_NAME_INVALID:

                TraceMessage(
                    TRACE_LEVEL_INFORMATION,
                    TRACE_FLAG_CONTROLLER,
                    "CREATE on controller %p target %wZ invalid filename, "
                    "failing CREATE request - %!STATUS!",
                    FxDevice,
                    WdfFileObjectGetFileName(FxFileObject),
                    status
                    );
                createHandled = TRUE;
                goto exit;

            //
            // Other - the filename is not recognized. Handoff the request to
            //     the controller without creating a target.
            //

            default:

                TraceMessage(
                    TRACE_LEVEL_INFORMATION,
                    TRACE_FLAG_CONTROLLER,
                    "CREATE on controller %p target %wZ failed to parse "
                    "filename, handoff to controller - %!STATUS!",
                    FxDevice,
                    WdfFileObjectGetFileName(FxFileObject),
                    status
                    );
                goto exit;
        }
    }

    //
    // Fetch the BIOS descriptor for this connection.
    //

    status = CScxTarget::QueryDescriptorFromResourceHub(
                 controller,
                 connectionid,
                 &spbConnection);

    if (!NT_SUCCESS(status))
    {
        TraceMessage(
            TRACE_LEVEL_ERROR,
            TRACE_FLAG_CONTROLLER,
            "CREATE on controller %p target %wZ failed to fetch BIOS "
            "descriptor for connection %I64x - %!STATUS!",
            FxDevice,
            WdfFileObjectGetFileName(FxFileObject),
            connectionid.QuadPart,
            status
            );
        goto exit;
    }

    NT_ASSERT(spbConnection->PropertiesLength >=
                  sizeof(PNP_SERIAL_BUS_DESCRIPTOR));

    spbDescriptor =
        (PPNP_SERIAL_BUS_DESCRIPTOR)spbConnection->ConnectionProperties;

    NT_ASSERT(spbConnection->PropertiesLength >= spbDescriptor->Length);

    spbTypeSpecificData =
        Add2Ptr(spbDescriptor, sizeof(PNP_SERIAL_BUS_DESCRIPTOR));

    //
    // Lock the list of target objects and see if the specified connection id exists
    // in the list.  If not then attach the target to the file object
    // and insert the target into the list.
    //

    PLIST_ENTRY entry;
    PCScxTarget target = NULL;

    WdfWaitLockAcquire(controller->m_TargetListLock, NULL);

    status = STATUS_SUCCESS;

    for(entry = controller->m_TargetListHead.Flink;
        entry != &(controller->m_TargetListHead);
        entry = entry->Flink)
    {
        PCScxTarget t = CScxTarget::_FromListEntry(entry);

        LARGE_INTEGER id = t->GetConnectionId();
        LPWSTR tag = t->GetScopeTag();

        PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER currentConnection;
        PPNP_SERIAL_BUS_DESCRIPTOR currentSpbDescriptor;

        currentConnection = t->GetConnectionProperties();

        NT_ASSERT(currentConnection->PropertiesLength >
                    sizeof(PNP_SERIAL_BUS_DESCRIPTOR));

        currentSpbDescriptor =
            (PPNP_SERIAL_BUS_DESCRIPTOR)
                currentConnection->ConnectionProperties;

        NT_ASSERT(currentConnection->PropertiesLength >=
                      currentSpbDescriptor->Length);

        //
        // Retain old behavior for SPBv1 descriptors. If either of the
        // descriptors is newer, then enforce new sharing behavior.
        //

        if ((spbDescriptor->RevisionId < SERIAL_BUS_DESCRIPTOR_REVISION_V2) &&
            (currentSpbDescriptor->RevisionId < SERIAL_BUS_DESCRIPTOR_REVISION_V2))
        {

            if ((id.QuadPart == connectionid.QuadPart) &&
                (wcscmp(tag, scopeTag) == 0))
            {
                status = STATUS_DEVICE_BUSY;
                break;
            }
        }
        else
        {
            PVOID currentTypeSpecificData;
            BOOLEAN TargetMatch;

            //
            // Check whether the supplied connection and the current connection
            // refer to the same target device. The target is considered to
            // be the same if either the descriptors are completely identical
            // (inferred by connection ID match) or their bus types and
            // type specific fields are identical.
            //

            if (id.QuadPart == connectionid.QuadPart)
            {
                TargetMatch = TRUE;

            } else if ((currentSpbDescriptor->SerialBusType ==
                            spbDescriptor->SerialBusType) &&
                       (currentSpbDescriptor->TypeDataLength ==
                            spbDescriptor->TypeDataLength))
            {
                currentTypeSpecificData =
                    Add2Ptr(currentSpbDescriptor, sizeof(PNP_SERIAL_BUS_DESCRIPTOR));

                TargetMatch = RtlEqualMemory(currentTypeSpecificData,
                                             spbTypeSpecificData,
                                             spbDescriptor->TypeDataLength);
            }
            else
            {
                TargetMatch = FALSE;
            }

            //
            // Only allow shared connection if BIOS descriptors for both
            // connections have the shared flag set.
            //
            // N.B. If one of the descriptors is v1 descriptor, then it is
            //      assumed that it supports sharing by default. In such case,
            //      sharing will be determined by the attribute in the
            //      newer v2 descriptor.
            //

            if ((TargetMatch != FALSE) &&
                ((SCX_IS_DESCRIPTOR_SHARED(currentSpbDescriptor) == FALSE) ||
                 (SCX_IS_DESCRIPTOR_SHARED(spbDescriptor) == FALSE)))
            {
                status = STATUS_DEVICE_BUSY;
                break;
            }
        }
    }

    if (NT_SUCCESS(status))
    {
        //
        // If this is the first  reference then check to see if there are any
        // function config connections for this controller. If yes, open
        // these connections.
        //
        
        controller->m_ControllerRefCount++;
        refCounted = TRUE;

        if ((controller->m_ControllerRefCount == 1) &&
            (controller->m_FunctionConfigConnectionCount != 0))
        {
            status = controller->OpenFunctionConfigConnections();

            if (!NT_SUCCESS(status))
            {
                //
                // Log the failure. No need to disconnect since
                // cleanup is already done inside OpenFunctionConfigConnections().
                //

                controller->m_ControllerRefCount--;
                refCounted = FALSE;

                TraceMessage(
                    TRACE_LEVEL_ERROR,
                    TRACE_FLAG_TARGET,
                    "Error opening function config connections."
                    "Failing CREATE request - %!STATUS!%",
                    status
                    );

                goto LockRelease;
            }
        }

        //
        // Create a target object attached to the file.
        //

        status = CScxTarget::_CreateAndInitialize(
                                controller,
                                FxFileObject,
                                &connectionid,
                                scopeTag,
                                (WdfRequestGetRequestorMode(FxRequest) ==
                                 UserMode),
                                spbConnection,
                                &target
                                );

        if (NT_SUCCESS(status))
        {
            InsertTailList(entry, target->GetTargetListEntry());

            //
            // Mark the buffer as being owned by the target object from now on.
            //

            spbConnection = NULL;
        }
        else
        {
            //
            // Log the failure.  Handle the error below after dropping
            // the lock.
            //

            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Error creating target. Failing CREATE request - %!STATUS!%",
                status
                );
        }
    }

LockRelease:

    //
    // If an error occured while holding the lock then exit. If
    // we have already ref-counted on the controller, drop the
    // ref-count here and close function config connections
    // if needed.
    //

    if (!NT_SUCCESS(status))
    {
        if (refCounted == TRUE)
        {
            controller->m_ControllerRefCount--;

            if ((controller->m_ControllerRefCount == 0) &&
                (controller->m_FunctionConfigConnectionCount != 0))
            {
                controller->CloseFunctionConfigConnections();
            }
        }

        //
        // Release the wait lock.
        //

        WdfWaitLockRelease(controller->m_TargetListLock);
    }
    else
    {
        //
        // Release the wait lock.
        //

        WdfWaitLockRelease(controller->m_TargetListLock);


        //
        // Invoke the target's OnCreate method
        //

        status = target->OnCreate();

        //
        // If that fails then remove the target from the target list
        // and let the create fail
        //

        if (!NT_SUCCESS(status))
        {
            WdfWaitLockAcquire(controller->m_TargetListLock, NULL);
            RemoveEntryList(target->GetTargetListEntry());

            if (refCounted == TRUE)
            {
                controller->m_ControllerRefCount--;

                if ((controller->m_ControllerRefCount == 0) &&
                    (controller->m_FunctionConfigConnectionCount != 0))
                {
                    controller->CloseFunctionConfigConnections();
                }
            }

            WdfWaitLockRelease(controller->m_TargetListLock);
        }
    }

exit:

    if (createHandled == TRUE)
    {
        WdfRequestComplete(FxRequest, status);
    }

    if (spbConnection != NULL)
    {
        ExFreePoolWithTag(spbConnection, SCX_POOL_TAG);
    }

    return createHandled;
}

VOID
CScxController::_OnFileCleanup(
    __in WDFFILEOBJECT FxFileObject
    )
{
    PCScxTarget target = GetTargetFromFxFile(FxFileObject);

    if (target->IsConstructed() == true)
    {
        target->OnCleanup();
    }

    return;
}

VOID
CScxController::_OnFileClose(
    __in WDFFILEOBJECT FxFileObject
    )
{
    PCScxTarget target = GetTargetFromFxFile(FxFileObject);

    if (target->IsConstructed() == true)
    {
        PCScxController controller = target->GetController();

        //
        // Close the target down.
        //

        target->OnClose();

        //
        // Remove from the list of active targets.
        //

        WdfWaitLockAcquire(controller->m_TargetListLock, NULL);

        //
        // Decrement the controller file ref count. If this is the last outstanding
        // ref count, close any oustanding function config connections.
        //

        controller->m_ControllerRefCount--;

        if (controller->m_ControllerRefCount == 0)
        {
            controller->CloseFunctionConfigConnections();
        }

        RemoveEntryList(target->GetTargetListEntry());
        WdfWaitLockRelease(controller->m_TargetListLock);
    }

    return;
}

NTSTATUS
CScxController::ParseFunctionConfigResources(
    _In_ PCM_RESOURCE_LIST ResourceList
    )
/*++

Routine Description:

    This routine parses the list of translated resources for the SPB
    controller and stores the function config descriptors.

Arguments:

    ResourceList - the list of translated resources for the SPB controller.

Return Value:

    NTSTATUS

--*/
{
    ULONG index;
    ULONG index2;
    PCM_FULL_RESOURCE_DESCRIPTOR list;
    ULONG numberResources = 0;
    USHORT pass = 0;
    NTSTATUS status = STATUS_SUCCESS;

    if (ResourceList == NULL)
    {
        goto exit;
    }

    //
    // Clear any stale function config connection allocations.
    //

    m_FunctionConfigConnectionCount = 0;

    if (m_FunctionConfigConnectionIds != NULL) 
    {
        ExFreePoolWithTag(m_FunctionConfigConnectionIds, SCX_POOL_TAG);
        m_FunctionConfigConnectionIds = NULL;
    }

    if (m_FunctionConfigConnectionTarget != NULL) 
    {
        ExFreePoolWithTag(m_FunctionConfigConnectionTarget, SCX_POOL_TAG);
        m_FunctionConfigConnectionTarget = NULL;
    }

    for (pass = 0; pass < 2; pass++)
    {
        list = ResourceList->List;

        //
        // During pass 0, determine the number many function config resources
        // there are in the resource list. Then allocate the connection id and
        // iotarget arrays to populate for pass 1.
        //

        if (pass == 1) 
        {
            m_FunctionConfigConnectionIds = (PLARGE_INTEGER) ExAllocatePoolWithTag(
                NonPagedPoolNx,
                sizeof(LARGE_INTEGER) * m_FunctionConfigConnectionCount,
                SCX_POOL_TAG
                );

            if (m_FunctionConfigConnectionIds == NULL)
            {
                status = STATUS_INSUFFICIENT_RESOURCES;
                TraceMessage(
                    TRACE_LEVEL_ERROR,
                    TRACE_FLAG_TARGET,
                    "Failure allocating buffer for connection ids"
                    "%!STATUS!",
                    status
                    );

                goto exit;
            }

            m_FunctionConfigConnectionTarget = (WDFIOTARGET *) ExAllocatePoolWithTag(
                NonPagedPoolNx,
                sizeof(WDFIOTARGET)* m_FunctionConfigConnectionCount,
                SCX_POOL_TAG
                );

            if (m_FunctionConfigConnectionTarget == NULL)
            {
                status = STATUS_INSUFFICIENT_RESOURCES;
                TraceMessage(
                    TRACE_LEVEL_ERROR,
                    TRACE_FLAG_TARGET,
                    "Failure allocating buffer for connection io targets"
                    "%!STATUS!",
                    status
                    );

                goto exit;
            }

            RtlZeroMemory(
                m_FunctionConfigConnectionTarget,
                sizeof(WDFIOTARGET) * m_FunctionConfigConnectionCount
                );
        
        }

        numberResources = 0;
    
        for (index = 0; index < ResourceList->Count; ++index) 
        {
            for (index2 = 0; index2 < list->PartialResourceList.Count; ++index2) 
            {
                PCM_PARTIAL_RESOURCE_DESCRIPTOR descriptor =
                    list->PartialResourceList.PartialDescriptors + index2;

                //
                // Store function config connection ids.
                //

                switch (descriptor->Type)
                {
                    case CmResourceTypeConnection:
                    {
                        if ((descriptor->u.Connection.Class ==
                             CM_RESOURCE_CONNECTION_CLASS_FUNCTION_CONFIG) &&
                            (descriptor->u.Connection.Type ==
                             CM_RESOURCE_CONNECTION_TYPE_FUNCTION_CONFIG))
                        {

                            if (pass == 1)
                            {
                                m_FunctionConfigConnectionIds[numberResources].LowPart =
                                    descriptor->u.Connection.IdLowPart;
                                m_FunctionConfigConnectionIds[numberResources].HighPart =
                                    descriptor->u.Connection.IdHighPart;
                            }

                            numberResources++;
                        }

                        break;
                    }
                    default:
                    {
                        break;
                    }
                }
            }

            //
            // Advance to the next CM_FULL_RESOURCE_DESCRIPTOR block in memory.
            //

            list = (PCM_FULL_RESOURCE_DESCRIPTOR)
                (list->PartialResourceList.PartialDescriptors +
                list->PartialResourceList.Count);
        }

        if (numberResources == 0) {
            goto exit;
        }

        if (pass == 0) 
        {
            m_FunctionConfigConnectionCount = numberResources;
        }
    }

exit:

    if (!NT_SUCCESS(status))
    {
        if (m_FunctionConfigConnectionIds != NULL)
        {
            ExFreePoolWithTag(m_FunctionConfigConnectionIds, SCX_POOL_TAG);
            m_FunctionConfigConnectionIds = NULL;
        }

        if (m_FunctionConfigConnectionTarget != NULL)
        {
            ExFreePoolWithTag(m_FunctionConfigConnectionTarget, SCX_POOL_TAG);
            m_FunctionConfigConnectionTarget = NULL;
        }
    }

    return status;
}

NTSTATUS
CScxController::OpenFunctionConfigConnections(
    )
/*++

Routine Description:

    This routine opens the function config connections parsed from device
    start resource list.

Arguments:

    N/A

Return Value:

    NTSTATUS

--*/
{
    ULONG index;
    NTSTATUS status;
    WDFIOTARGET ioTarget = NULL;
    WDF_IO_TARGET_OPEN_PARAMS openParams;
    WDFDEVICE device;

    DECLARE_UNICODE_STRING_SIZE(connectionString, RESOURCE_HUB_PATH_SIZE);

    NT_ASSERTMSG("Unexpected m_ControllerRefCount",
        (m_ControllerRefCount == 1));

    status = STATUS_SUCCESS;
    device = GetFxObject();

    //
    // Function config connections are established in two phases:
    //
    // Phase 0: An I/O target is opened on the connection id. This will cause
    //     the function config resource associated with the connection id to
    //     be reserved by the resource provider. Currently these connection IDs
    //     are mapped to physical pins managed by the GPIO stack(GPIOCLX).
    //
    // Phase 1: IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS is sent to the I/O
    //     target created in phase 0. Currently this IOCTL is fielded by GPIOCLX
    //     and causes the invocation of the GPIO client driver through its 
    //     alternate function configuration interface. In response, the client
    //     driver configures the  actual H/W. Currently this involves GPIOCLX
    //
    //
    // The two-phase approach is used to reduce the failure frequency in the
    // configuration of the actual function config hardware. Phase 0
    // ensures that from a software perspective, the specified function config pins
    // can be configured and reserved. Phase 1 merely commits the configuration to
    // H/W registers.
    //

    for (index = 0; index < m_FunctionConfigConnectionCount; index++)
    {
        //
        // Create function config resource connection string.
        //
     
        status = RESOURCE_HUB_CREATE_PATH_FROM_ID(
            &connectionString,
            m_FunctionConfigConnectionIds[index].LowPart,
            m_FunctionConfigConnectionIds[index].HighPart
            );

        if (!NT_SUCCESS(status)) 
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Failure creating path from function config connection id"
                "%!STATUS!",
                status
                );

            goto exit;
        }

        //
        // Create I/O target for function config connection.
        //

        ioTarget = NULL;
        status = WdfIoTargetCreate(
            device,
            WDF_NO_OBJECT_ATTRIBUTES,
            &ioTarget
            );

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Failure creating io target for function config connection"
                "%!STATUS!",
                status
                );

            goto exit;
        }

        m_FunctionConfigConnectionTarget[index] = ioTarget;

        WDF_IO_TARGET_OPEN_PARAMS_INIT_OPEN_BY_NAME(
            &openParams,
            &connectionString,
            FILE_GENERIC_READ | FILE_GENERIC_WRITE
            );

        //
        // Open I/O target for function config connection.
        //

        status = WdfIoTargetOpen(ioTarget, &openParams);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Failure opening io target for function config connection"
                "%!STATUS!",
                status
                );

            goto exit;
        }
    }

    //
    // Commit phase
    //

    for (index = 0; index < m_FunctionConfigConnectionCount; index++)
    {
        //
        // Send IOCTL to commit function config settings to H/W
        //

        status = WdfIoTargetSendIoctlSynchronously(
            m_FunctionConfigConnectionTarget[index],
            NULL,
            IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS,
            NULL,
            0,
            NULL,
            0
            );

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Failure formatting IOCTL to commit function config pins"
                "%!STATUS!",
                status
                );

            goto exit;
        }
    }

exit:
    //
    // Cleanup on failure.
    //

    if (!NT_SUCCESS(status))
    {
        for (index = 0; index < m_FunctionConfigConnectionCount; index++)
        {
            if (m_FunctionConfigConnectionTarget[index] != NULL)
            {
                WdfObjectDelete(m_FunctionConfigConnectionTarget[index]);
                m_FunctionConfigConnectionTarget[index] = NULL;
            }
        }
    }

    return status;
}

VOID
CScxController::CloseFunctionConfigConnections(
    )
/*++

Routine Description:

    This routine closes the function config connections (if any) established
    in the OpenFunctionConfigConnections routine.

Arguments:

    N/A

Return Value:

    N/A

--*/
{
    ULONG index;

    NT_ASSERTMSG("Unexpected m_ControllerRefCount",
        (m_ControllerRefCount == 0));

    for (index = 0; index < m_FunctionConfigConnectionCount; index++)
    {
        if (m_FunctionConfigConnectionTarget[index] != NULL)
        {
            WdfObjectDelete(m_FunctionConfigConnectionTarget[index]);
            m_FunctionConfigConnectionTarget[index] = NULL;
        }
    }
}

NTSTATUS
CScxController::_OnIrpPreprocess(
    __in    WDFDEVICE  FxDevice,
    __inout PIRP       Irp,
    __in    WDFCONTEXT DispatchContext
    )
/*++

  Routine Description:

    This routine is called to preprocess all power IRPs and some
    PNP IRPs. D0->Dx transitions are not allowed while the controller
    is locked. For Dx IRPs this routine blocks until all outstanding
    locked IO completes and then sets the synchronization event
    in the completion routine to continue lock processing.

    In addition, this routine is called to preprocess IRP_MN_DEVICE_START.
    This is used to locate function config resource descriptors
    used by SpbCx. These descriptors should be ignored by the
    SPB client driver.

    For all IRPs the routine calls WdfDeviceWdmDispatchIrp()
    to continue processing.

  Arguments:

    FxDevice - the WDF device object

    Irp - the IRP

    DispatchContext - the driver supplied context

  Return Value:

    Status from WdfDeviceWdmDispatchIrp().

--*/
{
    PIO_STACK_LOCATION stackLocation = IoGetCurrentIrpStackLocation(Irp);
    PCM_RESOURCE_LIST resourceList;
    PCScxController controller = GetControllerFromFxDevice(FxDevice);
    NTSTATUS status;

    //
    // Log the IRP.
    //

    _WriteIrpPreprocessEvent(FxDevice, Irp);

    //
    // Process Dx IRPs. Ensure that we do not power transition
    // during a locked IO sequence. This implementation is
    // required beyond basic idle detection for reasons (1) and (2).
    //

    if ((controller != NULL) &&
        (controller->IsConstructed() == true) &&
        (controller->DriverPowerManaged() == WdfTrue) &&
        (stackLocation->MajorFunction == IRP_MJ_POWER) &&
        (stackLocation->MinorFunction == IRP_MN_SET_POWER) &&
        (stackLocation->Parameters.Power.Type == DevicePowerState) &&
        (stackLocation->Parameters.Power.State.DeviceState > PowerDeviceD0))
    {
        //
        // Entering Dx, stop processing of lock queue
        // until the IRP completes.
        //

#if DBG
        NTSTATUS waitStatus;

        waitStatus =
#endif
        KeWaitForSingleObject(
            &controller->m_LockedIoSynchronizationEvent,
            Executive,
            KernelMode,
            FALSE,
            0);

        NT_ASSERTMSG("Unexpected status from KeWaitForSingleObject",
                     (waitStatus == STATUS_SUCCESS));

        IoCopyCurrentIrpStackLocationToNext(Irp);

        IoSetCompletionRoutine(
            Irp,
            _OnDxIrpComplete,
            FxDevice,
            TRUE,   // InvokeOnSuccess
            TRUE,   // InvokeOnError
            TRUE);  // InvokeOnCancel
    }
    else if ((controller != NULL) &&
             (controller->IsConstructed() == true) &&
             (stackLocation->MajorFunction == IRP_MJ_PNP) &&
             (stackLocation->MinorFunction == IRP_MN_START_DEVICE))
    {
        //
        // If this is a start device IRP, attempt to parse any function
        // config resource descriptors from the resource list.
        //

        resourceList =
            stackLocation->Parameters.StartDevice.AllocatedResourcesTranslated;

        status = controller->ParseFunctionConfigResources(resourceList);

        if (!NT_SUCCESS(status))
        {
            TraceMessage(
                TRACE_LEVEL_ERROR,
                TRACE_FLAG_TARGET,
                "Failed to parse function config resources"
                "%!STATUS!",
                status
                );

            Irp->IoStatus.Status = status;
            Irp->IoStatus.Information = 0;
            IoCompleteRequest(Irp, IO_NO_INCREMENT);
            goto exit;
        }

        IoSkipCurrentIrpStackLocation(Irp);
    }
    else
    {
        IoSkipCurrentIrpStackLocation(Irp);
    }

    status = WdfDeviceWdmDispatchIrp(FxDevice, Irp, DispatchContext);

exit:
    return status;
}

NTSTATUS
CScxController::_OnDxIrpComplete(
    PDEVICE_OBJECT  /* DeviceObject */,
    PIRP            Irp,
    PVOID           Context
    )
/*++

  Routine Description:

    This routine is called when the Dx IRP completes.
    It sets an event to continue processing of the lock queue.

  Arguments:

    DeviceObject - the underlying device object

    Irp - the IRP

    Context - the completion context, set to be the WDFDEVICE

  Return Value:

    STATUS_CONTINUE_COMPLETION to continue processing
    the IRP completion.

--*/
{
    PCScxController controller = GetControllerFromFxDevice((WDFDEVICE)Context);

    //
    // Continue processing of lock queue.
    //

    KeSetEvent(
        &controller->m_LockedIoSynchronizationEvent,
        IO_NO_INCREMENT,
        FALSE);

    //
    // This optimization avoids grabbing the dispatcher lock,
    // and improves perf.
    //

    if (Irp->PendingReturned == TRUE)
    {
        IoMarkIrpPending(Irp);
    }

    return STATUS_CONTINUE_COMPLETION;
}

NTSTATUS
CScxController::_OnIrpDispatch(
    __in WDFDEVICE  FxDevice,
    __in UCHAR      /* MajorFunction */,
    __in UCHAR      /* MinorFunction */,
    __in ULONG      /* Code */,
    __in WDFCONTEXT /* DriverContext */,
    __inout PIRP    Irp,
    __in WDFCONTEXT DispatchContext
    )
/*++

  Routine Description:

    This routine intercepts I/O operations before they are handed to WDF.
    It determines if they are intended for an SPB target and forwards
    to the appropriate target queue.

  Arguments:

    FxDevice - the device the IRP was sent to

    MajorFunction - the IRP major function

    MinorFunction - the IRP minor function

    Code - the IRP code

    DriverContext - the driver assigned context

    Irp - the IRP

    DispatchContext - the dispatch context, this should be passed on
                      when calling WdfDeviceWdmDispatchIrp

  Return Value:

    Status from one of
      -  WdfDeviceWdmDispatchIrpToIoQueue if class extension will handle IRP
      -  WdfDeviceWdmDispatchIrp if dispatching to controller

--*/
{
    PFILE_OBJECT wdmFile = IoGetCurrentIrpStackLocation(Irp)->FileObject;
    WDFFILEOBJECT fxFile = WdfDeviceGetFileObject(FxDevice, wdmFile);
    PCScxTarget target;
    NTSTATUS status;

    //
    // Check the file the IRP was sent to.  If it's not one of
    // the SPB targets then dispatch to the controller.
    //

    if (fxFile != NULL)
    {
        target = GetTargetFromFxFile(fxFile);

        if ((target != NULL) &&
            (target->IsConstructed() == true))
        {
            //
            // IRP intended for an SPB target, forward to the target queue.
            //

            TraceMessage(
                TRACE_LEVEL_INFORMATION,
                TRACE_FLAG_REQUEST,
                "Forward IRP %p to SPBTARGET %p queue for controller %p",
                Irp,
                target->GetSpbObject(),
                FxDevice
                );

            status = target->ForwardIrpToTargetQueue(Irp);
            goto exit;
        }
    }

    //
    // IRP not intended for SPB target, dispatch to controller.
    //

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "IRP %p not intended for SPB target, handoff to controller %p",
        Irp,
        FxDevice
        );

    status = WdfDeviceWdmDispatchIrp(FxDevice, Irp, DispatchContext);

exit:

    return status;
}

VOID
CScxController::_OnLockedIoWatchdogExpired(
    __in WDFTIMER FxTimer
    )
/*++

  Routine Description:

    This routine is called when the locked IO watchdog timer
    expires. This is likely the result of a misbehaving peripheral.
    If verifier is enabled it issues a verifier break.

  Arguments:

    FxTimer - the WDF timer object

  Return Value:

    none

--*/
{
    WDFDEVICE fxDevice = (WDFDEVICE)WdfTimerGetParentObject(FxTimer);
    PCScxController controller = GetControllerFromFxDevice(fxDevice);

    NT_ASSERTMSG("Locked IO watchdog expired. This likely indicates \
                  misuse of SPB lock/unlock IOCTLs",
                  FALSE);

    TraceMessage(
        TRACE_LEVEL_WARNING,
        TRACE_FLAG_CONTROLLER,
        "Locked IO watchdog expired for target %p, controller %p",
        controller,
        controller->m_ExclusiveOwner
        );

    EventWrite_SPBCX_LOCK_TIMEOUT(
        NULL,
        controller,
        controller->m_ExclusiveOwner->GetConnectionTag(),
        controller->m_ExclusiveOwner->GetScopeTag(),
        controller->m_ExclusiveOwner);

    WdfVerifierDbgBreakPoint();

    //
    // Restart the locked IO watchdog timer.
    //

    WdfTimerStart(
        FxTimer,
        WDF_REL_TIMEOUT_IN_MS(s_LockedIoWatchdogTimeoutMs));
}

VOID
CScxController::_OnIoInCallerContext(
    __in WDFDEVICE  FxDevice,
    __in WDFREQUEST FxRequest
    )
/*++

  Routine Description:

    This routine is called in the I/O path before dispatching to a queue
    and allows SPBCx to capture buffers in the user-mode address space of the
    calling process.

    When invoked for an IOCTL_SPB_PACKET ioctl this routine will create the
    CScxRequest structure for the request and capture the array of transfer
    descriptors.

    When invoked for an "other" IOCTL this routine will invoke the controller
    driver's OnInCallerContext callback (if set).

    Otherwise it will dispatch the request to the top-level queue.

  Arguments:

    FxDevice - the device the request was sent to

    FxRequest - the request

  Return Value:

    none

--*/
{
    PCScxController controller = GetControllerFromFxDevice(FxDevice);
    PCScxTarget target = GetTargetFromFxFile(
                            WdfRequestGetFileObject(FxRequest)
                            );

    PCScxRequest requestContext = GetRequestFromFxRequest(FxRequest);
    WDF_REQUEST_PARAMETERS fxParams;

    enum {
        ActionEnqueue,
        ActionFail,
        ActionHandoff
    } action;

    NTSTATUS status = STATUS_INTERNAL_ERROR;

    //
    // Check and see if the request is an SPB request.  If not
    // then hand off to the controller driver's InCallerContext
    // callback if it set one.
    //

    WDF_REQUEST_PARAMETERS_INIT(&fxParams);

    WdfRequestGetParameters(FxRequest, &fxParams);

    if ((fxParams.Type == WdfRequestTypeDeviceControl) ||
        (fxParams.Type == WdfRequestTypeDeviceControlInternal))
    {
        SpbIoctl controlCode;

        controlCode = (SpbIoctl)
                        fxParams.Parameters.DeviceIoControl.IoControlCode;

        switch(controlCode)
        {
            case IOCTL_SPB_LOCK_CONTROLLER:
            case IOCTL_SPB_UNLOCK_CONTROLLER:
            case IOCTL_SPB_LOCK_CONNECTION:
            case IOCTL_SPB_UNLOCK_CONNECTION:
            {
                action = ActionEnqueue;
                break;
            }

            case IOCTL_SPB_EXECUTE_SEQUENCE:
            {
                NT_ASSERTMSG("Request must not be constructed before in-caller \
                             context",
                             (requestContext->IsConstructed() == false));

                //
                // Create the request.
                //

                status = CScxRequest::_CreateAndInitialize(
                                        target,
                                        FxRequest,
                                        ScxRequestFlagsNone,
                                        &requestContext
                                        );

                if (!NT_SUCCESS(status))
                {
                    TraceMessage(
                        TRACE_LEVEL_ERROR,
                        TRACE_FLAG_TARGET,
                        "Unable to allocate context for request %p to target "
                        "%p - %!STATUS!",
                        FxRequest,
                        target->GetFxObject(),
                        status
                        );
                    action = ActionFail;
                }
                else
                {
                    action = ActionEnqueue;
                }
                break;
            }

            default:
            {
                action = ActionHandoff;
                break;
            }
        }
    }
    else
    {
        action = ActionEnqueue;
    }

    //
    // If the action is to hand-off to the controller only hand-off
    // when the driver provides a callback.
    //

    if (action == ActionHandoff)
    {
        if (controller->m_DriverOtherIoInCallerContextCallback == NULL)
        {
            action = ActionEnqueue;
        }
        else
        {
            controller->m_DriverOtherIoInCallerContextCallback(FxDevice,
                                                               FxRequest);
        }
    }

    if (action == ActionEnqueue)
    {
        status = WdfDeviceEnqueueRequest(FxDevice, FxRequest);

        if (!NT_SUCCESS(status))
        {
            action = ActionFail;
        }
    }

    if (action == ActionFail)
    {
        WdfRequestComplete(FxRequest, status);
    }

    return;
}

NTSTATUS
CScxController::InvokeSpbTargetConnect(
    __in PCScxTarget Target
    )
/*++

  Routine Description:

    This method invokes the controller driver's handler for TargetConnect
    events if one is registered.

  Arguments:

    Target - the target to pass to the controller driver

  Return Value:

    STATUS_SUCCESS if no target connect method is registered.

    Otherwise returns the result returned by the connect method

--*/
{
    NTSTATUS status = STATUS_SUCCESS;

    if (m_DriverConfig.EvtSpbTargetConnect != NULL)
    {
        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_TARGET,
            "Invoking EvtSpbTargetConnect callback for controller %p, "
            "file/target %p (ConnectionTag %S)",
            GetFxObject(),
            Target->GetSpbObject(),
            Target->GetConnectionTag()
            );

        status = m_DriverConfig.EvtSpbTargetConnect(
                    GetFxObject(),
                    Target->GetSpbObject()
                    );

        TraceMessage(
            (NT_SUCCESS(status) ? TRACE_LEVEL_INFORMATION : TRACE_LEVEL_ERROR),
            TRACE_FLAG_TARGET,
            "EvtSpbTargetConnect callback for controller %p, "
            "file/target %p (ConnectionTag %S) returned %!STATUS!",
            GetFxObject(),
            Target->GetSpbObject(),
            Target->GetConnectionTag(),
            status
            );
    }

    return status;
}

VOID
CScxController::InvokeSpbTargetDisconnect(
    __in PCScxTarget Target
    )
/*++

  Routine Description:

    This method invokes the controller driver's handler for TargetDisconnect
    events if one is registered.

  Arguments:

    Target - the target to pass to the controller driver

  Return Value:

    None

--*/
{
    if (m_DriverConfig.EvtSpbTargetDisconnect != NULL)
    {
        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_TARGET,
            "Invoking EvtSpbTargetDisconnect callback for controller %p, "
            "file/target %p (ConnectionTag %S)",
            GetFxObject(),
            Target->GetSpbObject(),
            Target->GetConnectionTag()
            );

        m_DriverConfig.EvtSpbTargetDisconnect(
            GetFxObject(),
            Target->GetSpbObject()
            );

        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_TARGET,
            "EvtSpbTargetDisconnect callback for controller %p, "
            "file/target %p (ConnectionTag %S) returned",
            GetFxObject(),
            Target->GetSpbObject(),
            Target->GetConnectionTag()
            );
    }
}

VOID
CScxController::_OnControllerIoDefault(
    __in WDFQUEUE   /* FxQueue */,
    __in WDFREQUEST FxRequest
    )
/*++

  Routine Description:

    This routine receives I/O operations which have passed down through the
    controller queue and which are ready to be sent to the user-mode driver.

  Arguments:

    FxQueue - the controller queue

    FxRequest - the request which is ready to be sent to the i/o controller
                driver

  Return Value:

    None

--*/
{
    PCScxRequest request = GetRequestFromFxRequest(FxRequest);

    request->GetController()->ProcessRequest(request);
}


VOID
CScxController::_OnIoTargetRemoveComplete(
    __in  WDFIOTARGET IoTarget
    )
/*++

  Routine Description:

    This routine is called when when the removal of a specified remote I/O target
    is complete. This

  Arguments:

    IoTarget - A handle to the Resource Hub I/O target object.

  Return Value:

    None

--*/
{
    if (IoTarget != NULL)
    {
        WdfIoTargetClose(IoTarget);
        WdfObjectDelete(IoTarget);
        IoTarget = NULL;
    }
}

VOID
CScxController::ProcessRequest(
    __in PCScxRequest Request
    )
/*++

  Routine Description:

    This is the dispatch routine for the controller level.  This receives
    requests which come through the controller queue, as well as requests
    recieved directly from the target queue from a target with exclusive
    ownership.

  Arguments:

    Request - the request

  Return Value:

    none

--*/
{
    SPB_REQUEST_TYPE spbIoType = Request->GetSpbIoType();
    PCScxTarget target = Request->GetTarget();
    PCScxController controller = Request->GetController();

    SPBREQUEST spbRequest = Request->GetSpbObject();

    NTSTATUS status;

    EventWrite_SPBCX_IO_DISPATCH_TO_CONTROLLER(Request->GetActivityId());

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "SPBREQUEST %p to SPBTARGET %p dispatched from controller %p queue",
        spbRequest,
        target->GetSpbObject(),
        GetFxObject()
        );

    //
    // If this is a lock request then forward it to the lock queue.
    //

    if (spbIoType == SpbRequestTypeLockController)
    {
        if (target->IsExclusiveConnectionOwner())
        {
            //
            // Continue processing the lock if target is already the
            // exclusive connection owner. The controller queue has
            // already been stopped and we are dispatching from the
            // target queue directly.
            //

            controller->CompleteLock(Request);
        }
        else
        {
            //
            // Otherwise forward the lock controller request to the lock queue.
            //

            status = WdfRequestForwardToIoQueue(
                        spbRequest,
                        m_LockQueue
                        );

            if (!NT_SUCCESS(status))
            {
                NT_ASSERTMSG("Lock queue should always accept requests",
                            NT_SUCCESS(status));
            }
        }
    }
    else if (spbIoType == SpbRequestTypeLockConnection)
    {
        //
        // A lock connection request will never be preceded by
        // another lock request. Just forward the lock request to
        // the lock queue.
        //
        // NOTE: For now connection locking is treated the same
        //       as controller locking in terms of exclusivity.
        //       A great deal of logic is shared between the two
        //       lock paths. Relevant code will need to be reevaluated
        //       if more fine-grained connection locking is implemented
        //       in the future.
        //

        status = WdfRequestForwardToIoQueue(
                    spbRequest,
                    m_LockQueue
                    );

        if (!NT_SUCCESS(status))
        {
            NT_ASSERTMSG("Lock queue should always accept requests",
                        NT_SUCCESS(status));
        }
    }
    else
    {
        //
        // Invoke the driver's I/O callback
        //

        PSCX_CONTROLLER_IO_CALLBACK invokeCallback;

        EventWrite_SPBCX_IO_TO_DRIVER(Request->GetActivityId());

        invokeCallback = m_InvokeIoCallbacks[spbIoType];

        (this->*invokeCallback)(Request);
    }
}

VOID
CScxController::_OnLockQueueReady(
    __in WDFQUEUE   /* FxQueue */,
    __in WDFCONTEXT Context
    )
/*++

  Routine Description:

    This routine receives lock operations which have passed down through the
    controller queue and the lock queue.  In response the controller queue
    is stopped.

  Arguments:

    FxQueue - the controller queue

    FxRequest - the lock request.

  Return Value:

    None

--*/
{
    PCScxController controller = (PCScxController) Context;

    WDFREQUEST fxRequest;
    PCScxRequest request;
    LARGE_INTEGER timeoutNoWait;

    NTSTATUS status;

    //
    // Check the locked IO sync event.
    //

    timeoutNoWait.QuadPart = 0;

    status = KeWaitForSingleObject(
        &controller->m_LockedIoSynchronizationEvent,
        Executive,
        KernelMode,
        FALSE,
        &timeoutNoWait);

    if (status == STATUS_SUCCESS)
    {
        //
        // Retrieve the lock request from the queue.
        //

        status = WdfIoQueueRetrieveNextRequest(controller->m_LockQueue,
                                               &fxRequest);

        if (NT_SUCCESS(status))
        {
            request = GetRequestFromFxRequest(fxRequest);

            NT_ASSERTMSG("OnLockIoDefault should only be used for lock requests",
                         ((request->GetSpbIoType() == SpbRequestTypeLockController) ||
                          (request->GetSpbIoType() == SpbRequestTypeLockConnection)));

            //
            // Stop the controller queue.
            //

            WdfIoQueueStop(
                controller->m_ControllerQueue,
                _OnLockStopComplete,
                request
                );
        }
        else
        {
            NT_ASSERTMSG("Unexpected status from WdfIoQueueRetreiveNextRequest for \
                         lock queue",
                         (status == STATUS_NO_MORE_ENTRIES));

            //
            // We aren't processing a lock, set the locked IO sync event
            //

            KeSetEvent(
                &controller->m_LockedIoSynchronizationEvent,
                IO_NO_INCREMENT,
                FALSE);
        }
    }
    else
    {
        NT_ASSERTMSG("Unexpected status from KeWaitForSingleObject",
                     (status == STATUS_TIMEOUT));

        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_CONTROLLER,
            "Lock queue %p is currently blocked, request processing deferred",
            controller->m_LockQueue
            );
    }

    return;
}

VOID
CScxController::_OnLockStopComplete(
    __in WDFQUEUE   /* FxQueue */,
    __in WDFCONTEXT Context
    )
/*++

  Routine Description:

    This method is invoked when the controller queue stops as a result of an
    active lock operation.  The lock process is continued at this point.

  Arguments:

    FxQueue - the controller queue

    Context - the lock request

  Return Value:

    none

--*/
{
    PCScxRequest request = (PCScxRequest) Context;
    PCScxController controller = request->GetController();

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "WDFDEVICE %p controller queue has been stopped for lock operation",
        controller->GetFxObject()
        );

    controller->CompleteLock(request);
}

VOID
CScxController::CompleteLock(
    __in PCScxRequest LockRequest
    )
/*++

  Routine Description:

    This method is invoked to complete the lock request.  It invokes the
    controller driver's lock callback as appropriate.

  Arguments:

    LockRequest - the lock request which the controller driver should complete

  Return Value:

    None

--*/
{
    PCScxTarget target = LockRequest->GetTarget();
    SPB_REQUEST_TYPE spbIoType = LockRequest->GetSpbIoType();
    PSCX_CONTROLLER_IO_CALLBACK invokeCallback;

    if (spbIoType == SpbRequestTypeLockController)
    {
        //
        // If the controller driver is the PPO then stop idle detection so as
        // to force the device to remain in D0.
        //
        // Stop idle at the controller level before forwarding to the lock
        // queue.  The reference will be released when we receive an unlock
        // (one unlock per lock)
        //

        if (DriverPowerManaged() == WdfTrue)
        {
            TraceMessage(
                TRACE_LEVEL_INFORMATION,
                TRACE_FLAG_CONTROLLER,
                "Disabling idle detection on controller %p due to lock "
                "request for target %p",
                GetFxObject(),
                target->GetSpbObject()
                );

            NTSTATUS status = WdfDeviceStopIdle(GetFxObject(), false);

            if (!NT_SUCCESS(status))
            {
                NT_ASSERTMSG("Controller is powered on now - stopping IDLE \
                                detection should not fail",
                                NT_SUCCESS(status));
            }
        }

        //
        // Mark which target has exclusive ownership of the controller.
        //

        target->SetIsExclusiveControllerOwner(true);
    }
    else if (spbIoType == SpbRequestTypeLockConnection)
    {
        //
        // Mark which target has exclusive ownership of the connection.
        //

        target->SetIsExclusiveConnectionOwner(true);
    }
    else
    {
        NT_ASSERTMSG("CompleteLock should only be used for lock requests",
                     ((spbIoType == SpbRequestTypeLockController) ||
                      (spbIoType == SpbRequestTypeLockConnection)));
    }

    SetExclusiveOwner(target);

    //
    // Start the locked IO watchdog timer.
    //

    WdfTimerStart(
        m_LockedIoWatchdog,
        WDF_REL_TIMEOUT_IN_MS(s_LockedIoWatchdogTimeoutMs));

    //
    // Call the controller driver to let it know of the lock operation.
    //

    invokeCallback = m_InvokeIoCallbacks[spbIoType];

    (this->*invokeCallback)(LockRequest);

}

VOID
CScxController::InvokeUnsupportedIo(
    __in PCScxRequest Request
    )
{
    TraceMessage(
        TRACE_LEVEL_ERROR,
        TRACE_FLAG_CONTROLLER,
        "SPBREQUEST %p is an unsupported I/O type.  Completing with "
        "status %!STATUS!",
        Request->GetSpbObject(),
        STATUS_NOT_SUPPORTED
        );
    WdfRequestComplete(Request->GetFxObject(), STATUS_NOT_SUPPORTED);
}

VOID
CScxController::InvokeSpbIoRead(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();
    size_t     length = Request->GetTotalCb();

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Invoking EvtSpbIoRead callback for controller %p, "
        "file/target %p (ConnectionTag %S) request %p",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest
        );

    m_DriverConfig.EvtSpbIoRead(fxController,
                                fxTarget,
                                fxRequest,
                                length);

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "EvtSpbIoRead callback for controller %p, target %p req %p returned",
        fxController,
        fxTarget,
        fxRequest
        );
}

VOID
CScxController::InvokeSpbIoWrite(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();
    size_t     length = Request->GetTotalCb();

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Invoking EvtSpbIoWrite callback for controller %p, "
        "file/target %p (ConnectionTag %S) request %p",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest
        );

    m_DriverConfig.EvtSpbIoWrite(fxController,
                                 fxTarget,
                                 fxRequest,
                                 length);

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "EvtSpbIoWrite callback for controller %p, target %p req %p returned",
        fxController,
        fxTarget,
        fxRequest
        );
}

VOID
CScxController::InvokeSpbIoSequence(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Invoking EvtSpbIoSequence callback for controller %p, "
        "file/target %p (ConnectionTag %S) request %p",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest
        );

    m_DriverConfig.EvtSpbIoSequence(fxController,
                                    fxTarget,
                                    fxRequest,
                                    Request->GetTransferCount());

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "EvtSpbIoSequence callback for controller %p, target %p req %p "
        "returned",
        fxController,
        fxTarget,
        fxRequest
        );
}

VOID
CScxController::InvokeSpbIoOther(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    WDF_REQUEST_PARAMETERS fxParams;

    WDF_REQUEST_PARAMETERS_INIT(&fxParams);

    WdfRequestGetParameters(fxRequest, &fxParams);

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Invoking EvtSpbIoOther callback for controller %p, "
        "file/target %p (ConnectionTag %S) request %p (control code 0x%08x)",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest,
        fxParams.Parameters.DeviceIoControl.IoControlCode
        );

    m_DriverOtherIoCallback(
        fxController,
        fxTarget,
        fxRequest,
        fxParams.Parameters.DeviceIoControl.OutputBufferLength,
        fxParams.Parameters.DeviceIoControl.InputBufferLength,
        fxParams.Parameters.DeviceIoControl.IoControlCode
        );

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "EvtSpbIoOther callback for controller %p, target %p req %p returned",
        fxController,
        fxTarget,
        fxRequest
        );
}

VOID
CScxController::InvokeSpbControllerLock(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    if (m_DriverConfig.EvtSpbControllerLock != NULL)
    {
        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_CONTROLLER,
            "Invoking EvtSpbControllerLock callback for controller %p, "
            "file/target %p (ConnectionTag %S) request %p",
            fxController,
            fxTarget,
            Request->GetTarget()->GetConnectionTag(),
            fxRequest
            );

        m_DriverConfig.EvtSpbControllerLock(fxController,
                                            fxTarget,
                                            fxRequest);

        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_CONTROLLER,
            "EvtSpbControllerLock callback for controller %p, target %p req %p "
            "returned",
            fxController,
            fxTarget,
            fxRequest
            );
    }
    else
    {
        TraceMessage(
            TRACE_LEVEL_INFORMATION,
            TRACE_FLAG_CONTROLLER,
            "No EvtSpbControllerLock callback provided for controller %p, "
            "file/target %p (ConnectionTag %S) request %p. Completing with "
            "status %!STATUS!",
            fxController,
            fxTarget,
            Request->GetTarget()->GetConnectionTag(),
            fxRequest,
            STATUS_SUCCESS
            );

        WdfRequestComplete(fxRequest, STATUS_SUCCESS);
    }
}

VOID
CScxController::InvokeSpbControllerUnlock(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Invoking EvtSpbControllerUnlock callback for controller %p, "
        "file/target %p (ConnectionTag %S) request %p",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest
        );

    m_DriverConfig.EvtSpbControllerUnlock(fxController,
                                          fxTarget,
                                          fxRequest);

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "EvtSpbControllerUnlock callback for controller %p, target %p req %p "
        "returned",
        fxController,
        fxTarget,
        fxRequest
        );
}

VOID
CScxController::InvokeSpbConnectionLock(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    //
    // There is no EvtSpbConnectionLock callback.
    // Just complete the request.
    //

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Done with connection lock for controller %p, "
        "file/target %p (ConnectionTag %S) request %p. Completing with "
        "status %!STATUS!",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest,
        STATUS_SUCCESS
        );

    WdfRequestComplete(fxRequest, STATUS_SUCCESS);
}

VOID
CScxController::InvokeSpbConnectionUnlock(
    __in PCScxRequest Request
    )
{
    WDFDEVICE  fxController = this->GetFxObject();
    SPBTARGET  fxTarget = Request->GetTarget()->GetSpbObject();
    SPBREQUEST fxRequest = Request->GetSpbObject();

    //
    // There is no EvtSpbConnectionUnlock callback.
    // Just complete the request.
    //

    TraceMessage(
        TRACE_LEVEL_INFORMATION,
        TRACE_FLAG_CONTROLLER,
        "Done with connection unlock for controller %p, "
        "file/target %p (ConnectionTag %S) request %p. Completing with "
        "status %!STATUS!",
        fxController,
        fxTarget,
        Request->GetTarget()->GetConnectionTag(),
        fxRequest,
        STATUS_SUCCESS
        );

    WdfRequestComplete(fxRequest, STATUS_SUCCESS);
}

VOID
CScxController::ForwardRequestToControllerQueue(
    __in PCScxRequest Request
    )
/*++

  Routine Description:

    This routine forwards an I/O request to the controller queue

  Arguments:

    Request - the request to forward

  Return Value:

    none

--*/
{
    EventWrite_SPBCX_IO_FORWARD_TO_CONTROLLER(Request->GetActivityId());

#if DBG
    NTSTATUS status;

    status =
#endif
#pragma prefast(suppress:__WARNING_RETVAL_IGNORED_FUNC_COULD_FAIL, "controller queue should never be stopped, so this should never fail");
    WdfRequestForwardToIoQueue(Request->GetFxObject(), m_ControllerQueue);

    NT_ASSERTMSG("Controller queue should never be stopped, only paused",
                 NT_SUCCESS(status));
}

VOID
CScxController::BeginUnlock(
    __in PCScxRequest UnlockRequest
    )
/*++

  Routine Description:

    This method is invoked to unlock the controller or connection. It
    arranges for post processing of the unlock request on completion, then
    invokes the appropriate controller driver callback.

  Arguments:

    Request - the unlock request which the controller driver should complete

  Return Value:

    None

--*/
{
    SPB_REQUEST_TYPE spbIoType = UnlockRequest->GetSpbIoType();
    PSCX_CONTROLLER_IO_CALLBACK invokeCallback;

    //
    // Finish processing when this unlock completes.
    //

    UnlockRequest->SetPostProcessUnlockOnCompletion(true);

    //
    // Call the controller driver to let it know of the unlock operation.
    //

    invokeCallback = m_InvokeIoCallbacks[spbIoType];

    (this->*invokeCallback)(UnlockRequest);
}

VOID
CScxController::EndUnlock(
    VOID
    )
/*++

  Routine Description:

    This routine ends either a successful unlock or a failed lock command.  It
    restarts the controller queue (allowing the next I/O operation to run).

  Arguments:

    None

  Return Value:

    None

--*/
{
    WDFREQUEST lockFxRequest;
    NTSTATUS status;

    //
    // Pull the next request off the lock queue.
    //

    status = WdfIoQueueRetrieveNextRequest(m_LockQueue, &lockFxRequest);

    //
    // If there is a new lock request update the target which owns the
    // controller and inform the driver of the new owner.
    //
    // Otherwise inform the controller that it has been unlocked and return.
    //

    if (NT_SUCCESS(status))
    {
        PCScxRequest lockRequest = GetRequestFromFxRequest(lockFxRequest);
        PCScxController controller = lockRequest->GetController();

        //
        // Continue processing the lock request.
        //

        controller->CompleteLock(lockRequest);
    }
    else
    {
        NT_ASSERTMSG("Unexpected status from WdfIoQueueRetreiveNextRequest for \
                        lock queue",
                        (status == STATUS_NO_MORE_ENTRIES));

        //
        // Clear the exclusive owner.
        //

        SetExclusiveOwner(NULL);

        //
        // Stop the locked IO watchdog timer.
        //

        WdfTimerStop(m_LockedIoWatchdog, FALSE);

        //
        // Set the locked IO sync event.
        //

        KeSetEvent(
            &m_LockedIoSynchronizationEvent,
            IO_NO_INCREMENT,
            FALSE);

        //
        // Restart the controller queue.
        //

        WdfIoQueueStart(m_ControllerQueue);
    }

    return;
}

VOID
CScxController::_WriteIrpPreprocessEvent(
    __in WDFDEVICE FxDevice,
    __in PIRP Irp
    )
{
    PIO_STACK_LOCATION stackLocation = IoGetCurrentIrpStackLocation(Irp);
    GUID activityId;

    //
    // Log D-IRPs.
    //

    if ((stackLocation->MajorFunction == IRP_MJ_POWER) &&
        (stackLocation->MinorFunction == IRP_MN_SET_POWER) &&
        (stackLocation->Parameters.Power.Type == DevicePowerState))
    {
        if (MCGEN_ENABLE_CHECK(SPB_PROVIDER_Context, EVENT_SPBCX_DIRP_PREPROCESS))
        {
            //
            // Get the activity ID.
            //

            IoGetActivityIdIrp(Irp, &activityId);

            //
            // It's ok if the activity ID is NULL. Log the
            // D-IRP anyway.
            //

            EventWrite_SPBCX_DIRP_PREPROCESS(
                &activityId,
                FxDevice,
                stackLocation->Parameters.Power.State.DeviceState);
        }
    }
}
