//
// interaction.h
//
//---------------------------------------------------------------------------------------------------------------------------------
//

//---------------------------------------------------------------------------------------------------------------------------------
// Most of the test cases that emit data from one pipeline stage and consume it elsewhere follow the same pattern: A canonical
// pattern of data is written by the emitting stage, consumed by the receiver stage and written out (perhaps through the gfx 
// pipeline via the PS) to a volume texture of a well-defined size and format.
// The R, G, B and A components of all elements within any resource always contain the x,y,z and array-index values of the texel 
// coordinates of each element. For resources of less dimension than 3, the unused components are kept at zero. 
// There are general "write" and "read" routines whose shaders take constant arguments about the size over which to iterate
// in each dimension. Thus, for example, CSUAVWrite can take (n,1,1,1), meaning only fill in data appropriate for a one-dimensional
// resource of width n.
// The RGBA values have a further encoding that denotes mip level. The value written to each component is d / ( 1 ^ (mip+1) ).
// 12345.f is used as an "initial" value, in order to detect that a particular texel was never written (except by CPU init).
// This structure allows a general "validation" routine to exist that can validate the output for any tested resource dimension.
//
// The flow of the test is thus to
// -create the resource under test, fill it with initial values generated by the CPU
// -write to the resource
// -read from the resource, and push the data out to the volume texture bound to the pixel shader.
// -read the PS output back to the CPU and validate each pixel.
//
// The last step, pushing the data out, is accomplished by drawing a point list. The VS generates positions for each point based
// on vertex ID. The GS iterates over the depth values, using render target index to drive the pixel shader to write to various
// slices of the output volume. 
// 
// The test performs the last two steps once per array index.


//Include shader code to get struct definitions
#define NOHLSL
#include "interaction.sh"
#undef NOHLSL

class CInteractionTest : public CD3D11Test
{
public:
    CInteractionTest() : m_OutputBufferInitialValue( 123 ) {}

public: //CD3D11Test overrides
    TEST_RESULT     Setup();
    void            Cleanup();
    void            InitTestParameters();
    TEST_RESULT     ExecuteTestCase();

protected:
    //Helpers:
    //void            CompileShader(LPCSTR pResource, LPCSTR pFunctionName, LPCSTR pProfile, UINT uFlags, ID3D10Blob **ppShader); //throw TRException
    void            UpdateResourceDimensionCB(); //throw TRException
    void            InitializeOutputVolume();

    //These Create* functions setup resource dimensions for later *Read/*Write functions
    void            CreateBuffer( UINT Bind, bool bStructured = false, bool bRaw = false ); //throw TRException
    void            CreateTexture1D( UINT Bind, UINT ArraySize ); //throw TRException
    void            CreateTexture2D( UINT Bind, UINT ArraySize ); //throw TRException
    void            CreateTexture3D( UINT Bind ); //throw TRException
    
    //Read/Write functions 
    void            CPUWrite( ID3D11Resource * pDefault, ID3D11Resource * pStaging );//throw TRException
    void            CSUAVWrite( ); //throw TRException
    void            PSUAVWrite( ); //throw TRException
    void            VSUAVWrite( ) { assert(false); } //throw TRException
    void            SOWrite( ) { assert(false); } //throw TRException
    void            PSRTWrite( ); //throw TRException
    void            VSSRVRead( ); //throw TRException
    void            VSUAVRead( ) { assert(false); } //throw TRException
    void            PSSRVRead( ); //throw TRException
    void            PSUAVRead( ) { assert(false); } //throw TRException
    void            CSUAVRead( ); //throw TRException
    void            IAVBRead( ); //throw TRException

    void            Validate( ); //throw TRException

protected:

    //All these resources are created (by one of the Create* methods above) and destroyed once per test case
    WindowsTest::CHandle< ID3D11Buffer >                 m_pStagingBuffer;
    WindowsTest::CHandle< ID3D11Buffer >                 m_pTestBuffer;
    WindowsTest::CHandle< ID3D11Texture1D >              m_pTest1D;
    WindowsTest::CHandle< ID3D11Texture1D >              m_pStaging1D;
    WindowsTest::CHandle< ID3D11Texture2D >              m_pTest2D;
    WindowsTest::CHandle< ID3D11Texture2D >              m_pStaging2D;
    WindowsTest::CHandle< ID3D11Texture3D >              m_pTest3D;
    WindowsTest::CHandle< ID3D11Texture3D >              m_pStaging3D;
    WindowsTest::CHandle< ID3D11UnorderedAccessView >    m_pTestWriteUAV;
    WindowsTest::CHandle< ID3D11UnorderedAccessView >    m_pTestReadUAV;
    WindowsTest::CHandle< ID3D11ShaderResourceView >     m_pTestSRV;
    WindowsTest::CHandle< ID3D11RenderTargetView >       m_pTestRTV;

    //These resources are created at Setup
    WindowsTest::CHandle< ID3D11Texture3D >              m_pDefaultOutputResource;
    WindowsTest::CHandle< ID3D11Texture3D >              m_pStagingOutputResource;
    WindowsTest::CHandle< ID3D11Texture3D >              m_pStagingOutputResourceWithInitialValues;
    WindowsTest::CHandle< ID3D11UnorderedAccessView >    m_pOutputResourceUAV;
    WindowsTest::CHandle< ID3D11RenderTargetView >       m_pOutputResourceRTV;
    WindowsTest::CHandle< ID3D11Buffer >                 m_pResourceDimensionConstants;
    WindowsTest::CHandle< ID3D11Buffer >                 m_pDummyDataBuffer;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVS;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRV1DSample;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRV1DSampleArray;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRV2DSample;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRV2DSampleArray;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRV3DSample;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSRVBufferSample;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSampleStructured;
    WindowsTest::CHandle< ID3D11VertexShader >           m_pVSSampleRawSRV;
    WindowsTest::CHandle< ID3D11GeometryShader >         m_pGS;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSPassthrough;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRVBufferSample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRV1DSample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRV2DSample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRV1DArraySample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRV2DArraySample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSRV3DSample;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAVStructuredBuffer;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAVTypedBuffer;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAVRawBuffer;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAV1D;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAV1DArray;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAV2D;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAV2DArray;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSUAV3D;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSampleStructured;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSSampleRawSRV;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSWriteRT;
    WindowsTest::CHandle< ID3D11PixelShader >            m_pPSWriteRTArray;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUAVStructuredBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUAVTypedBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavRawBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavTexture1D;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavTexture1DArray;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavTexture2D;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavTexture2DArray;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavTexture3D;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadRawBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTypedBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadStructuredBuffer;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTexture1D;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTexture1DArray;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTexture2D;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTexture2DArray;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pCSUavReadTexture3D;
    struct PreComputedElementCoordinates
    {
        D3DXVECTOR3 PointSamplePosition;
        D3DXVECTOR3 TargetPointPosition;
        D3DXVECTOR3 RawPointCoordinates;
    };
    WindowsTest::CHandle< ID3D11Buffer >                 m_pPointSampleLocations;
    WindowsTest::CHandle< ID3D11InputLayout >            m_InputLayout;

    //These are set by Create*. 
    bool                        m_bStructured;  //Only CreateBuffer sets it to true
    bool                        m_bRaw;         //Only CreateBuffer sets it to true
    bool                        m_ValidateInTestedBuffer; //Only set to true for CS_4_0 read (raw and structured) cases 

    //These dimensions are set by the Create* functions above"
    UINT                        m_ResourceWidth;
    UINT                        m_ResourceHeight;
    UINT                        m_ResourceDepth;
    UINT                        m_ResourceArraySize;
    UINT                        m_CurrentArraySlice;

    //Constants that control tested resource size
    static const UINT           m_MaxWidth     = 4;
    static const UINT           m_MaxHeight    = 3;
    static const UINT           m_MaxDepth     = 2;
    static const UINT           m_MaxArraySize = 2;
    static const UINT           m_SizeMultiplier = 2;

    const BYTE                  m_OutputBufferInitialValue;

    static const DXGI_FORMAT    m_BufferTypelessFormat          = DXGI_FORMAT_R8G8B8A8_TYPELESS;
    static const DXGI_FORMAT    m_BufferTypedFormat             = DXGI_FORMAT_R32_UINT;
    static const DXGI_FORMAT    m_BufferSRVFormat               = DXGI_FORMAT_R8G8B8A8_UNORM;
    static const UINT           m_BufferElementSize             = sizeof(BYTE) * 4;
    static const UINT           m_BufferUnusedLeadingElements   = 0;//4;
public: //factors
    typedef enum { TypedUAVWrite, RawUAVWrite, StructUAVWrite, StreamOutWrite, RTWrite } e_WriteMethod;
    typedef enum { TypedUAVRead,  RawUAVRead,  StructUAVRead, TypedSRVRead, RawSRVRead, StructSRVRead, VBRead, IBRead } e_ReadMethod;
    typedef enum { StructuredBuffer, RawBuffer, TypedBuffer, Texture1D, Texture1DArray, Texture2D, Texture2DArray, Texture3D } e_ResourceType;
    typedef enum { InputAssembler, VertexShader, ComputeShader, PixelShader, StreamOutStage } e_Stage;

    e_WriteMethod               m_WriteMethod;
    e_ReadMethod                m_ReadMethod;
    e_ResourceType              m_InteractionResourceType;
    e_Stage                     m_WriteStage;
    e_Stage                     m_ReadStage;
    bool                        m_bRequires_5_0;
};

tstring __stdcall ToString( CInteractionTest::e_WriteMethod value );
bool __stdcall FromString( CInteractionTest::e_WriteMethod *value, const tstring &str );

tstring __stdcall ToString( CInteractionTest::e_ReadMethod value );
bool __stdcall FromString( CInteractionTest::e_ReadMethod *value, const tstring &str );

tstring __stdcall ToString( CInteractionTest::e_ResourceType value );
bool __stdcall FromString( CInteractionTest::e_ResourceType *value, const tstring &str );

tstring __stdcall ToString( CInteractionTest::e_Stage value );
bool __stdcall FromString( CInteractionTest::e_Stage *value, const tstring &str );



//---------------------------------------------------------------------------------------------------------------------------------
// This tests implements the classic "game of life" cellular automata dealio. The CS and the CPU run identical frames, and the
// frames are compared over 100 iterations for differences. 
// On the GPU, the source buffer is the swapchain's buffer 1, and the dest buffer is the swap chain's buffer 0, thereby testing
// correct rotation of UAVs under presentation.

class CInteractionPresentTest : public CD3D11Test
{
public: //CD3D11Test overrides
    TEST_RESULT     Setup();
    void            Cleanup();
    void            InitTestParameters();
    TEST_RESULT     ExecuteTestCase();

    int GetNumRenderWindows() const 
    { 
        return 0; 
    }
    HRESULT CreateBuffers(WindowsTest::CHandle< ID3D11Texture2D > &pBB0,
                          WindowsTest::CHandle< ID3D11Texture2D > &pBB1);
    HRESULT Present();
protected:
    WindowsTest::CHandle< ID3D11UnorderedAccessView >    m_pTestWriteUAV;
    WindowsTest::CHandle< ID3D11ShaderResourceView >     m_pTestReadSRV;
    WindowsTest::CHandle< ID3D11Texture2D >              m_pStaging2D0;
    WindowsTest::CHandle< ID3D11Texture2D >              m_pStaging2D1;
    WindowsTest::CHandle< ID3D11Texture2D >              m_pStagingCompare;
    WindowsTest::CHandle< ID3D11ComputeShader >          m_pComputeLife;
};


