/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    intdebounce.c

Abstract:

    This file implements the enhanced software debouncing algorithm for
    Windows 8.1 and beyond. At the high-level, the approach works as follows:
        <When the interrupt fires>
            1. Mask the interrupt line
            2. Schedule a timer for the debounce interval

        <When the debounce timer fires>
            3. Reprogram the line for the opposite polarity (will internally
                clear pending status)

            4. Schedule a high resolution timer for noise filtering interval
                (500 usec)

         <when the noise filtering timer fires>
            5. If no interrupt fired since the debounce timer fired, then
                invoke the ISR. Otherwise, ignore the interrupt.

    Although the 8.1 implementation will be preferred, Windows could run
    on platforms (esp. built for Windows 8) that may not support the
    requirements for enhanced software debouncing. In those cases, the
    debouncing logic implemented in Windows 8 will be used as a fallback.


    Assumptions:

       Interrupts that have been requested to be masked are not presented to
       to the debounce engine until the line is unmasked again. For
       primary controllers and memory-mapped GPIO controllers, a mask request
       is dispatched synchronously and cannot fail, and thus always TRUE.
       For off-SoC GPIO controllers, the mask request may be processed
       asynchronously and could fail. Thus interrupts could still fire for
       such pins. However, GPIO class extension already handles both these
       cases and would not present the interrupt to the debounce engine.
       On the same lines, this module assumes that unmask requests cannot fail.
       If they fail, it is the responsibility of the GPIO class extension to
       try again.

    The main control flow of this module is as follows:

      {GpiopDebounceOnInterrupt, GpiopDebounceOnDebounceTimer, GpiopDebounceOnNoiseTimer}
        -> GpiopDebounceInterruptEngine - loop of:
           -> GpiopDebounceStateMachineTransition
           -> Callbacks into the interrupt.c module


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"
#include "privdefs.h"

#if defined(EVENT_TRACING)
#include "intdebounce.tmh"         // auto-generated by WPP
#endif

//
// -------------------------------------------------------------------- Defines
//

//
// Macro to save previous state for diagnosis.
//

#define GPIO_DEBOUNCE_SAVE_PREVIOUS_STATE(Context, Event)                    \
    (Context)->PreviousState[(Context)->StateIndex] = (Context)->State;      \
    (Context)->PreviousEvent[(Context)->StateIndex] = (Event);               \
    (Context)->StateIndex = ((Context)->StateIndex + 1) % GPIO_DEBOUNCE_HISTORY_SIZE;

//
// Macro to deal with an invalid state transition. Assumes Status and
// ReturnStatus locals are defined.
//

#define GPIO_DEBOUNCE_INVALID_STATE_TRANSITION(Context)                     \
                                                                            \
    GPIO_ASSERT(FALSE);                                                     \
                                                                            \
    ReturnStatus = STATUS_UNSUCCESSFUL;                                     \
    Status = STATUS_UNSUCCESSFUL;                                           \
    if ((Context)->State != DebounceStateInvalid) {                         \
        (Context)->State = DebounceStateOnInitialInterrupt;                 \
    }                                                                       \

//
// -------------------------------------------------------------------- Globals
//

//
// Define a global override flag. This allows various debouncing models to be
// forced easily for testing.
//

GPIO_DEBOUNCE_MODEL GpioGlobalDebounceModelOverride = DebounceModelNone;


//
// ----------------------------------------------------------------- Prototypes
//

_IRQL_requires_min_(DISPATCH_LEVEL)
NTSTATUS
GpiopDebounceStateMachineTransition (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in GPIO_DEBOUNCE_EVENT_TYPE PrimaryEventType,
    __in PGPIO_DEBOUNCE_INPUT_EVENT InputEvent,
    __out PGPIO_DEBOUNCE_OUTPUT_ACTION OutputAction
    );

__drv_when((DebounceContext->Flags.MemoryMapped == TRUE),
    _IRQL_requires_min_(DISPATCH_LEVEL))
NTSTATUS
GpiopDebounceInterruptEngine (
    __inout PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in PVOID ControllerContext,
    __in GPIO_DEBOUNCE_EVENT_TYPE PrimaryEventType,
    __in_opt PGPIO_DEBOUNCE_OUTPUT_ACTION AssociatedOutputAction,
    __out PBOOLEAN DebounceStateMachineCompleted
    );

_IRQL_requires_min_(DISPATCH_LEVEL)
BOOLEAN
GpiopDebounceIsPinAtCompletedState (
     __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext
     );

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(PAGE, GpiopDebounceInitializeContext)
#pragma alloc_text(PAGE, GpiopDebounceDestroyContext)

//
// --------------------------------------------------------- Exported routines
//

_IRQL_requires_(PASSIVE_LEVEL)
VOID
GpiopDebounceInitializeContext (
    __in BOOLEAN MemoryMappedController,
    __in BOOLEAN EmulatedDebounce,
    __in BOOLEAN EmulatedActiveBoth,
    __in WDF_TRI_STATE LevelReconfigurationSupported,
    __in GPIO_DEBOUNCE_MODEL DebounceModelOverride,
    __out PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext
    )

/*++

Routine Description:

    This routine initializes the context used by the debounce state machine.
    It is called when an interrupt is being connected (or re-connected).

Arguments:

    MemoryMappedController - Supplies a flag indicating whether the controller
        is memory-mapped or not.

    EmulatedDebounce - Supplies whether the pin is enabled for debounce
        emulation or not.

    EmulatedActiveBoth - Supplies whether the pin is enabled for ActiveBoth
        emulation.

    LevelReconfigurationSupported - Supplies whether the pin can be configured
        for [Level, <opposite polarity>]. WdfUseDefault implies the value is
        non-deterministic at this time (i.e. [Level, <opposite polarity>]
        reconfiguration may or may not be possible).

    DebounceModelOverride - Supplies an override debounce model supplied by
        internal test clients. A value of DebounceModelNone implies no
        override.

    DebounceContext - Supplies a pointer that receives the initialized debounce
        context.

Return Value:

    None.

--*/

{

    GPIO_DEBOUNCE_MODEL DebounceModel;

    PAGED_CODE();

    RtlZeroMemory(DebounceContext, sizeof(GPIO_DEBOUNCE_PIN_CONTEXT));
    DebounceContext->State = DebounceStateOnInitialInterrupt;
    DebounceContext->Flags.MemoryMapped = MemoryMappedController;

    //
    // 1. If debouncing doesn't need to be emulated, then force debounce model
    //    to be none.
    //
    // 2. If the GPIO controller supports reconfiguration to [Level,
    //    <opposite polarity>], then use enhanced debouncing model.
    //
    // 3. If [Level, <opposite polarity>] reconfiguration value is
    //    non-determistic, then mark it as "possibly enhanced".
    //
    // 4. Otherwise fallback to legacy debouncing model.
    //

    if (EmulatedDebounce != FALSE) {
        if (LevelReconfigurationSupported == WdfTrue) {
            DebounceModel = DebounceModelEnhanced;

        } else if (LevelReconfigurationSupported == WdfUseDefault) {
            DebounceModel = DebounceModelPossiblyEnhanced;

        } else {
            DebounceModel = DebounceModelLegacy;
        }

        //
        // Force the test-supplied override debounce model provided it is
        // compatible with the model determined by GPIO class extension.
        //

        if (DebounceModelOverride != DebounceModelNone) {
            if ((DebounceModel == DebounceModelLegacy) &&
                (DebounceModelOverride == DebounceModelEnhanced)) {

                GPIO_ASSERT(FALSE);

            } else {
                DebounceModel = DebounceModelOverride;
            }

        //
        // If there is a global debounce model override specified, then force
        // that model. For controllers that are not capable of supporting
        // enhanced debounce model, ignore the global override if specified as
        // such.
        //

        } else if (GpioGlobalDebounceModelOverride != DebounceModelNone) {
            if ((DebounceModel != DebounceModelLegacy) ||
                (GpioGlobalDebounceModelOverride != DebounceModelEnhanced)) {

                DebounceModel = GpioGlobalDebounceModelOverride;
            }
        }

    } else {
        DebounceModel = DebounceModelNone;
    }

    DebounceContext->Flags.DebounceModel = DebounceModel;
    DebounceContext->Flags.NoReconfigurePostValidInterrupt = EmulatedActiveBoth;
    DebounceContext->InterruptEpoch = 0x1;
    return;
}

_IRQL_requires_(PASSIVE_LEVEL)
VOID
GpiopDebounceDestroyContext (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext
    )

/*++

Routine Description:

    This routine destroys the debounce context.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

Return Value:

    None.

--*/

{

    PAGED_CODE();

    RtlZeroMemory(DebounceContext, sizeof(GPIO_DEBOUNCE_PIN_CONTEXT));
    return;
}

NTSTATUS
GpiopDebounceOnInterrupt (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in PVOID ControllerContext,
    __out PBOOLEAN DebounceStateMachineCompleted
    )

/*++

Routine Description:

    This routine is an entry point into the debounce engine. It is called
    when an interrupt fires on the specified pin.

    N.B. Entry IRQL: D-IRQL for memory-mapped GPIOs and PASSIVE_LEVEL for
         off-SoC GPIOs.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

    DebounceStateMachineCompleted - Supplies a pointer to a boolean that
        indicates whether this routine moved the state machine through
        to completion.

        If this is set to TRUE, the caller should invoke
        GpiopDebounceSignalStateMachineCompletion().

Return Value:

    NTSTATUS code.

--*/

{

    NTSTATUS Status;

    Status = GpiopDebounceInterruptEngine(DebounceContext,
                                          ControllerContext,
                                          DebounceEventOnInterrupt,
                                          NULL,
                                          DebounceStateMachineCompleted);

    return Status;
}

_IRQL_requires_max_(DISPATCH_LEVEL)
NTSTATUS
GpiopDebounceOnDebounceTimer (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in PVOID ControllerContext,
    __out PBOOLEAN DebounceStateMachineCompleted
    )

/*++

Routine Description:

    This routine is an entry point into the debounce engine. It is called
    when the debounce interval timer expires.

    N.B. 1. Entry IRQL: D-IRQL for memory-mapped GPIOs and PASSIVE_LEVEL for
            off-SoC GPIOs.

         2. This routine is invoked with the D-IRQL bank interrupt lock *NOT*
            held for memory-mapped GPIO controllers. The passive-level bank lock
            *is held* for off-SoC GPIO controllers.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

    DebounceStateMachineCompleted - Supplies a pointer to a boolean that
        indicates whether this routine moved the state machine through
        to completion.

        If this is set to TRUE, the caller should invoke
        GpiopDebounceSignalStateMachineCompletion().

Return Value:

    NTSTATUS code.

--*/

{

    return GpiopDebounceInterruptEngine(DebounceContext,
                                        ControllerContext,
                                        DebounceEventOnDebounceTimerFire,
                                        NULL,
                                        DebounceStateMachineCompleted);
}

_IRQL_requires_max_(DISPATCH_LEVEL)
NTSTATUS
GpiopDebounceOnNoiseTimer (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in PVOID ControllerContext,
    __in PGPIO_DEBOUNCE_OUTPUT_ACTION AssociatedOutputAction,
    __out PBOOLEAN DebounceStateMachineCompleted
    )

/*++

Routine Description:

    This routine is an entry point into the debounce engine. It is called
    when the noise filter interval expires.

    N.B. 1. Entry IRQL: DISPATCH_LEVEL for memory-mapped GPIOs and PASSIVE_LEVEL for
            off-SoC GPIOs.

         2. This routine is invoked with the D-IRQL bank interrupt lock *NOT*
            held for memory-mapped GPIO controllers. The passive-level bank lock
            *is held* for off-SoC GPIO controllers.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

    AssociatedOutputAction - Supplies a pointer to debounce output action buffer
        that was returned when noise timers were scheduled.

    DebounceStateMachineCompleted - Supplies a pointer to a boolean that
        indicates whether this routine moved the state machine through
        to completion.

        If this is set to TRUE, the caller should invoke
        GpiopDebounceSignalStateMachineCompletion().

Return Value:

    NTSTATUS code.

--*/

{

    return GpiopDebounceInterruptEngine(DebounceContext,
                                        ControllerContext,
                                        DebounceEventOnNoiseTimerFire,
                                        AssociatedOutputAction,
                                        DebounceStateMachineCompleted);
}

ULONG
GpiopDebounceQueryInterruptEpoch (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext
    )

/*++

Routine Description:

    This routine returns the current interrupt epoch associated with the
    specified debounce pin.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

Return Value:

    ULONG - the current interrupt epoch.

--*/

{

    return (ULONG)InterlockedCompareExchange(
                    (volatile LONG *)&DebounceContext->InterruptEpoch, 0, 0);
}

//
// ---------------------------------------------------------- Internal routines
//

_IRQL_requires_min_(DISPATCH_LEVEL)
NTSTATUS
GpiopDebounceStateMachineTransition (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in GPIO_DEBOUNCE_EVENT_TYPE PrimaryEventType,
    __in PGPIO_DEBOUNCE_INPUT_EVENT InputEvent,
    __out PGPIO_DEBOUNCE_OUTPUT_ACTION OutputAction
    )

/*++

Routine Description:

    This routine handles the debounce state machine. It is invoked when:
    a) an interrupt fires,
    b) the debounce interval timer expires, or
    c) the noise filter duration expires.
    d) a request (DebounceActionXyz) completes.

    This routine assumes that replay interrupts are never presented to the
    debounce state machine even if emulated debouncing is active on that
    interrupt pin.

    N.B. 1. Entry IRQL: D-IRQL or DISPATCH_LEVEL for memory-mapped GPIOs.
            DISPATCH_LEVEL or PASSIVE_LEVEL for off-SoC GPIOs.

         2. If the PrimaryEventType is DebounceEventOnInterrupt, then this
            routine assumes the caller has already acquired the bank interrupt
            lock for memory-mapped controllers.

Arguments:

    DebounceContext - Supplies a pointer to per-pin debounce context.

    PrimaryEventType - Supplies the type of primary event the debounce engine is
        being invoked for.

    EventType - Supplies the type of secondary event the debounce engine is
        being invoked for and its associated status.

    OutputAction - Supplies a pointer that receives the output action to be
        performed.

Return Value:

    NTSTATUS code.

--*/

{


    GPIO_DEBOUNCE_ACTION_TYPE ActionType;
    ULONG CurrentEpoch;
    GPIO_DEBOUNCE_MODEL DebounceModel;
    GPIO_DEBOUNCE_EVENT_TYPE EventType;
    NTSTATUS EventStatus;
    GPIO_DEBOUNCE_STATE NextState;
    PGPIO_DEBOUNCE_OUTPUT_ACTION PreviousOutputAction;
    NTSTATUS ReturnStatus;
    BOOLEAN StateChangeValid;
    NTSTATUS Status;

    GPIO_ASSERT(KeGetCurrentIrql() >= DISPATCH_LEVEL);

    ActionType = DebounceActionNone;
    DebounceModel = DebounceContext->Flags.DebounceModel;
    EventType = InputEvent->EventType;
    EventStatus = InputEvent->EventStatus;
    PreviousOutputAction = &InputEvent->AssociatedOutputAction;
    ReturnStatus = STATUS_SUCCESS;
    Status = STATUS_MORE_PROCESSING_REQUIRED;

    //
    // By default assume that any output action will be executed synchronously
    // with the state machine still locked. Thus a state change is not possible.
    //

    StateChangeValid = FALSE;

    //
    // The primary and secondary event types supplied must be valid. These are
    // supplied from within the debounce module and thus trusted.
    //

    GPIO_ASSERT((PrimaryEventType == DebounceEventOnInterrupt) ||
                (PrimaryEventType == DebounceEventOnDebounceTimerFire) ||
                (PrimaryEventType == DebounceEventOnNoiseTimerFire));

    GPIO_ASSERT((EventType == DebounceEventNone) ||
                (EventType == DebounceEventRequestCompletion) ||
                (EventType == DebounceEventRequestPending));

    //
    // If the debounce state machine could have moved ahead before the previous
    // request completes, then force an epoch check first. Otherwise, resume
    // from the current state.
    //

    if (PreviousOutputAction->StateChangeValid != FALSE) {

        GPIO_ASSERT(PrimaryEventType != DebounceEventOnInterrupt);

        NextState = DebounceStateCheckEpochMatch;

    } else {
        NextState = DebounceContext->State;
    }

    //
    // Loop through the state machine until a "callback" state is reached or
    // a "termination" state is reached. A callback state is one where the
    // GPIO class extension needs to be called back for some operation. A
    // termination state is one where the state machine ends.
    //

    do {
        switch(NextState) {

        case DebounceStateInvalid:

            GPIO_DEBOUNCE_INVALID_STATE_TRANSITION(DebounceContext);

            break;

            //
            // If the current interrupt epoch doesn't match the epoch at the
            // time the action was issued, then it implies that a different
            // event occurred in-between and advanced the state machine. The
            // incoming event is to be treated as invalid.
            //
            // N.B. The state machine may have since progressed and thus
            //      should not be disturbed. Just copy the existing state
            //      and terminate.
            //

        case DebounceStateCheckEpochMatch:

            CurrentEpoch = GpiopDebounceQueryInterruptEpoch(DebounceContext);
            if (PreviousOutputAction->AssociatedEpoch != CurrentEpoch) {
                NextState = DebounceContext->State;
                ReturnStatus = STATUS_SUCCESS;
                Status = STATUS_SUCCESS;

            } else {

                GPIO_ASSERT(PreviousOutputAction->ExpectedState ==
                    DebounceContext->State);

                NextState = DebounceContext->State;
            }

            break;

            //
            // The initial interrupt has arrived. The next step is to update
            // the interrupt epoch.
            //

        case DebounceStateOnInitialInterrupt:

            //
            // Pins that do not need any debouncing should not be presented to
            // the debounce engine.
            //

            GPIO_ASSERT((DebounceModel != DebounceModelNone) &&
                        (DebounceModel < DebounceModelMaximum) &&
                        (PrimaryEventType == DebounceEventOnInterrupt));

            NextState = DebounceStateUpdateEpochOnInitialInterrupt;
            break;

            //
            // Update the interrupt epoch. The next step depends on the type
            // of interrupt.
            //
            // - On initial interrupt: Next step is to mask the interrupt.
            //
            // - On noise interrupt: Next step is to cancel the noise timer.
            //

        case DebounceStateUpdateEpochOnNoiseTimerInterrupt:

            __fallthrough;  // -> DebounceStateUpdateEpochOnInitialInterrupt

        case DebounceStateUpdateEpochOnInitialInterrupt:

            GPIO_ASSERT(PrimaryEventType == DebounceEventOnInterrupt);

            InterlockedIncrement(
                (volatile LONG *)&DebounceContext->InterruptEpoch);

            if (NextState == DebounceStateUpdateEpochOnInitialInterrupt) {
                NextState = DebounceStateMaskInterrupt;

            } else if (NextState == DebounceStateUpdateEpochOnNoiseTimerInterrupt) {
                NextState = DebounceStateCancelNoiseTimer;

            } else {

                GPIO_DEBOUNCE_INVALID_STATE_TRANSITION(DebounceContext);
            }

            break;

            //
            // DebounceStateMaskInterrupt:
            //     Mask the interrupt. The next step is to schedule a timer for
            //     the debounce interval.
            //
            // DebounceStateMaskRestorePolarity:
            //      Mask the interrupt. The next step is to reconfigure the
            //      interrupt.
            //

        case DebounceStateMaskRestorePolarity:

            __fallthrough; // -> DebounceStateMaskInterrupt

        case DebounceStateMaskInterrupt:

            GPIO_ASSERT((PrimaryEventType == DebounceEventOnInterrupt) ||
                        (PrimaryEventType == DebounceEventOnNoiseTimerFire));

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held. Thus no state change is possible.
                //

                ActionType = DebounceActionMaskInterrupt;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;

                //
                // A mask request effectively cannot fail. Refer to the
                // assumptions at the top of this file.
                //

                GPIO_ASSERT(NT_SUCCESS(EventStatus));

                if (NextState == DebounceStateMaskInterrupt) {
                    NextState = DebounceStateScheduleDebounceTimer;

                } else {
                    NextState = DebounceStateReconfigureRestorePolarity;
                }
            }

            break;

            //
            // Schedule a timer for the debounce interval. The next step
            // depends on the debounce model:
            // - Legacy debounce model: Next step is to invoke the ISR.
            //
            // - Enhanced debounce model: Next step is to wait for the
            //     debounce interval to elapse.
            //
            // - PossiblyEnhanced debounce model: This is treated just like
            //     Enhanced debounce model case at this point in the state
            //     machine.
            //

        case DebounceStateScheduleDebounceTimer:

            GPIO_ASSERT(PrimaryEventType == DebounceEventOnInterrupt);

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held. Thus no state change is possible.
                //

                ActionType = DebounceActionScheduleDebounceTimer;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;
                if (DebounceModel == DebounceModelLegacy) {
                    NextState = DebounceStateInvokeTargetIsr;

                } else {
                    NextState = DebounceStateOnDebounceTimerFire;
                    ReturnStatus = STATUS_SUCCESS;
                    Status = STATUS_SUCCESS;
                }
            }

            break;

            //
            // Invoke the target ISR. This state can be called for all three
            // types of primary events. The next step depends on the primary
            // event:
            //  - On interrupt: Next step is to wait for debounce timer to fire.
            //
            //  - On debounce timer: Next step is to update the debounce model.
            //
            //  - On noise timer: Next step is to check if reconfiguration is
            //      needed.
            //

        case DebounceStateInvokeTargetIsr:

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held or passive-level bank lock held for
                // off-SoC GPIOs. Thus no state change is possible.
                //

                ActionType = DebounceActionInvokeTargetIsr;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;
                if (PrimaryEventType == DebounceEventOnInterrupt) {
                    NextState = DebounceStateOnDebounceTimerFire;
                    ReturnStatus = STATUS_SUCCESS;
                    Status = STATUS_SUCCESS;

                } else if (PrimaryEventType ==
                                            DebounceEventOnDebounceTimerFire) {

                    GPIO_ASSERT(DebounceModel == DebounceModelPossiblyEnhanced);

                    NextState = DebounceStateUpdateDebounceModel;

                } else {
                    NextState = DebounceStateCheckRestorePolarity;
                }
            }

            break;

            //
            // The debounce interval has elapsed. The next step depends on the
            // debounce model:
            // Legacy debounce model: The next step is to unmask the interrupt.
            //
            // Enhanced debounce model: The next step is to reconfigure the
            //     interrupt for opposite polarity.
            //
            // PossiblyEnhanced debounce model: This is treated just like
            //     Enhanced debounce model case at this point.
            //

        case DebounceStateOnDebounceTimerFire:

            GPIO_ASSERT((EventType == DebounceEventNone) &&
                        (PrimaryEventType == DebounceEventOnDebounceTimerFire));

            if (DebounceModel == DebounceModelLegacy) {
                NextState = DebounceStateUnmaskInterrupt;

            } else {
                NextState = DebounceStateReconfigureInterrupt;
            }

            break;

            //
            // DebounceStateUnmaskInterrupt:
            //     Unmask the interrupt. The next step is to wait for initial
            //     interrupt or the unmask to complete.
            //
            //     In the legacy debounce model case, the next step after unmask
            //     is to wait for the initial interrupt. Thus it is implemented
            //     as an action + termination state.
            //
            //     In the enhanced debounce model case:
            //     Since the interrupt is unmasked prior to scheduling
            //     the noise filter timer, there is a possiblity that an
            //     interrupt arrives before the state machine is entered again.
            //     This is only possible for off-SoC GPIO controllers as the
            //     interrupt lock should be held for on-SoC GPIO controllers.
            //     Snapshot the current interrupt epoch prior and record the
            //     fact that a state change is possible prior to re-entry.
            //
            //      N.B. The interrupt is unmasked prior to scheduling noise
            //           filter timer as such an operation could take arbitray
            //           interval of time and the noise filter interval
            //           (extremely small) could elapse even before the
            //           operation is even scheduled.
            //
            //
            // DebounceStateUnmaskRestorePolarity:
            //     Unmask the interrupt. The next step is to wait for initial
            //     interrupt. Thus it is implemented as an action + termination
            //     state.
            //
            // No need to check for unmask completion status as it effectively
            // cannot fail. Refer to the assumptions at the top of this file.
            //
            // If this is an action + termination state, then assume the
            // operation will be completed synchronously and thus no state
            // change is possible. It doesn't really matter.
            //
            // Otherwise, record the fact that a state change is possible.
            //

        case DebounceStateUnmaskRestorePolarity:

            GPIO_ASSERT(PrimaryEventType != DebounceEventOnDebounceTimerFire);

            __fallthrough; // -> DebounceStateUnmaskInterrupt

        case DebounceStateUnmaskInterrupt:

            GPIO_ASSERT((EventType == DebounceEventNone) &&
                        (DebounceModel != DebounceModelPossiblyEnhanced));

            ActionType = DebounceActionUnmaskInterrupt;
            if ((DebounceModel == DebounceModelLegacy) ||
                (NextState == DebounceStateUnmaskRestorePolarity)) {

                NextState = DebounceStateOnInitialInterrupt;
                ReturnStatus = STATUS_SUCCESS;

            } else {
                NextState = DebounceStateWaitForUnmaskCompletion;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;

                //
                // State change possible prior to re-entry.
                //

                StateChangeValid = TRUE;
            }

            Status = STATUS_SUCCESS;
            break;

            //
            // The debounce state machine could be called back here when either
            // the unmask request pends, completes or the interrupt fires.
            //
            // If the interrupt fires, then treat this as the same event as
            // interrupt firing after noise timer has been queued.
            //
            // If the unmask completed, then proceed to the next state, which
            // is scheduling noise timers. Since the current interrupt epoch
            // matches the epoch prior to the unmask operation, it implies the
            // line has been stable since unmask request completed (i.e. no
            // intervening interrupts fired). Proceed to scheduling the noise
            // filter timer.
            //
            // If unmask is still pending, then simply bail out. An interrupt
            // could still arrive before the unmask completion and advance the
            // state machine. Record the fact that a state change is possible.
            //
            // Note the current GPIO implementation always performs the unmask
            // synchronously and thus doesn't return pending.
            //

        case DebounceStateWaitForUnmaskCompletion:

            if (PrimaryEventType == DebounceEventOnInterrupt) {
                NextState = DebounceStateOnNoiseTimerFire;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;

                //
                // An unmask request effectively cannot fail. Refer to the
                // assumptions at the top of this file.
                //

                GPIO_ASSERT(NT_SUCCESS(EventStatus));

                NextState = DebounceStateScheduleNoiseTimer;

            } else {

                GPIO_ASSERT(EventType == DebounceEventRequestPending);

                //
                // State change possible prior to re-entry.
                //

                StateChangeValid = TRUE;
                ReturnStatus = STATUS_SUCCESS;
                Status = STATUS_SUCCESS;
            }

            break;

            //
            // DebounceStateReconfigureInterrupt:
            //     Regconfigure the interrupt. The next step depends on a
            //     couple of factors: a) whether the request completed
            //     successfully or not; b) the current debounce model.
            //
            //     (Enhanced, STATUS_SUCCESS) -> next step is to unmask the
            //         interrupt.
            //
            //     (Enhanced, !NT_SUCCESS) -> next step is to retry
            //          reconfiguration.
            //
            //     (PossiblyEhanced, STATUS_SUCCESS) -> next step is to
            //          update the debounce model.
            //
            //     (PossiblyEhanced, !NT_SUCCESS) -> next step is to invoke the
            //          target ISR.
            //
            //      Note failures can only occur with non-memory mapped GPIO
            //      controllers.
            //
            // DebounceStateReconfigureRestorePolarity:
            //      Regconfigure the interrupt. The next step is to unmask the
            //      interrupt. If the reconfigure fails, the next step is
            //      to retry reconfiguration.
            //
            // The interrupt is masked prior to reconfiguration and thus
            // the state machine cannot advance due to an interrupt.
            //

        case DebounceStateReconfigureRestorePolarity:

            GPIO_ASSERT((PrimaryEventType == DebounceEventOnInterrupt) ||
                        (PrimaryEventType == DebounceEventOnNoiseTimerFire));

            __fallthrough;

        case DebounceStateReconfigureInterrupt:
            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held or passive-level bank lock held for
                // off-SoC GPIOs. Thus no state change is possible.
                //

                DebounceContext->Flags.ReconfigureFailed = FALSE;
                ActionType = DebounceActionReconfigureInterrupt;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;

                if (NT_SUCCESS(EventStatus)) {
                    DebounceContext->Flags.ReconfigureFailed = FALSE;
                    DebounceContext->ReconfigureFailureCount = 0;

                } else {

                    //
                    // Tolerate failures from memory-mapped controllers only
                    // while in PossiblyEnhanced model.
                    //

                    GPIO_ASSERT((DebounceContext->Flags.MemoryMapped == 0) ||
                                (DebounceModel == DebounceModelPossiblyEnhanced));

                    DebounceContext->Flags.ReconfigureFailed = TRUE;
                    DebounceContext->ReconfigureFailureCount += 1;
                }

                if (NextState == DebounceStateReconfigureRestorePolarity) {
                    if (NT_SUCCESS(EventStatus)) {
                        NextState = DebounceStateUnmaskRestorePolarity;

                    } else {
                        NextState =
                            DebounceStateRestoreReconfigureFailureReattempt;
                    }

                } else {  // NextState == DebounceStateReconfigureInterrupt


                    GPIO_ASSERT(PrimaryEventType ==
                                    DebounceEventOnDebounceTimerFire);

                    //
                    // Check whether the reconfigure succeeded or not and deal
                    // with failures appropriately.
                    //

                    if (NT_SUCCESS(EventStatus)) {
                        if (DebounceModel == DebounceModelPossiblyEnhanced) {
                            NextState = DebounceStateUpdateDebounceModel;

                        } else {
                            NextState = DebounceStateUnmaskInterrupt;
                        }

                    } else {

                        if (DebounceModel == DebounceModelPossiblyEnhanced) {
                            NextState = DebounceStateInvokeTargetIsr;

                        } else {
                            NextState =
                                    DebounceStateReconfigureFailureReattempt;
                        }
                    }
                }
            }

            break;

            //
            // Schedule the noise filter timer. The next step is to wait for
            // the noise timer to fire or an interrupt to arrive. Thus it is
            // implemented as an action + termination state.
            //
            // N.B. The current interrupt epoch must the epoch prior to the
            //      unmask operation for the state machine to get here. This
            //      implies the line has been stable since unmask request
            //      completed (i.e. no intervening interrupts fired).
            //

        case DebounceStateScheduleNoiseTimer:

            GPIO_ASSERT((PrimaryEventType == DebounceEventOnInterrupt) ||
                        (PrimaryEventType == DebounceEventOnDebounceTimerFire));

            //
            // The action is done with bank interrupt lock not held.
            // Thus an incoming interrupt could advance the state.
            //

            ActionType = DebounceActionScheduleNoiseTimer;
            NextState = DebounceStateOnNoiseTimerFire;
            ReturnStatus = STATUS_SUCCESS;
            Status = STATUS_SUCCESS;

            //
            // State change possible prior to re-entry.
            //

            StateChangeValid = TRUE;
            break;

            //
            // Update the debounce model to either legacy or enhanced depending
            // on whether the reconfigure succeeded or not. The next step is to
            // unmask the interrupt.
            //

        case DebounceStateUpdateDebounceModel:

            GPIO_ASSERT((DebounceModel == DebounceModelPossiblyEnhanced) &&
                        (PrimaryEventType == DebounceEventOnDebounceTimerFire));

            if (DebounceContext->Flags.ReconfigureFailed == FALSE) {
                DebounceContext->Flags.DebounceModel = DebounceModelEnhanced;

            } else {
                DebounceContext->Flags.DebounceModel = DebounceModelLegacy;
            }

            DebounceModel = DebounceContext->Flags.DebounceModel;
            NextState = DebounceStateUnmaskInterrupt;
            break;

            //
            // This state is reached when either the noise filter timer has
            // elapsed or a new interrupt arrived while we were waiting for the
            // timer to fire. The state depends on the event:
            //  - If the timer fired, query the interrupt epoch to ensure it is
            //    still associated with the correct epoch.
            //
            //  - If a new interrupt arrived, then update the epoch.
            //

        case DebounceStateOnNoiseTimerFire:

            GPIO_ASSERT(
                (PrimaryEventType == DebounceEventOnNoiseTimerFire) ||
                (PrimaryEventType == DebounceEventOnInterrupt));

            if (PrimaryEventType == DebounceEventOnNoiseTimerFire) {
                NextState = DebounceStateQueryInterruptEpoch;

            } else if (PrimaryEventType == DebounceEventOnInterrupt) {
                NextState = DebounceStateUpdateEpochOnNoiseTimerInterrupt;
            }

            EventType = DebounceEventNone;
            break;

            //
            // Query the interrupt epoch associated with the noise filter timer.
            // The next step is to invoke the target ISR provided the epoch
            // matches the current interrupt epoch.
            //

        case DebounceStateQueryInterruptEpoch:

            GPIO_ASSERT(PrimaryEventType == DebounceEventOnNoiseTimerFire);

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held. Thus no state change is possible.
                //

                ActionType = DebounceActionQueryNoiseFilterEpoch;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;

                //
                // Query the current interrupt epoch value and the epoch
                // at the time the noise filter timer was queued.
                //
                // N.B. Current epoch is queried outside of the lock and thus
                //      manipulated interlocked. The noise filter epoch is
                //      always manipulated under the bank interrupt lock and
                //      hence doesn't need to be interlocked.
                //

                CurrentEpoch =
                    GpiopDebounceQueryInterruptEpoch(DebounceContext);

                //
                // If the current interrupt epoch matches the noise filter
                // epoch, then it implies the line was stable (no intervening
                // interrupts fired). Proceed to invoke the target ISR in
                // such case.
                //

                if (CurrentEpoch == DebounceContext->NoiseFilterEpoch) {
                    NextState = DebounceStateInvokeTargetIsr;

                //
                // If the current interrupt epoch does not match the noise
                // filter epoch, then it implies the is for a spurious interrupt
                // and thus should be ignored.
                //
                // N.B. The state machine may have since progressed and thus
                //      should not be disturbed. Just copy the existing state
                //      and terminate.
                //

                } else {
                    NextState = DebounceContext->State;
                    ReturnStatus = STATUS_SUCCESS;
                    Status = STATUS_SUCCESS;
                }
            }

            break;

            //
            // Cancel the noise filter timer. The next step is to record the
            // interrupt as spurious.
            //

        case DebounceStateCancelNoiseTimer:

            GPIO_ASSERT(PrimaryEventType == DebounceEventOnInterrupt);

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held. Thus no state change is possible.
                //

                ActionType = DebounceActionCancelNoiseTimer;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;
                NextState = DebounceStateLogSpuriousInterrupt;
            }

            break;

            //
            // Record the interrupt as spurious. The next step is to reconfigure
            // the interrupt.
            //

        case DebounceStateLogSpuriousInterrupt:

            if (EventType == DebounceEventNone) {

                //
                // This action is required to be completed synchronously with
                // bank interrupt lock held. Thus no state change is possible.
                //

                ActionType = DebounceActionLogSpuriousInterrupt;
                ReturnStatus = STATUS_MORE_PROCESSING_REQUIRED;
                Status = STATUS_SUCCESS;

            } else if (EventType == DebounceEventRequestCompletion) {
                EventType = DebounceEventNone;
                NextState = DebounceStateMaskRestorePolarity;
            }

            break;

            //
            // Check if the polarity needs to be restored after debounce
            // completion. It doesn't need to be restored for emulated
            // ActiveBoth interrupts.
            //

        case DebounceStateCheckRestorePolarity:

            GPIO_ASSERT(DebounceModel == DebounceModelEnhanced);

            if (DebounceContext->Flags.NoReconfigurePostValidInterrupt == 1) {
                NextState = DebounceStateOnInitialInterrupt;
                ReturnStatus = STATUS_SUCCESS;
                Status = STATUS_SUCCESS;

            } else {
                NextState = DebounceStateMaskRestorePolarity;
            }

            break;

        //
        // This state can be entered either because a reconfigure attempt
        // just failed or a previously failed reconfiguration attempt needs
        // to be retried after retry timer expiration.
        //
        // On failed reconfiguration: Next step is to retry after waiting
        //     for an interval.
        //
        // On retry timer expiry: re-attempt the reconfiguration operation.
        //
        // N.B. 1. Scheduling a timer for the noise interval is convenient and
        //         thus chosen.
        //
        //      2. The primary event can be DebounceEventOnDebounceTimerFire
        //         both in timer expiration and reconfiguration failure. Thus
        //         the failure status is checked rather than primary event type.
        //

        case DebounceStateRestoreReconfigureFailureReattempt:

            __fallthrough;  // -> DebounceStateReconfigureFailureReattempt

        case DebounceStateReconfigureFailureReattempt:

            if (DebounceContext->Flags.ReconfigureFailed == TRUE) {
                DebounceContext->Flags.ReconfigureFailed = FALSE;

                //
                // N.B. 1. If the consecutive reconfiguration failures exceed a
                //         predetermined value, then bail out without setting an
                //         action. This will cause the interrupt to be left
                //         masked and no further interrupts will be serviced.
                //         The state will be left as "reconfigure failure
                //         reattempt" for diagnosis.
                //
                //      2. StateChangeValid is not set as the interrupt is
                //         masked and thus the state machine cannot advance
                //         due to an interrupt.
                //

                if (DebounceContext->ReconfigureFailureCount <
                            GPIO_DEBOUNCE_RECONFIGURE_FAILURE_THRESHOLD) {

                    ActionType = DebounceActionScheduleNoiseTimer;
                }

                ReturnStatus = STATUS_SUCCESS;
                Status = STATUS_SUCCESS;

            } else if (NextState == DebounceStateReconfigureFailureReattempt) {
                NextState = DebounceStateReconfigureInterrupt;

            } else {
                NextState = DebounceStateReconfigureRestorePolarity;
            }

            break;

        default:

            GPIO_DEBOUNCE_INVALID_STATE_TRANSITION(DebounceContext);
        }

        GPIO_DEBOUNCE_SAVE_PREVIOUS_STATE(DebounceContext, PrimaryEventType);

        DebounceContext->State = NextState;

    } while (Status == STATUS_MORE_PROCESSING_REQUIRED);

    GPIO_ASSERT((Status != STATUS_MORE_PROCESSING_REQUIRED) ||
                (ActionType != DebounceActionNone));

    OutputAction->ActionType = ActionType;
    OutputAction->StateChangeValid = StateChangeValid;
    OutputAction->ExpectedState = DebounceContext->State;
    OutputAction->AssociatedEpoch =
        GpiopDebounceQueryInterruptEpoch(DebounceContext);;

    return ReturnStatus;
}

__drv_when((DebounceContext->Flags.MemoryMapped == TRUE),
    _IRQL_requires_min_(DISPATCH_LEVEL))
NTSTATUS
GpiopDebounceInterruptEngine (
    __inout PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext,
    __in PVOID ControllerContext,
    __in GPIO_DEBOUNCE_EVENT_TYPE PrimaryEventType,
    __in_opt PGPIO_DEBOUNCE_OUTPUT_ACTION AssociatedOutputAction,
    __out PBOOLEAN DebounceStateMachineCompleted
    )

/*++

Routine Description:

    This routine is the main entry point into the debounce engine.
    It is called when:
    a) an interrupt fires,
    b) the debounce interval timer expires, or
    c) the noise filter duration expires.

    N.B. 1. Entry IRQL: D-IRQL or DISPATCH_LEVEL for memory-mapped GPIOs and
            PASSIVE_LEVEL for off-SoC GPIOs.

         2. If the PrimaryEventType is DebounceEventOnInterrupt, then this
            routine assumes the caller has already acquired the bank interrupt
            lock for memory-mapped controllers.

         3. For off-SoC GPIOs, this routine is invoked with the passive-level
            bank lock held.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

    PrimaryEventType - Supplies the type of event the debounce engine is being
        invoked for.

    AssociatedOutputAction - Supplies a pointer to debounce output action buffer
        that was returned when noise timers were scheduled. This is only
        supplied when a noise timer fires.

    DebounceStateMachineCompleted - Supplies a pointer to a boolean that
        indicates whether this routine moved the state machine through
        to completion.

        If this is set to TRUE, the caller should invoke
        GpiopDebounceSignalStateMachineCompletion().

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireLock;
    ULONG EpochValue;
    NTSTATUS EventStatus;
    GPIO_DEBOUNCE_INPUT_EVENT InputEvent;
    GPIO_DEBOUNCE_OUTPUT_ACTION OutputAction;
    NTSTATUS Status;

    RtlZeroMemory(&InputEvent, sizeof(GPIO_DEBOUNCE_INPUT_EVENT));
    RtlZeroMemory(&OutputAction, sizeof(GPIO_DEBOUNCE_OUTPUT_ACTION));

    InputEvent.EventType = DebounceEventNone;
    InputEvent.EventStatus = STATUS_SUCCESS;
    OutputAction.ActionType = DebounceActionNone;

    //
    // If an associated output action buffer is supplied, pass that to the
    // debounce state machine as part of the input event buffer.
    //

    if (ARGUMENT_PRESENT(AssociatedOutputAction) != FALSE) {
        InputEvent.AssociatedOutputAction = *AssociatedOutputAction;
    }

    if ((DebounceContext->Flags.MemoryMapped == FALSE) ||
        (PrimaryEventType == DebounceEventOnDebounceTimerFire) ||
        (PrimaryEventType == DebounceEventOnNoiseTimerFire)) {

        AcquireLock = TRUE;

    } else {
        AcquireLock = FALSE;
    }

    if (AcquireLock != FALSE) {
        GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
    }

    GPIO_ASSERT(KeGetCurrentIrql() >= DISPATCH_LEVEL);

    do {

        Status = GpiopDebounceStateMachineTransition(
                     DebounceContext,
                     PrimaryEventType,
                     &InputEvent,
                     &OutputAction);

        switch (OutputAction.ActionType) {

        case DebounceActionNone:

            // NOTHING;

            break;

        case DebounceActionMaskInterrupt:

            //
            // Memory-mapped GPIOs: OK to hold the interrupt lock across the
            //      operation.
            //
            // Off-SoC GPIOs: Need to drop the interrupt lock across the
            //      operation. The passive-level bank lock is held by the
            //      caller.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {

                GPIO_ASSERT(AcquireLock != FALSE);

                GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
            }

            EventStatus = GpiopDebounceMaskInterruptCallback(
                              DebounceContext,
                              ControllerContext);

            InputEvent.EventStatus = EventStatus;
            InputEvent.EventType = DebounceEventRequestCompletion;

            //
            // Reacquire the interrupt lock prior to entering the state machine
            // again.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {
                GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
            }

            break;

        case DebounceActionScheduleDebounceTimer:

            //
            // OK to hold the interrupt lock across the operation.
            //

            EventStatus = GpiopDebouceScheduleDebounceTimerCallback(
                              DebounceContext,
                              ControllerContext);

            InputEvent.EventStatus = EventStatus;
            InputEvent.EventType = DebounceEventRequestCompletion;
            break;

        case DebounceActionScheduleNoiseTimer:

            //
            // OK to hold the interrupt lock across the operation.
            //
            // Note this action can be requested when the primary event is
            // DebounceEventOnInterrupt, DebounceEventOnDebounceTimerFire or
            // DebounceEventOnNoiseTimerFire. These can happen when the
            // noise timer is rescheduled (the retry timer) after a
            // reconfiguration failure.
            //

            GpiopDebouceScheduleNoiseFilterTimerCallback(DebounceContext,
                                                         ControllerContext,
                                                         &OutputAction);

            InputEvent.EventStatus = STATUS_SUCCESS;
            InputEvent.EventType = DebounceEventRequestCompletion;
            break;

        case DebounceActionInvokeTargetIsr:

            //
            // Memory-mapped GPIOs: OK to hold the interrupt lock across the
            //      operation.
            //
            // Off-SoC GPIOs: Need to drop the interrupt lock across the
            //      operation. The passive-level bank lock is held by the
            //      caller.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {

                GPIO_ASSERT(AcquireLock != FALSE);

                GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
            }

            GpiopDebounceInvokeTargetIsrCallback(DebounceContext,
                                                 ControllerContext);

            //
            // Reacquire the interrupt lock prior to entering the state machine
            // again.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {
                GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
            }

            InputEvent.EventStatus = STATUS_SUCCESS;
            InputEvent.EventType = DebounceEventRequestCompletion;
            break;

        case DebounceActionCancelNoiseTimer:

            //
            // OK to hold the interrupt lock across the operation.
            //

            GpiopDebouceCancelNoiseFilterTimerCallback(DebounceContext,
                                                       ControllerContext);

            InputEvent.EventStatus = STATUS_SUCCESS;
            InputEvent.EventType = DebounceEventRequestCompletion;
            break;

        case DebounceActionUnmaskInterrupt:

            //
            // Memory-mapped GPIOs: OK to hold the interrupt lock across the
            //      operation.
            //
            // Off-SoC GPIOs: Need to drop the interrupt lock across the
            //      operation. The passive-level bank lock is held by the
            //      caller.
            //
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {

                GPIO_ASSERT(AcquireLock != FALSE);

                GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
            }

            EventStatus = GpiopDebounceUnmaskInterruptCallback(
                              DebounceContext,
                              ControllerContext);

            InputEvent.EventStatus = EventStatus;
            InputEvent.EventType = DebounceEventRequestCompletion;

            //
            // Reacquire the interrupt lock prior to entering the state machine
            // again.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {
                GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
            }

            break;

        case DebounceActionReconfigureInterrupt:

            //
            // Memory-mapped GPIOs: OK to hold the interrupt lock across the
            //      operation.
            //
            // Off-SoC GPIOs: Need to drop the interrupt lock across the
            //      operation. The passive-level bank lock is held by the
            //      caller.
            //
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {

                GPIO_ASSERT(AcquireLock != FALSE);

                GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
            }

            EventStatus = GpiopDebounceReconfigureInterruptCallback(
                              DebounceContext,
                              ControllerContext);

            InputEvent.EventStatus = EventStatus;
            InputEvent.EventType = DebounceEventRequestCompletion;

            //
            // Reacquire the interrupt lock prior to entering the state machine
            // again.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {
                GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
            }

            break;

        case DebounceActionQueryNoiseFilterEpoch:

            //
            // OK to hold the interrupt lock across the operation.
            //

            EpochValue = GpiopDebounceQueryNoiseFilterEpoch(
                              DebounceContext,
                              ControllerContext);

            DebounceContext->NoiseFilterEpoch = EpochValue;
            InputEvent.EventStatus = STATUS_SUCCESS;
            InputEvent.EventType = DebounceEventRequestCompletion;
            break;

        case DebounceActionLogSpuriousInterrupt:

            //
            // Memory-mapped GPIOs: OK to hold the interrupt lock across the
            //      operation.
            //
            // Off-SoC GPIOs: Need to drop the interrupt lock across the
            //      operation. The passive-level bank lock is held by the
            //      caller.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {

                GPIO_ASSERT(AcquireLock != FALSE);

                GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
            }

            GpiopDebounceLogSpuriousInterrupt(DebounceContext,
                                              ControllerContext);

            InputEvent.EventStatus = STATUS_SUCCESS;
            InputEvent.EventType = DebounceEventRequestCompletion;

            //
            // Reacquire the interrupt lock prior to entering the state machine
            // again.
            //

            if (DebounceContext->Flags.MemoryMapped == FALSE) {
                GpiopDebounceAcquireLock(DebounceContext, ControllerContext);
            }

            break;

        default:

            GPIO_ASSERT(FALSE);

            Status = STATUS_UNSUCCESSFUL;
        }

        //
        // Supply the previous output action back to the debounce state machine.
        //

        InputEvent.AssociatedOutputAction = OutputAction;

    } while (Status == STATUS_MORE_PROCESSING_REQUIRED);

    //
    // If the state machine completed...
    //

    if (GpiopDebounceIsPinAtCompletedState(DebounceContext) != FALSE) {
        *DebounceStateMachineCompleted = TRUE;

    } else {
        *DebounceStateMachineCompleted = FALSE;
    }

    if (AcquireLock != FALSE) {
        GpiopDebounceReleaseLock(DebounceContext, ControllerContext);
    }

    return Status;
}

_IRQL_requires_min_(DISPATCH_LEVEL)
BOOLEAN
GpiopDebounceIsPinAtCompletedState (
     __in PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext
    )

/*++

Routine Description:

    This routine returns whether a pin is in a state that was reached by
    completing the state machine.  As a quirk, this includes the very first
    DebounceStateOnInitialInterrupt when no interrupt activity has occurred yet.
    
    N.B. This assumes that the debounce lock has been acquired.

Arguments:

    DebounceContext - Supplies a pointer that receives the initialized debounce
        context.

Return Value:

    NTSTATUS code.

--*/

{

    GPIO_ASSERT(KeGetCurrentIrql() >= DISPATCH_LEVEL);
    GPIO_ASSERT(DebounceContext->State != DebounceStateInvalid);

    if (DebounceContext->State == DebounceStateOnInitialInterrupt) {
        return TRUE;
    }

    if (((DebounceContext->State == DebounceStateRestoreReconfigureFailureReattempt) ||
         (DebounceContext->State == DebounceStateReconfigureFailureReattempt)) &&
        (DebounceContext->ReconfigureFailureCount >= GPIO_DEBOUNCE_RECONFIGURE_FAILURE_THRESHOLD)) {

        return TRUE;
    }

    return FALSE;
}

_IRQL_requires_(PASSIVE_LEVEL)
VOID
GpiopDebounceWaitForStateMachineCompletionOfMaskedPin (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebouncePinContext,
    __in PVOID ControllerContext,
    __in BOOLEAN WaitForTimerDpcCompletion
    )

/*++

Routine Description:

    This routine waits for a per-pin debounce state machine to move through
    to completion (e.g. debounce timer, noise timer).

    N.B. 1. This must only be used on pins that are debounced in software.

         2. This must only be used on pins that have been masked in hardware.
            This means that when the routine returns, the interrupt.c state
            machine would have also completed.

         3. This must not be called in parallel with itself on the same pin.

         4. The caller must guarantee that pin is not being simultaneously
            disabled.

            This is true when the primary WDF queue is paused (e.g. when called
            from GpioClxEvtDevicePreInterruptsDisabled() or
            GpioClxEvtDevicePostInterruptsEnabled()).
         
    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

    WaitForTimerDpcCompletion - Supplies whether to wait for the debounce
        and/or noise timer DPCs to complete.  If FALSE, the caller should
        subsequently call KeFlushQueuedDpcs().

Return Value:

    None.

--*/

{

    KEVENT CompletionEvent;
    
    GpiopDebounceAcquireLock(DebouncePinContext, ControllerContext);

    GPIO_ASSERT(KeGetCurrentIrql() >= DISPATCH_LEVEL);
    GPIO_ASSERT(DebouncePinContext->CompletionEvent == NULL);

    //
    // Easy case - no debounce activity is ongoing and a masked pin will not
    // initiate any debounce activity.
    //

    if (GpiopDebounceIsPinAtCompletedState(DebouncePinContext) != FALSE) {
        GpiopDebounceReleaseLock(DebouncePinContext, ControllerContext);

    //
    // Difficult case - it's not possible that the pin fires now (because it's
    // masked) but it could be that it previously fired and is still going
    // through debounce.
    //

    } else {

        KeInitializeEvent(&CompletionEvent, SynchronizationEvent, FALSE);
        DebouncePinContext->CompletionEvent = &CompletionEvent;

        //
        // Drop the lock so that debounce activity can move forward.
        //

        GpiopDebounceReleaseLock(DebouncePinContext, ControllerContext);

        //
        // Wait till after the state machine has completed.
        //

        GpiopDebounceTracePin(DebouncePinContext,
                              ControllerContext,
                              "Waiting for debounce completion; State=",
                              1,
                              DebouncePinContext->State);

        KeWaitForSingleObject(&CompletionEvent,
                              Executive,
                              KernelMode,
                              FALSE,
                              NULL);


        GpiopDebounceTracePin(DebouncePinContext,
                              ControllerContext,
                              "Debounce completed",
                              0,
                              0);

        if (WaitForTimerDpcCompletion != FALSE) {
            KeFlushQueuedDpcs();
        }
    }

    return;
}

_IRQL_requires_(PASSIVE_LEVEL)
VOID
GpiopDebounceSignalStateMachineCompletion (
    __in PGPIO_DEBOUNCE_PIN_CONTEXT DebouncePinContext,
    __in PVOID ControllerContext
    )

/*++

Routine Description:

    This routine is called after a per-pin debounce state machine has moved
    through to completion.  This signals DebounceContext->CompletionEvent if
    it is non-NULL.
    
    Currently:
    
      1. The waiter is GpiopDebounceWaitForStateMachineCompletionOfMaskedPin().

      2. This is only called for masked pins, since the sole waiter only
         considers masked pins.

    N.B. The caller must guarantee that pin is not being simultaneously
         disabled.  For instance, the caller could hold the bank lock.

Arguments:

    DebouncePinContext - Supplies a pointer to per-pin debounce context.

    ControllerContext - Supplies a pointer to an opaque controller context.

Return Value:

    None.

--*/

{

    PKEVENT CompletionEvent;

    GpiopDebounceAcquireLock(DebouncePinContext, ControllerContext);
    CompletionEvent = DebouncePinContext->CompletionEvent;
    DebouncePinContext->CompletionEvent = NULL;
    GpiopDebounceReleaseLock(DebouncePinContext, ControllerContext);

    if (CompletionEvent != NULL) {
        KeSetEvent(CompletionEvent, IO_NO_INCREMENT, FALSE);
    }
}

