ddresses != FALSE) {
            MappedBasePage = Intersection.Maximum - (Pages - 1);
            MappedBasePage -= AlignDownOffset;

            ASSERT(MappedBasePage >= Intersection.Minimum);

        } else {
            MappedBasePage = Intersection.Minimum;
        }
    }

    //
    // Check to make sure the region's attributes are adequate.  There are four
    // types of allocations; cachability, protection, allocation and
    // descriptive.  The first two (cachability and protection) describe
    // support.  For a region to be sufficient for a request, a memory region
    // needs to support *at least* the specified attributes.  The allocation
    // attributes require an *exact* match to assure the desired memory region
    // is being allocated for the caller.  The descriptive attributes are
    // descriptive only and do not need to match.
    //

    AttrSub1 = MemoryRegion->Attributes & MEMORY_ATTRIBUTE_CACHE_MASK;
    AttrSub2 = Attributes & MEMORY_ATTRIBUTE_CACHE_MASK;

    //
    // Caller requires a memory region to support cache attributes.
    //

    if (CHECK_FLAG(AttrSub1, AttrSub2) != AttrSub2) {
        return FALSE;
    }

    AttrSub1 = MemoryRegion->Attributes & MEMORY_ATTRIBUTE_PROTECT_MASK;
    AttrSub2 = Attributes & MEMORY_ATTRIBUTE_PROTECT_MASK;

    //
    // Caller requires a memory region to support protection attributes.
    //

    if (CHECK_FLAG(AttrSub1, AttrSub2) != AttrSub2) {
        return FALSE;
    }

    AttrSub1 = Attributes & MEMORY_ATTRIBUTE_ALLOCATION_ALLOCATE_MASK;
    AttrSub2 =
        MemoryRegion->Attributes & MEMORY_ATTRIBUTE_ALLOCATION_ALLOCATE_MASK;

    //
    // Caller requires a memory region to have some of the same allocation
    // attributes.  Allocation attributes not covered by the match mask indicate
    // address policy, which has already been encoded into the address ranges.
    //

    if (AttrSub1 != AttrSub2) {
        return FALSE;
    }

    //
    // Finally, check that the memory type is that which the caller requested.
    //

    if (MemoryRegion->MemoryType != RequiredMemoryType) {
        return FALSE;
    }

    //
    // Record the allocated region.
    //

    AllocateRegion->BasePage = BasePage;
    AllocateRegion->MappedBasePage = MappedBasePage;
    AllocateRegion->PageCount = Pages;
    AllocateRegion->Attributes = MemoryRegion->Attributes;
    AllocateRegion->MemoryType = MemoryRegion->MemoryType;
    return TRUE;
}

ULONG
MmMdCountList (
    __in PMEMORY_DESCRIPTOR_LIST Mdl
    )

/*++

Routine Description:

    Counts the number of descriptors in a memory descriptor list.

Arguments:

    Mdl - Supplies the memory descriptor list to count.

Return Value:

    The number of descriptors in the MDL.

--*/

{

    ULONG Count;
    PLIST_ENTRY NextEntry;

    Count = 0;
    NextEntry = Mdl->Head->Flink;
    while (NextEntry != Mdl->Head) {
        Count += 1;
        NextEntry = NextEntry->Flink;
    }

    return Count;
}

VOID
MmMdPrintList (
    __in PMEMORY_DESCRIPTOR_LIST DescriptorList
    )

/*++

Routine Description:

    Prints to the console the contents of each memory descriptor in the
    specified memory descriptor list..

Arguments:

    DescriptorList - Supplies the list to print.

Return Value:

    None.

--*/

{

    ULONGLONG EndPage;
    ULONGLONG MappedEndPage;
    PMEMORY_DESCRIPTOR MemoryDescriptor;
    PLIST_ENTRY NextEntry;

    NextEntry = DescriptorList->Head->Flink;
    while (NextEntry != DescriptorList->Head) {
        MemoryDescriptor = CONTAINING_RECORD(NextEntry,
                                             MEMORY_DESCRIPTOR,
                                             ListEntry);

        EndPage = MemoryDescriptor->BasePage + MemoryDescriptor->PageCount;
        if (MemoryDescriptor->MappedBasePage != 0) {
            MappedEndPage = MemoryDescriptor->MappedBasePage +
                            MemoryDescriptor->PageCount;

        } else {
            MappedEndPage = 0;
        }


        //
        // Print [PA BasePage:PA EndPage] [VA BasePage:VA EndPage]:MemoryType:Attributes
        //

        ConsolePrint(L"[%08x%08x:%08x%08x] [%08x%08x:%08x%08x]:%x:%x\r\n",
                     (ULONG)(MemoryDescriptor->BasePage >> 32),
                     (ULONG)MemoryDescriptor->BasePage,
                     (ULONG)(EndPage >> 32),
                     (ULONG)EndPage,
                     (ULONG)(MemoryDescriptor->MappedBasePage >> 32),
                     (ULONG)MemoryDescriptor->MappedBasePage,
                     (ULONG)(MappedEndPage >> 32),
                     (ULONG)MappedEndPage,
                     (ULONG)MemoryDescriptor->MemoryType,
                     (ULONG)MemoryDescriptor->Attributes);

        NextEntry = NextEntry->Flink;
    }
}

NTSTATUS
MmMdValidateList (
    __in PMEMORY_DESCRIPTOR_LIST DescriptorList
    )

/*++

Routine Description:

    Validates a memory descriptor list, asserting that the list does not
    contain any overlapping memory descriptors.

Arguments:

    DescriptorList - Supplies the list to validate.

Return Value:

    STATUS_UNSUCCESSFUL if there are any overlapping descriptors.

    STATUS_SUCCESS when no errors are found.

--*/

{

    ULONGLONG EndPage;
    PMEMORY_DESCRIPTOR MemoryDescriptor;
    PLIST_ENTRY NextEntry;

    //
    // Ensure there are no overlapping descriptor.  This can be done with a
    // single pass through the descriptor list, asserting that the previous
    // descriptor's memory range does not overlap with the current descriptor.
    //
    // N.B. This routine does not check for overlaps in mapped addresses in
    //      physical MDLs, since these addresses can be arbitrary and are not
    //      sorted.
    //

    EndPage = 0;
    NextEntry = DescriptorList->Head->Flink;
    while (NextEntry != DescriptorList->Head) {
        MemoryDescriptor = CONTAINING_RECORD(NextEntry,
                                             MEMORY_DESCRIPTOR,
                                             ListEntry);

        ASSERT(EndPage <= MemoryDescriptor->BasePage);

        if (EndPage > MemoryDescriptor->BasePage){
            return STATUS_UNSUCCESSFUL;
        }

        EndPage = MemoryDescriptor->BasePage + MemoryDescriptor->PageCount;
        NextEntry = NextEntry->Flink;
    }

    return STATUS_SUCCESS;
}

// --------------------------------------------------------- Internal Functions

VOID
MmMdpSaveCurrentListPointer (
    __inout PMEMORY_DESCRIPTOR_LIST Mdl,
    __in PLIST_ENTRY ListEntry
    )

/*++

Routine Description:

    This routine caches the given memory descriptor as the currently
    accessed element of an MDL.  Subsequent MDL searches may then start
    from this descriptor to reduce access time.

Arguments:

    Mdl - Supplies a pointer to the MDL to cache the pointer in.

    ListEntry - Supplies a pointer to the list entry in the MDL corresponding
        to the memory descriptor to be cached.

Return Value:

    None.

--*/

{

    PMEMORY_DESCRIPTOR EndAddress;

    //
    // Do not cache global memory descriptors.  These are reclaimed and freed
    // outside of normal MDL processing, and it is therefore possible to get
    // stale cache entries if these are cached.
    //

    EndAddress = &MmGlobalMemoryDescriptors[MAX_GLOBAL_DESCRIPTOR_COUNT];
    if (((ULONG_PTR)ListEntry >= (ULONG_PTR)MmGlobalMemoryDescriptors) &&
        ((ULONG_PTR)ListEntry < (ULONG_PTR)EndAddress)) {

        return;
    }

    //
    // Do not cache a list head.  This does not correspond to a memory
    // descriptor.
    //

    if (ListEntry != Mdl->Head) {
        Mdl->Current = ListEntry;
    }

    return;
}

VOID
MmMdpSwitchToDynamicDescriptors (
    __in ULONG NumberOfDescriptors
    )

/*++

Routine Description:

    This routine allocates a heap for dynamic descriptors and switches
    the global state to allocate new descriptors out of the dynamic descriptor
    buffer. If the buffer allocation fails, the application continues to
    use the static descriptors.

Arguments:

    NumberOfDescriptors - Supplies the total number of dynamic descriptors
        required.

Return Value:

    None.

--*/

{

    ULONG Size;
    NTSTATUS Status;

    Status = RtlULongMult(NumberOfDescriptors,
                          sizeof(MEMORY_DESCRIPTOR),
                          &Size);

    if (!NT_SUCCESS(Status)) {
        goto SwitchToDynamicDescriptorsEnd;
    }

    MmDynamicMemoryDescriptors = BlMmAllocateHeap(Size);

    //
    // Switch to dynamic descriptors only if the buffer could be allocated.
    //

    if (MmDynamicMemoryDescriptors == NULL) {
        goto SwitchToDynamicDescriptorsEnd;
    }

    //
    // All static descriptors generated during the heap allocation of dynamic
    // descriptor buffer should be first migrated to heap based descriptors.
    //
    // N.B. MmDescriptorCallTreeCount might have changed during the heap
    //      allocation above. But it should be 1 at this point to correspond
    //      with the increment done inside BlMmInitialize.
    //

    if (MmGlobalMemoryDescriptorsUsed != 0) {

        ASSERT(MmDescriptorCallTreeCount == 1);

        MmMdFreeGlobalDescriptors();

        //
        // Ensure that all the descriptors were correctly converted.
        //

        ASSERT((MmDescriptorCallTreeCount == 1) &&
               (MmGlobalMemoryDescriptorsUsed == 0));
    }

    //
    // Switch the application to use the dynamic descriptors.
    //

    MmGlobalMemoryDescriptors = MmDynamicMemoryDescriptors;
    MmDynamicMemoryDescriptorCount = NumberOfDescriptors;
    MmGlobalMemoryDescriptorCount = NumberOfDescriptors;
    RtlZeroMemory(MmGlobalMemoryDescriptors,
                  sizeof(MEMORY_DESCRIPTOR) * MAX_GLOBAL_DESCRIPTOR_COUNT);

    MmGlobalMemoryDescriptorsUsed = 0;

SwitchToDynamicDescriptorsEnd:
    return;
}

NTSTATUS
MmMdpSwitchToStaticDescriptors (
    VOID
    )

/*++

Routine Description:

    This routine switches the application to use static memory descriptors.

Arguments:

    None.

Return Value:

    STATUS_SUCCESS if the application is using only static descriptors.

    Appropriate NTSTATUS code otherwise.

--*/

{

    NTSTATUS Status;

    //
    // Switching to static descriptors is only required if the application is
    // currently using dynamic descriptors.
    //

    if (MmGlobalMemoryDescriptors != MmDynamicMemoryDescriptors) {
        Status = STATUS_SUCCESS;
        goto SwitchToStaticDescriptorsEnd;
    }

    //
    // Memory operations that occur before the attempt to switch to
    // static descriptors might generate new descriptors. All such
    // descriptors will be resident in the dynamic descriptor buffer.
    // Convert such descriptors to heap-based descriptors in preparation
    // to free the dynamic descriptor buffer.
    //

    if (MmGlobalMemoryDescriptorsUsed != 0) {

        ASSERT(MmDescriptorCallTreeCount == 1);

        MmMdFreeGlobalDescriptors();

        //
        // Ensure that all the descriptors were correctly converted.
        //

        ASSERT((MmDescriptorCallTreeCount == 1) &&
               (MmGlobalMemoryDescriptorsUsed == 0));
    }

    //
    // Switch to static descriptors before freeing the dynamic descriptor
    // buffer. This is done to prevent new descriptors from being allocated
    // from the dynamic descriptor buffer that is being freed!
    //

    MmGlobalMemoryDescriptors = MmStaticMemoryDescriptors;
    MmGlobalMemoryDescriptorCount = MAX_STATIC_DESCRIPTOR_COUNT;
    RtlZeroMemory(MmGlobalMemoryDescriptors,
                  sizeof(MEMORY_DESCRIPTOR) * MAX_GLOBAL_DESCRIPTOR_COUNT);

    MmGlobalMemoryDescriptorsUsed = 0;

    //
    // Free the dynamic descriptor buffer.
    //

    Status = BlMmFreeHeap(MmDynamicMemoryDescriptors);

    ASSERT(Status == STATUS_SUCCESS);

    MmDynamicMemoryDescriptors = NULL;
    MmDynamicMemoryDescriptorCount = 0;

SwitchToStaticDescriptorsEnd:
    return Status;
}

BOOLEAN
MmMdpTruncateDescriptor (
    __in PMEMORY_DESCRIPTOR_LIST Mdl,
    __in PMEMORY_DESCRIPTOR MemoryDescriptor,
    __in ULONG InsertionFlags
    )

/*++

Routine Description:

    Truncates any overlapping regions between the specified memory
    descriptor and its adjacent descriptors.  This process is beneficial
    for platforms whose firmware used a different memory page size than
    the Boot Environment Library.  If the granularity of the firmware page
    size is smaller than the OS's, then there could be memory
    described by the firmware that is only partially in a memory region
    described by the OS.  If a firmware stub lets each partial page look
    like a full page, this routine can properly decide which memory type
    should take precedence.

    N.B. If this routine has to split a descriptor, it will use one of the
         descriptors from the global array.  It will be the caller's
         responsibility to free the descriptor, since this routine can be
         called during initialization before the heap allocator has been
         initialized.

Arguments:

    Mdl - Supplies the memory descriptor list containing the given memory
        descriptor.

    MemoryDescriptor - Supplies the memory descriptor to search the MDL for
        overlaps.

    InsertionFlags - Supplies flags to use when inserting split descriptors
        as part of the truncation.

Return Value:

    TRUE if MemoryDescriptor was deleted from list.

--*/

{

    PMEMORY_DESCRIPTOR BlinkDescriptor;
    ULONGLONG BlinkDescriptorEnd;
    ULONGLONG Difference;
    PMEMORY_DESCRIPTOR FlinkDescriptor;
    ULONGLONG FlinkDescriptorEnd;
    PLIST_ENTRY ListBlink;
    PLIST_ENTRY ListFlink;
    ULONGLONG MappedBasePage;
    ULONGLONG MemoryDescriptorEnd;
    PMEMORY_DESCRIPTOR SplitDescriptor;

    //
    // Get the adjacent memory descriptors.
    //

    ListBlink = MemoryDescriptor->ListEntry.Blink;
    ListFlink = MemoryDescriptor->ListEntry.Flink;
    BlinkDescriptor = CONTAINING_RECORD(ListBlink,
                                        MEMORY_DESCRIPTOR,
                                        ListEntry);

    FlinkDescriptor = CONTAINING_RECORD(ListFlink,
                                        MEMORY_DESCRIPTOR,
                                        ListEntry);

    MemoryDescriptorEnd = MemoryDescriptor->BasePage +
                          MemoryDescriptor->PageCount;

    BlinkDescriptorEnd = BlinkDescriptor->BasePage + BlinkDescriptor->PageCount;
    FlinkDescriptorEnd = FlinkDescriptor->BasePage + FlinkDescriptor->PageCount;

    //
    // Check for overlapping regions with previous descriptor.
    //

    if ((ListBlink != Mdl->Head) &&
        (MemoryDescriptor->BasePage < BlinkDescriptorEnd)) {

        //
        // If two descriptors have the same memory type, give precedence to
        // the descriptor with the lowest base page (first arguement to
        // MmMdpHasPrecedence).
        //

        if (MmMdpHasPrecedence(MemoryDescriptor->MemoryType,
                               BlinkDescriptor->MemoryType) != FALSE) {

            //
            // The current memory descriptor has precedence.  Therefore,
            // the overlapping region in the previous descriptor needs to
            // be removed from the previous descriptor.
            //
            // N.B. This operation could potentially split the previous
            //      descriptor into two regions.  The MDL is ordered by
            //      base page.  However, this does not say anything about
            //      the end page.
            //
            // Adjust the end page of the overlapping descriptor.
            //

            BlinkDescriptor->PageCount =
                MemoryDescriptor->BasePage - BlinkDescriptor->BasePage;

            //
            // Split the previous descriptor if necessary.
            //

            if (MemoryDescriptorEnd < BlinkDescriptorEnd) {
                if (BlinkDescriptor->MappedBasePage != 0) {
                    MappedBasePage =
                        BlinkDescriptor->MappedBasePage +
                        MemoryDescriptorEnd -
                        BlinkDescriptor->BasePage;

                } else {
                    MappedBasePage = 0;
                }

                SplitDescriptor =
                    MmMdInitDescriptor(MemoryDescriptorEnd,
                                       MappedBasePage,
                                       BlinkDescriptorEnd - MemoryDescriptorEnd,
                                       BlinkDescriptor->Attributes,
                                       BlinkDescriptor->MemoryType);

                if (SplitDescriptor != NULL) {
                    MmMdAddDescriptorToList(Mdl,
                                            SplitDescriptor,
                                            InsertionFlags);
                }
            }

            //
            // Remove the previous descriptor from the list if the current
            // descriptor overlaps the entire region.
            //

            if (BlinkDescriptor->PageCount == 0) {
                MmMdRemoveDescriptorFromList(Mdl, BlinkDescriptor);
                MmMdFreeDescriptor(BlinkDescriptor);
            }

        } else {

            //
            // The previous descriptor has precedence.  The overlapping
            // region in the current memory descriptor needs to be removed.
            //

            if (MemoryDescriptorEnd <= BlinkDescriptorEnd) {

                //
                // If the previous descriptor completely overlaps with the
                // current descriptor, remove it from the MDL.
                //

                MmMdRemoveDescriptorFromList(Mdl, MemoryDescriptor);
                MmMdFreeDescriptor(MemoryDescriptor);

                //
                // There is no reason to fall through to see if the current
                // descriptor can be truncated with the next descriptor.
                // Return now.
                //

                return TRUE;
            }

            //
            // Adjust start page for current descriptor.
            //

            Difference = BlinkDescriptorEnd - MemoryDescriptor->BasePage;
            MemoryDescriptor->BasePage += Difference;
            MemoryDescriptor->PageCount -= Difference;
            if (MemoryDescriptor->MappedBasePage != 0) {
                MemoryDescriptor->MappedBasePage += Difference;
            }
        }
    }

    //
    // Check for overlapping region with next descriptor.
    //

    if ((ListFlink != Mdl->Head) &&
        (FlinkDescriptor->BasePage < MemoryDescriptorEnd)) {

        //
        // If two descriptors have the same memory type, give precedence to
        // the descriptor with the lowest base page (first argument to
        // MmMdpHasPrecedence).
        //

        if (MmMdpHasPrecedence(FlinkDescriptor->MemoryType,
                               MemoryDescriptor->MemoryType) != FALSE) {

            //
            // The next memory descriptor has precedence.  Therefore, the
            // overlapping region in the current memory descriptor needs
            // to be removed.
            //
            // N.B. This operation could potentially split the previous
            //      descriptor into two regions.  The MDL is ordered by
            //      base page.  However, this does not say anything about
            //      the end page.
            //
            // Adjust the end page of the current descriptor.
            //

            MemoryDescriptor->PageCount =
                FlinkDescriptor->BasePage - MemoryDescriptor->BasePage;

            //
            // Split the current descriptor if necessary.
            //

            if (FlinkDescriptorEnd < MemoryDescriptorEnd) {
                if (MemoryDescriptor->MappedBasePage != 0) {
                    MappedBasePage =
                        MemoryDescriptor->MappedBasePage +
                        FlinkDescriptorEnd -
                        MemoryDescriptor->BasePage;

                } else {
                    MappedBasePage = 0;
                }

                SplitDescriptor =
                    MmMdInitDescriptor(FlinkDescriptorEnd,
                                       MappedBasePage,
                                       MemoryDescriptorEnd - FlinkDescriptorEnd,
                                       MemoryDescriptor->Attributes,
                                       MemoryDescriptor->MemoryType);

                if (SplitDescriptor != NULL) {
                    MmMdAddDescriptorToList(Mdl, SplitDescriptor, InsertionFlags);
                }
            }

            //
            // Remove the current descriptor from the list if the next
            // descriptor completely overlaps with the current descriptor.
            //

            if (MemoryDescriptor->PageCount == 0) {
                MmMdRemoveDescriptorFromList(Mdl, MemoryDescriptor);
                MmMdFreeDescriptor(MemoryDescriptor);

                //
                // The current descriptor has been deleted.  Return this
                // status to the caller.
                //

                return TRUE;
            }

        } else {

            //
            // The current memory descriptor has precedence.  Remove the
            // overlapping region in the next descriptor.
            //

            if (FlinkDescriptorEnd <= MemoryDescriptorEnd) {

                //
                // If the next descriptor completely overlaps with the current
                // descriptor, remove it from the MDL.
                //

                MmMdRemoveDescriptorFromList(Mdl, FlinkDescriptor);
                MmMdFreeDescriptor(FlinkDescriptor);

            } else {

                //
                // Adjust the start page for the next descriptor.
                //

                Difference = MemoryDescriptorEnd - FlinkDescriptor->BasePage;
                FlinkDescriptor->BasePage += Difference;
                FlinkDescriptor->PageCount -= Difference;
                if (FlinkDescriptor->MappedBasePage != 0) {
                    FlinkDescriptor->MappedBasePage += Difference;
                }
            }
        }
    }

    return FALSE;
}

BOOLEAN
MmMdpCoalesceDescriptor (
    __in PMEMORY_DESCRIPTOR_LIST Mdl,
    __in PMEMORY_DESCRIPTOR MemoryDescriptor,
    __in ULONG InsertionFlags
    )

/*++

Routine Description:

    Attempts to coalesce the specified memory descriptor with its adjacent
    descriptors.  Coalescing will occur if and only if the memory descriptor
    shares a memory boundary with an adjacent descriptor and the two also
    share the same memory type.

Arguments:

    Mdl - Supplies a memory descriptor list containing the given descriptor.

    MemoryDescriptor - Supplies the memory descriptor  to attempt to coalesce
        with adjacent descriptors.

    InsertionFlags - Insertion flags.  Not currently used.

Return Value:

    TRUE if MemoryDescriptor was deleted from list as a result of being
    coalesced with an adjacent descriptor.
    FALSE otherwise.

--*/

{

    PMEMORY_DESCRIPTOR BlinkDescriptor;
    BOOLEAN CoalescedDescriptor;
    PMEMORY_DESCRIPTOR FlinkDescriptor;
    PLIST_ENTRY ListBlink;
    PLIST_ENTRY ListFlink;

    UNREFERENCED_PARAMETER(InsertionFlags);

    CoalescedDescriptor = FALSE;

    //
    // Get adjacent descriptors.
    //

    ListBlink = MemoryDescriptor->ListEntry.Blink;
    ListFlink = MemoryDescriptor->ListEntry.Flink;
    BlinkDescriptor = CONTAINING_RECORD(ListBlink,
                                        MEMORY_DESCRIPTOR,
                                        ListEntry);

    FlinkDescriptor = CONTAINING_RECORD(ListFlink,
                                        MEMORY_DESCRIPTOR,
                                        ListEntry);

    //
    // Attempt to coalesce with previous descriptor.
    //

    if ((ListBlink != Mdl->Head) &&

        //
        // In order to coalesce, the newly inserted descriptor and the
        // previous descriptor must have the same memory type.
        //

        (BlinkDescriptor->MemoryType == MemoryDescriptor->MemoryType) &&

        //
        // Memory ranges with different attributes must remain partitioned
        // and can not be coalesced.
        //

        (DO_COALESCE_ATTRIBUTES_MATCH(BlinkDescriptor->Attributes,
                                      MemoryDescriptor->Attributes) != FALSE) &&

        //
        // And descriptors must describe adjacent memory regions (both
        // physically and virtually).
        //

        ((BlinkDescriptor->BasePage + BlinkDescriptor->PageCount) ==
         MemoryDescriptor->BasePage) &&

        //
        // Only compare for mapped address adjacency if both regions are mapped.
        //

        (((MemoryDescriptor->MappedBasePage == 0) &&
          (BlinkDescriptor->MappedBasePage == 0))

         ||

         ((MemoryDescriptor->MappedBasePage != 0) &&
          (BlinkDescriptor->MappedBasePage != 0) &&
          (BlinkDescriptor->MappedBasePage + BlinkDescriptor->PageCount ==
           MemoryDescriptor->MappedBasePage)))) {

        //
        // The specified memory descriptor can be coalesced with the
        // previous descriptor.
        //

        BlinkDescriptor->PageCount += MemoryDescriptor->PageCount;
        MmMdRemoveDescriptorFromList(Mdl, MemoryDescriptor);
        MmMdFreeDescriptor(MemoryDescriptor);

        //
        // Fall through to next case, using the newly formed BlinkDescriptor.
        //

        CoalescedDescriptor = TRUE;
        MemoryDescriptor = BlinkDescriptor;
    }

    //
    // Attempt to coalesce with next descriptor.
    //

    if ((ListFlink != Mdl->Head) &&

        //
        // In order to coalesce, the newly inserted descriptor and the
        // next descriptor must have the same memory type.
        //

        (FlinkDescriptor->MemoryType == MemoryDescriptor->MemoryType) &&

        //
        // Memory ranges with different attributes must remain partitioned
        // and can not be coalesced.
        //

        (DO_COALESCE_ATTRIBUTES_MATCH(FlinkDescriptor->Attributes,
                                      MemoryDescriptor->Attributes) != FALSE) &&

        //
        // And descriptors must describe adjacent memory regions (both
        // physically and virtually).
        //

        ((MemoryDescriptor->BasePage + MemoryDescriptor->PageCount) ==
         FlinkDescriptor->BasePage) &&

        //
        // Only compare for virtual adjacency if both regions are virtually
        // mapped.
        //

        (((MemoryDescriptor->MappedBasePage == 0) &&
          (FlinkDescriptor->MappedBasePage == 0))

         ||

         ((MemoryDescriptor->MappedBasePage != 0) &&
          (FlinkDescriptor->MappedBasePage != 0) &&
          ((MemoryDescriptor->MappedBasePage + MemoryDescriptor->PageCount) ==
           FlinkDescriptor->MappedBasePage)))) {

        //
        // The specified memory descriptor can be coalesced with the
        // next descriptor.
        //

        FlinkDescriptor->BasePage = MemoryDescriptor->BasePage;
        FlinkDescriptor->MappedBasePage  = MemoryDescriptor->MappedBasePage;
        FlinkDescriptor->PageCount += MemoryDescriptor->PageCount;
        MmMdRemoveDescriptorFromList(Mdl, MemoryDescriptor);
        MmMdFreeDescriptor(MemoryDescriptor);
        CoalescedDescriptor = TRUE;
    }

    return CoalescedDescriptor;
}

BOOLEAN
MmMdpHasPrecedence (
    __in MEMORY_TYPE Type1,
    __in MEMORY_TYPE Type2
    )

/*++

Routine Description:

    Determines if a memory region with Type1 memory has precedence over a
    memory region with Type2 memory.  Precedence rules are as follows:

    - Free memory has the lowest precedence.
    - In use, platform memory has highest precedence.
    - Library memory has precendence over application memory.

Arguments:

    Type1 - First memory type to compare.

    Type2 - Second memory type to compare.

Return Value:

    TRUE if Memory type 1 has precedence over Memory type 2.
    Tie goes to Type1.

--*/

{

    ULONG PrecedenceIndex1;
    ULONG PrecedenceIndex2;
    UNPACKED_MEMORY_TYPE UnpackedType1;
    UNPACKED_MEMORY_TYPE UnpackedType2;

    //
    // Take a shortcut in case if one of types is 'free' or 'zeroed'
    // as this types have the lowest precedence.
    //

    if (Type2 == MEMORY_TYPE_FREE_ZEROED) {
        return TRUE;

    } else if (Type1 == MEMORY_TYPE_FREE_ZEROED) {
        return FALSE;
    }

    if (Type2 == MEMORY_TYPE_FREE) {
        return TRUE;

    } else if (Type1 == MEMORY_TYPE_FREE) {
        return FALSE;
    }

    //
    // If either memory type is an unknown class, assume it has precedence.
    //

    UnpackedType1.MemoryType = Type1;
    UnpackedType2.MemoryType = Type2;
    if ((UnpackedType1.Class != MEMORY_CLASS_PLATFORM) &&
        (UnpackedType1.Class != MEMORY_CLASS_LIBRARY) &&
        (UnpackedType1.Class != MEMORY_CLASS_APPLICATION)) {

        return TRUE;
    }

    if ((UnpackedType2.Class != MEMORY_CLASS_PLATFORM) &&
        (UnpackedType2.Class != MEMORY_CLASS_LIBRARY) &&
        (UnpackedType2.Class != MEMORY_CLASS_APPLICATION)) {

        return FALSE;
    }

    //
    // Both memory types are of known memory class.  Compare each and determine
    // which has precendence.  If both memory types have the same memory class,
    // precendence goes to Type1.
    //

    if (UnpackedType1.Class == MEMORY_CLASS_PLATFORM) {
        if (UnpackedType2.Class != MEMORY_CLASS_PLATFORM) {
            return TRUE;

        } else {

            //
            // Both memory types are of platform class.  Finer grained
            // comparison needs to be done based on the specific types in
            // question.  Types with lower precedence indices carry
            // higher precedence.  Ties go to Type1.
            //

            PrecedenceIndex1 = MmMdpLookupTypePrecedenceIndex(Type1);
            PrecedenceIndex2 = MmMdpLookupTypePrecedenceIndex(Type2);
            if (PrecedenceIndex1 == MM_MAXIMAL_PRECEDENCE) {
                return TRUE;

            } else if (PrecedenceIndex2 == MM_MAXIMAL_PRECEDENCE) {
                return FALSE;

            } else {
                if (PrecedenceIndex1 <= PrecedenceIndex2) {
                    return TRUE;

                } else {
                    return FALSE;
                }
            }
        }

    } else if (UnpackedType2.Class == MEMORY_CLASS_PLATFORM) {
        return FALSE;
    }

    if (UnpackedType1.Class == MEMORY_CLASS_LIBRARY) {
        return TRUE;

    } else if (UnpackedType2.Class == MEMORY_CLASS_LIBRARY) {
        return FALSE;
    }

    //
    // Both are of class, MEMORY_CLASS_APPLICATION.  Precedence goes to Type1.
    //

    return TRUE;
}

ULONG
MmMdpLookupTypePrecedenceIndex (
    __in MEMORY_TYPE Type
    )

/*++

Routine Description:

    This routine determines the index of the given memory type within the
    platform memory type precedence list.

Arguments:

    Type - Supplies the type to look up in the precedence list.

Return Value:

    The index of the given memory type within the platform type precedence
    list or MM_MAXIMAL_PRECEDENCE if the given type is not listed and should
    thus be given maximum precedence.

--*/

{

    MEMORY_TYPE CurrentType;
    ULONG Index;

    for (Index = 0;
         Index < RTL_NUMBER_OF(MmPlatformMemoryTypePrecedence);
         Index += 1) {

        CurrentType = MmPlatformMemoryTypePrecedence[Index];
        if (CurrentType == Type) {
            return Index;
        }
    }

    return MM_MAXIMAL_PRECEDENCE;
}

#include "Test_Include.fx"

int Test_Count = 3022;

string TestInfo
<
	string TestType = "PS";
>
= "ps_2_x_inst_texldd";

PS_CRITERIA PS_001_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_001_Desc = "ps_2_x : texldd source reg combination r0, s0, v0, r1 is allowed";
string PS_001 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_002_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_002_Desc = "ps_2_x : texldd source reg combination r0, s0, v0, c0 is allowed";
string PS_002 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, v0, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_003_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_003_Desc = "ps_2_x : texldd source reg combination r0, s0, r1, v0 is allowed";
string PS_003 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, r1, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_004_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_004_Desc = "ps_2_x : texldd source reg combination r0, s0, r1, r2 is allowed";
string PS_004 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"mov r2, c0 "
	"texldd r0, r0, s0, r1, r2 "
	"mov oC0, c0 ";

PS_CRITERIA PS_005_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_005_Desc = "ps_2_x : texldd source reg combination r0, s0, r1, c0 is allowed";
string PS_005 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, r1, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_006_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_006_Desc = "ps_2_x : texldd source reg combination r0, s0, r1, t0 is allowed";
string PS_006 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, r1, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_007_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_007_Desc = "ps_2_x : texldd source reg combination r0, s0, c0, v0 is allowed";
string PS_007 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, c0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_008_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_008_Desc = "ps_2_x : texldd source reg combination r0, s0, c0, r1 is allowed";
string PS_008 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, c0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_009_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_009_Desc = "ps_2_x : texldd source reg combination r0, s0, c0, t0 is allowed";
string PS_009 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"dcl t0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, c0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_010_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_010_Desc = "ps_2_x : texldd source reg combination r0, s0, t0, r1 is allowed";
string PS_010 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, t0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_011_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_011_Desc = "ps_2_x : texldd source reg combination r0, s0, t0, c0 is allowed";
string PS_011 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"dcl t0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, t0, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_012_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_012_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1 is allowed";
string PS_012 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_013_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_013_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, c0 is allowed";
string PS_013 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, r0, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_014_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_014_Desc = "ps_2_x : texldd source reg combination t0, s0, c0, r0 is allowed";
string PS_014 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, c0, r0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_015_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_015_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1 is allowed";
string PS_015 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_016_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_016_Desc = "ps_2_x : texldd source reg combination r0, s0, v0, v1 is NOT allowed";
string PS_016 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"dcl v1 "
	"mov r0, c0 "
	"texldd r0, r0, s0, v0, v1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_017_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_017_Desc = "ps_2_x : texldd source reg combination r0, s0, v0, t0 is NOT allowed";
string PS_017 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"dcl t0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, v0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_018_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_018_Desc = "ps_2_x : texldd source reg combination r0, s0, c0, c1 is NOT allowed";
string PS_018 = 
	"ps_2_x "
	"def c1, 1, 1, 1, 1 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, c0, c1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_019_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_019_Desc = "ps_2_x : texldd source reg combination r0, s0, t0, v0 is NOT allowed";
string PS_019 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, t0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_020_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_020_Desc = "ps_2_x : texldd source reg combination r0, s0, t0, t0 is NOT allowed";
string PS_020 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"mov r0, c0 "
	"texldd r0, r0, s0, t0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_021_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_021_Desc = "ps_2_x : texldd source reg combination r0, s0, t0, t1 is NOT allowed";
string PS_021 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"dcl t1 "
	"mov r0, c0 "
	"texldd r0, r0, s0, t0, t1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_022_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_022_Desc = "ps_2_x : texldd source reg combination t0, s0, v0, v0 is NOT allowed";
string PS_022 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, t0, s0, v0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_023_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_023_Desc = "ps_2_x : texldd source reg combination t0, s0, v0, r0 is NOT allowed";
string PS_023 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, v0, r0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_024_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_024_Desc = "ps_2_x : texldd source reg combination t0, s0, v0, c0 is NOT allowed";
string PS_024 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, t0, s0, v0, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_025_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_025_Desc = "ps_2_x : texldd source reg combination t0, s0, v0, t0 is NOT allowed";
string PS_025 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, t0, s0, v0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_026_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_026_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, v0 is NOT allowed";
string PS_026 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, r0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_027_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_027_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, t0 is NOT allowed";
string PS_027 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, r0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_028_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_028_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, t1 is NOT allowed";
string PS_028 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl t1 "
	"mov r0, c0 "
	"texldd r0, t0, s0, r0, t1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_029_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_029_Desc = "ps_2_x : texldd source reg combination t0, s0, c0, v0 is NOT allowed";
string PS_029 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, t0, s0, c0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_030_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_030_Desc = "ps_2_x : texldd source reg combination t0, s0, c0, c1 is NOT allowed";
string PS_030 = 
	"ps_2_x "
	"def c1, 1, 1, 1, 1 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, t0, s0, c0, c1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_031_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_031_Desc = "ps_2_x : texldd source reg combination t0, s0, c0, t0 is NOT allowed";
string PS_031 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, t0, s0, c0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_032_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_032_Desc = "ps_2_x : texldd source reg combination t0, s0, c0, t1 is NOT allowed";
string PS_032 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl t1 "
	"texldd r0, t0, s0, c0, t1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_033_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_033_Desc = "ps_2_x : texldd source reg combination t0, s0, t0, v0 is NOT allowed";
string PS_033 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, t0, s0, t0, v0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_034_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_034_Desc = "ps_2_x : texldd source reg combination t0, s0, t0, r0 is NOT allowed";
string PS_034 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, t0, s0, t0, r0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_035_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_035_Desc = "ps_2_x : texldd source reg combination t0, s0, t1, r0 is NOT allowed";
string PS_035 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl t1 "
	"mov r0, c0 "
	"texldd r0, t0, s0, t1, r0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_036_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_036_Desc = "ps_2_x : texldd source reg combination t0, s0, t0, c0 is NOT allowed";
string PS_036 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, t0, s0, t0, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_037_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_037_Desc = "ps_2_x : texldd source reg combination t0, s0, t1, c0 is NOT allowed";
string PS_037 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl t1 "
	"texldd r0, t0, s0, t1, c0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_038_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_038_Desc = "ps_2_x : texldd source reg combination t0, s0, t0, t0 is NOT allowed";
string PS_038 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, t0, s0, t0, t0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_039_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_039_Desc = "ps_2_x : texldd source reg combination t0, s0, t1, t2 is NOT allowed";
string PS_039 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"dcl t1 "
	"dcl t2 "
	"texldd r0, t0, s0, t1, t2 "
	"mov oC0, c0 ";

PS_CRITERIA PS_040_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_040_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, b0 is NOT allowed";
string PS_040 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_041_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_041_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, i0 is NOT allowed";
string PS_041 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_042_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_042_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, p0 is NOT allowed";
string PS_042 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_043_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_043_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, s0 is NOT allowed";
string PS_043 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_044_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_044_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, oC0 is NOT allowed";
string PS_044 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_045_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_045_Desc = "ps_2_x : texldd source reg combination v0, v0, b0, oDepth is NOT allowed";
string PS_045 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_046_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_046_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, b0 is NOT allowed";
string PS_046 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_047_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_047_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, i0 is NOT allowed";
string PS_047 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_048_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_048_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, p0 is NOT allowed";
string PS_048 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_049_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_049_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, s0 is NOT allowed";
string PS_049 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_050_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_050_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, oC0 is NOT allowed";
string PS_050 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_051_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_051_Desc = "ps_2_x : texldd source reg combination v0, v0, i0, oDepth is NOT allowed";
string PS_051 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_052_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_052_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, b0 is NOT allowed";
string PS_052 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_053_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_053_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, i0 is NOT allowed";
string PS_053 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_054_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_054_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, p0 is NOT allowed";
string PS_054 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_055_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_055_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, s0 is NOT allowed";
string PS_055 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_056_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_056_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, oC0 is NOT allowed";
string PS_056 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_057_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_057_Desc = "ps_2_x : texldd source reg combination v0, v0, p0, oDepth is NOT allowed";
string PS_057 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_058_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_058_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, b0 is NOT allowed";
string PS_058 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_059_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_059_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, i0 is NOT allowed";
string PS_059 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_060_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_060_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, p0 is NOT allowed";
string PS_060 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_061_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_061_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, s0 is NOT allowed";
string PS_061 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_062_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_062_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, oC0 is NOT allowed";
string PS_062 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_063_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_063_Desc = "ps_2_x : texldd source reg combination v0, v0, s0, oDepth is NOT allowed";
string PS_063 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_064_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_064_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, b0 is NOT allowed";
string PS_064 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_065_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_065_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, i0 is NOT allowed";
string PS_065 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_066_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_066_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, p0 is NOT allowed";
string PS_066 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_067_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_067_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, s0 is NOT allowed";
string PS_067 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_068_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_068_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, oC0 is NOT allowed";
string PS_068 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_069_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_069_Desc = "ps_2_x : texldd source reg combination v0, v0, oC0, oDepth is NOT allowed";
string PS_069 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_070_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_070_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, b0 is NOT allowed";
string PS_070 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_071_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_071_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, i0 is NOT allowed";
string PS_071 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_072_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_072_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, p0 is NOT allowed";
string PS_072 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_073_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_073_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, s0 is NOT allowed";
string PS_073 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_074_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_074_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, oC0 is NOT allowed";
string PS_074 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_075_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_075_Desc = "ps_2_x : texldd source reg combination v0, v0, oDepth, oDepth is NOT allowed";
string PS_075 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_076_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_076_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, b0 is NOT allowed";
string PS_076 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_077_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_077_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, i0 is NOT allowed";
string PS_077 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_078_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_078_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, p0 is NOT allowed";
string PS_078 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_079_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_079_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, s0 is NOT allowed";
string PS_079 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_080_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_080_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, oC0 is NOT allowed";
string PS_080 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_081_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_081_Desc = "ps_2_x : texldd source reg combination v0, r0, b0, oDepth is NOT allowed";
string PS_081 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_082_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_082_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, b0 is NOT allowed";
string PS_082 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_083_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_083_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, i0 is NOT allowed";
string PS_083 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_084_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_084_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, p0 is NOT allowed";
string PS_084 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_085_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_085_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, s0 is NOT allowed";
string PS_085 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_086_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_086_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, oC0 is NOT allowed";
string PS_086 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_087_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_087_Desc = "ps_2_x : texldd source reg combination v0, r0, i0, oDepth is NOT allowed";
string PS_087 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_088_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_088_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, b0 is NOT allowed";
string PS_088 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_089_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_089_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, i0 is NOT allowed";
string PS_089 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_090_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_090_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, p0 is NOT allowed";
string PS_090 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_091_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_091_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, s0 is NOT allowed";
string PS_091 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_092_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_092_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, oC0 is NOT allowed";
string PS_092 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_093_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_093_Desc = "ps_2_x : texldd source reg combination v0, r0, p0, oDepth is NOT allowed";
string PS_093 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_094_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_094_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, b0 is NOT allowed";
string PS_094 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_095_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_095_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, i0 is NOT allowed";
string PS_095 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_096_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_096_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, p0 is NOT allowed";
string PS_096 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_097_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_097_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, s0 is NOT allowed";
string PS_097 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_098_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_098_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, oC0 is NOT allowed";
string PS_098 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_099_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_099_Desc = "ps_2_x : texldd source reg combination v0, r0, s0, oDepth is NOT allowed";
string PS_099 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_100_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_100_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, b0 is NOT allowed";
string PS_100 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_101_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_101_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, i0 is NOT allowed";
string PS_101 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_102_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_102_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, p0 is NOT allowed";
string PS_102 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_103_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_103_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, s0 is NOT allowed";
string PS_103 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_104_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_104_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, oC0 is NOT allowed";
string PS_104 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_105_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_105_Desc = "ps_2_x : texldd source reg combination v0, r0, oC0, oDepth is NOT allowed";
string PS_105 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_106_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_106_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, b0 is NOT allowed";
string PS_106 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_107_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_107_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, i0 is NOT allowed";
string PS_107 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_108_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_108_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, p0 is NOT allowed";
string PS_108 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_109_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_109_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, s0 is NOT allowed";
string PS_109 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_110_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_110_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, oC0 is NOT allowed";
string PS_110 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_111_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_111_Desc = "ps_2_x : texldd source reg combination v0, r0, oDepth, oDepth is NOT allowed";
string PS_111 = 
	"ps_2_x "
	"dcl v0 "
	"mov r0, c0 "
	"texldd r0, v0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_112_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_112_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, b0 is NOT allowed";
string PS_112 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_113_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_113_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, i0 is NOT allowed";
string PS_113 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_114_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_114_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, p0 is NOT allowed";
string PS_114 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_115_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_115_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, s0 is NOT allowed";
string PS_115 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_116_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_116_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, oC0 is NOT allowed";
string PS_116 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_117_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_117_Desc = "ps_2_x : texldd source reg combination v0, c0, b0, oDepth is NOT allowed";
string PS_117 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_118_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_118_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, b0 is NOT allowed";
string PS_118 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_119_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_119_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, i0 is NOT allowed";
string PS_119 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_120_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_120_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, p0 is NOT allowed";
string PS_120 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_121_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_121_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, s0 is NOT allowed";
string PS_121 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_122_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_122_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, oC0 is NOT allowed";
string PS_122 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_123_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_123_Desc = "ps_2_x : texldd source reg combination v0, c0, i0, oDepth is NOT allowed";
string PS_123 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_124_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_124_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, b0 is NOT allowed";
string PS_124 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_125_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_125_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, i0 is NOT allowed";
string PS_125 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_126_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_126_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, p0 is NOT allowed";
string PS_126 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_127_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_127_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, s0 is NOT allowed";
string PS_127 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_128_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_128_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, oC0 is NOT allowed";
string PS_128 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_129_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_129_Desc = "ps_2_x : texldd source reg combination v0, c0, p0, oDepth is NOT allowed";
string PS_129 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_130_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_130_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, b0 is NOT allowed";
string PS_130 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_131_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_131_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, i0 is NOT allowed";
string PS_131 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_132_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_132_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, p0 is NOT allowed";
string PS_132 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_133_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_133_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, s0 is NOT allowed";
string PS_133 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_134_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_134_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, oC0 is NOT allowed";
string PS_134 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_135_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_135_Desc = "ps_2_x : texldd source reg combination v0, c0, s0, oDepth is NOT allowed";
string PS_135 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_136_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_136_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, b0 is NOT allowed";
string PS_136 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_137_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_137_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, i0 is NOT allowed";
string PS_137 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_138_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_138_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, p0 is NOT allowed";
string PS_138 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_139_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_139_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, s0 is NOT allowed";
string PS_139 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_140_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_140_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, oC0 is NOT allowed";
string PS_140 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_141_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_141_Desc = "ps_2_x : texldd source reg combination v0, c0, oC0, oDepth is NOT allowed";
string PS_141 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_142_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_142_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, b0 is NOT allowed";
string PS_142 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_143_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_143_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, i0 is NOT allowed";
string PS_143 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_144_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_144_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, p0 is NOT allowed";
string PS_144 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_145_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_145_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, s0 is NOT allowed";
string PS_145 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_146_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_146_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, oC0 is NOT allowed";
string PS_146 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_147_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_147_Desc = "ps_2_x : texldd source reg combination v0, c0, oDepth, oDepth is NOT allowed";
string PS_147 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, v0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_148_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_148_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, b0 is NOT allowed";
string PS_148 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_149_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_149_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, i0 is NOT allowed";
string PS_149 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_150_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_150_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, p0 is NOT allowed";
string PS_150 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_151_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_151_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, s0 is NOT allowed";
string PS_151 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_152_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_152_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, oC0 is NOT allowed";
string PS_152 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_153_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_153_Desc = "ps_2_x : texldd source reg combination v0, b0, b0, oDepth is NOT allowed";
string PS_153 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_154_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_154_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, b0 is NOT allowed";
string PS_154 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_155_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_155_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, i0 is NOT allowed";
string PS_155 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_156_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_156_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, p0 is NOT allowed";
string PS_156 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_157_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_157_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, s0 is NOT allowed";
string PS_157 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_158_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_158_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, oC0 is NOT allowed";
string PS_158 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_159_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_159_Desc = "ps_2_x : texldd source reg combination v0, b0, i0, oDepth is NOT allowed";
string PS_159 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_160_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_160_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, b0 is NOT allowed";
string PS_160 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_161_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_161_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, i0 is NOT allowed";
string PS_161 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_162_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_162_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, p0 is NOT allowed";
string PS_162 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_163_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_163_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, s0 is NOT allowed";
string PS_163 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_164_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_164_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, oC0 is NOT allowed";
string PS_164 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_165_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_165_Desc = "ps_2_x : texldd source reg combination v0, b0, p0, oDepth is NOT allowed";
string PS_165 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_166_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_166_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, b0 is NOT allowed";
string PS_166 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_167_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_167_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, i0 is NOT allowed";
string PS_167 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_168_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_168_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, p0 is NOT allowed";
string PS_168 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_169_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_169_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, s0 is NOT allowed";
string PS_169 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_170_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_170_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, oC0 is NOT allowed";
string PS_170 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_171_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_171_Desc = "ps_2_x : texldd source reg combination v0, b0, s0, oDepth is NOT allowed";
string PS_171 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_172_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_172_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, b0 is NOT allowed";
string PS_172 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_173_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_173_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, i0 is NOT allowed";
string PS_173 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_174_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_174_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, p0 is NOT allowed";
string PS_174 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_175_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_175_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, s0 is NOT allowed";
string PS_175 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_176_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_176_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, oC0 is NOT allowed";
string PS_176 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_177_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_177_Desc = "ps_2_x : texldd source reg combination v0, b0, oC0, oDepth is NOT allowed";
string PS_177 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_178_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_178_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, b0 is NOT allowed";
string PS_178 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_179_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_179_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, i0 is NOT allowed";
string PS_179 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_180_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_180_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, p0 is NOT allowed";
string PS_180 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_181_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_181_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, s0 is NOT allowed";
string PS_181 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_182_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_182_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, oC0 is NOT allowed";
string PS_182 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_183_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_183_Desc = "ps_2_x : texldd source reg combination v0, b0, oDepth, oDepth is NOT allowed";
string PS_183 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_184_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_184_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, b0 is NOT allowed";
string PS_184 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_185_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_185_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, i0 is NOT allowed";
string PS_185 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_186_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_186_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, p0 is NOT allowed";
string PS_186 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_187_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_187_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, s0 is NOT allowed";
string PS_187 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_188_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_188_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, oC0 is NOT allowed";
string PS_188 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_189_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_189_Desc = "ps_2_x : texldd source reg combination v0, i0, b0, oDepth is NOT allowed";
string PS_189 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_190_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_190_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, b0 is NOT allowed";
string PS_190 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_191_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_191_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, i0 is NOT allowed";
string PS_191 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_192_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_192_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, p0 is NOT allowed";
string PS_192 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_193_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_193_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, s0 is NOT allowed";
string PS_193 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_194_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_194_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, oC0 is NOT allowed";
string PS_194 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_195_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_195_Desc = "ps_2_x : texldd source reg combination v0, i0, i0, oDepth is NOT allowed";
string PS_195 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_196_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_196_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, b0 is NOT allowed";
string PS_196 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_197_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_197_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, i0 is NOT allowed";
string PS_197 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_198_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_198_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, p0 is NOT allowed";
string PS_198 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_199_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_199_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, s0 is NOT allowed";
string PS_199 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_200_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_200_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, oC0 is NOT allowed";
string PS_200 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_201_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_201_Desc = "ps_2_x : texldd source reg combination v0, i0, p0, oDepth is NOT allowed";
string PS_201 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_202_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_202_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, b0 is NOT allowed";
string PS_202 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_203_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_203_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, i0 is NOT allowed";
string PS_203 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_204_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_204_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, p0 is NOT allowed";
string PS_204 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_205_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_205_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, s0 is NOT allowed";
string PS_205 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_206_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_206_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, oC0 is NOT allowed";
string PS_206 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_207_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_207_Desc = "ps_2_x : texldd source reg combination v0, i0, s0, oDepth is NOT allowed";
string PS_207 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_208_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_208_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, b0 is NOT allowed";
string PS_208 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_209_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_209_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, i0 is NOT allowed";
string PS_209 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_210_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_210_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, p0 is NOT allowed";
string PS_210 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_211_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_211_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, s0 is NOT allowed";
string PS_211 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_212_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_212_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, oC0 is NOT allowed";
string PS_212 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_213_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_213_Desc = "ps_2_x : texldd source reg combination v0, i0, oC0, oDepth is NOT allowed";
string PS_213 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_214_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_214_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, b0 is NOT allowed";
string PS_214 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_215_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_215_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, i0 is NOT allowed";
string PS_215 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_216_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_216_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, p0 is NOT allowed";
string PS_216 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_217_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_217_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, s0 is NOT allowed";
string PS_217 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_218_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_218_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, oC0 is NOT allowed";
string PS_218 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_219_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_219_Desc = "ps_2_x : texldd source reg combination v0, i0, oDepth, oDepth is NOT allowed";
string PS_219 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_220_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_220_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, b0 is NOT allowed";
string PS_220 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_221_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_221_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, i0 is NOT allowed";
string PS_221 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_222_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_222_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, p0 is NOT allowed";
string PS_222 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_223_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_223_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, s0 is NOT allowed";
string PS_223 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_224_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_224_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, oC0 is NOT allowed";
string PS_224 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_225_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_225_Desc = "ps_2_x : texldd source reg combination v0, p0, b0, oDepth is NOT allowed";
string PS_225 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_226_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_226_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, b0 is NOT allowed";
string PS_226 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_227_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_227_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, i0 is NOT allowed";
string PS_227 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_228_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_228_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, p0 is NOT allowed";
string PS_228 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_229_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_229_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, s0 is NOT allowed";
string PS_229 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_230_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_230_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, oC0 is NOT allowed";
string PS_230 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_231_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_231_Desc = "ps_2_x : texldd source reg combination v0, p0, i0, oDepth is NOT allowed";
string PS_231 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_232_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_232_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, b0 is NOT allowed";
string PS_232 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_233_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_233_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, i0 is NOT allowed";
string PS_233 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_234_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_234_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, p0 is NOT allowed";
string PS_234 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_235_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_235_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, s0 is NOT allowed";
string PS_235 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_236_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_236_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, oC0 is NOT allowed";
string PS_236 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_237_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_237_Desc = "ps_2_x : texldd source reg combination v0, p0, p0, oDepth is NOT allowed";
string PS_237 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_238_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_238_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, b0 is NOT allowed";
string PS_238 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_239_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_239_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, i0 is NOT allowed";
string PS_239 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_240_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_240_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, p0 is NOT allowed";
string PS_240 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_241_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_241_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, s0 is NOT allowed";
string PS_241 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_242_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_242_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, oC0 is NOT allowed";
string PS_242 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_243_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_243_Desc = "ps_2_x : texldd source reg combination v0, p0, s0, oDepth is NOT allowed";
string PS_243 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_244_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_244_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, b0 is NOT allowed";
string PS_244 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_245_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_245_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, i0 is NOT allowed";
string PS_245 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_246_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_246_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, p0 is NOT allowed";
string PS_246 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_247_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_247_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, s0 is NOT allowed";
string PS_247 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_248_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_248_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, oC0 is NOT allowed";
string PS_248 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_249_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_249_Desc = "ps_2_x : texldd source reg combination v0, p0, oC0, oDepth is NOT allowed";
string PS_249 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_250_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_250_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, b0 is NOT allowed";
string PS_250 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_251_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_251_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, i0 is NOT allowed";
string PS_251 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_252_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_252_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, p0 is NOT allowed";
string PS_252 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_253_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_253_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, s0 is NOT allowed";
string PS_253 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, v0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_254_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_254_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, oC0 is NOT allowed";
string PS_254 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_255_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_255_Desc = "ps_2_x : texldd source reg combination v0, p0, oDepth, oDepth is NOT allowed";
string PS_255 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_256_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_256_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, b0 is NOT allowed";
string PS_256 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_257_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_257_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, i0 is NOT allowed";
string PS_257 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_258_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_258_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, p0 is NOT allowed";
string PS_258 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_259_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_259_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, s0 is NOT allowed";
string PS_259 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_260_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_260_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, oC0 is NOT allowed";
string PS_260 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_261_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_261_Desc = "ps_2_x : texldd source reg combination v0, t0, b0, oDepth is NOT allowed";
string PS_261 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_262_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_262_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, b0 is NOT allowed";
string PS_262 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_263_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_263_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, i0 is NOT allowed";
string PS_263 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_264_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_264_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, p0 is NOT allowed";
string PS_264 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_265_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_265_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, s0 is NOT allowed";
string PS_265 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_266_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_266_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, oC0 is NOT allowed";
string PS_266 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_267_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_267_Desc = "ps_2_x : texldd source reg combination v0, t0, i0, oDepth is NOT allowed";
string PS_267 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_268_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_268_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, b0 is NOT allowed";
string PS_268 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_269_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_269_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, i0 is NOT allowed";
string PS_269 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_270_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_270_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, p0 is NOT allowed";
string PS_270 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_271_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_271_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, s0 is NOT allowed";
string PS_271 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_272_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_272_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, oC0 is NOT allowed";
string PS_272 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_273_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_273_Desc = "ps_2_x : texldd source reg combination v0, t0, p0, oDepth is NOT allowed";
string PS_273 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_274_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_274_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, b0 is NOT allowed";
string PS_274 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_275_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_275_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, i0 is NOT allowed";
string PS_275 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_276_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_276_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, p0 is NOT allowed";
string PS_276 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_277_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_277_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, s0 is NOT allowed";
string PS_277 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_278_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_278_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, oC0 is NOT allowed";
string PS_278 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_279_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_279_Desc = "ps_2_x : texldd source reg combination v0, t0, s0, oDepth is NOT allowed";
string PS_279 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, v0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_280_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_280_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, b0 is NOT allowed";
string PS_280 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_281_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_281_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, i0 is NOT allowed";
string PS_281 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_282_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_282_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, p0 is NOT allowed";
string PS_282 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_283_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_283_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, s0 is NOT allowed";
string PS_283 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, v0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_284_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_284_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, oC0 is NOT allowed";
string PS_284 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_285_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_285_Desc = "ps_2_x : texldd source reg combination v0, t0, oC0, oDepth is NOT allowed";
string PS_285 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_286_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_286_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, b0 is NOT allowed";
string PS_286 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_287_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_287_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, i0 is NOT allowed";
string PS_287 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_288_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_288_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, p0 is NOT allowed";
string PS_288 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_289_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_289_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, s0 is NOT allowed";
string PS_289 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, v0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_290_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_290_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, oC0 is NOT allowed";
string PS_290 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_291_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_291_Desc = "ps_2_x : texldd source reg combination v0, t0, oDepth, oDepth is NOT allowed";
string PS_291 = 
	"ps_2_x "
	"dcl v0 "
	"dcl t0 "
	"texldd r0, v0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_292_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_292_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, b0 is NOT allowed";
string PS_292 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_293_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_293_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, i0 is NOT allowed";
string PS_293 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_294_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_294_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, p0 is NOT allowed";
string PS_294 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_295_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_295_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, s0 is NOT allowed";
string PS_295 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_296_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_296_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, oC0 is NOT allowed";
string PS_296 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_297_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_297_Desc = "ps_2_x : texldd source reg combination v0, oC0, b0, oDepth is NOT allowed";
string PS_297 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_298_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_298_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, b0 is NOT allowed";
string PS_298 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_299_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_299_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, i0 is NOT allowed";
string PS_299 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_300_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_300_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, p0 is NOT allowed";
string PS_300 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_301_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_301_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, s0 is NOT allowed";
string PS_301 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_302_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_302_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, oC0 is NOT allowed";
string PS_302 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_303_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_303_Desc = "ps_2_x : texldd source reg combination v0, oC0, i0, oDepth is NOT allowed";
string PS_303 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_304_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_304_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, b0 is NOT allowed";
string PS_304 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_305_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_305_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, i0 is NOT allowed";
string PS_305 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_306_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_306_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, p0 is NOT allowed";
string PS_306 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_307_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_307_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, s0 is NOT allowed";
string PS_307 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_308_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_308_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, oC0 is NOT allowed";
string PS_308 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_309_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_309_Desc = "ps_2_x : texldd source reg combination v0, oC0, p0, oDepth is NOT allowed";
string PS_309 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_310_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_310_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, b0 is NOT allowed";
string PS_310 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_311_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_311_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, i0 is NOT allowed";
string PS_311 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_312_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_312_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, p0 is NOT allowed";
string PS_312 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_313_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_313_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, s0 is NOT allowed";
string PS_313 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_314_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_314_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, oC0 is NOT allowed";
string PS_314 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_315_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_315_Desc = "ps_2_x : texldd source reg combination v0, oC0, s0, oDepth is NOT allowed";
string PS_315 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_316_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_316_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, b0 is NOT allowed";
string PS_316 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_317_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_317_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, i0 is NOT allowed";
string PS_317 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_318_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_318_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, p0 is NOT allowed";
string PS_318 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_319_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_319_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, s0 is NOT allowed";
string PS_319 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_320_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_320_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, oC0 is NOT allowed";
string PS_320 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_321_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_321_Desc = "ps_2_x : texldd source reg combination v0, oC0, oC0, oDepth is NOT allowed";
string PS_321 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_322_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_322_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, b0 is NOT allowed";
string PS_322 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_323_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_323_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, i0 is NOT allowed";
string PS_323 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_324_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_324_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, p0 is NOT allowed";
string PS_324 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_325_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_325_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, s0 is NOT allowed";
string PS_325 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_326_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_326_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, oC0 is NOT allowed";
string PS_326 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_327_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_327_Desc = "ps_2_x : texldd source reg combination v0, oC0, oDepth, oDepth is NOT allowed";
string PS_327 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_328_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_328_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, b0 is NOT allowed";
string PS_328 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_329_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_329_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, i0 is NOT allowed";
string PS_329 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_330_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_330_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, p0 is NOT allowed";
string PS_330 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_331_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_331_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, s0 is NOT allowed";
string PS_331 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_332_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_332_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, oC0 is NOT allowed";
string PS_332 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_333_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_333_Desc = "ps_2_x : texldd source reg combination v0, oDepth, b0, oDepth is NOT allowed";
string PS_333 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_334_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_334_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, b0 is NOT allowed";
string PS_334 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_335_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_335_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, i0 is NOT allowed";
string PS_335 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_336_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_336_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, p0 is NOT allowed";
string PS_336 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_337_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_337_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, s0 is NOT allowed";
string PS_337 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_338_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_338_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, oC0 is NOT allowed";
string PS_338 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_339_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_339_Desc = "ps_2_x : texldd source reg combination v0, oDepth, i0, oDepth is NOT allowed";
string PS_339 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_340_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_340_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, b0 is NOT allowed";
string PS_340 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_341_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_341_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, i0 is NOT allowed";
string PS_341 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_342_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_342_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, p0 is NOT allowed";
string PS_342 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_343_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_343_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, s0 is NOT allowed";
string PS_343 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_344_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_344_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, oC0 is NOT allowed";
string PS_344 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_345_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_345_Desc = "ps_2_x : texldd source reg combination v0, oDepth, p0, oDepth is NOT allowed";
string PS_345 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_346_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_346_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, b0 is NOT allowed";
string PS_346 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_347_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_347_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, i0 is NOT allowed";
string PS_347 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_348_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_348_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, p0 is NOT allowed";
string PS_348 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, v0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_349_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_349_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, s0 is NOT allowed";
string PS_349 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_350_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_350_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, oC0 is NOT allowed";
string PS_350 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_351_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_351_Desc = "ps_2_x : texldd source reg combination v0, oDepth, s0, oDepth is NOT allowed";
string PS_351 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_352_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_352_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, b0 is NOT allowed";
string PS_352 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_353_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_353_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, i0 is NOT allowed";
string PS_353 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_354_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_354_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, p0 is NOT allowed";
string PS_354 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_355_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_355_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, s0 is NOT allowed";
string PS_355 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_356_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_356_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, oC0 is NOT allowed";
string PS_356 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_357_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_357_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oC0, oDepth is NOT allowed";
string PS_357 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_358_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_358_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, b0 is NOT allowed";
string PS_358 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, v0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_359_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_359_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, i0 is NOT allowed";
string PS_359 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, v0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_360_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_360_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, p0 is NOT allowed";
string PS_360 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_361_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_361_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, s0 is NOT allowed";
string PS_361 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, v0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_362_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_362_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, oC0 is NOT allowed";
string PS_362 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_363_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_363_Desc = "ps_2_x : texldd source reg combination v0, oDepth, oDepth, oDepth is NOT allowed";
string PS_363 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, v0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_364_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_364_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, b0 is NOT allowed";
string PS_364 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_365_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_365_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, i0 is NOT allowed";
string PS_365 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_366_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_366_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, p0 is NOT allowed";
string PS_366 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_367_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_367_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, s0 is NOT allowed";
string PS_367 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, c0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_368_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_368_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, oC0 is NOT allowed";
string PS_368 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_369_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_369_Desc = "ps_2_x : texldd source reg combination c0, v0, b0, oDepth is NOT allowed";
string PS_369 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_370_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_370_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, b0 is NOT allowed";
string PS_370 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_371_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_371_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, i0 is NOT allowed";
string PS_371 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_372_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_372_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, p0 is NOT allowed";
string PS_372 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_373_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_373_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, s0 is NOT allowed";
string PS_373 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, c0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_374_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_374_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, oC0 is NOT allowed";
string PS_374 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_375_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_375_Desc = "ps_2_x : texldd source reg combination c0, v0, i0, oDepth is NOT allowed";
string PS_375 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_376_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_376_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, b0 is NOT allowed";
string PS_376 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_377_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_377_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, i0 is NOT allowed";
string PS_377 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_378_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_378_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, p0 is NOT allowed";
string PS_378 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_379_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_379_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, s0 is NOT allowed";
string PS_379 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, c0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_380_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_380_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, oC0 is NOT allowed";
string PS_380 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_381_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_381_Desc = "ps_2_x : texldd source reg combination c0, v0, p0, oDepth is NOT allowed";
string PS_381 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_382_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_382_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, b0 is NOT allowed";
string PS_382 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, c0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_383_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_383_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, i0 is NOT allowed";
string PS_383 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_384_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_384_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, p0 is NOT allowed";
string PS_384 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_385_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_385_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, s0 is NOT allowed";
string PS_385 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_386_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_386_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, oC0 is NOT allowed";
string PS_386 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_387_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_387_Desc = "ps_2_x : texldd source reg combination c0, v0, s0, oDepth is NOT allowed";
string PS_387 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_388_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_388_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, b0 is NOT allowed";
string PS_388 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_389_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_389_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, i0 is NOT allowed";
string PS_389 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_390_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_390_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, p0 is NOT allowed";
string PS_390 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_391_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_391_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, s0 is NOT allowed";
string PS_391 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_392_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_392_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, oC0 is NOT allowed";
string PS_392 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_393_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_393_Desc = "ps_2_x : texldd source reg combination c0, v0, oC0, oDepth is NOT allowed";
string PS_393 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_394_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_394_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, b0 is NOT allowed";
string PS_394 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_395_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_395_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, i0 is NOT allowed";
string PS_395 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_396_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_396_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, p0 is NOT allowed";
string PS_396 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_397_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_397_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, s0 is NOT allowed";
string PS_397 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, c0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_398_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_398_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, oC0 is NOT allowed";
string PS_398 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_399_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_399_Desc = "ps_2_x : texldd source reg combination c0, v0, oDepth, oDepth is NOT allowed";
string PS_399 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl v0 "
	"texldd r0, c0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_400_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_400_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, b0 is NOT allowed";
string PS_400 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_401_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_401_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, i0 is NOT allowed";
string PS_401 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_402_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_402_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, p0 is NOT allowed";
string PS_402 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_403_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_403_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, s0 is NOT allowed";
string PS_403 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_404_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_404_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, oC0 is NOT allowed";
string PS_404 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_405_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_405_Desc = "ps_2_x : texldd source reg combination c0, r0, b0, oDepth is NOT allowed";
string PS_405 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_406_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_406_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, b0 is NOT allowed";
string PS_406 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_407_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_407_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, i0 is NOT allowed";
string PS_407 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_408_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_408_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, p0 is NOT allowed";
string PS_408 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_409_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_409_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, s0 is NOT allowed";
string PS_409 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_410_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_410_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, oC0 is NOT allowed";
string PS_410 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_411_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_411_Desc = "ps_2_x : texldd source reg combination c0, r0, i0, oDepth is NOT allowed";
string PS_411 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_412_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_412_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, b0 is NOT allowed";
string PS_412 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_413_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_413_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, i0 is NOT allowed";
string PS_413 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_414_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_414_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, p0 is NOT allowed";
string PS_414 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_415_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_415_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, s0 is NOT allowed";
string PS_415 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_416_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_416_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, oC0 is NOT allowed";
string PS_416 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_417_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_417_Desc = "ps_2_x : texldd source reg combination c0, r0, p0, oDepth is NOT allowed";
string PS_417 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_418_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_418_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, b0 is NOT allowed";
string PS_418 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_419_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_419_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, i0 is NOT allowed";
string PS_419 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_420_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_420_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, p0 is NOT allowed";
string PS_420 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_421_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_421_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, s0 is NOT allowed";
string PS_421 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_422_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_422_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, oC0 is NOT allowed";
string PS_422 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_423_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_423_Desc = "ps_2_x : texldd source reg combination c0, r0, s0, oDepth is NOT allowed";
string PS_423 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_424_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_424_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, b0 is NOT allowed";
string PS_424 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_425_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_425_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, i0 is NOT allowed";
string PS_425 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_426_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_426_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, p0 is NOT allowed";
string PS_426 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_427_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_427_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, s0 is NOT allowed";
string PS_427 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_428_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_428_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, oC0 is NOT allowed";
string PS_428 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_429_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_429_Desc = "ps_2_x : texldd source reg combination c0, r0, oC0, oDepth is NOT allowed";
string PS_429 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_430_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_430_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, b0 is NOT allowed";
string PS_430 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_431_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_431_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, i0 is NOT allowed";
string PS_431 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_432_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_432_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, p0 is NOT allowed";
string PS_432 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_433_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_433_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, s0 is NOT allowed";
string PS_433 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_434_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_434_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, oC0 is NOT allowed";
string PS_434 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_435_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_435_Desc = "ps_2_x : texldd source reg combination c0, r0, oDepth, oDepth is NOT allowed";
string PS_435 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"mov r0, c0 "
	"texldd r0, c0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_436_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_436_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, b0 is NOT allowed";
string PS_436 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_437_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_437_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, i0 is NOT allowed";
string PS_437 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_438_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_438_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, p0 is NOT allowed";
string PS_438 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_439_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_439_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, s0 is NOT allowed";
string PS_439 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_440_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_440_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, oC0 is NOT allowed";
string PS_440 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_441_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_441_Desc = "ps_2_x : texldd source reg combination c0, c0, b0, oDepth is NOT allowed";
string PS_441 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_442_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_442_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, b0 is NOT allowed";
string PS_442 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_443_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_443_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, i0 is NOT allowed";
string PS_443 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_444_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_444_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, p0 is NOT allowed";
string PS_444 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_445_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_445_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, s0 is NOT allowed";
string PS_445 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_446_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_446_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, oC0 is NOT allowed";
string PS_446 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_447_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_447_Desc = "ps_2_x : texldd source reg combination c0, c0, i0, oDepth is NOT allowed";
string PS_447 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_448_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_448_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, b0 is NOT allowed";
string PS_448 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_449_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_449_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, i0 is NOT allowed";
string PS_449 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_450_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_450_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, p0 is NOT allowed";
string PS_450 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_451_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_451_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, s0 is NOT allowed";
string PS_451 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_452_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_452_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, oC0 is NOT allowed";
string PS_452 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_453_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_453_Desc = "ps_2_x : texldd source reg combination c0, c0, p0, oDepth is NOT allowed";
string PS_453 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_454_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_454_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, b0 is NOT allowed";
string PS_454 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_455_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_455_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, i0 is NOT allowed";
string PS_455 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_456_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_456_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, p0 is NOT allowed";
string PS_456 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_457_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_457_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, s0 is NOT allowed";
string PS_457 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_458_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_458_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, oC0 is NOT allowed";
string PS_458 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_459_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_459_Desc = "ps_2_x : texldd source reg combination c0, c0, s0, oDepth is NOT allowed";
string PS_459 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_460_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_460_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, b0 is NOT allowed";
string PS_460 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_461_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_461_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, i0 is NOT allowed";
string PS_461 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_462_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_462_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, p0 is NOT allowed";
string PS_462 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_463_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_463_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, s0 is NOT allowed";
string PS_463 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_464_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_464_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, oC0 is NOT allowed";
string PS_464 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_465_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_465_Desc = "ps_2_x : texldd source reg combination c0, c0, oC0, oDepth is NOT allowed";
string PS_465 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_466_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_466_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, b0 is NOT allowed";
string PS_466 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_467_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_467_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, i0 is NOT allowed";
string PS_467 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_468_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_468_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, p0 is NOT allowed";
string PS_468 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_469_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_469_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, s0 is NOT allowed";
string PS_469 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_470_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_470_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, oC0 is NOT allowed";
string PS_470 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_471_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_471_Desc = "ps_2_x : texldd source reg combination c0, c0, oDepth, oDepth is NOT allowed";
string PS_471 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_472_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_472_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, b0 is NOT allowed";
string PS_472 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_473_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_473_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, i0 is NOT allowed";
string PS_473 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_474_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_474_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, p0 is NOT allowed";
string PS_474 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_475_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_475_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, s0 is NOT allowed";
string PS_475 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_476_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_476_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, oC0 is NOT allowed";
string PS_476 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_477_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_477_Desc = "ps_2_x : texldd source reg combination c0, b0, b0, oDepth is NOT allowed";
string PS_477 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_478_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_478_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, b0 is NOT allowed";
string PS_478 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_479_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_479_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, i0 is NOT allowed";
string PS_479 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_480_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_480_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, p0 is NOT allowed";
string PS_480 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_481_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_481_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, s0 is NOT allowed";
string PS_481 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_482_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_482_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, oC0 is NOT allowed";
string PS_482 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_483_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_483_Desc = "ps_2_x : texldd source reg combination c0, b0, i0, oDepth is NOT allowed";
string PS_483 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_484_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_484_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, b0 is NOT allowed";
string PS_484 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_485_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_485_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, i0 is NOT allowed";
string PS_485 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_486_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_486_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, p0 is NOT allowed";
string PS_486 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_487_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_487_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, s0 is NOT allowed";
string PS_487 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_488_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_488_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, oC0 is NOT allowed";
string PS_488 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_489_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_489_Desc = "ps_2_x : texldd source reg combination c0, b0, p0, oDepth is NOT allowed";
string PS_489 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_490_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_490_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, b0 is NOT allowed";
string PS_490 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_491_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_491_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, i0 is NOT allowed";
string PS_491 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_492_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_492_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, p0 is NOT allowed";
string PS_492 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_493_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_493_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, s0 is NOT allowed";
string PS_493 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_494_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_494_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, oC0 is NOT allowed";
string PS_494 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_495_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_495_Desc = "ps_2_x : texldd source reg combination c0, b0, s0, oDepth is NOT allowed";
string PS_495 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_496_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_496_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, b0 is NOT allowed";
string PS_496 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_497_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_497_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, i0 is NOT allowed";
string PS_497 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_498_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_498_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, p0 is NOT allowed";
string PS_498 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_499_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_499_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, s0 is NOT allowed";
string PS_499 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_500_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_500_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, oC0 is NOT allowed";
string PS_500 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_501_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_501_Desc = "ps_2_x : texldd source reg combination c0, b0, oC0, oDepth is NOT allowed";
string PS_501 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_502_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_502_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, b0 is NOT allowed";
string PS_502 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_503_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_503_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, i0 is NOT allowed";
string PS_503 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_504_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_504_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, p0 is NOT allowed";
string PS_504 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_505_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_505_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, s0 is NOT allowed";
string PS_505 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_506_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_506_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, oC0 is NOT allowed";
string PS_506 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_507_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_507_Desc = "ps_2_x : texldd source reg combination c0, b0, oDepth, oDepth is NOT allowed";
string PS_507 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_508_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_508_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, b0 is NOT allowed";
string PS_508 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_509_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_509_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, i0 is NOT allowed";
string PS_509 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_510_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_510_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, p0 is NOT allowed";
string PS_510 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_511_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_511_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, s0 is NOT allowed";
string PS_511 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_512_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_512_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, oC0 is NOT allowed";
string PS_512 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_513_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_513_Desc = "ps_2_x : texldd source reg combination c0, i0, b0, oDepth is NOT allowed";
string PS_513 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_514_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_514_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, b0 is NOT allowed";
string PS_514 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_515_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_515_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, i0 is NOT allowed";
string PS_515 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_516_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_516_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, p0 is NOT allowed";
string PS_516 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_517_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_517_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, s0 is NOT allowed";
string PS_517 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_518_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_518_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, oC0 is NOT allowed";
string PS_518 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_519_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_519_Desc = "ps_2_x : texldd source reg combination c0, i0, i0, oDepth is NOT allowed";
string PS_519 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_520_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_520_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, b0 is NOT allowed";
string PS_520 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_521_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_521_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, i0 is NOT allowed";
string PS_521 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_522_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_522_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, p0 is NOT allowed";
string PS_522 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_523_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_523_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, s0 is NOT allowed";
string PS_523 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_524_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_524_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, oC0 is NOT allowed";
string PS_524 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_525_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_525_Desc = "ps_2_x : texldd source reg combination c0, i0, p0, oDepth is NOT allowed";
string PS_525 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_526_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_526_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, b0 is NOT allowed";
string PS_526 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_527_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_527_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, i0 is NOT allowed";
string PS_527 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_528_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_528_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, p0 is NOT allowed";
string PS_528 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_529_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_529_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, s0 is NOT allowed";
string PS_529 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_530_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_530_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, oC0 is NOT allowed";
string PS_530 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_531_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_531_Desc = "ps_2_x : texldd source reg combination c0, i0, s0, oDepth is NOT allowed";
string PS_531 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_532_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_532_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, b0 is NOT allowed";
string PS_532 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_533_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_533_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, i0 is NOT allowed";
string PS_533 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_534_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_534_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, p0 is NOT allowed";
string PS_534 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_535_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_535_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, s0 is NOT allowed";
string PS_535 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_536_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_536_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, oC0 is NOT allowed";
string PS_536 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_537_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_537_Desc = "ps_2_x : texldd source reg combination c0, i0, oC0, oDepth is NOT allowed";
string PS_537 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_538_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_538_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, b0 is NOT allowed";
string PS_538 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_539_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_539_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, i0 is NOT allowed";
string PS_539 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_540_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_540_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, p0 is NOT allowed";
string PS_540 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_541_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_541_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, s0 is NOT allowed";
string PS_541 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_542_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_542_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, oC0 is NOT allowed";
string PS_542 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_543_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_543_Desc = "ps_2_x : texldd source reg combination c0, i0, oDepth, oDepth is NOT allowed";
string PS_543 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_544_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_544_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, b0 is NOT allowed";
string PS_544 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_545_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_545_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, i0 is NOT allowed";
string PS_545 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_546_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_546_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, p0 is NOT allowed";
string PS_546 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_547_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_547_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, s0 is NOT allowed";
string PS_547 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_548_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_548_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, oC0 is NOT allowed";
string PS_548 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_549_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_549_Desc = "ps_2_x : texldd source reg combination c0, p0, b0, oDepth is NOT allowed";
string PS_549 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_550_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_550_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, b0 is NOT allowed";
string PS_550 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_551_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_551_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, i0 is NOT allowed";
string PS_551 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_552_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_552_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, p0 is NOT allowed";
string PS_552 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_553_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_553_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, s0 is NOT allowed";
string PS_553 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_554_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_554_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, oC0 is NOT allowed";
string PS_554 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_555_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_555_Desc = "ps_2_x : texldd source reg combination c0, p0, i0, oDepth is NOT allowed";
string PS_555 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_556_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_556_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, b0 is NOT allowed";
string PS_556 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_557_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_557_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, i0 is NOT allowed";
string PS_557 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_558_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_558_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, p0 is NOT allowed";
string PS_558 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_559_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_559_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, s0 is NOT allowed";
string PS_559 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_560_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_560_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, oC0 is NOT allowed";
string PS_560 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_561_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_561_Desc = "ps_2_x : texldd source reg combination c0, p0, p0, oDepth is NOT allowed";
string PS_561 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_562_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_562_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, b0 is NOT allowed";
string PS_562 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_563_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_563_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, i0 is NOT allowed";
string PS_563 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_564_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_564_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, p0 is NOT allowed";
string PS_564 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_565_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_565_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, s0 is NOT allowed";
string PS_565 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_566_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_566_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, oC0 is NOT allowed";
string PS_566 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_567_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_567_Desc = "ps_2_x : texldd source reg combination c0, p0, s0, oDepth is NOT allowed";
string PS_567 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_568_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_568_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, b0 is NOT allowed";
string PS_568 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_569_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_569_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, i0 is NOT allowed";
string PS_569 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_570_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_570_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, p0 is NOT allowed";
string PS_570 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_571_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_571_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, s0 is NOT allowed";
string PS_571 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_572_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_572_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, oC0 is NOT allowed";
string PS_572 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_573_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_573_Desc = "ps_2_x : texldd source reg combination c0, p0, oC0, oDepth is NOT allowed";
string PS_573 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_574_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_574_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, b0 is NOT allowed";
string PS_574 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_575_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_575_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, i0 is NOT allowed";
string PS_575 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_576_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_576_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, p0 is NOT allowed";
string PS_576 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_577_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_577_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, s0 is NOT allowed";
string PS_577 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_578_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_578_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, oC0 is NOT allowed";
string PS_578 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_579_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_579_Desc = "ps_2_x : texldd source reg combination c0, p0, oDepth, oDepth is NOT allowed";
string PS_579 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_580_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_580_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, b0 is NOT allowed";
string PS_580 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_581_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_581_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, i0 is NOT allowed";
string PS_581 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_582_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_582_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, p0 is NOT allowed";
string PS_582 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_583_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_583_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, s0 is NOT allowed";
string PS_583 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, c0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_584_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_584_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, oC0 is NOT allowed";
string PS_584 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_585_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_585_Desc = "ps_2_x : texldd source reg combination c0, t0, b0, oDepth is NOT allowed";
string PS_585 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_586_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_586_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, b0 is NOT allowed";
string PS_586 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_587_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_587_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, i0 is NOT allowed";
string PS_587 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_588_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_588_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, p0 is NOT allowed";
string PS_588 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_589_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_589_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, s0 is NOT allowed";
string PS_589 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, c0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_590_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_590_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, oC0 is NOT allowed";
string PS_590 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_591_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_591_Desc = "ps_2_x : texldd source reg combination c0, t0, i0, oDepth is NOT allowed";
string PS_591 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_592_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_592_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, b0 is NOT allowed";
string PS_592 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_593_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_593_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, i0 is NOT allowed";
string PS_593 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_594_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_594_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, p0 is NOT allowed";
string PS_594 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_595_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_595_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, s0 is NOT allowed";
string PS_595 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_596_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_596_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, oC0 is NOT allowed";
string PS_596 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_597_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_597_Desc = "ps_2_x : texldd source reg combination c0, t0, p0, oDepth is NOT allowed";
string PS_597 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_598_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_598_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, b0 is NOT allowed";
string PS_598 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_599_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_599_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, i0 is NOT allowed";
string PS_599 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_600_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_600_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, p0 is NOT allowed";
string PS_600 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_601_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_601_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, s0 is NOT allowed";
string PS_601 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_602_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_602_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, oC0 is NOT allowed";
string PS_602 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_603_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_603_Desc = "ps_2_x : texldd source reg combination c0, t0, s0, oDepth is NOT allowed";
string PS_603 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_604_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_604_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, b0 is NOT allowed";
string PS_604 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_605_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_605_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, i0 is NOT allowed";
string PS_605 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_606_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_606_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, p0 is NOT allowed";
string PS_606 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_607_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_607_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, s0 is NOT allowed";
string PS_607 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, c0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_608_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_608_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, oC0 is NOT allowed";
string PS_608 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_609_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_609_Desc = "ps_2_x : texldd source reg combination c0, t0, oC0, oDepth is NOT allowed";
string PS_609 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_610_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_610_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, b0 is NOT allowed";
string PS_610 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_611_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_611_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, i0 is NOT allowed";
string PS_611 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_612_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_612_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, p0 is NOT allowed";
string PS_612 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_613_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_613_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, s0 is NOT allowed";
string PS_613 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, c0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_614_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_614_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, oC0 is NOT allowed";
string PS_614 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_615_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_615_Desc = "ps_2_x : texldd source reg combination c0, t0, oDepth, oDepth is NOT allowed";
string PS_615 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl t0 "
	"texldd r0, c0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_616_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_616_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, b0 is NOT allowed";
string PS_616 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_617_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_617_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, i0 is NOT allowed";
string PS_617 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_618_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_618_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, p0 is NOT allowed";
string PS_618 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_619_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_619_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, s0 is NOT allowed";
string PS_619 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_620_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_620_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, oC0 is NOT allowed";
string PS_620 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_621_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_621_Desc = "ps_2_x : texldd source reg combination c0, oC0, b0, oDepth is NOT allowed";
string PS_621 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_622_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_622_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, b0 is NOT allowed";
string PS_622 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_623_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_623_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, i0 is NOT allowed";
string PS_623 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_624_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_624_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, p0 is NOT allowed";
string PS_624 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_625_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_625_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, s0 is NOT allowed";
string PS_625 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_626_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_626_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, oC0 is NOT allowed";
string PS_626 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_627_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_627_Desc = "ps_2_x : texldd source reg combination c0, oC0, i0, oDepth is NOT allowed";
string PS_627 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_628_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_628_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, b0 is NOT allowed";
string PS_628 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_629_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_629_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, i0 is NOT allowed";
string PS_629 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_630_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_630_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, p0 is NOT allowed";
string PS_630 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_631_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_631_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, s0 is NOT allowed";
string PS_631 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_632_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_632_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, oC0 is NOT allowed";
string PS_632 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_633_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_633_Desc = "ps_2_x : texldd source reg combination c0, oC0, p0, oDepth is NOT allowed";
string PS_633 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_634_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_634_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, b0 is NOT allowed";
string PS_634 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_635_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_635_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, i0 is NOT allowed";
string PS_635 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_636_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_636_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, p0 is NOT allowed";
string PS_636 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_637_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_637_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, s0 is NOT allowed";
string PS_637 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_638_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_638_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, oC0 is NOT allowed";
string PS_638 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_639_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_639_Desc = "ps_2_x : texldd source reg combination c0, oC0, s0, oDepth is NOT allowed";
string PS_639 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_640_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_640_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, b0 is NOT allowed";
string PS_640 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_641_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_641_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, i0 is NOT allowed";
string PS_641 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_642_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_642_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, p0 is NOT allowed";
string PS_642 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_643_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_643_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, s0 is NOT allowed";
string PS_643 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_644_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_644_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, oC0 is NOT allowed";
string PS_644 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_645_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_645_Desc = "ps_2_x : texldd source reg combination c0, oC0, oC0, oDepth is NOT allowed";
string PS_645 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_646_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_646_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, b0 is NOT allowed";
string PS_646 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_647_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_647_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, i0 is NOT allowed";
string PS_647 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_648_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_648_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, p0 is NOT allowed";
string PS_648 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_649_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_649_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, s0 is NOT allowed";
string PS_649 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_650_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_650_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, oC0 is NOT allowed";
string PS_650 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_651_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_651_Desc = "ps_2_x : texldd source reg combination c0, oC0, oDepth, oDepth is NOT allowed";
string PS_651 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_652_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_652_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, b0 is NOT allowed";
string PS_652 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_653_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_653_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, i0 is NOT allowed";
string PS_653 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_654_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_654_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, p0 is NOT allowed";
string PS_654 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_655_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_655_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, s0 is NOT allowed";
string PS_655 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, c0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_656_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_656_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, oC0 is NOT allowed";
string PS_656 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_657_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_657_Desc = "ps_2_x : texldd source reg combination c0, oDepth, b0, oDepth is NOT allowed";
string PS_657 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_658_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_658_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, b0 is NOT allowed";
string PS_658 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_659_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_659_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, i0 is NOT allowed";
string PS_659 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_660_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_660_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, p0 is NOT allowed";
string PS_660 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_661_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_661_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, s0 is NOT allowed";
string PS_661 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_662_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_662_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, oC0 is NOT allowed";
string PS_662 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_663_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_663_Desc = "ps_2_x : texldd source reg combination c0, oDepth, i0, oDepth is NOT allowed";
string PS_663 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_664_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_664_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, b0 is NOT allowed";
string PS_664 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_665_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_665_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, i0 is NOT allowed";
string PS_665 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_666_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_666_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, p0 is NOT allowed";
string PS_666 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_667_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_667_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, s0 is NOT allowed";
string PS_667 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_668_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_668_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, oC0 is NOT allowed";
string PS_668 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_669_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_669_Desc = "ps_2_x : texldd source reg combination c0, oDepth, p0, oDepth is NOT allowed";
string PS_669 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_670_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_670_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, b0 is NOT allowed";
string PS_670 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_671_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_671_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, i0 is NOT allowed";
string PS_671 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_672_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_672_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, p0 is NOT allowed";
string PS_672 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_673_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_673_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, s0 is NOT allowed";
string PS_673 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_674_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_674_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, oC0 is NOT allowed";
string PS_674 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_675_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_675_Desc = "ps_2_x : texldd source reg combination c0, oDepth, s0, oDepth is NOT allowed";
string PS_675 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_676_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_676_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, b0 is NOT allowed";
string PS_676 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_677_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_677_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, i0 is NOT allowed";
string PS_677 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_678_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_678_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, p0 is NOT allowed";
string PS_678 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_679_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_679_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, s0 is NOT allowed";
string PS_679 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, c0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_680_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_680_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, oC0 is NOT allowed";
string PS_680 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_681_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_681_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oC0, oDepth is NOT allowed";
string PS_681 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_682_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_682_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, b0 is NOT allowed";
string PS_682 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_683_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_683_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, i0 is NOT allowed";
string PS_683 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_684_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_684_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, p0 is NOT allowed";
string PS_684 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_685_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_685_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, s0 is NOT allowed";
string PS_685 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, c0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_686_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_686_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, oC0 is NOT allowed";
string PS_686 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_687_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_687_Desc = "ps_2_x : texldd source reg combination c0, oDepth, oDepth, oDepth is NOT allowed";
string PS_687 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, c0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_688_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_688_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, b0 is NOT allowed";
string PS_688 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_689_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_689_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, i0 is NOT allowed";
string PS_689 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_690_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_690_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, p0 is NOT allowed";
string PS_690 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_691_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_691_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, s0 is NOT allowed";
string PS_691 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_692_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_692_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, oC0 is NOT allowed";
string PS_692 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_693_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_693_Desc = "ps_2_x : texldd source reg combination b0, v0, b0, oDepth is NOT allowed";
string PS_693 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_694_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_694_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, b0 is NOT allowed";
string PS_694 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_695_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_695_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, i0 is NOT allowed";
string PS_695 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_696_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_696_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, p0 is NOT allowed";
string PS_696 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_697_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_697_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, s0 is NOT allowed";
string PS_697 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, b0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_698_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_698_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, oC0 is NOT allowed";
string PS_698 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_699_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_699_Desc = "ps_2_x : texldd source reg combination b0, v0, i0, oDepth is NOT allowed";
string PS_699 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_700_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_700_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, b0 is NOT allowed";
string PS_700 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_701_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_701_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, i0 is NOT allowed";
string PS_701 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_702_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_702_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, p0 is NOT allowed";
string PS_702 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_703_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_703_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, s0 is NOT allowed";
string PS_703 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, b0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_704_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_704_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, oC0 is NOT allowed";
string PS_704 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_705_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_705_Desc = "ps_2_x : texldd source reg combination b0, v0, p0, oDepth is NOT allowed";
string PS_705 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_706_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_706_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, b0 is NOT allowed";
string PS_706 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, b0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_707_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_707_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, i0 is NOT allowed";
string PS_707 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_708_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_708_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, p0 is NOT allowed";
string PS_708 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_709_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_709_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, s0 is NOT allowed";
string PS_709 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_710_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_710_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, oC0 is NOT allowed";
string PS_710 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_711_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_711_Desc = "ps_2_x : texldd source reg combination b0, v0, s0, oDepth is NOT allowed";
string PS_711 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_712_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_712_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, b0 is NOT allowed";
string PS_712 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_713_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_713_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, i0 is NOT allowed";
string PS_713 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_714_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_714_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, p0 is NOT allowed";
string PS_714 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_715_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_715_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, s0 is NOT allowed";
string PS_715 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_716_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_716_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, oC0 is NOT allowed";
string PS_716 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_717_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_717_Desc = "ps_2_x : texldd source reg combination b0, v0, oC0, oDepth is NOT allowed";
string PS_717 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_718_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_718_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, b0 is NOT allowed";
string PS_718 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_719_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_719_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, i0 is NOT allowed";
string PS_719 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_720_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_720_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, p0 is NOT allowed";
string PS_720 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_721_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_721_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, s0 is NOT allowed";
string PS_721 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, b0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_722_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_722_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, oC0 is NOT allowed";
string PS_722 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_723_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_723_Desc = "ps_2_x : texldd source reg combination b0, v0, oDepth, oDepth is NOT allowed";
string PS_723 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, b0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_724_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_724_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, b0 is NOT allowed";
string PS_724 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_725_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_725_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, i0 is NOT allowed";
string PS_725 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_726_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_726_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, p0 is NOT allowed";
string PS_726 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_727_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_727_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, s0 is NOT allowed";
string PS_727 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_728_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_728_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, oC0 is NOT allowed";
string PS_728 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_729_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_729_Desc = "ps_2_x : texldd source reg combination b0, r0, b0, oDepth is NOT allowed";
string PS_729 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_730_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_730_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, b0 is NOT allowed";
string PS_730 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_731_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_731_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, i0 is NOT allowed";
string PS_731 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_732_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_732_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, p0 is NOT allowed";
string PS_732 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_733_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_733_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, s0 is NOT allowed";
string PS_733 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_734_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_734_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, oC0 is NOT allowed";
string PS_734 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_735_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_735_Desc = "ps_2_x : texldd source reg combination b0, r0, i0, oDepth is NOT allowed";
string PS_735 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_736_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_736_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, b0 is NOT allowed";
string PS_736 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_737_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_737_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, i0 is NOT allowed";
string PS_737 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_738_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_738_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, p0 is NOT allowed";
string PS_738 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_739_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_739_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, s0 is NOT allowed";
string PS_739 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_740_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_740_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, oC0 is NOT allowed";
string PS_740 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_741_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_741_Desc = "ps_2_x : texldd source reg combination b0, r0, p0, oDepth is NOT allowed";
string PS_741 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_742_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_742_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, b0 is NOT allowed";
string PS_742 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_743_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_743_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, i0 is NOT allowed";
string PS_743 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_744_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_744_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, p0 is NOT allowed";
string PS_744 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_745_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_745_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, s0 is NOT allowed";
string PS_745 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_746_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_746_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, oC0 is NOT allowed";
string PS_746 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_747_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_747_Desc = "ps_2_x : texldd source reg combination b0, r0, s0, oDepth is NOT allowed";
string PS_747 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_748_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_748_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, b0 is NOT allowed";
string PS_748 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_749_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_749_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, i0 is NOT allowed";
string PS_749 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_750_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_750_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, p0 is NOT allowed";
string PS_750 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_751_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_751_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, s0 is NOT allowed";
string PS_751 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_752_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_752_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, oC0 is NOT allowed";
string PS_752 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_753_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_753_Desc = "ps_2_x : texldd source reg combination b0, r0, oC0, oDepth is NOT allowed";
string PS_753 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_754_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_754_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, b0 is NOT allowed";
string PS_754 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_755_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_755_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, i0 is NOT allowed";
string PS_755 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_756_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_756_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, p0 is NOT allowed";
string PS_756 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_757_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_757_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, s0 is NOT allowed";
string PS_757 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_758_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_758_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, oC0 is NOT allowed";
string PS_758 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_759_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_759_Desc = "ps_2_x : texldd source reg combination b0, r0, oDepth, oDepth is NOT allowed";
string PS_759 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, b0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_760_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_760_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, b0 is NOT allowed";
string PS_760 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_761_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_761_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, i0 is NOT allowed";
string PS_761 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_762_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_762_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, p0 is NOT allowed";
string PS_762 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_763_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_763_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, s0 is NOT allowed";
string PS_763 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_764_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_764_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, oC0 is NOT allowed";
string PS_764 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_765_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_765_Desc = "ps_2_x : texldd source reg combination b0, c0, b0, oDepth is NOT allowed";
string PS_765 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_766_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_766_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, b0 is NOT allowed";
string PS_766 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_767_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_767_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, i0 is NOT allowed";
string PS_767 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_768_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_768_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, p0 is NOT allowed";
string PS_768 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_769_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_769_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, s0 is NOT allowed";
string PS_769 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_770_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_770_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, oC0 is NOT allowed";
string PS_770 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_771_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_771_Desc = "ps_2_x : texldd source reg combination b0, c0, i0, oDepth is NOT allowed";
string PS_771 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_772_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_772_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, b0 is NOT allowed";
string PS_772 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_773_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_773_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, i0 is NOT allowed";
string PS_773 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_774_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_774_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, p0 is NOT allowed";
string PS_774 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_775_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_775_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, s0 is NOT allowed";
string PS_775 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_776_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_776_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, oC0 is NOT allowed";
string PS_776 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_777_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_777_Desc = "ps_2_x : texldd source reg combination b0, c0, p0, oDepth is NOT allowed";
string PS_777 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_778_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_778_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, b0 is NOT allowed";
string PS_778 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_779_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_779_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, i0 is NOT allowed";
string PS_779 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_780_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_780_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, p0 is NOT allowed";
string PS_780 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_781_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_781_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, s0 is NOT allowed";
string PS_781 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_782_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_782_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, oC0 is NOT allowed";
string PS_782 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_783_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_783_Desc = "ps_2_x : texldd source reg combination b0, c0, s0, oDepth is NOT allowed";
string PS_783 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_784_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_784_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, b0 is NOT allowed";
string PS_784 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_785_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_785_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, i0 is NOT allowed";
string PS_785 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_786_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_786_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, p0 is NOT allowed";
string PS_786 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_787_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_787_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, s0 is NOT allowed";
string PS_787 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_788_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_788_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, oC0 is NOT allowed";
string PS_788 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_789_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_789_Desc = "ps_2_x : texldd source reg combination b0, c0, oC0, oDepth is NOT allowed";
string PS_789 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_790_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_790_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, b0 is NOT allowed";
string PS_790 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_791_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_791_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, i0 is NOT allowed";
string PS_791 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_792_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_792_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, p0 is NOT allowed";
string PS_792 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_793_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_793_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, s0 is NOT allowed";
string PS_793 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_794_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_794_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, oC0 is NOT allowed";
string PS_794 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_795_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_795_Desc = "ps_2_x : texldd source reg combination b0, c0, oDepth, oDepth is NOT allowed";
string PS_795 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defb b0, true "
	"texldd r0, b0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_796_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_796_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, b0 is NOT allowed";
string PS_796 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_797_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_797_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, i0 is NOT allowed";
string PS_797 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_798_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_798_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, p0 is NOT allowed";
string PS_798 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_799_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_799_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, s0 is NOT allowed";
string PS_799 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_800_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_800_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, oC0 is NOT allowed";
string PS_800 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_801_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_801_Desc = "ps_2_x : texldd source reg combination b0, b0, b0, oDepth is NOT allowed";
string PS_801 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_802_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_802_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, b0 is NOT allowed";
string PS_802 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_803_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_803_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, i0 is NOT allowed";
string PS_803 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_804_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_804_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, p0 is NOT allowed";
string PS_804 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_805_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_805_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, s0 is NOT allowed";
string PS_805 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_806_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_806_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, oC0 is NOT allowed";
string PS_806 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_807_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_807_Desc = "ps_2_x : texldd source reg combination b0, b0, i0, oDepth is NOT allowed";
string PS_807 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_808_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_808_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, b0 is NOT allowed";
string PS_808 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_809_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_809_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, i0 is NOT allowed";
string PS_809 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_810_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_810_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, p0 is NOT allowed";
string PS_810 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_811_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_811_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, s0 is NOT allowed";
string PS_811 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_812_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_812_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, oC0 is NOT allowed";
string PS_812 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_813_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_813_Desc = "ps_2_x : texldd source reg combination b0, b0, p0, oDepth is NOT allowed";
string PS_813 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_814_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_814_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, b0 is NOT allowed";
string PS_814 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_815_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_815_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, i0 is NOT allowed";
string PS_815 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_816_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_816_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, p0 is NOT allowed";
string PS_816 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_817_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_817_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, s0 is NOT allowed";
string PS_817 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_818_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_818_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, oC0 is NOT allowed";
string PS_818 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_819_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_819_Desc = "ps_2_x : texldd source reg combination b0, b0, s0, oDepth is NOT allowed";
string PS_819 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_820_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_820_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, b0 is NOT allowed";
string PS_820 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_821_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_821_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, i0 is NOT allowed";
string PS_821 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_822_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_822_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, p0 is NOT allowed";
string PS_822 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_823_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_823_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, s0 is NOT allowed";
string PS_823 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_824_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_824_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, oC0 is NOT allowed";
string PS_824 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_825_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_825_Desc = "ps_2_x : texldd source reg combination b0, b0, oC0, oDepth is NOT allowed";
string PS_825 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_826_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_826_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, b0 is NOT allowed";
string PS_826 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_827_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_827_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, i0 is NOT allowed";
string PS_827 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_828_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_828_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, p0 is NOT allowed";
string PS_828 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_829_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_829_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, s0 is NOT allowed";
string PS_829 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_830_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_830_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, oC0 is NOT allowed";
string PS_830 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_831_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_831_Desc = "ps_2_x : texldd source reg combination b0, b0, oDepth, oDepth is NOT allowed";
string PS_831 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_832_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_832_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, b0 is NOT allowed";
string PS_832 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_833_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_833_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, i0 is NOT allowed";
string PS_833 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_834_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_834_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, p0 is NOT allowed";
string PS_834 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_835_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_835_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, s0 is NOT allowed";
string PS_835 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_836_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_836_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, oC0 is NOT allowed";
string PS_836 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_837_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_837_Desc = "ps_2_x : texldd source reg combination b0, i0, b0, oDepth is NOT allowed";
string PS_837 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_838_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_838_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, b0 is NOT allowed";
string PS_838 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_839_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_839_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, i0 is NOT allowed";
string PS_839 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_840_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_840_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, p0 is NOT allowed";
string PS_840 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_841_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_841_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, s0 is NOT allowed";
string PS_841 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_842_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_842_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, oC0 is NOT allowed";
string PS_842 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_843_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_843_Desc = "ps_2_x : texldd source reg combination b0, i0, i0, oDepth is NOT allowed";
string PS_843 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_844_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_844_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, b0 is NOT allowed";
string PS_844 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_845_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_845_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, i0 is NOT allowed";
string PS_845 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_846_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_846_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, p0 is NOT allowed";
string PS_846 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_847_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_847_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, s0 is NOT allowed";
string PS_847 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_848_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_848_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, oC0 is NOT allowed";
string PS_848 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_849_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_849_Desc = "ps_2_x : texldd source reg combination b0, i0, p0, oDepth is NOT allowed";
string PS_849 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_850_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_850_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, b0 is NOT allowed";
string PS_850 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_851_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_851_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, i0 is NOT allowed";
string PS_851 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_852_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_852_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, p0 is NOT allowed";
string PS_852 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_853_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_853_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, s0 is NOT allowed";
string PS_853 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_854_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_854_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, oC0 is NOT allowed";
string PS_854 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_855_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_855_Desc = "ps_2_x : texldd source reg combination b0, i0, s0, oDepth is NOT allowed";
string PS_855 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_856_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_856_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, b0 is NOT allowed";
string PS_856 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_857_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_857_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, i0 is NOT allowed";
string PS_857 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_858_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_858_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, p0 is NOT allowed";
string PS_858 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_859_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_859_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, s0 is NOT allowed";
string PS_859 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_860_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_860_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, oC0 is NOT allowed";
string PS_860 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_861_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_861_Desc = "ps_2_x : texldd source reg combination b0, i0, oC0, oDepth is NOT allowed";
string PS_861 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_862_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_862_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, b0 is NOT allowed";
string PS_862 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_863_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_863_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, i0 is NOT allowed";
string PS_863 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_864_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_864_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, p0 is NOT allowed";
string PS_864 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_865_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_865_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, s0 is NOT allowed";
string PS_865 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_866_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_866_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, oC0 is NOT allowed";
string PS_866 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_867_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_867_Desc = "ps_2_x : texldd source reg combination b0, i0, oDepth, oDepth is NOT allowed";
string PS_867 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_868_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_868_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, b0 is NOT allowed";
string PS_868 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_869_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_869_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, i0 is NOT allowed";
string PS_869 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_870_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_870_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, p0 is NOT allowed";
string PS_870 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_871_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_871_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, s0 is NOT allowed";
string PS_871 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_872_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_872_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, oC0 is NOT allowed";
string PS_872 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_873_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_873_Desc = "ps_2_x : texldd source reg combination b0, p0, b0, oDepth is NOT allowed";
string PS_873 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_874_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_874_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, b0 is NOT allowed";
string PS_874 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_875_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_875_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, i0 is NOT allowed";
string PS_875 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_876_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_876_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, p0 is NOT allowed";
string PS_876 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_877_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_877_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, s0 is NOT allowed";
string PS_877 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_878_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_878_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, oC0 is NOT allowed";
string PS_878 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_879_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_879_Desc = "ps_2_x : texldd source reg combination b0, p0, i0, oDepth is NOT allowed";
string PS_879 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_880_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_880_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, b0 is NOT allowed";
string PS_880 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_881_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_881_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, i0 is NOT allowed";
string PS_881 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_882_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_882_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, p0 is NOT allowed";
string PS_882 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_883_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_883_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, s0 is NOT allowed";
string PS_883 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_884_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_884_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, oC0 is NOT allowed";
string PS_884 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_885_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_885_Desc = "ps_2_x : texldd source reg combination b0, p0, p0, oDepth is NOT allowed";
string PS_885 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_886_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_886_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, b0 is NOT allowed";
string PS_886 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_887_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_887_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, i0 is NOT allowed";
string PS_887 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_888_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_888_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, p0 is NOT allowed";
string PS_888 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_889_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_889_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, s0 is NOT allowed";
string PS_889 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_890_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_890_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, oC0 is NOT allowed";
string PS_890 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_891_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_891_Desc = "ps_2_x : texldd source reg combination b0, p0, s0, oDepth is NOT allowed";
string PS_891 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_892_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_892_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, b0 is NOT allowed";
string PS_892 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_893_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_893_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, i0 is NOT allowed";
string PS_893 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_894_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_894_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, p0 is NOT allowed";
string PS_894 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_895_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_895_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, s0 is NOT allowed";
string PS_895 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_896_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_896_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, oC0 is NOT allowed";
string PS_896 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_897_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_897_Desc = "ps_2_x : texldd source reg combination b0, p0, oC0, oDepth is NOT allowed";
string PS_897 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_898_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_898_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, b0 is NOT allowed";
string PS_898 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_899_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_899_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, i0 is NOT allowed";
string PS_899 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_900_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_900_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, p0 is NOT allowed";
string PS_900 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_901_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_901_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, s0 is NOT allowed";
string PS_901 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_902_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_902_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, oC0 is NOT allowed";
string PS_902 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_903_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_903_Desc = "ps_2_x : texldd source reg combination b0, p0, oDepth, oDepth is NOT allowed";
string PS_903 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_904_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_904_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, b0 is NOT allowed";
string PS_904 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_905_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_905_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, i0 is NOT allowed";
string PS_905 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_906_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_906_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, p0 is NOT allowed";
string PS_906 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_907_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_907_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, s0 is NOT allowed";
string PS_907 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_908_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_908_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, oC0 is NOT allowed";
string PS_908 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_909_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_909_Desc = "ps_2_x : texldd source reg combination b0, t0, b0, oDepth is NOT allowed";
string PS_909 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_910_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_910_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, b0 is NOT allowed";
string PS_910 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_911_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_911_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, i0 is NOT allowed";
string PS_911 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_912_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_912_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, p0 is NOT allowed";
string PS_912 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_913_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_913_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, s0 is NOT allowed";
string PS_913 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, b0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_914_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_914_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, oC0 is NOT allowed";
string PS_914 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_915_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_915_Desc = "ps_2_x : texldd source reg combination b0, t0, i0, oDepth is NOT allowed";
string PS_915 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_916_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_916_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, b0 is NOT allowed";
string PS_916 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_917_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_917_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, i0 is NOT allowed";
string PS_917 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_918_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_918_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, p0 is NOT allowed";
string PS_918 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_919_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_919_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, s0 is NOT allowed";
string PS_919 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, b0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_920_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_920_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, oC0 is NOT allowed";
string PS_920 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_921_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_921_Desc = "ps_2_x : texldd source reg combination b0, t0, p0, oDepth is NOT allowed";
string PS_921 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_922_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_922_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, b0 is NOT allowed";
string PS_922 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_923_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_923_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, i0 is NOT allowed";
string PS_923 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_924_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_924_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, p0 is NOT allowed";
string PS_924 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_925_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_925_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, s0 is NOT allowed";
string PS_925 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_926_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_926_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, oC0 is NOT allowed";
string PS_926 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_927_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_927_Desc = "ps_2_x : texldd source reg combination b0, t0, s0, oDepth is NOT allowed";
string PS_927 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, b0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_928_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_928_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, b0 is NOT allowed";
string PS_928 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_929_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_929_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, i0 is NOT allowed";
string PS_929 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_930_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_930_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, p0 is NOT allowed";
string PS_930 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_931_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_931_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, s0 is NOT allowed";
string PS_931 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, b0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_932_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_932_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, oC0 is NOT allowed";
string PS_932 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_933_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_933_Desc = "ps_2_x : texldd source reg combination b0, t0, oC0, oDepth is NOT allowed";
string PS_933 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_934_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_934_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, b0 is NOT allowed";
string PS_934 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_935_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_935_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, i0 is NOT allowed";
string PS_935 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_936_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_936_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, p0 is NOT allowed";
string PS_936 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_937_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_937_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, s0 is NOT allowed";
string PS_937 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, b0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_938_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_938_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, oC0 is NOT allowed";
string PS_938 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_939_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_939_Desc = "ps_2_x : texldd source reg combination b0, t0, oDepth, oDepth is NOT allowed";
string PS_939 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, b0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_940_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_940_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, b0 is NOT allowed";
string PS_940 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_941_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_941_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, i0 is NOT allowed";
string PS_941 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_942_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_942_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, p0 is NOT allowed";
string PS_942 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_943_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_943_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, s0 is NOT allowed";
string PS_943 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_944_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_944_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, oC0 is NOT allowed";
string PS_944 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_945_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_945_Desc = "ps_2_x : texldd source reg combination b0, oC0, b0, oDepth is NOT allowed";
string PS_945 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_946_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_946_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, b0 is NOT allowed";
string PS_946 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_947_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_947_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, i0 is NOT allowed";
string PS_947 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_948_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_948_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, p0 is NOT allowed";
string PS_948 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_949_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_949_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, s0 is NOT allowed";
string PS_949 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_950_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_950_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, oC0 is NOT allowed";
string PS_950 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_951_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_951_Desc = "ps_2_x : texldd source reg combination b0, oC0, i0, oDepth is NOT allowed";
string PS_951 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_952_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_952_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, b0 is NOT allowed";
string PS_952 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_953_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_953_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, i0 is NOT allowed";
string PS_953 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_954_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_954_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, p0 is NOT allowed";
string PS_954 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_955_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_955_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, s0 is NOT allowed";
string PS_955 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_956_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_956_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, oC0 is NOT allowed";
string PS_956 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_957_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_957_Desc = "ps_2_x : texldd source reg combination b0, oC0, p0, oDepth is NOT allowed";
string PS_957 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_958_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_958_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, b0 is NOT allowed";
string PS_958 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_959_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_959_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, i0 is NOT allowed";
string PS_959 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_960_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_960_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, p0 is NOT allowed";
string PS_960 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_961_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_961_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, s0 is NOT allowed";
string PS_961 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_962_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_962_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, oC0 is NOT allowed";
string PS_962 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_963_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_963_Desc = "ps_2_x : texldd source reg combination b0, oC0, s0, oDepth is NOT allowed";
string PS_963 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_964_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_964_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, b0 is NOT allowed";
string PS_964 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_965_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_965_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, i0 is NOT allowed";
string PS_965 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_966_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_966_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, p0 is NOT allowed";
string PS_966 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_967_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_967_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, s0 is NOT allowed";
string PS_967 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_968_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_968_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, oC0 is NOT allowed";
string PS_968 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_969_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_969_Desc = "ps_2_x : texldd source reg combination b0, oC0, oC0, oDepth is NOT allowed";
string PS_969 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_970_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_970_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, b0 is NOT allowed";
string PS_970 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_971_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_971_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, i0 is NOT allowed";
string PS_971 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_972_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_972_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, p0 is NOT allowed";
string PS_972 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_973_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_973_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, s0 is NOT allowed";
string PS_973 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_974_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_974_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, oC0 is NOT allowed";
string PS_974 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_975_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_975_Desc = "ps_2_x : texldd source reg combination b0, oC0, oDepth, oDepth is NOT allowed";
string PS_975 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_976_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_976_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, b0 is NOT allowed";
string PS_976 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_977_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_977_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, i0 is NOT allowed";
string PS_977 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_978_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_978_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, p0 is NOT allowed";
string PS_978 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_979_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_979_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, s0 is NOT allowed";
string PS_979 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_980_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_980_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, oC0 is NOT allowed";
string PS_980 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_981_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_981_Desc = "ps_2_x : texldd source reg combination b0, oDepth, b0, oDepth is NOT allowed";
string PS_981 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_982_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_982_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, b0 is NOT allowed";
string PS_982 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_983_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_983_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, i0 is NOT allowed";
string PS_983 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_984_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_984_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, p0 is NOT allowed";
string PS_984 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_985_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_985_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, s0 is NOT allowed";
string PS_985 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_986_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_986_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, oC0 is NOT allowed";
string PS_986 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_987_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_987_Desc = "ps_2_x : texldd source reg combination b0, oDepth, i0, oDepth is NOT allowed";
string PS_987 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_988_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_988_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, b0 is NOT allowed";
string PS_988 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_989_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_989_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, i0 is NOT allowed";
string PS_989 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_990_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_990_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, p0 is NOT allowed";
string PS_990 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_991_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_991_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, s0 is NOT allowed";
string PS_991 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_992_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_992_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, oC0 is NOT allowed";
string PS_992 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_993_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_993_Desc = "ps_2_x : texldd source reg combination b0, oDepth, p0, oDepth is NOT allowed";
string PS_993 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_994_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_994_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, b0 is NOT allowed";
string PS_994 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_995_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_995_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, i0 is NOT allowed";
string PS_995 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_996_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_996_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, p0 is NOT allowed";
string PS_996 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_997_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_997_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, s0 is NOT allowed";
string PS_997 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_998_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_998_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, oC0 is NOT allowed";
string PS_998 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, b0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_999_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_999_Desc = "ps_2_x : texldd source reg combination b0, oDepth, s0, oDepth is NOT allowed";
string PS_999 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1000_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1000_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, b0 is NOT allowed";
string PS_1000 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1001_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1001_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, i0 is NOT allowed";
string PS_1001 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1002_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1002_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, p0 is NOT allowed";
string PS_1002 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1003_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1003_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, s0 is NOT allowed";
string PS_1003 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, b0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1004_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1004_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, oC0 is NOT allowed";
string PS_1004 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1005_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1005_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oC0, oDepth is NOT allowed";
string PS_1005 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1006_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1006_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, b0 is NOT allowed";
string PS_1006 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1007_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1007_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, i0 is NOT allowed";
string PS_1007 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, b0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1008_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1008_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, p0 is NOT allowed";
string PS_1008 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1009_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1009_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, s0 is NOT allowed";
string PS_1009 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, b0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1010_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1010_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, oC0 is NOT allowed";
string PS_1010 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1011_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1011_Desc = "ps_2_x : texldd source reg combination b0, oDepth, oDepth, oDepth is NOT allowed";
string PS_1011 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, b0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1012_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1012_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, b0 is NOT allowed";
string PS_1012 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1013_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1013_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, i0 is NOT allowed";
string PS_1013 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1014_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1014_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, p0 is NOT allowed";
string PS_1014 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1015_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1015_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, s0 is NOT allowed";
string PS_1015 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, i0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1016_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1016_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, oC0 is NOT allowed";
string PS_1016 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1017_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1017_Desc = "ps_2_x : texldd source reg combination i0, v0, b0, oDepth is NOT allowed";
string PS_1017 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1018_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1018_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, b0 is NOT allowed";
string PS_1018 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1019_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1019_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, i0 is NOT allowed";
string PS_1019 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1020_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1020_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, p0 is NOT allowed";
string PS_1020 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1021_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1021_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, s0 is NOT allowed";
string PS_1021 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, i0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1022_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1022_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, oC0 is NOT allowed";
string PS_1022 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1023_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1023_Desc = "ps_2_x : texldd source reg combination i0, v0, i0, oDepth is NOT allowed";
string PS_1023 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1024_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1024_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, b0 is NOT allowed";
string PS_1024 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1025_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1025_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, i0 is NOT allowed";
string PS_1025 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1026_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1026_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, p0 is NOT allowed";
string PS_1026 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1027_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1027_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, s0 is NOT allowed";
string PS_1027 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1028_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1028_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, oC0 is NOT allowed";
string PS_1028 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1029_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1029_Desc = "ps_2_x : texldd source reg combination i0, v0, p0, oDepth is NOT allowed";
string PS_1029 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1030_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1030_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, b0 is NOT allowed";
string PS_1030 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1031_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1031_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, i0 is NOT allowed";
string PS_1031 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1032_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1032_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, p0 is NOT allowed";
string PS_1032 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1033_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1033_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, s0 is NOT allowed";
string PS_1033 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1034_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1034_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, oC0 is NOT allowed";
string PS_1034 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1035_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1035_Desc = "ps_2_x : texldd source reg combination i0, v0, s0, oDepth is NOT allowed";
string PS_1035 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1036_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1036_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, b0 is NOT allowed";
string PS_1036 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1037_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1037_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, i0 is NOT allowed";
string PS_1037 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1038_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1038_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, p0 is NOT allowed";
string PS_1038 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1039_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1039_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, s0 is NOT allowed";
string PS_1039 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1040_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1040_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, oC0 is NOT allowed";
string PS_1040 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1041_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1041_Desc = "ps_2_x : texldd source reg combination i0, v0, oC0, oDepth is NOT allowed";
string PS_1041 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1042_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1042_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, b0 is NOT allowed";
string PS_1042 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1043_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1043_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, i0 is NOT allowed";
string PS_1043 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1044_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1044_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, p0 is NOT allowed";
string PS_1044 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1045_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1045_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, s0 is NOT allowed";
string PS_1045 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, i0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1046_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1046_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, oC0 is NOT allowed";
string PS_1046 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1047_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1047_Desc = "ps_2_x : texldd source reg combination i0, v0, oDepth, oDepth is NOT allowed";
string PS_1047 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, i0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1048_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1048_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, b0 is NOT allowed";
string PS_1048 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1049_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1049_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, i0 is NOT allowed";
string PS_1049 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1050_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1050_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, p0 is NOT allowed";
string PS_1050 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1051_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1051_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, s0 is NOT allowed";
string PS_1051 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1052_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1052_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, oC0 is NOT allowed";
string PS_1052 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1053_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1053_Desc = "ps_2_x : texldd source reg combination i0, r0, b0, oDepth is NOT allowed";
string PS_1053 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1054_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1054_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, b0 is NOT allowed";
string PS_1054 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1055_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1055_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, i0 is NOT allowed";
string PS_1055 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1056_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1056_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, p0 is NOT allowed";
string PS_1056 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1057_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1057_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, s0 is NOT allowed";
string PS_1057 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1058_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1058_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, oC0 is NOT allowed";
string PS_1058 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1059_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1059_Desc = "ps_2_x : texldd source reg combination i0, r0, i0, oDepth is NOT allowed";
string PS_1059 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1060_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1060_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, b0 is NOT allowed";
string PS_1060 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1061_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1061_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, i0 is NOT allowed";
string PS_1061 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1062_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1062_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, p0 is NOT allowed";
string PS_1062 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1063_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1063_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, s0 is NOT allowed";
string PS_1063 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1064_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1064_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, oC0 is NOT allowed";
string PS_1064 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1065_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1065_Desc = "ps_2_x : texldd source reg combination i0, r0, p0, oDepth is NOT allowed";
string PS_1065 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1066_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1066_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, b0 is NOT allowed";
string PS_1066 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1067_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1067_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, i0 is NOT allowed";
string PS_1067 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1068_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1068_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, p0 is NOT allowed";
string PS_1068 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1069_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1069_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, s0 is NOT allowed";
string PS_1069 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1070_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1070_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, oC0 is NOT allowed";
string PS_1070 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1071_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1071_Desc = "ps_2_x : texldd source reg combination i0, r0, s0, oDepth is NOT allowed";
string PS_1071 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1072_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1072_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, b0 is NOT allowed";
string PS_1072 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1073_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1073_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, i0 is NOT allowed";
string PS_1073 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1074_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1074_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, p0 is NOT allowed";
string PS_1074 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1075_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1075_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, s0 is NOT allowed";
string PS_1075 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1076_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1076_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, oC0 is NOT allowed";
string PS_1076 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1077_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1077_Desc = "ps_2_x : texldd source reg combination i0, r0, oC0, oDepth is NOT allowed";
string PS_1077 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1078_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1078_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, b0 is NOT allowed";
string PS_1078 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1079_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1079_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, i0 is NOT allowed";
string PS_1079 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1080_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1080_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, p0 is NOT allowed";
string PS_1080 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1081_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1081_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, s0 is NOT allowed";
string PS_1081 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1082_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1082_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, oC0 is NOT allowed";
string PS_1082 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1083_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1083_Desc = "ps_2_x : texldd source reg combination i0, r0, oDepth, oDepth is NOT allowed";
string PS_1083 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, i0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1084_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1084_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, b0 is NOT allowed";
string PS_1084 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1085_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1085_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, i0 is NOT allowed";
string PS_1085 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1086_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1086_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, p0 is NOT allowed";
string PS_1086 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1087_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1087_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, s0 is NOT allowed";
string PS_1087 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1088_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1088_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, oC0 is NOT allowed";
string PS_1088 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1089_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1089_Desc = "ps_2_x : texldd source reg combination i0, c0, b0, oDepth is NOT allowed";
string PS_1089 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1090_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1090_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, b0 is NOT allowed";
string PS_1090 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1091_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1091_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, i0 is NOT allowed";
string PS_1091 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1092_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1092_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, p0 is NOT allowed";
string PS_1092 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1093_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1093_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, s0 is NOT allowed";
string PS_1093 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1094_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1094_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, oC0 is NOT allowed";
string PS_1094 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1095_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1095_Desc = "ps_2_x : texldd source reg combination i0, c0, i0, oDepth is NOT allowed";
string PS_1095 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1096_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1096_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, b0 is NOT allowed";
string PS_1096 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1097_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1097_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, i0 is NOT allowed";
string PS_1097 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1098_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1098_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, p0 is NOT allowed";
string PS_1098 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1099_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1099_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, s0 is NOT allowed";
string PS_1099 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1100_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1100_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, oC0 is NOT allowed";
string PS_1100 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1101_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1101_Desc = "ps_2_x : texldd source reg combination i0, c0, p0, oDepth is NOT allowed";
string PS_1101 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1102_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1102_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, b0 is NOT allowed";
string PS_1102 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1103_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1103_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, i0 is NOT allowed";
string PS_1103 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1104_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1104_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, p0 is NOT allowed";
string PS_1104 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1105_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1105_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, s0 is NOT allowed";
string PS_1105 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1106_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1106_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, oC0 is NOT allowed";
string PS_1106 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1107_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1107_Desc = "ps_2_x : texldd source reg combination i0, c0, s0, oDepth is NOT allowed";
string PS_1107 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1108_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1108_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, b0 is NOT allowed";
string PS_1108 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1109_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1109_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, i0 is NOT allowed";
string PS_1109 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1110_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1110_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, p0 is NOT allowed";
string PS_1110 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1111_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1111_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, s0 is NOT allowed";
string PS_1111 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1112_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1112_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, oC0 is NOT allowed";
string PS_1112 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1113_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1113_Desc = "ps_2_x : texldd source reg combination i0, c0, oC0, oDepth is NOT allowed";
string PS_1113 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1114_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1114_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, b0 is NOT allowed";
string PS_1114 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1115_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1115_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, i0 is NOT allowed";
string PS_1115 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1116_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1116_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, p0 is NOT allowed";
string PS_1116 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1117_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1117_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, s0 is NOT allowed";
string PS_1117 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1118_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1118_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, oC0 is NOT allowed";
string PS_1118 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1119_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1119_Desc = "ps_2_x : texldd source reg combination i0, c0, oDepth, oDepth is NOT allowed";
string PS_1119 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1120_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1120_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, b0 is NOT allowed";
string PS_1120 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1121_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1121_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, i0 is NOT allowed";
string PS_1121 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1122_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1122_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, p0 is NOT allowed";
string PS_1122 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1123_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1123_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, s0 is NOT allowed";
string PS_1123 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1124_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1124_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, oC0 is NOT allowed";
string PS_1124 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1125_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1125_Desc = "ps_2_x : texldd source reg combination i0, b0, b0, oDepth is NOT allowed";
string PS_1125 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1126_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1126_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, b0 is NOT allowed";
string PS_1126 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1127_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1127_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, i0 is NOT allowed";
string PS_1127 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1128_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1128_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, p0 is NOT allowed";
string PS_1128 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1129_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1129_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, s0 is NOT allowed";
string PS_1129 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1130_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1130_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, oC0 is NOT allowed";
string PS_1130 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1131_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1131_Desc = "ps_2_x : texldd source reg combination i0, b0, i0, oDepth is NOT allowed";
string PS_1131 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1132_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1132_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, b0 is NOT allowed";
string PS_1132 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1133_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1133_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, i0 is NOT allowed";
string PS_1133 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1134_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1134_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, p0 is NOT allowed";
string PS_1134 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1135_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1135_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, s0 is NOT allowed";
string PS_1135 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1136_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1136_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, oC0 is NOT allowed";
string PS_1136 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1137_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1137_Desc = "ps_2_x : texldd source reg combination i0, b0, p0, oDepth is NOT allowed";
string PS_1137 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1138_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1138_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, b0 is NOT allowed";
string PS_1138 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1139_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1139_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, i0 is NOT allowed";
string PS_1139 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1140_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1140_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, p0 is NOT allowed";
string PS_1140 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1141_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1141_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, s0 is NOT allowed";
string PS_1141 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1142_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1142_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, oC0 is NOT allowed";
string PS_1142 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1143_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1143_Desc = "ps_2_x : texldd source reg combination i0, b0, s0, oDepth is NOT allowed";
string PS_1143 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1144_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1144_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, b0 is NOT allowed";
string PS_1144 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1145_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1145_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, i0 is NOT allowed";
string PS_1145 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1146_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1146_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, p0 is NOT allowed";
string PS_1146 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1147_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1147_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, s0 is NOT allowed";
string PS_1147 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1148_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1148_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, oC0 is NOT allowed";
string PS_1148 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1149_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1149_Desc = "ps_2_x : texldd source reg combination i0, b0, oC0, oDepth is NOT allowed";
string PS_1149 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1150_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1150_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, b0 is NOT allowed";
string PS_1150 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1151_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1151_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, i0 is NOT allowed";
string PS_1151 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1152_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1152_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, p0 is NOT allowed";
string PS_1152 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1153_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1153_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, s0 is NOT allowed";
string PS_1153 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1154_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1154_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, oC0 is NOT allowed";
string PS_1154 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1155_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1155_Desc = "ps_2_x : texldd source reg combination i0, b0, oDepth, oDepth is NOT allowed";
string PS_1155 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1156_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1156_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, b0 is NOT allowed";
string PS_1156 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1157_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1157_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, i0 is NOT allowed";
string PS_1157 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1158_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1158_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, p0 is NOT allowed";
string PS_1158 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1159_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1159_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, s0 is NOT allowed";
string PS_1159 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1160_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1160_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, oC0 is NOT allowed";
string PS_1160 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1161_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1161_Desc = "ps_2_x : texldd source reg combination i0, i0, b0, oDepth is NOT allowed";
string PS_1161 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1162_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1162_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, b0 is NOT allowed";
string PS_1162 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1163_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1163_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, i0 is NOT allowed";
string PS_1163 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1164_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1164_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, p0 is NOT allowed";
string PS_1164 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1165_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1165_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, s0 is NOT allowed";
string PS_1165 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1166_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1166_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, oC0 is NOT allowed";
string PS_1166 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1167_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1167_Desc = "ps_2_x : texldd source reg combination i0, i0, i0, oDepth is NOT allowed";
string PS_1167 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1168_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1168_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, b0 is NOT allowed";
string PS_1168 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1169_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1169_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, i0 is NOT allowed";
string PS_1169 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1170_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1170_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, p0 is NOT allowed";
string PS_1170 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1171_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1171_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, s0 is NOT allowed";
string PS_1171 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1172_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1172_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, oC0 is NOT allowed";
string PS_1172 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1173_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1173_Desc = "ps_2_x : texldd source reg combination i0, i0, p0, oDepth is NOT allowed";
string PS_1173 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1174_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1174_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, b0 is NOT allowed";
string PS_1174 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1175_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1175_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, i0 is NOT allowed";
string PS_1175 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1176_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1176_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, p0 is NOT allowed";
string PS_1176 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1177_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1177_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, s0 is NOT allowed";
string PS_1177 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1178_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1178_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, oC0 is NOT allowed";
string PS_1178 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1179_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1179_Desc = "ps_2_x : texldd source reg combination i0, i0, s0, oDepth is NOT allowed";
string PS_1179 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1180_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1180_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, b0 is NOT allowed";
string PS_1180 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1181_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1181_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, i0 is NOT allowed";
string PS_1181 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1182_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1182_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, p0 is NOT allowed";
string PS_1182 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1183_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1183_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, s0 is NOT allowed";
string PS_1183 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1184_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1184_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, oC0 is NOT allowed";
string PS_1184 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1185_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1185_Desc = "ps_2_x : texldd source reg combination i0, i0, oC0, oDepth is NOT allowed";
string PS_1185 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1186_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1186_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, b0 is NOT allowed";
string PS_1186 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1187_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1187_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, i0 is NOT allowed";
string PS_1187 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1188_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1188_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, p0 is NOT allowed";
string PS_1188 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1189_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1189_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, s0 is NOT allowed";
string PS_1189 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1190_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1190_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, oC0 is NOT allowed";
string PS_1190 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1191_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1191_Desc = "ps_2_x : texldd source reg combination i0, i0, oDepth, oDepth is NOT allowed";
string PS_1191 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1192_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1192_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, b0 is NOT allowed";
string PS_1192 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1193_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1193_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, i0 is NOT allowed";
string PS_1193 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1194_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1194_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, p0 is NOT allowed";
string PS_1194 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1195_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1195_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, s0 is NOT allowed";
string PS_1195 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1196_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1196_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, oC0 is NOT allowed";
string PS_1196 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1197_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1197_Desc = "ps_2_x : texldd source reg combination i0, p0, b0, oDepth is NOT allowed";
string PS_1197 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1198_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1198_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, b0 is NOT allowed";
string PS_1198 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1199_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1199_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, i0 is NOT allowed";
string PS_1199 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1200_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1200_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, p0 is NOT allowed";
string PS_1200 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1201_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1201_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, s0 is NOT allowed";
string PS_1201 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1202_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1202_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, oC0 is NOT allowed";
string PS_1202 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1203_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1203_Desc = "ps_2_x : texldd source reg combination i0, p0, i0, oDepth is NOT allowed";
string PS_1203 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1204_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1204_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, b0 is NOT allowed";
string PS_1204 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1205_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1205_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, i0 is NOT allowed";
string PS_1205 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1206_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1206_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, p0 is NOT allowed";
string PS_1206 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1207_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1207_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, s0 is NOT allowed";
string PS_1207 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1208_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1208_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, oC0 is NOT allowed";
string PS_1208 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1209_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1209_Desc = "ps_2_x : texldd source reg combination i0, p0, p0, oDepth is NOT allowed";
string PS_1209 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1210_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1210_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, b0 is NOT allowed";
string PS_1210 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1211_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1211_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, i0 is NOT allowed";
string PS_1211 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1212_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1212_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, p0 is NOT allowed";
string PS_1212 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1213_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1213_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, s0 is NOT allowed";
string PS_1213 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1214_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1214_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, oC0 is NOT allowed";
string PS_1214 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1215_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1215_Desc = "ps_2_x : texldd source reg combination i0, p0, s0, oDepth is NOT allowed";
string PS_1215 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1216_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1216_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, b0 is NOT allowed";
string PS_1216 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1217_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1217_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, i0 is NOT allowed";
string PS_1217 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1218_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1218_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, p0 is NOT allowed";
string PS_1218 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1219_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1219_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, s0 is NOT allowed";
string PS_1219 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1220_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1220_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, oC0 is NOT allowed";
string PS_1220 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1221_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1221_Desc = "ps_2_x : texldd source reg combination i0, p0, oC0, oDepth is NOT allowed";
string PS_1221 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1222_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1222_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, b0 is NOT allowed";
string PS_1222 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1223_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1223_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, i0 is NOT allowed";
string PS_1223 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1224_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1224_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, p0 is NOT allowed";
string PS_1224 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1225_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1225_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, s0 is NOT allowed";
string PS_1225 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1226_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1226_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, oC0 is NOT allowed";
string PS_1226 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1227_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1227_Desc = "ps_2_x : texldd source reg combination i0, p0, oDepth, oDepth is NOT allowed";
string PS_1227 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1228_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1228_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, b0 is NOT allowed";
string PS_1228 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1229_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1229_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, i0 is NOT allowed";
string PS_1229 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1230_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1230_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, p0 is NOT allowed";
string PS_1230 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1231_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1231_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, s0 is NOT allowed";
string PS_1231 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, i0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1232_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1232_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, oC0 is NOT allowed";
string PS_1232 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1233_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1233_Desc = "ps_2_x : texldd source reg combination i0, t0, b0, oDepth is NOT allowed";
string PS_1233 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1234_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1234_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, b0 is NOT allowed";
string PS_1234 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1235_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1235_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, i0 is NOT allowed";
string PS_1235 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1236_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1236_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, p0 is NOT allowed";
string PS_1236 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1237_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1237_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, s0 is NOT allowed";
string PS_1237 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, i0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1238_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1238_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, oC0 is NOT allowed";
string PS_1238 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1239_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1239_Desc = "ps_2_x : texldd source reg combination i0, t0, i0, oDepth is NOT allowed";
string PS_1239 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1240_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1240_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, b0 is NOT allowed";
string PS_1240 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1241_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1241_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, i0 is NOT allowed";
string PS_1241 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1242_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1242_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, p0 is NOT allowed";
string PS_1242 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1243_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1243_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, s0 is NOT allowed";
string PS_1243 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, i0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1244_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1244_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, oC0 is NOT allowed";
string PS_1244 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1245_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1245_Desc = "ps_2_x : texldd source reg combination i0, t0, p0, oDepth is NOT allowed";
string PS_1245 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1246_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1246_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, b0 is NOT allowed";
string PS_1246 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1247_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1247_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, i0 is NOT allowed";
string PS_1247 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1248_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1248_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, p0 is NOT allowed";
string PS_1248 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1249_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1249_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, s0 is NOT allowed";
string PS_1249 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1250_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1250_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, oC0 is NOT allowed";
string PS_1250 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1251_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1251_Desc = "ps_2_x : texldd source reg combination i0, t0, s0, oDepth is NOT allowed";
string PS_1251 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, i0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1252_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1252_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, b0 is NOT allowed";
string PS_1252 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1253_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1253_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, i0 is NOT allowed";
string PS_1253 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1254_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1254_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, p0 is NOT allowed";
string PS_1254 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1255_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1255_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, s0 is NOT allowed";
string PS_1255 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, i0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1256_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1256_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, oC0 is NOT allowed";
string PS_1256 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1257_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1257_Desc = "ps_2_x : texldd source reg combination i0, t0, oC0, oDepth is NOT allowed";
string PS_1257 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1258_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1258_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, b0 is NOT allowed";
string PS_1258 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1259_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1259_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, i0 is NOT allowed";
string PS_1259 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1260_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1260_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, p0 is NOT allowed";
string PS_1260 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1261_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1261_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, s0 is NOT allowed";
string PS_1261 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, i0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1262_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1262_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, oC0 is NOT allowed";
string PS_1262 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1263_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1263_Desc = "ps_2_x : texldd source reg combination i0, t0, oDepth, oDepth is NOT allowed";
string PS_1263 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, i0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1264_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1264_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, b0 is NOT allowed";
string PS_1264 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1265_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1265_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, i0 is NOT allowed";
string PS_1265 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1266_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1266_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, p0 is NOT allowed";
string PS_1266 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1267_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1267_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, s0 is NOT allowed";
string PS_1267 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1268_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1268_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, oC0 is NOT allowed";
string PS_1268 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1269_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1269_Desc = "ps_2_x : texldd source reg combination i0, oC0, b0, oDepth is NOT allowed";
string PS_1269 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1270_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1270_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, b0 is NOT allowed";
string PS_1270 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1271_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1271_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, i0 is NOT allowed";
string PS_1271 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1272_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1272_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, p0 is NOT allowed";
string PS_1272 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1273_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1273_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, s0 is NOT allowed";
string PS_1273 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1274_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1274_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, oC0 is NOT allowed";
string PS_1274 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1275_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1275_Desc = "ps_2_x : texldd source reg combination i0, oC0, i0, oDepth is NOT allowed";
string PS_1275 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1276_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1276_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, b0 is NOT allowed";
string PS_1276 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1277_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1277_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, i0 is NOT allowed";
string PS_1277 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1278_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1278_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, p0 is NOT allowed";
string PS_1278 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1279_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1279_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, s0 is NOT allowed";
string PS_1279 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1280_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1280_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, oC0 is NOT allowed";
string PS_1280 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1281_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1281_Desc = "ps_2_x : texldd source reg combination i0, oC0, p0, oDepth is NOT allowed";
string PS_1281 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1282_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1282_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, b0 is NOT allowed";
string PS_1282 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1283_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1283_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, i0 is NOT allowed";
string PS_1283 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1284_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1284_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, p0 is NOT allowed";
string PS_1284 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1285_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1285_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, s0 is NOT allowed";
string PS_1285 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1286_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1286_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, oC0 is NOT allowed";
string PS_1286 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1287_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1287_Desc = "ps_2_x : texldd source reg combination i0, oC0, s0, oDepth is NOT allowed";
string PS_1287 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1288_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1288_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, b0 is NOT allowed";
string PS_1288 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1289_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1289_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, i0 is NOT allowed";
string PS_1289 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1290_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1290_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, p0 is NOT allowed";
string PS_1290 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1291_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1291_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, s0 is NOT allowed";
string PS_1291 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1292_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1292_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, oC0 is NOT allowed";
string PS_1292 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1293_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1293_Desc = "ps_2_x : texldd source reg combination i0, oC0, oC0, oDepth is NOT allowed";
string PS_1293 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1294_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1294_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, b0 is NOT allowed";
string PS_1294 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1295_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1295_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, i0 is NOT allowed";
string PS_1295 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1296_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1296_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, p0 is NOT allowed";
string PS_1296 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1297_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1297_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, s0 is NOT allowed";
string PS_1297 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1298_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1298_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, oC0 is NOT allowed";
string PS_1298 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1299_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1299_Desc = "ps_2_x : texldd source reg combination i0, oC0, oDepth, oDepth is NOT allowed";
string PS_1299 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1300_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1300_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, b0 is NOT allowed";
string PS_1300 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1301_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1301_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, i0 is NOT allowed";
string PS_1301 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1302_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1302_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, p0 is NOT allowed";
string PS_1302 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1303_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1303_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, s0 is NOT allowed";
string PS_1303 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1304_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1304_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, oC0 is NOT allowed";
string PS_1304 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1305_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1305_Desc = "ps_2_x : texldd source reg combination i0, oDepth, b0, oDepth is NOT allowed";
string PS_1305 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1306_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1306_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, b0 is NOT allowed";
string PS_1306 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1307_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1307_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, i0 is NOT allowed";
string PS_1307 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1308_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1308_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, p0 is NOT allowed";
string PS_1308 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1309_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1309_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, s0 is NOT allowed";
string PS_1309 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1310_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1310_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, oC0 is NOT allowed";
string PS_1310 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1311_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1311_Desc = "ps_2_x : texldd source reg combination i0, oDepth, i0, oDepth is NOT allowed";
string PS_1311 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1312_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1312_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, b0 is NOT allowed";
string PS_1312 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1313_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1313_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, i0 is NOT allowed";
string PS_1313 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1314_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1314_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, p0 is NOT allowed";
string PS_1314 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1315_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1315_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, s0 is NOT allowed";
string PS_1315 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, i0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1316_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1316_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, oC0 is NOT allowed";
string PS_1316 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1317_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1317_Desc = "ps_2_x : texldd source reg combination i0, oDepth, p0, oDepth is NOT allowed";
string PS_1317 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1318_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1318_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, b0 is NOT allowed";
string PS_1318 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1319_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1319_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, i0 is NOT allowed";
string PS_1319 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1320_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1320_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, p0 is NOT allowed";
string PS_1320 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1321_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1321_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, s0 is NOT allowed";
string PS_1321 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1322_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1322_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, oC0 is NOT allowed";
string PS_1322 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1323_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1323_Desc = "ps_2_x : texldd source reg combination i0, oDepth, s0, oDepth is NOT allowed";
string PS_1323 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1324_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1324_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, b0 is NOT allowed";
string PS_1324 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1325_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1325_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, i0 is NOT allowed";
string PS_1325 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1326_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1326_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, p0 is NOT allowed";
string PS_1326 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1327_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1327_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, s0 is NOT allowed";
string PS_1327 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, i0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1328_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1328_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, oC0 is NOT allowed";
string PS_1328 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1329_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1329_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oC0, oDepth is NOT allowed";
string PS_1329 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1330_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1330_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, b0 is NOT allowed";
string PS_1330 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1331_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1331_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, i0 is NOT allowed";
string PS_1331 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1332_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1332_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, p0 is NOT allowed";
string PS_1332 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1333_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1333_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, s0 is NOT allowed";
string PS_1333 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, i0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1334_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1334_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, oC0 is NOT allowed";
string PS_1334 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1335_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1335_Desc = "ps_2_x : texldd source reg combination i0, oDepth, oDepth, oDepth is NOT allowed";
string PS_1335 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, i0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1336_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1336_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, b0 is NOT allowed";
string PS_1336 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1337_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1337_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, i0 is NOT allowed";
string PS_1337 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1338_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1338_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, p0 is NOT allowed";
string PS_1338 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1339_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1339_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, s0 is NOT allowed";
string PS_1339 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, p0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1340_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1340_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, oC0 is NOT allowed";
string PS_1340 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1341_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1341_Desc = "ps_2_x : texldd source reg combination p0, v0, b0, oDepth is NOT allowed";
string PS_1341 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1342_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1342_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, b0 is NOT allowed";
string PS_1342 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1343_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1343_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, i0 is NOT allowed";
string PS_1343 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1344_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1344_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, p0 is NOT allowed";
string PS_1344 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1345_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1345_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, s0 is NOT allowed";
string PS_1345 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, p0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1346_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1346_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, oC0 is NOT allowed";
string PS_1346 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1347_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1347_Desc = "ps_2_x : texldd source reg combination p0, v0, i0, oDepth is NOT allowed";
string PS_1347 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1348_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1348_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, b0 is NOT allowed";
string PS_1348 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1349_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1349_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, i0 is NOT allowed";
string PS_1349 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1350_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1350_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, p0 is NOT allowed";
string PS_1350 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1351_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1351_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, s0 is NOT allowed";
string PS_1351 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, p0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1352_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1352_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, oC0 is NOT allowed";
string PS_1352 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1353_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1353_Desc = "ps_2_x : texldd source reg combination p0, v0, p0, oDepth is NOT allowed";
string PS_1353 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1354_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1354_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, b0 is NOT allowed";
string PS_1354 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, p0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1355_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1355_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, i0 is NOT allowed";
string PS_1355 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1356_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1356_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, p0 is NOT allowed";
string PS_1356 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1357_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1357_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, s0 is NOT allowed";
string PS_1357 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1358_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1358_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, oC0 is NOT allowed";
string PS_1358 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1359_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1359_Desc = "ps_2_x : texldd source reg combination p0, v0, s0, oDepth is NOT allowed";
string PS_1359 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1360_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1360_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, b0 is NOT allowed";
string PS_1360 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1361_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1361_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, i0 is NOT allowed";
string PS_1361 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1362_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1362_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, p0 is NOT allowed";
string PS_1362 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1363_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1363_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, s0 is NOT allowed";
string PS_1363 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, p0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1364_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1364_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, oC0 is NOT allowed";
string PS_1364 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1365_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1365_Desc = "ps_2_x : texldd source reg combination p0, v0, oC0, oDepth is NOT allowed";
string PS_1365 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1366_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1366_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, b0 is NOT allowed";
string PS_1366 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, p0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1367_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1367_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, i0 is NOT allowed";
string PS_1367 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, p0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1368_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1368_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, p0 is NOT allowed";
string PS_1368 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1369_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1369_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, s0 is NOT allowed";
string PS_1369 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, p0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1370_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1370_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, oC0 is NOT allowed";
string PS_1370 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1371_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1371_Desc = "ps_2_x : texldd source reg combination p0, v0, oDepth, oDepth is NOT allowed";
string PS_1371 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, p0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1372_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1372_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, b0 is NOT allowed";
string PS_1372 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1373_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1373_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, i0 is NOT allowed";
string PS_1373 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1374_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1374_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, p0 is NOT allowed";
string PS_1374 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1375_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1375_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, s0 is NOT allowed";
string PS_1375 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1376_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1376_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, oC0 is NOT allowed";
string PS_1376 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1377_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1377_Desc = "ps_2_x : texldd source reg combination p0, r0, b0, oDepth is NOT allowed";
string PS_1377 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1378_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1378_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, b0 is NOT allowed";
string PS_1378 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1379_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1379_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, i0 is NOT allowed";
string PS_1379 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1380_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1380_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, p0 is NOT allowed";
string PS_1380 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1381_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1381_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, s0 is NOT allowed";
string PS_1381 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1382_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1382_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, oC0 is NOT allowed";
string PS_1382 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1383_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1383_Desc = "ps_2_x : texldd source reg combination p0, r0, i0, oDepth is NOT allowed";
string PS_1383 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1384_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1384_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, b0 is NOT allowed";
string PS_1384 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1385_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1385_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, i0 is NOT allowed";
string PS_1385 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1386_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1386_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, p0 is NOT allowed";
string PS_1386 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1387_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1387_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, s0 is NOT allowed";
string PS_1387 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1388_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1388_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, oC0 is NOT allowed";
string PS_1388 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1389_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1389_Desc = "ps_2_x : texldd source reg combination p0, r0, p0, oDepth is NOT allowed";
string PS_1389 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1390_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1390_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, b0 is NOT allowed";
string PS_1390 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1391_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1391_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, i0 is NOT allowed";
string PS_1391 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1392_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1392_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, p0 is NOT allowed";
string PS_1392 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1393_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1393_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, s0 is NOT allowed";
string PS_1393 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1394_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1394_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, oC0 is NOT allowed";
string PS_1394 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1395_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1395_Desc = "ps_2_x : texldd source reg combination p0, r0, s0, oDepth is NOT allowed";
string PS_1395 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1396_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1396_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, b0 is NOT allowed";
string PS_1396 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1397_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1397_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, i0 is NOT allowed";
string PS_1397 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1398_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1398_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, p0 is NOT allowed";
string PS_1398 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1399_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1399_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, s0 is NOT allowed";
string PS_1399 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1400_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1400_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, oC0 is NOT allowed";
string PS_1400 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1401_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1401_Desc = "ps_2_x : texldd source reg combination p0, r0, oC0, oDepth is NOT allowed";
string PS_1401 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1402_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1402_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, b0 is NOT allowed";
string PS_1402 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1403_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1403_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, i0 is NOT allowed";
string PS_1403 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1404_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1404_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, p0 is NOT allowed";
string PS_1404 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1405_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1405_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, s0 is NOT allowed";
string PS_1405 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1406_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1406_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, oC0 is NOT allowed";
string PS_1406 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1407_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1407_Desc = "ps_2_x : texldd source reg combination p0, r0, oDepth, oDepth is NOT allowed";
string PS_1407 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, p0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1408_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1408_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, b0 is NOT allowed";
string PS_1408 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1409_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1409_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, i0 is NOT allowed";
string PS_1409 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1410_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1410_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, p0 is NOT allowed";
string PS_1410 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1411_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1411_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, s0 is NOT allowed";
string PS_1411 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, p0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1412_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1412_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, oC0 is NOT allowed";
string PS_1412 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1413_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1413_Desc = "ps_2_x : texldd source reg combination p0, c0, b0, oDepth is NOT allowed";
string PS_1413 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1414_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1414_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, b0 is NOT allowed";
string PS_1414 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1415_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1415_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, i0 is NOT allowed";
string PS_1415 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1416_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1416_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, p0 is NOT allowed";
string PS_1416 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1417_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1417_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, s0 is NOT allowed";
string PS_1417 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1418_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1418_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, oC0 is NOT allowed";
string PS_1418 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1419_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1419_Desc = "ps_2_x : texldd source reg combination p0, c0, i0, oDepth is NOT allowed";
string PS_1419 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1420_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1420_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, b0 is NOT allowed";
string PS_1420 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1421_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1421_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, i0 is NOT allowed";
string PS_1421 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1422_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1422_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, p0 is NOT allowed";
string PS_1422 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1423_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1423_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, s0 is NOT allowed";
string PS_1423 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, p0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1424_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1424_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, oC0 is NOT allowed";
string PS_1424 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1425_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1425_Desc = "ps_2_x : texldd source reg combination p0, c0, p0, oDepth is NOT allowed";
string PS_1425 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1426_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1426_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, b0 is NOT allowed";
string PS_1426 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1427_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1427_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, i0 is NOT allowed";
string PS_1427 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1428_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1428_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, p0 is NOT allowed";
string PS_1428 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1429_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1429_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, s0 is NOT allowed";
string PS_1429 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1430_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1430_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, oC0 is NOT allowed";
string PS_1430 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1431_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1431_Desc = "ps_2_x : texldd source reg combination p0, c0, s0, oDepth is NOT allowed";
string PS_1431 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, p0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1432_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1432_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, b0 is NOT allowed";
string PS_1432 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1433_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1433_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, i0 is NOT allowed";
string PS_1433 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1434_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1434_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, p0 is NOT allowed";
string PS_1434 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1435_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1435_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, s0 is NOT allowed";
string PS_1435 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, p0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1436_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1436_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, oC0 is NOT allowed";
string PS_1436 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1437_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1437_Desc = "ps_2_x : texldd source reg combination p0, c0, oC0, oDepth is NOT allowed";
string PS_1437 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1438_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1438_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, b0 is NOT allowed";
string PS_1438 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1439_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1439_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, i0 is NOT allowed";
string PS_1439 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1440_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1440_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, p0 is NOT allowed";
string PS_1440 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1441_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1441_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, s0 is NOT allowed";
string PS_1441 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, p0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1442_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1442_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, oC0 is NOT allowed";
string PS_1442 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1443_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1443_Desc = "ps_2_x : texldd source reg combination p0, c0, oDepth, oDepth is NOT allowed";
string PS_1443 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, p0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1444_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1444_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, b0 is NOT allowed";
string PS_1444 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1445_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1445_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, i0 is NOT allowed";
string PS_1445 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1446_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1446_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, p0 is NOT allowed";
string PS_1446 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1447_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1447_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, s0 is NOT allowed";
string PS_1447 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, p0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1448_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1448_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, oC0 is NOT allowed";
string PS_1448 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1449_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1449_Desc = "ps_2_x : texldd source reg combination p0, b0, b0, oDepth is NOT allowed";
string PS_1449 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1450_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1450_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, b0 is NOT allowed";
string PS_1450 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1451_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1451_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, i0 is NOT allowed";
string PS_1451 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1452_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1452_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, p0 is NOT allowed";
string PS_1452 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1453_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1453_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, s0 is NOT allowed";
string PS_1453 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, p0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1454_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1454_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, oC0 is NOT allowed";
string PS_1454 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1455_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1455_Desc = "ps_2_x : texldd source reg combination p0, b0, i0, oDepth is NOT allowed";
string PS_1455 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1456_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1456_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, b0 is NOT allowed";
string PS_1456 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1457_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1457_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, i0 is NOT allowed";
string PS_1457 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1458_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1458_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, p0 is NOT allowed";
string PS_1458 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1459_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1459_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, s0 is NOT allowed";
string PS_1459 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1460_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1460_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, oC0 is NOT allowed";
string PS_1460 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1461_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1461_Desc = "ps_2_x : texldd source reg combination p0, b0, p0, oDepth is NOT allowed";
string PS_1461 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1462_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1462_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, b0 is NOT allowed";
string PS_1462 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1463_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1463_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, i0 is NOT allowed";
string PS_1463 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1464_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1464_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, p0 is NOT allowed";
string PS_1464 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1465_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1465_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, s0 is NOT allowed";
string PS_1465 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1466_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1466_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, oC0 is NOT allowed";
string PS_1466 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1467_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1467_Desc = "ps_2_x : texldd source reg combination p0, b0, s0, oDepth is NOT allowed";
string PS_1467 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1468_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1468_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, b0 is NOT allowed";
string PS_1468 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1469_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1469_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, i0 is NOT allowed";
string PS_1469 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1470_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1470_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, p0 is NOT allowed";
string PS_1470 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1471_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1471_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, s0 is NOT allowed";
string PS_1471 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, p0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1472_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1472_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, oC0 is NOT allowed";
string PS_1472 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1473_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1473_Desc = "ps_2_x : texldd source reg combination p0, b0, oC0, oDepth is NOT allowed";
string PS_1473 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1474_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1474_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, b0 is NOT allowed";
string PS_1474 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1475_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1475_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, i0 is NOT allowed";
string PS_1475 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1476_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1476_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, p0 is NOT allowed";
string PS_1476 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1477_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1477_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, s0 is NOT allowed";
string PS_1477 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1478_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1478_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, oC0 is NOT allowed";
string PS_1478 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1479_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1479_Desc = "ps_2_x : texldd source reg combination p0, b0, oDepth, oDepth is NOT allowed";
string PS_1479 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1480_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1480_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, b0 is NOT allowed";
string PS_1480 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1481_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1481_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, i0 is NOT allowed";
string PS_1481 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1482_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1482_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, p0 is NOT allowed";
string PS_1482 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1483_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1483_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, s0 is NOT allowed";
string PS_1483 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, p0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1484_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1484_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, oC0 is NOT allowed";
string PS_1484 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1485_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1485_Desc = "ps_2_x : texldd source reg combination p0, i0, b0, oDepth is NOT allowed";
string PS_1485 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1486_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1486_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, b0 is NOT allowed";
string PS_1486 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1487_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1487_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, i0 is NOT allowed";
string PS_1487 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1488_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1488_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, p0 is NOT allowed";
string PS_1488 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1489_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1489_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, s0 is NOT allowed";
string PS_1489 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1490_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1490_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, oC0 is NOT allowed";
string PS_1490 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1491_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1491_Desc = "ps_2_x : texldd source reg combination p0, i0, i0, oDepth is NOT allowed";
string PS_1491 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1492_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1492_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, b0 is NOT allowed";
string PS_1492 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1493_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1493_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, i0 is NOT allowed";
string PS_1493 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1494_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1494_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, p0 is NOT allowed";
string PS_1494 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1495_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1495_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, s0 is NOT allowed";
string PS_1495 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1496_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1496_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, oC0 is NOT allowed";
string PS_1496 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1497_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1497_Desc = "ps_2_x : texldd source reg combination p0, i0, p0, oDepth is NOT allowed";
string PS_1497 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1498_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1498_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, b0 is NOT allowed";
string PS_1498 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1499_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1499_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, i0 is NOT allowed";
string PS_1499 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1500_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1500_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, p0 is NOT allowed";
string PS_1500 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1501_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1501_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, s0 is NOT allowed";
string PS_1501 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1502_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1502_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, oC0 is NOT allowed";
string PS_1502 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1503_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1503_Desc = "ps_2_x : texldd source reg combination p0, i0, s0, oDepth is NOT allowed";
string PS_1503 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1504_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1504_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, b0 is NOT allowed";
string PS_1504 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1505_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1505_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, i0 is NOT allowed";
string PS_1505 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1506_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1506_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, p0 is NOT allowed";
string PS_1506 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1507_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1507_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, s0 is NOT allowed";
string PS_1507 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1508_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1508_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, oC0 is NOT allowed";
string PS_1508 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1509_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1509_Desc = "ps_2_x : texldd source reg combination p0, i0, oC0, oDepth is NOT allowed";
string PS_1509 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1510_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1510_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, b0 is NOT allowed";
string PS_1510 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1511_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1511_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, i0 is NOT allowed";
string PS_1511 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1512_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1512_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, p0 is NOT allowed";
string PS_1512 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1513_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1513_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, s0 is NOT allowed";
string PS_1513 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, p0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1514_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1514_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, oC0 is NOT allowed";
string PS_1514 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1515_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1515_Desc = "ps_2_x : texldd source reg combination p0, i0, oDepth, oDepth is NOT allowed";
string PS_1515 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1516_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1516_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, b0 is NOT allowed";
string PS_1516 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1517_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1517_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, i0 is NOT allowed";
string PS_1517 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1518_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1518_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, p0 is NOT allowed";
string PS_1518 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1519_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1519_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, s0 is NOT allowed";
string PS_1519 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, p0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1520_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1520_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, oC0 is NOT allowed";
string PS_1520 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1521_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1521_Desc = "ps_2_x : texldd source reg combination p0, p0, b0, oDepth is NOT allowed";
string PS_1521 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1522_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1522_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, b0 is NOT allowed";
string PS_1522 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1523_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1523_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, i0 is NOT allowed";
string PS_1523 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1524_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1524_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, p0 is NOT allowed";
string PS_1524 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1525_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1525_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, s0 is NOT allowed";
string PS_1525 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1526_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1526_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, oC0 is NOT allowed";
string PS_1526 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1527_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1527_Desc = "ps_2_x : texldd source reg combination p0, p0, i0, oDepth is NOT allowed";
string PS_1527 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1528_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1528_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, b0 is NOT allowed";
string PS_1528 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1529_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1529_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, i0 is NOT allowed";
string PS_1529 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1530_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1530_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, p0 is NOT allowed";
string PS_1530 = 
	"ps_2_x "
	"texldd r0, p0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1531_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1531_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, s0 is NOT allowed";
string PS_1531 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, p0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1532_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1532_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, oC0 is NOT allowed";
string PS_1532 = 
	"ps_2_x "
	"texldd r0, p0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1533_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1533_Desc = "ps_2_x : texldd source reg combination p0, p0, p0, oDepth is NOT allowed";
string PS_1533 = 
	"ps_2_x "
	"texldd r0, p0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1534_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1534_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, b0 is NOT allowed";
string PS_1534 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, p0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1535_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1535_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, i0 is NOT allowed";
string PS_1535 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1536_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1536_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, p0 is NOT allowed";
string PS_1536 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1537_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1537_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, s0 is NOT allowed";
string PS_1537 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1538_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1538_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, oC0 is NOT allowed";
string PS_1538 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1539_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1539_Desc = "ps_2_x : texldd source reg combination p0, p0, s0, oDepth is NOT allowed";
string PS_1539 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1540_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1540_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, b0 is NOT allowed";
string PS_1540 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1541_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1541_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, i0 is NOT allowed";
string PS_1541 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1542_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1542_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, p0 is NOT allowed";
string PS_1542 = 
	"ps_2_x "
	"texldd r0, p0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1543_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1543_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, s0 is NOT allowed";
string PS_1543 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1544_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1544_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, oC0 is NOT allowed";
string PS_1544 = 
	"ps_2_x "
	"texldd r0, p0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1545_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1545_Desc = "ps_2_x : texldd source reg combination p0, p0, oC0, oDepth is NOT allowed";
string PS_1545 = 
	"ps_2_x "
	"texldd r0, p0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1546_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1546_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, b0 is NOT allowed";
string PS_1546 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1547_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1547_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, i0 is NOT allowed";
string PS_1547 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1548_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1548_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, p0 is NOT allowed";
string PS_1548 = 
	"ps_2_x "
	"texldd r0, p0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1549_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1549_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, s0 is NOT allowed";
string PS_1549 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1550_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1550_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, oC0 is NOT allowed";
string PS_1550 = 
	"ps_2_x "
	"texldd r0, p0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1551_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1551_Desc = "ps_2_x : texldd source reg combination p0, p0, oDepth, oDepth is NOT allowed";
string PS_1551 = 
	"ps_2_x "
	"texldd r0, p0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1552_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1552_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, b0 is NOT allowed";
string PS_1552 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1553_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1553_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, i0 is NOT allowed";
string PS_1553 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1554_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1554_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, p0 is NOT allowed";
string PS_1554 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1555_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1555_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, s0 is NOT allowed";
string PS_1555 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, p0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1556_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1556_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, oC0 is NOT allowed";
string PS_1556 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1557_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1557_Desc = "ps_2_x : texldd source reg combination p0, t0, b0, oDepth is NOT allowed";
string PS_1557 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1558_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1558_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, b0 is NOT allowed";
string PS_1558 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1559_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1559_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, i0 is NOT allowed";
string PS_1559 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1560_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1560_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, p0 is NOT allowed";
string PS_1560 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1561_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1561_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, s0 is NOT allowed";
string PS_1561 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1562_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1562_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, oC0 is NOT allowed";
string PS_1562 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1563_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1563_Desc = "ps_2_x : texldd source reg combination p0, t0, i0, oDepth is NOT allowed";
string PS_1563 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1564_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1564_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, b0 is NOT allowed";
string PS_1564 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1565_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1565_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, i0 is NOT allowed";
string PS_1565 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1566_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1566_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, p0 is NOT allowed";
string PS_1566 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1567_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1567_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, s0 is NOT allowed";
string PS_1567 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, p0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1568_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1568_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, oC0 is NOT allowed";
string PS_1568 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1569_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1569_Desc = "ps_2_x : texldd source reg combination p0, t0, p0, oDepth is NOT allowed";
string PS_1569 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1570_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1570_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, b0 is NOT allowed";
string PS_1570 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1571_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1571_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, i0 is NOT allowed";
string PS_1571 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1572_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1572_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, p0 is NOT allowed";
string PS_1572 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1573_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1573_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, s0 is NOT allowed";
string PS_1573 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1574_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1574_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, oC0 is NOT allowed";
string PS_1574 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1575_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1575_Desc = "ps_2_x : texldd source reg combination p0, t0, s0, oDepth is NOT allowed";
string PS_1575 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1576_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1576_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, b0 is NOT allowed";
string PS_1576 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1577_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1577_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, i0 is NOT allowed";
string PS_1577 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1578_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1578_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, p0 is NOT allowed";
string PS_1578 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1579_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1579_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, s0 is NOT allowed";
string PS_1579 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, p0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1580_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1580_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, oC0 is NOT allowed";
string PS_1580 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1581_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1581_Desc = "ps_2_x : texldd source reg combination p0, t0, oC0, oDepth is NOT allowed";
string PS_1581 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1582_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1582_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, b0 is NOT allowed";
string PS_1582 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, p0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1583_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1583_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, i0 is NOT allowed";
string PS_1583 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, p0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1584_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1584_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, p0 is NOT allowed";
string PS_1584 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1585_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1585_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, s0 is NOT allowed";
string PS_1585 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, p0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1586_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1586_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, oC0 is NOT allowed";
string PS_1586 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1587_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1587_Desc = "ps_2_x : texldd source reg combination p0, t0, oDepth, oDepth is NOT allowed";
string PS_1587 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, p0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1588_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1588_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, b0 is NOT allowed";
string PS_1588 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1589_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1589_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, i0 is NOT allowed";
string PS_1589 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1590_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1590_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, p0 is NOT allowed";
string PS_1590 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1591_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1591_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, s0 is NOT allowed";
string PS_1591 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, p0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1592_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1592_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, oC0 is NOT allowed";
string PS_1592 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1593_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1593_Desc = "ps_2_x : texldd source reg combination p0, oC0, b0, oDepth is NOT allowed";
string PS_1593 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1594_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1594_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, b0 is NOT allowed";
string PS_1594 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1595_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1595_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, i0 is NOT allowed";
string PS_1595 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1596_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1596_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, p0 is NOT allowed";
string PS_1596 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1597_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1597_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, s0 is NOT allowed";
string PS_1597 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, p0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1598_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1598_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, oC0 is NOT allowed";
string PS_1598 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1599_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1599_Desc = "ps_2_x : texldd source reg combination p0, oC0, i0, oDepth is NOT allowed";
string PS_1599 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1600_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1600_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, b0 is NOT allowed";
string PS_1600 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1601_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1601_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, i0 is NOT allowed";
string PS_1601 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1602_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1602_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, p0 is NOT allowed";
string PS_1602 = 
	"ps_2_x "
	"texldd r0, p0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1603_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1603_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, s0 is NOT allowed";
string PS_1603 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1604_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1604_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, oC0 is NOT allowed";
string PS_1604 = 
	"ps_2_x "
	"texldd r0, p0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1605_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1605_Desc = "ps_2_x : texldd source reg combination p0, oC0, p0, oDepth is NOT allowed";
string PS_1605 = 
	"ps_2_x "
	"texldd r0, p0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1606_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1606_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, b0 is NOT allowed";
string PS_1606 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, p0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1607_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1607_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, i0 is NOT allowed";
string PS_1607 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1608_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1608_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, p0 is NOT allowed";
string PS_1608 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1609_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1609_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, s0 is NOT allowed";
string PS_1609 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, p0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1610_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1610_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, oC0 is NOT allowed";
string PS_1610 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, p0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1611_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1611_Desc = "ps_2_x : texldd source reg combination p0, oC0, s0, oDepth is NOT allowed";
string PS_1611 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, p0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1612_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1612_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, b0 is NOT allowed";
string PS_1612 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1613_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1613_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, i0 is NOT allowed";
string PS_1613 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1614_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1614_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, p0 is NOT allowed";
string PS_1614 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1615_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1615_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, s0 is NOT allowed";
string PS_1615 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1616_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1616_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, oC0 is NOT allowed";
string PS_1616 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1617_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1617_Desc = "ps_2_x : texldd source reg combination p0, oC0, oC0, oDepth is NOT allowed";
string PS_1617 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1618_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1618_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, b0 is NOT allowed";
string PS_1618 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1619_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1619_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, i0 is NOT allowed";
string PS_1619 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1620_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1620_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, p0 is NOT allowed";
string PS_1620 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1621_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1621_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, s0 is NOT allowed";
string PS_1621 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1622_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1622_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, oC0 is NOT allowed";
string PS_1622 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1623_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1623_Desc = "ps_2_x : texldd source reg combination p0, oC0, oDepth, oDepth is NOT allowed";
string PS_1623 = 
	"ps_2_x "
	"texldd r0, p0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1624_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1624_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, b0 is NOT allowed";
string PS_1624 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1625_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1625_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, i0 is NOT allowed";
string PS_1625 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, p0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1626_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1626_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, p0 is NOT allowed";
string PS_1626 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1627_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1627_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, s0 is NOT allowed";
string PS_1627 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, p0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1628_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1628_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, oC0 is NOT allowed";
string PS_1628 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1629_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1629_Desc = "ps_2_x : texldd source reg combination p0, oDepth, b0, oDepth is NOT allowed";
string PS_1629 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1630_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1630_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, b0 is NOT allowed";
string PS_1630 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1631_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1631_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, i0 is NOT allowed";
string PS_1631 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1632_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1632_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, p0 is NOT allowed";
string PS_1632 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1633_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1633_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, s0 is NOT allowed";
string PS_1633 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, p0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1634_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1634_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, oC0 is NOT allowed";
string PS_1634 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1635_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1635_Desc = "ps_2_x : texldd source reg combination p0, oDepth, i0, oDepth is NOT allowed";
string PS_1635 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1636_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1636_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, b0 is NOT allowed";
string PS_1636 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1637_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1637_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, i0 is NOT allowed";
string PS_1637 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1638_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1638_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, p0 is NOT allowed";
string PS_1638 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1639_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1639_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, s0 is NOT allowed";
string PS_1639 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1640_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1640_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, oC0 is NOT allowed";
string PS_1640 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1641_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1641_Desc = "ps_2_x : texldd source reg combination p0, oDepth, p0, oDepth is NOT allowed";
string PS_1641 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1642_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1642_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, b0 is NOT allowed";
string PS_1642 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1643_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1643_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, i0 is NOT allowed";
string PS_1643 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1644_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1644_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, p0 is NOT allowed";
string PS_1644 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1645_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1645_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, s0 is NOT allowed";
string PS_1645 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1646_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1646_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, oC0 is NOT allowed";
string PS_1646 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1647_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1647_Desc = "ps_2_x : texldd source reg combination p0, oDepth, s0, oDepth is NOT allowed";
string PS_1647 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1648_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1648_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, b0 is NOT allowed";
string PS_1648 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1649_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1649_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, i0 is NOT allowed";
string PS_1649 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1650_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1650_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, p0 is NOT allowed";
string PS_1650 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1651_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1651_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, s0 is NOT allowed";
string PS_1651 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, p0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1652_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1652_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, oC0 is NOT allowed";
string PS_1652 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1653_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1653_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oC0, oDepth is NOT allowed";
string PS_1653 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1654_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1654_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, b0 is NOT allowed";
string PS_1654 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, p0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1655_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1655_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, i0 is NOT allowed";
string PS_1655 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, p0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1656_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1656_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, p0 is NOT allowed";
string PS_1656 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1657_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1657_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, s0 is NOT allowed";
string PS_1657 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, p0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1658_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1658_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, oC0 is NOT allowed";
string PS_1658 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1659_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1659_Desc = "ps_2_x : texldd source reg combination p0, oDepth, oDepth, oDepth is NOT allowed";
string PS_1659 = 
	"ps_2_x "
	"texldd r0, p0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1660_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1660_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, b0 is NOT allowed";
string PS_1660 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1661_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1661_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, i0 is NOT allowed";
string PS_1661 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1662_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1662_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, p0 is NOT allowed";
string PS_1662 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1663_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1663_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, s0 is NOT allowed";
string PS_1663 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1664_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1664_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, oC0 is NOT allowed";
string PS_1664 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1665_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1665_Desc = "ps_2_x : texldd source reg combination s0, v0, b0, oDepth is NOT allowed";
string PS_1665 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1666_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1666_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, b0 is NOT allowed";
string PS_1666 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1667_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1667_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, i0 is NOT allowed";
string PS_1667 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1668_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1668_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, p0 is NOT allowed";
string PS_1668 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1669_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1669_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, s0 is NOT allowed";
string PS_1669 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1670_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1670_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, oC0 is NOT allowed";
string PS_1670 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1671_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1671_Desc = "ps_2_x : texldd source reg combination s0, v0, i0, oDepth is NOT allowed";
string PS_1671 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1672_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1672_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, b0 is NOT allowed";
string PS_1672 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1673_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1673_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, i0 is NOT allowed";
string PS_1673 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1674_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1674_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, p0 is NOT allowed";
string PS_1674 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1675_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1675_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, s0 is NOT allowed";
string PS_1675 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1676_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1676_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, oC0 is NOT allowed";
string PS_1676 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1677_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1677_Desc = "ps_2_x : texldd source reg combination s0, v0, p0, oDepth is NOT allowed";
string PS_1677 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, s0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1678_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1678_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, b0 is NOT allowed";
string PS_1678 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1679_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1679_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, i0 is NOT allowed";
string PS_1679 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1680_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1680_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, p0 is NOT allowed";
string PS_1680 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1681_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1681_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, s0 is NOT allowed";
string PS_1681 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1682_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1682_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, oC0 is NOT allowed";
string PS_1682 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1683_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1683_Desc = "ps_2_x : texldd source reg combination s0, v0, s0, oDepth is NOT allowed";
string PS_1683 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1684_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1684_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, b0 is NOT allowed";
string PS_1684 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1685_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1685_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, i0 is NOT allowed";
string PS_1685 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1686_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1686_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, p0 is NOT allowed";
string PS_1686 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1687_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1687_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, s0 is NOT allowed";
string PS_1687 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1688_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1688_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, oC0 is NOT allowed";
string PS_1688 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1689_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1689_Desc = "ps_2_x : texldd source reg combination s0, v0, oC0, oDepth is NOT allowed";
string PS_1689 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1690_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1690_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, b0 is NOT allowed";
string PS_1690 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1691_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1691_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, i0 is NOT allowed";
string PS_1691 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1692_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1692_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, p0 is NOT allowed";
string PS_1692 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1693_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1693_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, s0 is NOT allowed";
string PS_1693 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1694_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1694_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, oC0 is NOT allowed";
string PS_1694 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1695_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1695_Desc = "ps_2_x : texldd source reg combination s0, v0, oDepth, oDepth is NOT allowed";
string PS_1695 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl v0 "
	"texldd r0, s0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1696_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1696_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, b0 is NOT allowed";
string PS_1696 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1697_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1697_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, i0 is NOT allowed";
string PS_1697 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1698_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1698_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, p0 is NOT allowed";
string PS_1698 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1699_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1699_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, s0 is NOT allowed";
string PS_1699 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1700_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1700_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, oC0 is NOT allowed";
string PS_1700 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1701_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1701_Desc = "ps_2_x : texldd source reg combination s0, r0, b0, oDepth is NOT allowed";
string PS_1701 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1702_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1702_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, b0 is NOT allowed";
string PS_1702 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1703_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1703_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, i0 is NOT allowed";
string PS_1703 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1704_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1704_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, p0 is NOT allowed";
string PS_1704 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1705_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1705_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, s0 is NOT allowed";
string PS_1705 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1706_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1706_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, oC0 is NOT allowed";
string PS_1706 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1707_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1707_Desc = "ps_2_x : texldd source reg combination s0, r0, i0, oDepth is NOT allowed";
string PS_1707 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1708_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1708_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, b0 is NOT allowed";
string PS_1708 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1709_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1709_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, i0 is NOT allowed";
string PS_1709 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1710_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1710_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, p0 is NOT allowed";
string PS_1710 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1711_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1711_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, s0 is NOT allowed";
string PS_1711 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1712_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1712_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, oC0 is NOT allowed";
string PS_1712 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1713_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1713_Desc = "ps_2_x : texldd source reg combination s0, r0, p0, oDepth is NOT allowed";
string PS_1713 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1714_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1714_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, b0 is NOT allowed";
string PS_1714 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1715_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1715_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, i0 is NOT allowed";
string PS_1715 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1716_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1716_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, p0 is NOT allowed";
string PS_1716 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1717_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1717_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, s0 is NOT allowed";
string PS_1717 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1718_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1718_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, oC0 is NOT allowed";
string PS_1718 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1719_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1719_Desc = "ps_2_x : texldd source reg combination s0, r0, s0, oDepth is NOT allowed";
string PS_1719 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1720_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1720_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, b0 is NOT allowed";
string PS_1720 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1721_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1721_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, i0 is NOT allowed";
string PS_1721 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1722_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1722_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, p0 is NOT allowed";
string PS_1722 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1723_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1723_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, s0 is NOT allowed";
string PS_1723 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1724_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1724_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, oC0 is NOT allowed";
string PS_1724 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1725_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1725_Desc = "ps_2_x : texldd source reg combination s0, r0, oC0, oDepth is NOT allowed";
string PS_1725 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1726_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1726_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, b0 is NOT allowed";
string PS_1726 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1727_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1727_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, i0 is NOT allowed";
string PS_1727 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1728_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1728_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, p0 is NOT allowed";
string PS_1728 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1729_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1729_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, s0 is NOT allowed";
string PS_1729 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1730_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1730_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, oC0 is NOT allowed";
string PS_1730 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1731_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1731_Desc = "ps_2_x : texldd source reg combination s0, r0, oDepth, oDepth is NOT allowed";
string PS_1731 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, s0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1732_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1732_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, b0 is NOT allowed";
string PS_1732 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1733_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1733_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, i0 is NOT allowed";
string PS_1733 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1734_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1734_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, p0 is NOT allowed";
string PS_1734 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1735_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1735_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, s0 is NOT allowed";
string PS_1735 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1736_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1736_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, oC0 is NOT allowed";
string PS_1736 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1737_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1737_Desc = "ps_2_x : texldd source reg combination s0, c0, b0, oDepth is NOT allowed";
string PS_1737 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1738_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1738_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, b0 is NOT allowed";
string PS_1738 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1739_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1739_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, i0 is NOT allowed";
string PS_1739 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1740_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1740_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, p0 is NOT allowed";
string PS_1740 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1741_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1741_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, s0 is NOT allowed";
string PS_1741 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1742_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1742_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, oC0 is NOT allowed";
string PS_1742 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1743_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1743_Desc = "ps_2_x : texldd source reg combination s0, c0, i0, oDepth is NOT allowed";
string PS_1743 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1744_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1744_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, b0 is NOT allowed";
string PS_1744 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1745_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1745_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, i0 is NOT allowed";
string PS_1745 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1746_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1746_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, p0 is NOT allowed";
string PS_1746 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1747_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1747_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, s0 is NOT allowed";
string PS_1747 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1748_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1748_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, oC0 is NOT allowed";
string PS_1748 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1749_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1749_Desc = "ps_2_x : texldd source reg combination s0, c0, p0, oDepth is NOT allowed";
string PS_1749 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1750_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1750_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, b0 is NOT allowed";
string PS_1750 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, s0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1751_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1751_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, i0 is NOT allowed";
string PS_1751 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1752_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1752_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, p0 is NOT allowed";
string PS_1752 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1753_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1753_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, s0 is NOT allowed";
string PS_1753 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1754_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1754_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, oC0 is NOT allowed";
string PS_1754 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1755_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1755_Desc = "ps_2_x : texldd source reg combination s0, c0, s0, oDepth is NOT allowed";
string PS_1755 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1756_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1756_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, b0 is NOT allowed";
string PS_1756 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1757_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1757_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, i0 is NOT allowed";
string PS_1757 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1758_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1758_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, p0 is NOT allowed";
string PS_1758 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1759_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1759_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, s0 is NOT allowed";
string PS_1759 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1760_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1760_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, oC0 is NOT allowed";
string PS_1760 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1761_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1761_Desc = "ps_2_x : texldd source reg combination s0, c0, oC0, oDepth is NOT allowed";
string PS_1761 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1762_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1762_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, b0 is NOT allowed";
string PS_1762 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1763_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1763_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, i0 is NOT allowed";
string PS_1763 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1764_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1764_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, p0 is NOT allowed";
string PS_1764 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, s0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1765_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1765_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, s0 is NOT allowed";
string PS_1765 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1766_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1766_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, oC0 is NOT allowed";
string PS_1766 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1767_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1767_Desc = "ps_2_x : texldd source reg combination s0, c0, oDepth, oDepth is NOT allowed";
string PS_1767 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, s0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1768_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1768_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, b0 is NOT allowed";
string PS_1768 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1769_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1769_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, i0 is NOT allowed";
string PS_1769 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1770_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1770_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, p0 is NOT allowed";
string PS_1770 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1771_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1771_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, s0 is NOT allowed";
string PS_1771 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1772_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1772_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, oC0 is NOT allowed";
string PS_1772 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1773_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1773_Desc = "ps_2_x : texldd source reg combination s0, b0, b0, oDepth is NOT allowed";
string PS_1773 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1774_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1774_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, b0 is NOT allowed";
string PS_1774 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1775_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1775_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, i0 is NOT allowed";
string PS_1775 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1776_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1776_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, p0 is NOT allowed";
string PS_1776 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1777_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1777_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, s0 is NOT allowed";
string PS_1777 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1778_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1778_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, oC0 is NOT allowed";
string PS_1778 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1779_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1779_Desc = "ps_2_x : texldd source reg combination s0, b0, i0, oDepth is NOT allowed";
string PS_1779 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1780_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1780_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, b0 is NOT allowed";
string PS_1780 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1781_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1781_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, i0 is NOT allowed";
string PS_1781 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1782_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1782_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, p0 is NOT allowed";
string PS_1782 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1783_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1783_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, s0 is NOT allowed";
string PS_1783 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1784_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1784_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, oC0 is NOT allowed";
string PS_1784 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1785_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1785_Desc = "ps_2_x : texldd source reg combination s0, b0, p0, oDepth is NOT allowed";
string PS_1785 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1786_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1786_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, b0 is NOT allowed";
string PS_1786 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1787_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1787_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, i0 is NOT allowed";
string PS_1787 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1788_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1788_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, p0 is NOT allowed";
string PS_1788 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1789_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1789_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, s0 is NOT allowed";
string PS_1789 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1790_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1790_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, oC0 is NOT allowed";
string PS_1790 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1791_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1791_Desc = "ps_2_x : texldd source reg combination s0, b0, s0, oDepth is NOT allowed";
string PS_1791 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1792_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1792_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, b0 is NOT allowed";
string PS_1792 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1793_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1793_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, i0 is NOT allowed";
string PS_1793 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1794_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1794_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, p0 is NOT allowed";
string PS_1794 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1795_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1795_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, s0 is NOT allowed";
string PS_1795 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1796_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1796_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, oC0 is NOT allowed";
string PS_1796 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1797_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1797_Desc = "ps_2_x : texldd source reg combination s0, b0, oC0, oDepth is NOT allowed";
string PS_1797 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1798_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1798_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, b0 is NOT allowed";
string PS_1798 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1799_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1799_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, i0 is NOT allowed";
string PS_1799 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1800_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1800_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, p0 is NOT allowed";
string PS_1800 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1801_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1801_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, s0 is NOT allowed";
string PS_1801 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1802_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1802_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, oC0 is NOT allowed";
string PS_1802 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1803_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1803_Desc = "ps_2_x : texldd source reg combination s0, b0, oDepth, oDepth is NOT allowed";
string PS_1803 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1804_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1804_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, b0 is NOT allowed";
string PS_1804 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1805_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1805_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, i0 is NOT allowed";
string PS_1805 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1806_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1806_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, p0 is NOT allowed";
string PS_1806 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1807_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1807_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, s0 is NOT allowed";
string PS_1807 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1808_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1808_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, oC0 is NOT allowed";
string PS_1808 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1809_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1809_Desc = "ps_2_x : texldd source reg combination s0, i0, b0, oDepth is NOT allowed";
string PS_1809 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1810_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1810_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, b0 is NOT allowed";
string PS_1810 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1811_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1811_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, i0 is NOT allowed";
string PS_1811 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1812_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1812_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, p0 is NOT allowed";
string PS_1812 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1813_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1813_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, s0 is NOT allowed";
string PS_1813 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1814_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1814_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, oC0 is NOT allowed";
string PS_1814 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1815_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1815_Desc = "ps_2_x : texldd source reg combination s0, i0, i0, oDepth is NOT allowed";
string PS_1815 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1816_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1816_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, b0 is NOT allowed";
string PS_1816 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1817_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1817_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, i0 is NOT allowed";
string PS_1817 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1818_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1818_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, p0 is NOT allowed";
string PS_1818 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1819_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1819_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, s0 is NOT allowed";
string PS_1819 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1820_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1820_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, oC0 is NOT allowed";
string PS_1820 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1821_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1821_Desc = "ps_2_x : texldd source reg combination s0, i0, p0, oDepth is NOT allowed";
string PS_1821 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1822_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1822_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, b0 is NOT allowed";
string PS_1822 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1823_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1823_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, i0 is NOT allowed";
string PS_1823 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1824_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1824_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, p0 is NOT allowed";
string PS_1824 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1825_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1825_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, s0 is NOT allowed";
string PS_1825 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1826_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1826_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, oC0 is NOT allowed";
string PS_1826 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1827_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1827_Desc = "ps_2_x : texldd source reg combination s0, i0, s0, oDepth is NOT allowed";
string PS_1827 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1828_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1828_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, b0 is NOT allowed";
string PS_1828 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1829_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1829_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, i0 is NOT allowed";
string PS_1829 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1830_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1830_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, p0 is NOT allowed";
string PS_1830 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1831_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1831_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, s0 is NOT allowed";
string PS_1831 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1832_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1832_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, oC0 is NOT allowed";
string PS_1832 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1833_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1833_Desc = "ps_2_x : texldd source reg combination s0, i0, oC0, oDepth is NOT allowed";
string PS_1833 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1834_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1834_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, b0 is NOT allowed";
string PS_1834 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1835_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1835_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, i0 is NOT allowed";
string PS_1835 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1836_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1836_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, p0 is NOT allowed";
string PS_1836 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1837_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1837_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, s0 is NOT allowed";
string PS_1837 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1838_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1838_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, oC0 is NOT allowed";
string PS_1838 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1839_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1839_Desc = "ps_2_x : texldd source reg combination s0, i0, oDepth, oDepth is NOT allowed";
string PS_1839 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1840_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1840_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, b0 is NOT allowed";
string PS_1840 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1841_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1841_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, i0 is NOT allowed";
string PS_1841 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1842_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1842_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, p0 is NOT allowed";
string PS_1842 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1843_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1843_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, s0 is NOT allowed";
string PS_1843 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1844_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1844_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, oC0 is NOT allowed";
string PS_1844 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1845_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1845_Desc = "ps_2_x : texldd source reg combination s0, p0, b0, oDepth is NOT allowed";
string PS_1845 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1846_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1846_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, b0 is NOT allowed";
string PS_1846 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1847_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1847_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, i0 is NOT allowed";
string PS_1847 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1848_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1848_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, p0 is NOT allowed";
string PS_1848 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1849_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1849_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, s0 is NOT allowed";
string PS_1849 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1850_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1850_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, oC0 is NOT allowed";
string PS_1850 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1851_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1851_Desc = "ps_2_x : texldd source reg combination s0, p0, i0, oDepth is NOT allowed";
string PS_1851 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1852_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1852_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, b0 is NOT allowed";
string PS_1852 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1853_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1853_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, i0 is NOT allowed";
string PS_1853 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1854_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1854_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, p0 is NOT allowed";
string PS_1854 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1855_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1855_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, s0 is NOT allowed";
string PS_1855 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1856_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1856_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, oC0 is NOT allowed";
string PS_1856 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1857_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1857_Desc = "ps_2_x : texldd source reg combination s0, p0, p0, oDepth is NOT allowed";
string PS_1857 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1858_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1858_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, b0 is NOT allowed";
string PS_1858 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1859_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1859_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, i0 is NOT allowed";
string PS_1859 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1860_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1860_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, p0 is NOT allowed";
string PS_1860 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1861_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1861_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, s0 is NOT allowed";
string PS_1861 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1862_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1862_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, oC0 is NOT allowed";
string PS_1862 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1863_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1863_Desc = "ps_2_x : texldd source reg combination s0, p0, s0, oDepth is NOT allowed";
string PS_1863 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1864_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1864_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, b0 is NOT allowed";
string PS_1864 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1865_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1865_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, i0 is NOT allowed";
string PS_1865 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1866_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1866_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, p0 is NOT allowed";
string PS_1866 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1867_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1867_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, s0 is NOT allowed";
string PS_1867 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1868_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1868_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, oC0 is NOT allowed";
string PS_1868 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1869_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1869_Desc = "ps_2_x : texldd source reg combination s0, p0, oC0, oDepth is NOT allowed";
string PS_1869 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1870_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1870_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, b0 is NOT allowed";
string PS_1870 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1871_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1871_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, i0 is NOT allowed";
string PS_1871 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1872_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1872_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, p0 is NOT allowed";
string PS_1872 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1873_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1873_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, s0 is NOT allowed";
string PS_1873 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1874_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1874_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, oC0 is NOT allowed";
string PS_1874 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1875_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1875_Desc = "ps_2_x : texldd source reg combination s0, p0, oDepth, oDepth is NOT allowed";
string PS_1875 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1876_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1876_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, b0 is NOT allowed";
string PS_1876 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1877_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1877_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, i0 is NOT allowed";
string PS_1877 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1878_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1878_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, p0 is NOT allowed";
string PS_1878 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1879_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1879_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, s0 is NOT allowed";
string PS_1879 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1880_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1880_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, oC0 is NOT allowed";
string PS_1880 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1881_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1881_Desc = "ps_2_x : texldd source reg combination s0, t0, b0, oDepth is NOT allowed";
string PS_1881 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1882_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1882_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, b0 is NOT allowed";
string PS_1882 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1883_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1883_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, i0 is NOT allowed";
string PS_1883 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1884_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1884_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, p0 is NOT allowed";
string PS_1884 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1885_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1885_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, s0 is NOT allowed";
string PS_1885 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1886_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1886_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, oC0 is NOT allowed";
string PS_1886 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1887_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1887_Desc = "ps_2_x : texldd source reg combination s0, t0, i0, oDepth is NOT allowed";
string PS_1887 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1888_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1888_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, b0 is NOT allowed";
string PS_1888 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1889_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1889_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, i0 is NOT allowed";
string PS_1889 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1890_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1890_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, p0 is NOT allowed";
string PS_1890 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1891_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1891_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, s0 is NOT allowed";
string PS_1891 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1892_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1892_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, oC0 is NOT allowed";
string PS_1892 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1893_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1893_Desc = "ps_2_x : texldd source reg combination s0, t0, p0, oDepth is NOT allowed";
string PS_1893 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1894_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1894_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, b0 is NOT allowed";
string PS_1894 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1895_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1895_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, i0 is NOT allowed";
string PS_1895 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1896_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1896_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, p0 is NOT allowed";
string PS_1896 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1897_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1897_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, s0 is NOT allowed";
string PS_1897 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1898_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1898_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, oC0 is NOT allowed";
string PS_1898 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1899_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1899_Desc = "ps_2_x : texldd source reg combination s0, t0, s0, oDepth is NOT allowed";
string PS_1899 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1900_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1900_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, b0 is NOT allowed";
string PS_1900 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1901_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1901_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, i0 is NOT allowed";
string PS_1901 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1902_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1902_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, p0 is NOT allowed";
string PS_1902 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1903_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1903_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, s0 is NOT allowed";
string PS_1903 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1904_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1904_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, oC0 is NOT allowed";
string PS_1904 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1905_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1905_Desc = "ps_2_x : texldd source reg combination s0, t0, oC0, oDepth is NOT allowed";
string PS_1905 = 
	"ps_2_x "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1906_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1906_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, b0 is NOT allowed";
string PS_1906 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1907_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1907_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, i0 is NOT allowed";
string PS_1907 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1908_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1908_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, p0 is NOT allowed";
string PS_1908 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1909_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1909_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, s0 is NOT allowed";
string PS_1909 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1910_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1910_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, oC0 is NOT allowed";
string PS_1910 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1911_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1911_Desc = "ps_2_x : texldd source reg combination s0, t0, oDepth, oDepth is NOT allowed";
string PS_1911 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl t0 "
	"texldd r0, s0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1912_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1912_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, b0 is NOT allowed";
string PS_1912 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1913_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1913_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, i0 is NOT allowed";
string PS_1913 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1914_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1914_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, p0 is NOT allowed";
string PS_1914 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1915_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1915_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, s0 is NOT allowed";
string PS_1915 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1916_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1916_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, oC0 is NOT allowed";
string PS_1916 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1917_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1917_Desc = "ps_2_x : texldd source reg combination s0, oC0, b0, oDepth is NOT allowed";
string PS_1917 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1918_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1918_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, b0 is NOT allowed";
string PS_1918 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1919_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1919_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, i0 is NOT allowed";
string PS_1919 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1920_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1920_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, p0 is NOT allowed";
string PS_1920 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1921_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1921_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, s0 is NOT allowed";
string PS_1921 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1922_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1922_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, oC0 is NOT allowed";
string PS_1922 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1923_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1923_Desc = "ps_2_x : texldd source reg combination s0, oC0, i0, oDepth is NOT allowed";
string PS_1923 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1924_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1924_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, b0 is NOT allowed";
string PS_1924 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1925_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1925_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, i0 is NOT allowed";
string PS_1925 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1926_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1926_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, p0 is NOT allowed";
string PS_1926 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1927_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1927_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, s0 is NOT allowed";
string PS_1927 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1928_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1928_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, oC0 is NOT allowed";
string PS_1928 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1929_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1929_Desc = "ps_2_x : texldd source reg combination s0, oC0, p0, oDepth is NOT allowed";
string PS_1929 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1930_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1930_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, b0 is NOT allowed";
string PS_1930 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1931_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1931_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, i0 is NOT allowed";
string PS_1931 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1932_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1932_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, p0 is NOT allowed";
string PS_1932 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1933_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1933_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, s0 is NOT allowed";
string PS_1933 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1934_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1934_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, oC0 is NOT allowed";
string PS_1934 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1935_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1935_Desc = "ps_2_x : texldd source reg combination s0, oC0, s0, oDepth is NOT allowed";
string PS_1935 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1936_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1936_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, b0 is NOT allowed";
string PS_1936 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1937_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1937_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, i0 is NOT allowed";
string PS_1937 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1938_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1938_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, p0 is NOT allowed";
string PS_1938 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1939_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1939_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, s0 is NOT allowed";
string PS_1939 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1940_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1940_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, oC0 is NOT allowed";
string PS_1940 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1941_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1941_Desc = "ps_2_x : texldd source reg combination s0, oC0, oC0, oDepth is NOT allowed";
string PS_1941 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1942_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1942_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, b0 is NOT allowed";
string PS_1942 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1943_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1943_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, i0 is NOT allowed";
string PS_1943 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1944_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1944_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, p0 is NOT allowed";
string PS_1944 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1945_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1945_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, s0 is NOT allowed";
string PS_1945 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1946_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1946_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, oC0 is NOT allowed";
string PS_1946 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1947_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1947_Desc = "ps_2_x : texldd source reg combination s0, oC0, oDepth, oDepth is NOT allowed";
string PS_1947 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1948_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1948_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, b0 is NOT allowed";
string PS_1948 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1949_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1949_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, i0 is NOT allowed";
string PS_1949 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1950_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1950_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, p0 is NOT allowed";
string PS_1950 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1951_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1951_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, s0 is NOT allowed";
string PS_1951 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1952_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1952_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, oC0 is NOT allowed";
string PS_1952 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1953_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1953_Desc = "ps_2_x : texldd source reg combination s0, oDepth, b0, oDepth is NOT allowed";
string PS_1953 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1954_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1954_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, b0 is NOT allowed";
string PS_1954 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1955_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1955_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, i0 is NOT allowed";
string PS_1955 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1956_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1956_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, p0 is NOT allowed";
string PS_1956 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1957_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1957_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, s0 is NOT allowed";
string PS_1957 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1958_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1958_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, oC0 is NOT allowed";
string PS_1958 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1959_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1959_Desc = "ps_2_x : texldd source reg combination s0, oDepth, i0, oDepth is NOT allowed";
string PS_1959 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1960_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1960_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, b0 is NOT allowed";
string PS_1960 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1961_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1961_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, i0 is NOT allowed";
string PS_1961 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1962_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1962_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, p0 is NOT allowed";
string PS_1962 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1963_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1963_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, s0 is NOT allowed";
string PS_1963 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1964_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1964_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, oC0 is NOT allowed";
string PS_1964 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1965_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1965_Desc = "ps_2_x : texldd source reg combination s0, oDepth, p0, oDepth is NOT allowed";
string PS_1965 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1966_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1966_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, b0 is NOT allowed";
string PS_1966 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1967_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1967_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, i0 is NOT allowed";
string PS_1967 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1968_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1968_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, p0 is NOT allowed";
string PS_1968 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1969_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1969_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, s0 is NOT allowed";
string PS_1969 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1970_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1970_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, oC0 is NOT allowed";
string PS_1970 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1971_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1971_Desc = "ps_2_x : texldd source reg combination s0, oDepth, s0, oDepth is NOT allowed";
string PS_1971 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, s0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1972_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1972_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, b0 is NOT allowed";
string PS_1972 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1973_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1973_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, i0 is NOT allowed";
string PS_1973 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1974_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1974_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, p0 is NOT allowed";
string PS_1974 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1975_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1975_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, s0 is NOT allowed";
string PS_1975 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1976_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1976_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, oC0 is NOT allowed";
string PS_1976 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1977_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1977_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oC0, oDepth is NOT allowed";
string PS_1977 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1978_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1978_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, b0 is NOT allowed";
string PS_1978 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1979_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1979_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, i0 is NOT allowed";
string PS_1979 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1980_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1980_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, p0 is NOT allowed";
string PS_1980 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1981_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1981_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, s0 is NOT allowed";
string PS_1981 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1982_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1982_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, oC0 is NOT allowed";
string PS_1982 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, s0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1983_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1983_Desc = "ps_2_x : texldd source reg combination s0, oDepth, oDepth, oDepth is NOT allowed";
string PS_1983 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, s0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1984_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1984_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, b0 is NOT allowed";
string PS_1984 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1985_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1985_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, i0 is NOT allowed";
string PS_1985 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1986_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1986_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, p0 is NOT allowed";
string PS_1986 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1987_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1987_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, s0 is NOT allowed";
string PS_1987 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oC0, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1988_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1988_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, oC0 is NOT allowed";
string PS_1988 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1989_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1989_Desc = "ps_2_x : texldd source reg combination oC0, v0, b0, oDepth is NOT allowed";
string PS_1989 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1990_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1990_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, b0 is NOT allowed";
string PS_1990 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1991_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1991_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, i0 is NOT allowed";
string PS_1991 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1992_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1992_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, p0 is NOT allowed";
string PS_1992 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1993_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1993_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, s0 is NOT allowed";
string PS_1993 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oC0, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1994_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1994_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, oC0 is NOT allowed";
string PS_1994 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1995_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1995_Desc = "ps_2_x : texldd source reg combination oC0, v0, i0, oDepth is NOT allowed";
string PS_1995 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_1996_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1996_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, b0 is NOT allowed";
string PS_1996 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1997_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1997_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, i0 is NOT allowed";
string PS_1997 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1998_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1998_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, p0 is NOT allowed";
string PS_1998 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_1999_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_1999_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, s0 is NOT allowed";
string PS_1999 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oC0, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2000_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2000_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, oC0 is NOT allowed";
string PS_2000 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2001_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2001_Desc = "ps_2_x : texldd source reg combination oC0, v0, p0, oDepth is NOT allowed";
string PS_2001 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2002_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2002_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, b0 is NOT allowed";
string PS_2002 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2003_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2003_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, i0 is NOT allowed";
string PS_2003 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2004_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2004_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, p0 is NOT allowed";
string PS_2004 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2005_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2005_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, s0 is NOT allowed";
string PS_2005 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2006_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2006_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, oC0 is NOT allowed";
string PS_2006 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2007_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2007_Desc = "ps_2_x : texldd source reg combination oC0, v0, s0, oDepth is NOT allowed";
string PS_2007 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2008_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2008_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, b0 is NOT allowed";
string PS_2008 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2009_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2009_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, i0 is NOT allowed";
string PS_2009 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2010_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2010_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, p0 is NOT allowed";
string PS_2010 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2011_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2011_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, s0 is NOT allowed";
string PS_2011 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oC0, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2012_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2012_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, oC0 is NOT allowed";
string PS_2012 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2013_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2013_Desc = "ps_2_x : texldd source reg combination oC0, v0, oC0, oDepth is NOT allowed";
string PS_2013 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2014_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2014_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, b0 is NOT allowed";
string PS_2014 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oC0, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2015_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2015_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, i0 is NOT allowed";
string PS_2015 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oC0, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2016_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2016_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, p0 is NOT allowed";
string PS_2016 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2017_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2017_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, s0 is NOT allowed";
string PS_2017 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oC0, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2018_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2018_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, oC0 is NOT allowed";
string PS_2018 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2019_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2019_Desc = "ps_2_x : texldd source reg combination oC0, v0, oDepth, oDepth is NOT allowed";
string PS_2019 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oC0, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2020_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2020_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, b0 is NOT allowed";
string PS_2020 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2021_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2021_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, i0 is NOT allowed";
string PS_2021 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2022_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2022_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, p0 is NOT allowed";
string PS_2022 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2023_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2023_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, s0 is NOT allowed";
string PS_2023 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2024_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2024_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, oC0 is NOT allowed";
string PS_2024 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2025_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2025_Desc = "ps_2_x : texldd source reg combination oC0, r0, b0, oDepth is NOT allowed";
string PS_2025 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2026_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2026_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, b0 is NOT allowed";
string PS_2026 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2027_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2027_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, i0 is NOT allowed";
string PS_2027 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2028_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2028_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, p0 is NOT allowed";
string PS_2028 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2029_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2029_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, s0 is NOT allowed";
string PS_2029 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2030_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2030_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, oC0 is NOT allowed";
string PS_2030 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2031_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2031_Desc = "ps_2_x : texldd source reg combination oC0, r0, i0, oDepth is NOT allowed";
string PS_2031 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2032_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2032_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, b0 is NOT allowed";
string PS_2032 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2033_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2033_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, i0 is NOT allowed";
string PS_2033 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2034_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2034_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, p0 is NOT allowed";
string PS_2034 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2035_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2035_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, s0 is NOT allowed";
string PS_2035 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2036_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2036_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, oC0 is NOT allowed";
string PS_2036 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2037_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2037_Desc = "ps_2_x : texldd source reg combination oC0, r0, p0, oDepth is NOT allowed";
string PS_2037 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2038_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2038_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, b0 is NOT allowed";
string PS_2038 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2039_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2039_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, i0 is NOT allowed";
string PS_2039 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2040_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2040_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, p0 is NOT allowed";
string PS_2040 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2041_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2041_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, s0 is NOT allowed";
string PS_2041 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2042_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2042_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, oC0 is NOT allowed";
string PS_2042 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2043_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2043_Desc = "ps_2_x : texldd source reg combination oC0, r0, s0, oDepth is NOT allowed";
string PS_2043 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2044_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2044_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, b0 is NOT allowed";
string PS_2044 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2045_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2045_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, i0 is NOT allowed";
string PS_2045 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2046_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2046_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, p0 is NOT allowed";
string PS_2046 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2047_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2047_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, s0 is NOT allowed";
string PS_2047 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2048_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2048_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, oC0 is NOT allowed";
string PS_2048 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2049_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2049_Desc = "ps_2_x : texldd source reg combination oC0, r0, oC0, oDepth is NOT allowed";
string PS_2049 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2050_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2050_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, b0 is NOT allowed";
string PS_2050 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2051_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2051_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, i0 is NOT allowed";
string PS_2051 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2052_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2052_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, p0 is NOT allowed";
string PS_2052 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2053_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2053_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, s0 is NOT allowed";
string PS_2053 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2054_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2054_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, oC0 is NOT allowed";
string PS_2054 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2055_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2055_Desc = "ps_2_x : texldd source reg combination oC0, r0, oDepth, oDepth is NOT allowed";
string PS_2055 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oC0, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2056_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2056_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, b0 is NOT allowed";
string PS_2056 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2057_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2057_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, i0 is NOT allowed";
string PS_2057 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2058_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2058_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, p0 is NOT allowed";
string PS_2058 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2059_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2059_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, s0 is NOT allowed";
string PS_2059 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, oC0, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2060_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2060_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, oC0 is NOT allowed";
string PS_2060 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2061_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2061_Desc = "ps_2_x : texldd source reg combination oC0, c0, b0, oDepth is NOT allowed";
string PS_2061 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2062_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2062_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, b0 is NOT allowed";
string PS_2062 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2063_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2063_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, i0 is NOT allowed";
string PS_2063 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2064_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2064_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, p0 is NOT allowed";
string PS_2064 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2065_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2065_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, s0 is NOT allowed";
string PS_2065 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oC0, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2066_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2066_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, oC0 is NOT allowed";
string PS_2066 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2067_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2067_Desc = "ps_2_x : texldd source reg combination oC0, c0, i0, oDepth is NOT allowed";
string PS_2067 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2068_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2068_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, b0 is NOT allowed";
string PS_2068 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2069_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2069_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, i0 is NOT allowed";
string PS_2069 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2070_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2070_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, p0 is NOT allowed";
string PS_2070 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2071_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2071_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, s0 is NOT allowed";
string PS_2071 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oC0, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2072_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2072_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, oC0 is NOT allowed";
string PS_2072 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2073_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2073_Desc = "ps_2_x : texldd source reg combination oC0, c0, p0, oDepth is NOT allowed";
string PS_2073 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2074_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2074_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, b0 is NOT allowed";
string PS_2074 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oC0, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2075_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2075_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, i0 is NOT allowed";
string PS_2075 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oC0, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2076_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2076_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, p0 is NOT allowed";
string PS_2076 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oC0, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2077_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2077_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, s0 is NOT allowed";
string PS_2077 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oC0, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2078_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2078_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, oC0 is NOT allowed";
string PS_2078 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, oC0, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2079_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2079_Desc = "ps_2_x : texldd source reg combination oC0, c0, s0, oDepth is NOT allowed";
string PS_2079 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, oC0, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2080_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2080_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, b0 is NOT allowed";
string PS_2080 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2081_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2081_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, i0 is NOT allowed";
string PS_2081 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2082_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2082_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, p0 is NOT allowed";
string PS_2082 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2083_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2083_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, s0 is NOT allowed";
string PS_2083 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oC0, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2084_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2084_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, oC0 is NOT allowed";
string PS_2084 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2085_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2085_Desc = "ps_2_x : texldd source reg combination oC0, c0, oC0, oDepth is NOT allowed";
string PS_2085 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2086_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2086_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, b0 is NOT allowed";
string PS_2086 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2087_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2087_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, i0 is NOT allowed";
string PS_2087 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2088_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2088_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, p0 is NOT allowed";
string PS_2088 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2089_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2089_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, s0 is NOT allowed";
string PS_2089 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oC0, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2090_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2090_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, oC0 is NOT allowed";
string PS_2090 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2091_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2091_Desc = "ps_2_x : texldd source reg combination oC0, c0, oDepth, oDepth is NOT allowed";
string PS_2091 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oC0, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2092_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2092_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, b0 is NOT allowed";
string PS_2092 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2093_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2093_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, i0 is NOT allowed";
string PS_2093 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2094_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2094_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, p0 is NOT allowed";
string PS_2094 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2095_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2095_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, s0 is NOT allowed";
string PS_2095 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oC0, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2096_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2096_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, oC0 is NOT allowed";
string PS_2096 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2097_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2097_Desc = "ps_2_x : texldd source reg combination oC0, b0, b0, oDepth is NOT allowed";
string PS_2097 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2098_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2098_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, b0 is NOT allowed";
string PS_2098 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2099_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2099_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, i0 is NOT allowed";
string PS_2099 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2100_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2100_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, p0 is NOT allowed";
string PS_2100 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2101_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2101_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, s0 is NOT allowed";
string PS_2101 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oC0, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2102_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2102_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, oC0 is NOT allowed";
string PS_2102 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2103_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2103_Desc = "ps_2_x : texldd source reg combination oC0, b0, i0, oDepth is NOT allowed";
string PS_2103 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2104_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2104_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, b0 is NOT allowed";
string PS_2104 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2105_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2105_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, i0 is NOT allowed";
string PS_2105 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2106_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2106_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, p0 is NOT allowed";
string PS_2106 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2107_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2107_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, s0 is NOT allowed";
string PS_2107 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2108_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2108_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, oC0 is NOT allowed";
string PS_2108 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2109_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2109_Desc = "ps_2_x : texldd source reg combination oC0, b0, p0, oDepth is NOT allowed";
string PS_2109 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2110_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2110_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, b0 is NOT allowed";
string PS_2110 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2111_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2111_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, i0 is NOT allowed";
string PS_2111 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2112_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2112_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, p0 is NOT allowed";
string PS_2112 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2113_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2113_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, s0 is NOT allowed";
string PS_2113 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2114_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2114_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, oC0 is NOT allowed";
string PS_2114 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2115_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2115_Desc = "ps_2_x : texldd source reg combination oC0, b0, s0, oDepth is NOT allowed";
string PS_2115 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2116_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2116_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, b0 is NOT allowed";
string PS_2116 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2117_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2117_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, i0 is NOT allowed";
string PS_2117 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2118_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2118_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, p0 is NOT allowed";
string PS_2118 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2119_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2119_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, s0 is NOT allowed";
string PS_2119 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oC0, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2120_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2120_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, oC0 is NOT allowed";
string PS_2120 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2121_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2121_Desc = "ps_2_x : texldd source reg combination oC0, b0, oC0, oDepth is NOT allowed";
string PS_2121 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2122_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2122_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, b0 is NOT allowed";
string PS_2122 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2123_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2123_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, i0 is NOT allowed";
string PS_2123 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2124_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2124_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, p0 is NOT allowed";
string PS_2124 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2125_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2125_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, s0 is NOT allowed";
string PS_2125 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oC0, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2126_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2126_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, oC0 is NOT allowed";
string PS_2126 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2127_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2127_Desc = "ps_2_x : texldd source reg combination oC0, b0, oDepth, oDepth is NOT allowed";
string PS_2127 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2128_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2128_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, b0 is NOT allowed";
string PS_2128 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2129_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2129_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, i0 is NOT allowed";
string PS_2129 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2130_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2130_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, p0 is NOT allowed";
string PS_2130 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2131_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2131_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, s0 is NOT allowed";
string PS_2131 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2132_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2132_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, oC0 is NOT allowed";
string PS_2132 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2133_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2133_Desc = "ps_2_x : texldd source reg combination oC0, i0, b0, oDepth is NOT allowed";
string PS_2133 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2134_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2134_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, b0 is NOT allowed";
string PS_2134 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2135_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2135_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, i0 is NOT allowed";
string PS_2135 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2136_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2136_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, p0 is NOT allowed";
string PS_2136 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2137_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2137_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, s0 is NOT allowed";
string PS_2137 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oC0, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2138_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2138_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, oC0 is NOT allowed";
string PS_2138 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2139_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2139_Desc = "ps_2_x : texldd source reg combination oC0, i0, i0, oDepth is NOT allowed";
string PS_2139 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2140_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2140_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, b0 is NOT allowed";
string PS_2140 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2141_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2141_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, i0 is NOT allowed";
string PS_2141 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2142_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2142_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, p0 is NOT allowed";
string PS_2142 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2143_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2143_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, s0 is NOT allowed";
string PS_2143 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2144_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2144_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, oC0 is NOT allowed";
string PS_2144 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2145_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2145_Desc = "ps_2_x : texldd source reg combination oC0, i0, p0, oDepth is NOT allowed";
string PS_2145 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2146_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2146_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, b0 is NOT allowed";
string PS_2146 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2147_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2147_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, i0 is NOT allowed";
string PS_2147 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2148_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2148_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, p0 is NOT allowed";
string PS_2148 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2149_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2149_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, s0 is NOT allowed";
string PS_2149 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2150_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2150_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, oC0 is NOT allowed";
string PS_2150 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2151_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2151_Desc = "ps_2_x : texldd source reg combination oC0, i0, s0, oDepth is NOT allowed";
string PS_2151 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2152_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2152_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, b0 is NOT allowed";
string PS_2152 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2153_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2153_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, i0 is NOT allowed";
string PS_2153 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2154_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2154_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, p0 is NOT allowed";
string PS_2154 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2155_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2155_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, s0 is NOT allowed";
string PS_2155 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oC0, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2156_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2156_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, oC0 is NOT allowed";
string PS_2156 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2157_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2157_Desc = "ps_2_x : texldd source reg combination oC0, i0, oC0, oDepth is NOT allowed";
string PS_2157 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2158_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2158_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, b0 is NOT allowed";
string PS_2158 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2159_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2159_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, i0 is NOT allowed";
string PS_2159 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2160_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2160_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, p0 is NOT allowed";
string PS_2160 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2161_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2161_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, s0 is NOT allowed";
string PS_2161 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2162_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2162_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, oC0 is NOT allowed";
string PS_2162 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2163_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2163_Desc = "ps_2_x : texldd source reg combination oC0, i0, oDepth, oDepth is NOT allowed";
string PS_2163 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2164_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2164_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, b0 is NOT allowed";
string PS_2164 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2165_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2165_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, i0 is NOT allowed";
string PS_2165 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2166_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2166_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, p0 is NOT allowed";
string PS_2166 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2167_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2167_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, s0 is NOT allowed";
string PS_2167 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2168_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2168_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, oC0 is NOT allowed";
string PS_2168 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2169_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2169_Desc = "ps_2_x : texldd source reg combination oC0, p0, b0, oDepth is NOT allowed";
string PS_2169 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2170_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2170_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, b0 is NOT allowed";
string PS_2170 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2171_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2171_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, i0 is NOT allowed";
string PS_2171 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2172_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2172_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, p0 is NOT allowed";
string PS_2172 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2173_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2173_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, s0 is NOT allowed";
string PS_2173 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2174_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2174_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, oC0 is NOT allowed";
string PS_2174 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2175_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2175_Desc = "ps_2_x : texldd source reg combination oC0, p0, i0, oDepth is NOT allowed";
string PS_2175 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2176_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2176_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, b0 is NOT allowed";
string PS_2176 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2177_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2177_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, i0 is NOT allowed";
string PS_2177 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2178_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2178_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, p0 is NOT allowed";
string PS_2178 = 
	"ps_2_x "
	"texldd r0, oC0, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2179_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2179_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, s0 is NOT allowed";
string PS_2179 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oC0, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2180_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2180_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, oC0 is NOT allowed";
string PS_2180 = 
	"ps_2_x "
	"texldd r0, oC0, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2181_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2181_Desc = "ps_2_x : texldd source reg combination oC0, p0, p0, oDepth is NOT allowed";
string PS_2181 = 
	"ps_2_x "
	"texldd r0, oC0, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2182_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2182_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, b0 is NOT allowed";
string PS_2182 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oC0, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2183_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2183_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, i0 is NOT allowed";
string PS_2183 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2184_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2184_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, p0 is NOT allowed";
string PS_2184 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2185_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2185_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, s0 is NOT allowed";
string PS_2185 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2186_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2186_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, oC0 is NOT allowed";
string PS_2186 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2187_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2187_Desc = "ps_2_x : texldd source reg combination oC0, p0, s0, oDepth is NOT allowed";
string PS_2187 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2188_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2188_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, b0 is NOT allowed";
string PS_2188 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2189_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2189_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, i0 is NOT allowed";
string PS_2189 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2190_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2190_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, p0 is NOT allowed";
string PS_2190 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2191_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2191_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, s0 is NOT allowed";
string PS_2191 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2192_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2192_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, oC0 is NOT allowed";
string PS_2192 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2193_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2193_Desc = "ps_2_x : texldd source reg combination oC0, p0, oC0, oDepth is NOT allowed";
string PS_2193 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2194_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2194_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, b0 is NOT allowed";
string PS_2194 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2195_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2195_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, i0 is NOT allowed";
string PS_2195 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2196_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2196_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, p0 is NOT allowed";
string PS_2196 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2197_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2197_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, s0 is NOT allowed";
string PS_2197 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2198_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2198_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, oC0 is NOT allowed";
string PS_2198 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2199_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2199_Desc = "ps_2_x : texldd source reg combination oC0, p0, oDepth, oDepth is NOT allowed";
string PS_2199 = 
	"ps_2_x "
	"texldd r0, oC0, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2200_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2200_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, b0 is NOT allowed";
string PS_2200 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2201_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2201_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, i0 is NOT allowed";
string PS_2201 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2202_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2202_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, p0 is NOT allowed";
string PS_2202 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2203_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2203_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, s0 is NOT allowed";
string PS_2203 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oC0, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2204_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2204_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, oC0 is NOT allowed";
string PS_2204 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2205_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2205_Desc = "ps_2_x : texldd source reg combination oC0, t0, b0, oDepth is NOT allowed";
string PS_2205 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2206_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2206_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, b0 is NOT allowed";
string PS_2206 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2207_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2207_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, i0 is NOT allowed";
string PS_2207 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2208_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2208_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, p0 is NOT allowed";
string PS_2208 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2209_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2209_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, s0 is NOT allowed";
string PS_2209 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, oC0, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2210_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2210_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, oC0 is NOT allowed";
string PS_2210 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2211_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2211_Desc = "ps_2_x : texldd source reg combination oC0, t0, i0, oDepth is NOT allowed";
string PS_2211 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2212_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2212_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, b0 is NOT allowed";
string PS_2212 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2213_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2213_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, i0 is NOT allowed";
string PS_2213 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2214_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2214_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, p0 is NOT allowed";
string PS_2214 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2215_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2215_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, s0 is NOT allowed";
string PS_2215 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oC0, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2216_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2216_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, oC0 is NOT allowed";
string PS_2216 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2217_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2217_Desc = "ps_2_x : texldd source reg combination oC0, t0, p0, oDepth is NOT allowed";
string PS_2217 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2218_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2218_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, b0 is NOT allowed";
string PS_2218 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oC0, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2219_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2219_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, i0 is NOT allowed";
string PS_2219 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oC0, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2220_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2220_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, p0 is NOT allowed";
string PS_2220 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oC0, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2221_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2221_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, s0 is NOT allowed";
string PS_2221 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oC0, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2222_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2222_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, oC0 is NOT allowed";
string PS_2222 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oC0, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2223_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2223_Desc = "ps_2_x : texldd source reg combination oC0, t0, s0, oDepth is NOT allowed";
string PS_2223 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oC0, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2224_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2224_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, b0 is NOT allowed";
string PS_2224 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2225_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2225_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, i0 is NOT allowed";
string PS_2225 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2226_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2226_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, p0 is NOT allowed";
string PS_2226 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2227_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2227_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, s0 is NOT allowed";
string PS_2227 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oC0, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2228_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2228_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, oC0 is NOT allowed";
string PS_2228 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2229_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2229_Desc = "ps_2_x : texldd source reg combination oC0, t0, oC0, oDepth is NOT allowed";
string PS_2229 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2230_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2230_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, b0 is NOT allowed";
string PS_2230 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oC0, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2231_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2231_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, i0 is NOT allowed";
string PS_2231 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oC0, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2232_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2232_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, p0 is NOT allowed";
string PS_2232 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2233_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2233_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, s0 is NOT allowed";
string PS_2233 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, oC0, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2234_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2234_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, oC0 is NOT allowed";
string PS_2234 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2235_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2235_Desc = "ps_2_x : texldd source reg combination oC0, t0, oDepth, oDepth is NOT allowed";
string PS_2235 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oC0, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2236_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2236_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, b0 is NOT allowed";
string PS_2236 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2237_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2237_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, i0 is NOT allowed";
string PS_2237 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2238_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2238_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, p0 is NOT allowed";
string PS_2238 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2239_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2239_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, s0 is NOT allowed";
string PS_2239 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oC0, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2240_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2240_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, oC0 is NOT allowed";
string PS_2240 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2241_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2241_Desc = "ps_2_x : texldd source reg combination oC0, oC0, b0, oDepth is NOT allowed";
string PS_2241 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2242_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2242_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, b0 is NOT allowed";
string PS_2242 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2243_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2243_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, i0 is NOT allowed";
string PS_2243 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2244_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2244_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, p0 is NOT allowed";
string PS_2244 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2245_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2245_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, s0 is NOT allowed";
string PS_2245 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oC0, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2246_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2246_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, oC0 is NOT allowed";
string PS_2246 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2247_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2247_Desc = "ps_2_x : texldd source reg combination oC0, oC0, i0, oDepth is NOT allowed";
string PS_2247 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2248_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2248_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, b0 is NOT allowed";
string PS_2248 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2249_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2249_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, i0 is NOT allowed";
string PS_2249 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2250_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2250_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, p0 is NOT allowed";
string PS_2250 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2251_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2251_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, s0 is NOT allowed";
string PS_2251 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2252_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2252_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, oC0 is NOT allowed";
string PS_2252 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2253_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2253_Desc = "ps_2_x : texldd source reg combination oC0, oC0, p0, oDepth is NOT allowed";
string PS_2253 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2254_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2254_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, b0 is NOT allowed";
string PS_2254 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2255_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2255_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, i0 is NOT allowed";
string PS_2255 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2256_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2256_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, p0 is NOT allowed";
string PS_2256 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oC0, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2257_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2257_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, s0 is NOT allowed";
string PS_2257 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oC0, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2258_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2258_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, oC0 is NOT allowed";
string PS_2258 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2259_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2259_Desc = "ps_2_x : texldd source reg combination oC0, oC0, s0, oDepth is NOT allowed";
string PS_2259 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2260_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2260_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, b0 is NOT allowed";
string PS_2260 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2261_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2261_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, i0 is NOT allowed";
string PS_2261 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2262_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2262_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, p0 is NOT allowed";
string PS_2262 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2263_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2263_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, s0 is NOT allowed";
string PS_2263 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2264_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2264_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, oC0 is NOT allowed";
string PS_2264 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2265_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2265_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oC0, oDepth is NOT allowed";
string PS_2265 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2266_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2266_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, b0 is NOT allowed";
string PS_2266 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2267_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2267_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, i0 is NOT allowed";
string PS_2267 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2268_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2268_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, p0 is NOT allowed";
string PS_2268 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2269_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2269_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, s0 is NOT allowed";
string PS_2269 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2270_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2270_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, oC0 is NOT allowed";
string PS_2270 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2271_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2271_Desc = "ps_2_x : texldd source reg combination oC0, oC0, oDepth, oDepth is NOT allowed";
string PS_2271 = 
	"ps_2_x "
	"texldd r0, oC0, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2272_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2272_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, b0 is NOT allowed";
string PS_2272 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2273_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2273_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, i0 is NOT allowed";
string PS_2273 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oC0, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2274_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2274_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, p0 is NOT allowed";
string PS_2274 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2275_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2275_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, s0 is NOT allowed";
string PS_2275 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oC0, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2276_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2276_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, oC0 is NOT allowed";
string PS_2276 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2277_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2277_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, b0, oDepth is NOT allowed";
string PS_2277 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2278_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2278_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, b0 is NOT allowed";
string PS_2278 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2279_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2279_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, i0 is NOT allowed";
string PS_2279 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2280_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2280_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, p0 is NOT allowed";
string PS_2280 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2281_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2281_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, s0 is NOT allowed";
string PS_2281 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oC0, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2282_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2282_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, oC0 is NOT allowed";
string PS_2282 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2283_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2283_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, i0, oDepth is NOT allowed";
string PS_2283 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2284_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2284_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, b0 is NOT allowed";
string PS_2284 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2285_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2285_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, i0 is NOT allowed";
string PS_2285 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2286_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2286_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, p0 is NOT allowed";
string PS_2286 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2287_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2287_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, s0 is NOT allowed";
string PS_2287 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oC0, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2288_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2288_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, oC0 is NOT allowed";
string PS_2288 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2289_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2289_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, p0, oDepth is NOT allowed";
string PS_2289 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2290_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2290_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, b0 is NOT allowed";
string PS_2290 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oC0, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2291_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2291_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, i0 is NOT allowed";
string PS_2291 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oC0, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2292_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2292_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, p0 is NOT allowed";
string PS_2292 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2293_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2293_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, s0 is NOT allowed";
string PS_2293 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2294_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2294_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, oC0 is NOT allowed";
string PS_2294 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oC0, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2295_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2295_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, s0, oDepth is NOT allowed";
string PS_2295 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2296_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2296_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, b0 is NOT allowed";
string PS_2296 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2297_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2297_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, i0 is NOT allowed";
string PS_2297 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2298_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2298_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, p0 is NOT allowed";
string PS_2298 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2299_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2299_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, s0 is NOT allowed";
string PS_2299 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oC0, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2300_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2300_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, oC0 is NOT allowed";
string PS_2300 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2301_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2301_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oC0, oDepth is NOT allowed";
string PS_2301 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2302_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2302_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, b0 is NOT allowed";
string PS_2302 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oC0, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2303_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2303_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, i0 is NOT allowed";
string PS_2303 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oC0, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2304_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2304_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, p0 is NOT allowed";
string PS_2304 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2305_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2305_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, s0 is NOT allowed";
string PS_2305 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oC0, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2306_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2306_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, oC0 is NOT allowed";
string PS_2306 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2307_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2307_Desc = "ps_2_x : texldd source reg combination oC0, oDepth, oDepth, oDepth is NOT allowed";
string PS_2307 = 
	"ps_2_x "
	"texldd r0, oC0, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2308_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2308_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, b0 is NOT allowed";
string PS_2308 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2309_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2309_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, i0 is NOT allowed";
string PS_2309 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2310_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2310_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, p0 is NOT allowed";
string PS_2310 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2311_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2311_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, s0 is NOT allowed";
string PS_2311 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, v0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2312_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2312_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, oC0 is NOT allowed";
string PS_2312 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2313_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2313_Desc = "ps_2_x : texldd source reg combination oDepth, v0, b0, oDepth is NOT allowed";
string PS_2313 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2314_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2314_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, b0 is NOT allowed";
string PS_2314 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2315_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2315_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, i0 is NOT allowed";
string PS_2315 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2316_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2316_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, p0 is NOT allowed";
string PS_2316 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2317_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2317_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, s0 is NOT allowed";
string PS_2317 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, v0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2318_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2318_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, oC0 is NOT allowed";
string PS_2318 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2319_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2319_Desc = "ps_2_x : texldd source reg combination oDepth, v0, i0, oDepth is NOT allowed";
string PS_2319 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2320_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2320_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, b0 is NOT allowed";
string PS_2320 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2321_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2321_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, i0 is NOT allowed";
string PS_2321 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2322_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2322_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, p0 is NOT allowed";
string PS_2322 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2323_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2323_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, s0 is NOT allowed";
string PS_2323 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, v0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2324_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2324_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, oC0 is NOT allowed";
string PS_2324 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2325_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2325_Desc = "ps_2_x : texldd source reg combination oDepth, v0, p0, oDepth is NOT allowed";
string PS_2325 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2326_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2326_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, b0 is NOT allowed";
string PS_2326 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, v0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2327_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2327_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, i0 is NOT allowed";
string PS_2327 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, v0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2328_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2328_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, p0 is NOT allowed";
string PS_2328 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, v0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2329_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2329_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, s0 is NOT allowed";
string PS_2329 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, v0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2330_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2330_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, oC0 is NOT allowed";
string PS_2330 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, v0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2331_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2331_Desc = "ps_2_x : texldd source reg combination oDepth, v0, s0, oDepth is NOT allowed";
string PS_2331 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, v0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2332_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2332_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, b0 is NOT allowed";
string PS_2332 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2333_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2333_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, i0 is NOT allowed";
string PS_2333 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2334_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2334_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, p0 is NOT allowed";
string PS_2334 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2335_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2335_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, s0 is NOT allowed";
string PS_2335 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, v0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2336_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2336_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, oC0 is NOT allowed";
string PS_2336 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2337_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2337_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oC0, oDepth is NOT allowed";
string PS_2337 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2338_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2338_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, b0 is NOT allowed";
string PS_2338 = 
	"ps_2_x "
	"defb b0, true "
	"dcl v0 "
	"texldd r0, oDepth, v0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2339_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2339_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, i0 is NOT allowed";
string PS_2339 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl v0 "
	"texldd r0, oDepth, v0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2340_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2340_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, p0 is NOT allowed";
string PS_2340 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2341_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2341_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, s0 is NOT allowed";
string PS_2341 = 
	"ps_2_x "
	"dcl v0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, v0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2342_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2342_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, oC0 is NOT allowed";
string PS_2342 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2343_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2343_Desc = "ps_2_x : texldd source reg combination oDepth, v0, oDepth, oDepth is NOT allowed";
string PS_2343 = 
	"ps_2_x "
	"dcl v0 "
	"texldd r0, oDepth, v0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2344_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2344_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, b0 is NOT allowed";
string PS_2344 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2345_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2345_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, i0 is NOT allowed";
string PS_2345 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2346_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2346_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, p0 is NOT allowed";
string PS_2346 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2347_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2347_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, s0 is NOT allowed";
string PS_2347 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2348_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2348_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, oC0 is NOT allowed";
string PS_2348 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2349_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2349_Desc = "ps_2_x : texldd source reg combination oDepth, r0, b0, oDepth is NOT allowed";
string PS_2349 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2350_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2350_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, b0 is NOT allowed";
string PS_2350 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2351_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2351_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, i0 is NOT allowed";
string PS_2351 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2352_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2352_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, p0 is NOT allowed";
string PS_2352 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2353_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2353_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, s0 is NOT allowed";
string PS_2353 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2354_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2354_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, oC0 is NOT allowed";
string PS_2354 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2355_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2355_Desc = "ps_2_x : texldd source reg combination oDepth, r0, i0, oDepth is NOT allowed";
string PS_2355 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2356_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2356_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, b0 is NOT allowed";
string PS_2356 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2357_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2357_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, i0 is NOT allowed";
string PS_2357 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2358_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2358_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, p0 is NOT allowed";
string PS_2358 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2359_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2359_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, s0 is NOT allowed";
string PS_2359 = 
	"ps_2_x "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2360_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2360_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, oC0 is NOT allowed";
string PS_2360 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2361_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2361_Desc = "ps_2_x : texldd source reg combination oDepth, r0, p0, oDepth is NOT allowed";
string PS_2361 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2362_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2362_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, b0 is NOT allowed";
string PS_2362 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2363_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2363_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, i0 is NOT allowed";
string PS_2363 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2364_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2364_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, p0 is NOT allowed";
string PS_2364 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2365_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2365_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, s0 is NOT allowed";
string PS_2365 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2366_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2366_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, oC0 is NOT allowed";
string PS_2366 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2367_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2367_Desc = "ps_2_x : texldd source reg combination oDepth, r0, s0, oDepth is NOT allowed";
string PS_2367 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2368_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2368_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, b0 is NOT allowed";
string PS_2368 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2369_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2369_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, i0 is NOT allowed";
string PS_2369 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2370_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2370_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, p0 is NOT allowed";
string PS_2370 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2371_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2371_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, s0 is NOT allowed";
string PS_2371 = 
	"ps_2_x "
	"dcl_volume s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2372_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2372_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, oC0 is NOT allowed";
string PS_2372 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2373_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2373_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oC0, oDepth is NOT allowed";
string PS_2373 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2374_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2374_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, b0 is NOT allowed";
string PS_2374 = 
	"ps_2_x "
	"defb b0, true "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2375_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2375_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, i0 is NOT allowed";
string PS_2375 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2376_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2376_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, p0 is NOT allowed";
string PS_2376 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2377_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2377_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, s0 is NOT allowed";
string PS_2377 = 
	"ps_2_x "
	"dcl_2d s0 "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2378_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2378_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, oC0 is NOT allowed";
string PS_2378 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2379_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2379_Desc = "ps_2_x : texldd source reg combination oDepth, r0, oDepth, oDepth is NOT allowed";
string PS_2379 = 
	"ps_2_x "
	"mov r0, c0 "
	"texldd r0, oDepth, r0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2380_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2380_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, b0 is NOT allowed";
string PS_2380 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2381_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2381_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, i0 is NOT allowed";
string PS_2381 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2382_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2382_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, p0 is NOT allowed";
string PS_2382 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2383_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2383_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, s0 is NOT allowed";
string PS_2383 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oDepth, c0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2384_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2384_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, oC0 is NOT allowed";
string PS_2384 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2385_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2385_Desc = "ps_2_x : texldd source reg combination oDepth, c0, b0, oDepth is NOT allowed";
string PS_2385 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2386_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2386_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, b0 is NOT allowed";
string PS_2386 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2387_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2387_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, i0 is NOT allowed";
string PS_2387 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2388_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2388_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, p0 is NOT allowed";
string PS_2388 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2389_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2389_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, s0 is NOT allowed";
string PS_2389 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, oDepth, c0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2390_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2390_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, oC0 is NOT allowed";
string PS_2390 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2391_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2391_Desc = "ps_2_x : texldd source reg combination oDepth, c0, i0, oDepth is NOT allowed";
string PS_2391 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2392_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2392_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, b0 is NOT allowed";
string PS_2392 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2393_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2393_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, i0 is NOT allowed";
string PS_2393 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2394_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2394_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, p0 is NOT allowed";
string PS_2394 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2395_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2395_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, s0 is NOT allowed";
string PS_2395 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_volume s0 "
	"texldd r0, oDepth, c0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2396_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2396_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, oC0 is NOT allowed";
string PS_2396 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2397_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2397_Desc = "ps_2_x : texldd source reg combination oDepth, c0, p0, oDepth is NOT allowed";
string PS_2397 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2398_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2398_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, b0 is NOT allowed";
string PS_2398 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oDepth, c0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2399_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2399_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, i0 is NOT allowed";
string PS_2399 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oDepth, c0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2400_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2400_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, p0 is NOT allowed";
string PS_2400 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oDepth, c0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2401_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2401_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, s0 is NOT allowed";
string PS_2401 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_2d s0 "
	"texldd r0, oDepth, c0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2402_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2402_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, oC0 is NOT allowed";
string PS_2402 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oDepth, c0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2403_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2403_Desc = "ps_2_x : texldd source reg combination oDepth, c0, s0, oDepth is NOT allowed";
string PS_2403 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oDepth, c0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2404_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2404_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, b0 is NOT allowed";
string PS_2404 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2405_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2405_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, i0 is NOT allowed";
string PS_2405 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2406_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2406_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, p0 is NOT allowed";
string PS_2406 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2407_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2407_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, s0 is NOT allowed";
string PS_2407 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oDepth, c0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2408_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2408_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, oC0 is NOT allowed";
string PS_2408 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2409_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2409_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oC0, oDepth is NOT allowed";
string PS_2409 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2410_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2410_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, b0 is NOT allowed";
string PS_2410 = 
	"ps_2_x "
	"defb b0, true "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2411_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2411_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, i0 is NOT allowed";
string PS_2411 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2412_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2412_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, p0 is NOT allowed";
string PS_2412 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2413_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2413_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, s0 is NOT allowed";
string PS_2413 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"dcl_cube s0 "
	"texldd r0, oDepth, c0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2414_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2414_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, oC0 is NOT allowed";
string PS_2414 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2415_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2415_Desc = "ps_2_x : texldd source reg combination oDepth, c0, oDepth, oDepth is NOT allowed";
string PS_2415 = 
	"ps_2_x "
	"def c0, 1, 1, 1, 1 "
	"texldd r0, oDepth, c0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2416_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2416_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, b0 is NOT allowed";
string PS_2416 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2417_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2417_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, i0 is NOT allowed";
string PS_2417 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2418_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2418_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, p0 is NOT allowed";
string PS_2418 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2419_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2419_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, s0 is NOT allowed";
string PS_2419 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, b0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2420_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2420_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, oC0 is NOT allowed";
string PS_2420 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2421_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2421_Desc = "ps_2_x : texldd source reg combination oDepth, b0, b0, oDepth is NOT allowed";
string PS_2421 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2422_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2422_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, b0 is NOT allowed";
string PS_2422 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2423_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2423_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, i0 is NOT allowed";
string PS_2423 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2424_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2424_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, p0 is NOT allowed";
string PS_2424 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2425_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2425_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, s0 is NOT allowed";
string PS_2425 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, b0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2426_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2426_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, oC0 is NOT allowed";
string PS_2426 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2427_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2427_Desc = "ps_2_x : texldd source reg combination oDepth, b0, i0, oDepth is NOT allowed";
string PS_2427 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2428_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2428_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, b0 is NOT allowed";
string PS_2428 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2429_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2429_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, i0 is NOT allowed";
string PS_2429 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2430_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2430_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, p0 is NOT allowed";
string PS_2430 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2431_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2431_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, s0 is NOT allowed";
string PS_2431 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, b0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2432_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2432_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, oC0 is NOT allowed";
string PS_2432 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2433_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2433_Desc = "ps_2_x : texldd source reg combination oDepth, b0, p0, oDepth is NOT allowed";
string PS_2433 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2434_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2434_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, b0 is NOT allowed";
string PS_2434 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, b0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2435_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2435_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, i0 is NOT allowed";
string PS_2435 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, b0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2436_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2436_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, p0 is NOT allowed";
string PS_2436 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, b0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2437_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2437_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, s0 is NOT allowed";
string PS_2437 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, b0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2438_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2438_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, oC0 is NOT allowed";
string PS_2438 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, b0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2439_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2439_Desc = "ps_2_x : texldd source reg combination oDepth, b0, s0, oDepth is NOT allowed";
string PS_2439 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, b0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2440_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2440_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, b0 is NOT allowed";
string PS_2440 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2441_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2441_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, i0 is NOT allowed";
string PS_2441 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2442_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2442_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, p0 is NOT allowed";
string PS_2442 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2443_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2443_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, s0 is NOT allowed";
string PS_2443 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oDepth, b0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2444_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2444_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, oC0 is NOT allowed";
string PS_2444 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2445_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2445_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oC0, oDepth is NOT allowed";
string PS_2445 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2446_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2446_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, b0 is NOT allowed";
string PS_2446 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2447_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2447_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, i0 is NOT allowed";
string PS_2447 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, b0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2448_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2448_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, p0 is NOT allowed";
string PS_2448 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2449_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2449_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, s0 is NOT allowed";
string PS_2449 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, b0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2450_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2450_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, oC0 is NOT allowed";
string PS_2450 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2451_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2451_Desc = "ps_2_x : texldd source reg combination oDepth, b0, oDepth, oDepth is NOT allowed";
string PS_2451 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, b0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2452_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2452_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, b0 is NOT allowed";
string PS_2452 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2453_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2453_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, i0 is NOT allowed";
string PS_2453 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2454_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2454_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, p0 is NOT allowed";
string PS_2454 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2455_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2455_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, s0 is NOT allowed";
string PS_2455 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, i0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2456_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2456_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, oC0 is NOT allowed";
string PS_2456 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2457_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2457_Desc = "ps_2_x : texldd source reg combination oDepth, i0, b0, oDepth is NOT allowed";
string PS_2457 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2458_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2458_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, b0 is NOT allowed";
string PS_2458 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2459_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2459_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, i0 is NOT allowed";
string PS_2459 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2460_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2460_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, p0 is NOT allowed";
string PS_2460 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2461_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2461_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, s0 is NOT allowed";
string PS_2461 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, i0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2462_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2462_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, oC0 is NOT allowed";
string PS_2462 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2463_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2463_Desc = "ps_2_x : texldd source reg combination oDepth, i0, i0, oDepth is NOT allowed";
string PS_2463 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2464_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2464_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, b0 is NOT allowed";
string PS_2464 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2465_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2465_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, i0 is NOT allowed";
string PS_2465 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2466_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2466_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, p0 is NOT allowed";
string PS_2466 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2467_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2467_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, s0 is NOT allowed";
string PS_2467 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, i0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2468_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2468_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, oC0 is NOT allowed";
string PS_2468 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2469_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2469_Desc = "ps_2_x : texldd source reg combination oDepth, i0, p0, oDepth is NOT allowed";
string PS_2469 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2470_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2470_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, b0 is NOT allowed";
string PS_2470 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, i0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2471_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2471_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, i0 is NOT allowed";
string PS_2471 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, i0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2472_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2472_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, p0 is NOT allowed";
string PS_2472 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, i0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2473_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2473_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, s0 is NOT allowed";
string PS_2473 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, i0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2474_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2474_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, oC0 is NOT allowed";
string PS_2474 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, i0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2475_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2475_Desc = "ps_2_x : texldd source reg combination oDepth, i0, s0, oDepth is NOT allowed";
string PS_2475 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, i0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2476_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2476_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, b0 is NOT allowed";
string PS_2476 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2477_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2477_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, i0 is NOT allowed";
string PS_2477 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2478_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2478_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, p0 is NOT allowed";
string PS_2478 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2479_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2479_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, s0 is NOT allowed";
string PS_2479 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, i0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2480_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2480_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, oC0 is NOT allowed";
string PS_2480 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2481_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2481_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oC0, oDepth is NOT allowed";
string PS_2481 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2482_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2482_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, b0 is NOT allowed";
string PS_2482 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2483_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2483_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, i0 is NOT allowed";
string PS_2483 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2484_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2484_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, p0 is NOT allowed";
string PS_2484 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2485_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2485_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, s0 is NOT allowed";
string PS_2485 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, i0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2486_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2486_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, oC0 is NOT allowed";
string PS_2486 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2487_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2487_Desc = "ps_2_x : texldd source reg combination oDepth, i0, oDepth, oDepth is NOT allowed";
string PS_2487 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, i0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2488_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2488_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, b0 is NOT allowed";
string PS_2488 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2489_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2489_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, i0 is NOT allowed";
string PS_2489 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, p0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2490_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2490_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, p0 is NOT allowed";
string PS_2490 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2491_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2491_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, s0 is NOT allowed";
string PS_2491 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oDepth, p0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2492_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2492_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, oC0 is NOT allowed";
string PS_2492 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2493_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2493_Desc = "ps_2_x : texldd source reg combination oDepth, p0, b0, oDepth is NOT allowed";
string PS_2493 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2494_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2494_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, b0 is NOT allowed";
string PS_2494 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2495_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2495_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, i0 is NOT allowed";
string PS_2495 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2496_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2496_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, p0 is NOT allowed";
string PS_2496 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2497_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2497_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, s0 is NOT allowed";
string PS_2497 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, p0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2498_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2498_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, oC0 is NOT allowed";
string PS_2498 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2499_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2499_Desc = "ps_2_x : texldd source reg combination oDepth, p0, i0, oDepth is NOT allowed";
string PS_2499 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2500_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2500_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, b0 is NOT allowed";
string PS_2500 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2501_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2501_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, i0 is NOT allowed";
string PS_2501 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2502_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2502_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, p0 is NOT allowed";
string PS_2502 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2503_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2503_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, s0 is NOT allowed";
string PS_2503 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oDepth, p0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2504_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2504_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, oC0 is NOT allowed";
string PS_2504 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2505_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2505_Desc = "ps_2_x : texldd source reg combination oDepth, p0, p0, oDepth is NOT allowed";
string PS_2505 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2506_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2506_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, b0 is NOT allowed";
string PS_2506 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2507_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2507_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, i0 is NOT allowed";
string PS_2507 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2508_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2508_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, p0 is NOT allowed";
string PS_2508 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2509_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2509_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, s0 is NOT allowed";
string PS_2509 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2510_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2510_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, oC0 is NOT allowed";
string PS_2510 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2511_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2511_Desc = "ps_2_x : texldd source reg combination oDepth, p0, s0, oDepth is NOT allowed";
string PS_2511 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2512_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2512_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, b0 is NOT allowed";
string PS_2512 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2513_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2513_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, i0 is NOT allowed";
string PS_2513 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2514_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2514_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, p0 is NOT allowed";
string PS_2514 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2515_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2515_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, s0 is NOT allowed";
string PS_2515 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, p0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2516_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2516_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, oC0 is NOT allowed";
string PS_2516 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2517_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2517_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oC0, oDepth is NOT allowed";
string PS_2517 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2518_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2518_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, b0 is NOT allowed";
string PS_2518 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, p0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2519_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2519_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, i0 is NOT allowed";
string PS_2519 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, p0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2520_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2520_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, p0 is NOT allowed";
string PS_2520 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2521_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2521_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, s0 is NOT allowed";
string PS_2521 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oDepth, p0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2522_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2522_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, oC0 is NOT allowed";
string PS_2522 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2523_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2523_Desc = "ps_2_x : texldd source reg combination oDepth, p0, oDepth, oDepth is NOT allowed";
string PS_2523 = 
	"ps_2_x "
	"texldd r0, oDepth, p0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2524_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2524_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, b0 is NOT allowed";
string PS_2524 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2525_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2525_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, i0 is NOT allowed";
string PS_2525 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2526_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2526_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, p0 is NOT allowed";
string PS_2526 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2527_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2527_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, s0 is NOT allowed";
string PS_2527 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, t0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2528_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2528_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, oC0 is NOT allowed";
string PS_2528 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2529_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2529_Desc = "ps_2_x : texldd source reg combination oDepth, t0, b0, oDepth is NOT allowed";
string PS_2529 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2530_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2530_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, b0 is NOT allowed";
string PS_2530 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2531_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2531_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, i0 is NOT allowed";
string PS_2531 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2532_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2532_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, p0 is NOT allowed";
string PS_2532 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2533_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2533_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, s0 is NOT allowed";
string PS_2533 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_2d s0 "
	"texldd r0, oDepth, t0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2534_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2534_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, oC0 is NOT allowed";
string PS_2534 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2535_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2535_Desc = "ps_2_x : texldd source reg combination oDepth, t0, i0, oDepth is NOT allowed";
string PS_2535 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2536_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2536_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, b0 is NOT allowed";
string PS_2536 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2537_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2537_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, i0 is NOT allowed";
string PS_2537 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2538_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2538_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, p0 is NOT allowed";
string PS_2538 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2539_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2539_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, s0 is NOT allowed";
string PS_2539 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2540_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2540_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, oC0 is NOT allowed";
string PS_2540 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2541_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2541_Desc = "ps_2_x : texldd source reg combination oDepth, t0, p0, oDepth is NOT allowed";
string PS_2541 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2542_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2542_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, b0 is NOT allowed";
string PS_2542 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2543_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2543_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, i0 is NOT allowed";
string PS_2543 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2544_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2544_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, p0 is NOT allowed";
string PS_2544 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2545_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2545_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, s0 is NOT allowed";
string PS_2545 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2546_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2546_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, oC0 is NOT allowed";
string PS_2546 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2547_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2547_Desc = "ps_2_x : texldd source reg combination oDepth, t0, s0, oDepth is NOT allowed";
string PS_2547 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, t0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2548_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2548_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, b0 is NOT allowed";
string PS_2548 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2549_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2549_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, i0 is NOT allowed";
string PS_2549 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2550_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2550_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, p0 is NOT allowed";
string PS_2550 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2551_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2551_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, s0 is NOT allowed";
string PS_2551 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, t0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2552_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2552_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, oC0 is NOT allowed";
string PS_2552 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2553_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2553_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oC0, oDepth is NOT allowed";
string PS_2553 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2554_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2554_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, b0 is NOT allowed";
string PS_2554 = 
	"ps_2_x "
	"defb b0, true "
	"dcl t0 "
	"texldd r0, oDepth, t0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2555_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2555_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, i0 is NOT allowed";
string PS_2555 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl t0 "
	"texldd r0, oDepth, t0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2556_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2556_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, p0 is NOT allowed";
string PS_2556 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2557_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2557_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, s0 is NOT allowed";
string PS_2557 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, t0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2558_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2558_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, oC0 is NOT allowed";
string PS_2558 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2559_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2559_Desc = "ps_2_x : texldd source reg combination oDepth, t0, oDepth, oDepth is NOT allowed";
string PS_2559 = 
	"ps_2_x "
	"dcl t0 "
	"texldd r0, oDepth, t0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2560_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2560_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, b0 is NOT allowed";
string PS_2560 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2561_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2561_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, i0 is NOT allowed";
string PS_2561 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, oC0, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2562_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2562_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, p0 is NOT allowed";
string PS_2562 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2563_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2563_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, s0 is NOT allowed";
string PS_2563 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, oC0, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2564_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2564_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, oC0 is NOT allowed";
string PS_2564 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2565_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2565_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, b0, oDepth is NOT allowed";
string PS_2565 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2566_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2566_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, b0 is NOT allowed";
string PS_2566 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2567_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2567_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, i0 is NOT allowed";
string PS_2567 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2568_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2568_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, p0 is NOT allowed";
string PS_2568 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2569_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2569_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, s0 is NOT allowed";
string PS_2569 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, oC0, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2570_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2570_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, oC0 is NOT allowed";
string PS_2570 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2571_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2571_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, i0, oDepth is NOT allowed";
string PS_2571 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2572_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2572_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, b0 is NOT allowed";
string PS_2572 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2573_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2573_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, i0 is NOT allowed";
string PS_2573 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2574_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2574_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, p0 is NOT allowed";
string PS_2574 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2575_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2575_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, s0 is NOT allowed";
string PS_2575 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oDepth, oC0, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2576_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2576_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, oC0 is NOT allowed";
string PS_2576 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2577_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2577_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, p0, oDepth is NOT allowed";
string PS_2577 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2578_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2578_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, b0 is NOT allowed";
string PS_2578 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_volume s0 "
	"texldd r0, oDepth, oC0, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2579_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2579_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, i0 is NOT allowed";
string PS_2579 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, oC0, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2580_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2580_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, p0 is NOT allowed";
string PS_2580 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oDepth, oC0, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2581_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2581_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, s0 is NOT allowed";
string PS_2581 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, oC0, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2582_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2582_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, oC0 is NOT allowed";
string PS_2582 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, oC0, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2583_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2583_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, s0, oDepth is NOT allowed";
string PS_2583 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, oC0, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2584_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2584_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, b0 is NOT allowed";
string PS_2584 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2585_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2585_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, i0 is NOT allowed";
string PS_2585 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2586_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2586_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, p0 is NOT allowed";
string PS_2586 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2587_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2587_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, s0 is NOT allowed";
string PS_2587 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oC0, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2588_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2588_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, oC0 is NOT allowed";
string PS_2588 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2589_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2589_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oC0, oDepth is NOT allowed";
string PS_2589 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2590_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2590_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, b0 is NOT allowed";
string PS_2590 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oC0, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2591_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2591_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, i0 is NOT allowed";
string PS_2591 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oC0, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2592_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2592_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, p0 is NOT allowed";
string PS_2592 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2593_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2593_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, s0 is NOT allowed";
string PS_2593 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, oC0, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2594_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2594_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, oC0 is NOT allowed";
string PS_2594 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2595_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2595_Desc = "ps_2_x : texldd source reg combination oDepth, oC0, oDepth, oDepth is NOT allowed";
string PS_2595 = 
	"ps_2_x "
	"texldd r0, oDepth, oC0, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2596_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2596_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, b0 is NOT allowed";
string PS_2596 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, b0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2597_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2597_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, i0 is NOT allowed";
string PS_2597 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, b0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2598_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2598_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, p0 is NOT allowed";
string PS_2598 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, b0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2599_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2599_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, s0 is NOT allowed";
string PS_2599 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, b0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2600_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2600_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, oC0 is NOT allowed";
string PS_2600 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, b0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2601_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2601_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, b0, oDepth is NOT allowed";
string PS_2601 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, b0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2602_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2602_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, b0 is NOT allowed";
string PS_2602 = 
	"ps_2_x "
	"defb b0, true "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, i0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2603_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2603_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, i0 is NOT allowed";
string PS_2603 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, i0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2604_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2604_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, p0 is NOT allowed";
string PS_2604 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, i0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2605_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2605_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, s0 is NOT allowed";
string PS_2605 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_volume s0 "
	"texldd r0, oDepth, oDepth, i0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2606_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2606_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, oC0 is NOT allowed";
string PS_2606 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, i0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2607_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2607_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, i0, oDepth is NOT allowed";
string PS_2607 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, i0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2608_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2608_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, b0 is NOT allowed";
string PS_2608 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, p0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2609_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2609_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, i0 is NOT allowed";
string PS_2609 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, p0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2610_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2610_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, p0 is NOT allowed";
string PS_2610 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, p0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2611_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2611_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, s0 is NOT allowed";
string PS_2611 = 
	"ps_2_x "
	"dcl_2d s0 "
	"texldd r0, oDepth, oDepth, p0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2612_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2612_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, oC0 is NOT allowed";
string PS_2612 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, p0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2613_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2613_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, p0, oDepth is NOT allowed";
string PS_2613 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, p0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2614_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2614_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, b0 is NOT allowed";
string PS_2614 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2615_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2615_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, i0 is NOT allowed";
string PS_2615 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2616_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2616_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, p0 is NOT allowed";
string PS_2616 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2617_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2617_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, s0 is NOT allowed";
string PS_2617 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2618_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2618_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, oC0 is NOT allowed";
string PS_2618 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2619_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2619_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, s0, oDepth is NOT allowed";
string PS_2619 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, s0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2620_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2620_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, b0 is NOT allowed";
string PS_2620 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, oC0, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2621_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2621_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, i0 is NOT allowed";
string PS_2621 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, oC0, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2622_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2622_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, p0 is NOT allowed";
string PS_2622 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oC0, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2623_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2623_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, s0 is NOT allowed";
string PS_2623 = 
	"ps_2_x "
	"dcl_cube s0 "
	"texldd r0, oDepth, oDepth, oC0, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2624_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2624_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, oC0 is NOT allowed";
string PS_2624 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oC0, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2625_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2625_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oC0, oDepth is NOT allowed";
string PS_2625 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oC0, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2626_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2626_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, b0 is NOT allowed";
string PS_2626 = 
	"ps_2_x "
	"defb b0, true "
	"texldd r0, oDepth, oDepth, oDepth, b0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2627_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2627_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, i0 is NOT allowed";
string PS_2627 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"texldd r0, oDepth, oDepth, oDepth, i0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2628_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2628_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, p0 is NOT allowed";
string PS_2628 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oDepth, p0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2629_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2629_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, s0 is NOT allowed";
string PS_2629 = 
	"ps_2_x "
	"dcl_volume s0 "
	"texldd r0, oDepth, oDepth, oDepth, s0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2630_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2630_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, oC0 is NOT allowed";
string PS_2630 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oDepth, oC0 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2631_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2631_Desc = "ps_2_x : texldd source reg combination oDepth, oDepth, oDepth, oDepth is NOT allowed";
string PS_2631 = 
	"ps_2_x "
	"texldd r0, oDepth, oDepth, oDepth, oDepth "
	"mov oC0, c0 ";

PS_CRITERIA PS_2632_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2632_Desc = "ps_2_x : texldd source reg combination -t0, s0, r0, r1 is NOT allowed";
string PS_2632 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, -t0, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2633_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2633_Desc = "ps_2_x : texldd source reg combination t0.r, s0, r0, r1 is NOT allowed";
string PS_2633 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.r, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2634_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2634_Desc = "ps_2_x : texldd source reg combination t0.g, s0, r0, r1 is NOT allowed";
string PS_2634 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.g, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2635_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2635_Desc = "ps_2_x : texldd source reg combination t0.b, s0, r0, r1 is NOT allowed";
string PS_2635 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.b, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2636_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2636_Desc = "ps_2_x : texldd source reg combination t0.a, s0, r0, r1 is NOT allowed";
string PS_2636 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.a, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2637_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2637_Desc = "ps_2_x : texldd source reg combination t0.gbra, s0, r0, r1 is NOT allowed";
string PS_2637 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.gbra, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2638_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2638_Desc = "ps_2_x : texldd source reg combination t0.brga, s0, r0, r1 is NOT allowed";
string PS_2638 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.brga, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2639_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2639_Desc = "ps_2_x : texldd source reg combination t0.abgr, s0, r0, r1 is NOT allowed";
string PS_2639 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.abgr, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2640_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2640_Desc = "ps_2_x : texldd source reg combination t0.agrb, s0, r0, r1 is NOT allowed";
string PS_2640 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.agrb, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2641_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2641_Desc = "ps_2_x : texldd source reg combination t0.rbga, s0, r0, r1 is NOT allowed";
string PS_2641 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.rbga, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2642_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2642_Desc = "ps_2_x : texldd source reg combination t0.rgab, s0, r0, r1 is NOT allowed";
string PS_2642 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.rgab, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2643_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2643_Desc = "ps_2_x : texldd source reg combination t0.bgr, s0, r0, r1 is NOT allowed";
string PS_2643 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.bgr, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2644_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2644_Desc = "ps_2_x : texldd source reg combination t0.rbg, s0, r0, r1 is NOT allowed";
string PS_2644 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.rbg, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2645_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2645_Desc = "ps_2_x : texldd source reg combination t0.gar, s0, r0, r1 is NOT allowed";
string PS_2645 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.gar, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2646_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2646_Desc = "ps_2_x : texldd source reg combination t0.gr, s0, r0, r1 is NOT allowed";
string PS_2646 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.gr, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2647_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2647_Desc = "ps_2_x : texldd source reg combination t0.ab, s0, r0, r1 is NOT allowed";
string PS_2647 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.ab, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2648_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2648_Desc = "ps_2_x : texldd source reg combination t0.bg, s0, r0, r1 is NOT allowed";
string PS_2648 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0.bg, s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2649_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2649_Desc = "ps_2_x : texldd source reg combination t0, -s0, r0, r1 is NOT allowed";
string PS_2649 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, -s0, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2650_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2650_Desc = "ps_2_x : texldd source reg combination t0, s0.r, r0, r1 is NOT allowed";
string PS_2650 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.r, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2651_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2651_Desc = "ps_2_x : texldd source reg combination t0, s0.g, r0, r1 is NOT allowed";
string PS_2651 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.g, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2652_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2652_Desc = "ps_2_x : texldd source reg combination t0, s0.b, r0, r1 is NOT allowed";
string PS_2652 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.b, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2653_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2653_Desc = "ps_2_x : texldd source reg combination t0, s0.a, r0, r1 is NOT allowed";
string PS_2653 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.a, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2654_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2654_Desc = "ps_2_x : texldd source reg combination t0, s0.gbra, r0, r1 is NOT allowed";
string PS_2654 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.gbra, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2655_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2655_Desc = "ps_2_x : texldd source reg combination t0, s0.brga, r0, r1 is NOT allowed";
string PS_2655 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.brga, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2656_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2656_Desc = "ps_2_x : texldd source reg combination t0, s0.abgr, r0, r1 is NOT allowed";
string PS_2656 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.abgr, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2657_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2657_Desc = "ps_2_x : texldd source reg combination t0, s0.agrb, r0, r1 is NOT allowed";
string PS_2657 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.agrb, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2658_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2658_Desc = "ps_2_x : texldd source reg combination t0, s0.rbga, r0, r1 is NOT allowed";
string PS_2658 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.rbga, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2659_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2659_Desc = "ps_2_x : texldd source reg combination t0, s0.rgab, r0, r1 is NOT allowed";
string PS_2659 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.rgab, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2660_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2660_Desc = "ps_2_x : texldd source reg combination t0, s0.bgr, r0, r1 is NOT allowed";
string PS_2660 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.bgr, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2661_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2661_Desc = "ps_2_x : texldd source reg combination t0, s0.rbg, r0, r1 is NOT allowed";
string PS_2661 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.rbg, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2662_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2662_Desc = "ps_2_x : texldd source reg combination t0, s0.gar, r0, r1 is NOT allowed";
string PS_2662 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.gar, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2663_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2663_Desc = "ps_2_x : texldd source reg combination t0, s0.gr, r0, r1 is NOT allowed";
string PS_2663 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.gr, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2664_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2664_Desc = "ps_2_x : texldd source reg combination t0, s0.ab, r0, r1 is NOT allowed";
string PS_2664 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.ab, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2665_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2665_Desc = "ps_2_x : texldd source reg combination t0, s0.bg, r0, r1 is NOT allowed";
string PS_2665 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0.bg, r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2666_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2666_Desc = "ps_2_x : texldd source reg combination t0, s0, -r0, r1 is NOT allowed";
string PS_2666 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, -r0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2667_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2667_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.r, r1 is NOT allowed";
string PS_2667 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.r, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2668_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2668_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.g, r1 is NOT allowed";
string PS_2668 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.g, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2669_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2669_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.b, r1 is NOT allowed";
string PS_2669 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.b, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2670_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2670_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.a, r1 is NOT allowed";
string PS_2670 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.a, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2671_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2671_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.gbra, r1 is NOT allowed";
string PS_2671 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.gbra, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2672_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2672_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.brga, r1 is NOT allowed";
string PS_2672 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.brga, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2673_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2673_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.abgr, r1 is NOT allowed";
string PS_2673 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.abgr, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2674_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2674_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.agrb, r1 is NOT allowed";
string PS_2674 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.agrb, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2675_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2675_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.rbga, r1 is NOT allowed";
string PS_2675 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.rbga, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2676_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2676_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.rgab, r1 is NOT allowed";
string PS_2676 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.rgab, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2677_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2677_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.bgr, r1 is NOT allowed";
string PS_2677 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.bgr, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2678_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2678_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.rbg, r1 is NOT allowed";
string PS_2678 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.rbg, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2679_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2679_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.gar, r1 is NOT allowed";
string PS_2679 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.gar, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2680_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2680_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.gr, r1 is NOT allowed";
string PS_2680 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.gr, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2681_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2681_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.ab, r1 is NOT allowed";
string PS_2681 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.ab, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2682_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2682_Desc = "ps_2_x : texldd source reg combination t0, s0, r0.bg, r1 is NOT allowed";
string PS_2682 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0.bg, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2683_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2683_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, -r1 is NOT allowed";
string PS_2683 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, -r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2684_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2684_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.r is NOT allowed";
string PS_2684 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.r "
	"mov oC0, c0 ";

PS_CRITERIA PS_2685_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2685_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.g is NOT allowed";
string PS_2685 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.g "
	"mov oC0, c0 ";

PS_CRITERIA PS_2686_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2686_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.b is NOT allowed";
string PS_2686 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.b "
	"mov oC0, c0 ";

PS_CRITERIA PS_2687_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2687_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.a is NOT allowed";
string PS_2687 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.a "
	"mov oC0, c0 ";

PS_CRITERIA PS_2688_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2688_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.gbra is NOT allowed";
string PS_2688 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.gbra "
	"mov oC0, c0 ";

PS_CRITERIA PS_2689_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2689_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.brga is NOT allowed";
string PS_2689 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.brga "
	"mov oC0, c0 ";

PS_CRITERIA PS_2690_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2690_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.abgr is NOT allowed";
string PS_2690 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.abgr "
	"mov oC0, c0 ";

PS_CRITERIA PS_2691_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2691_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.agrb is NOT allowed";
string PS_2691 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.agrb "
	"mov oC0, c0 ";

PS_CRITERIA PS_2692_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2692_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.rbga is NOT allowed";
string PS_2692 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_2d s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.rbga "
	"mov oC0, c0 ";

PS_CRITERIA PS_2693_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2693_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.rgab is NOT allowed";
string PS_2693 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.rgab "
	"mov oC0, c0 ";

PS_CRITERIA PS_2694_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2694_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.bgr is NOT allowed";
string PS_2694 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.bgr "
	"mov oC0, c0 ";

PS_CRITERIA PS_2695_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2695_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.rbg is NOT allowed";
string PS_2695 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.rbg "
	"mov oC0, c0 ";

PS_CRITERIA PS_2696_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2696_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.gar is NOT allowed";
string PS_2696 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_cube s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.gar "
	"mov oC0, c0 ";

PS_CRITERIA PS_2697_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2697_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.gr is NOT allowed";
string PS_2697 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.gr "
	"mov oC0, c0 ";

PS_CRITERIA PS_2698_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2698_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.ab is NOT allowed";
string PS_2698 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.ab "
	"mov oC0, c0 ";

PS_CRITERIA PS_2699_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_ARBITRARYSWIZZLE + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2699_Desc = "ps_2_x : texldd source reg combination t0, s0, r0, r1.bg is NOT allowed";
string PS_2699 = 
	"ps_2_x "
	"dcl t0 "
	"dcl_volume s0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, t0, s0, r0, r1.bg "
	"mov oC0, c0 ";

PS_CRITERIA PS_2700_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2700_Desc = "ps_2_x : texldd dest r# is allowed";
string PS_2700 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2701_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2701_Desc = "ps_2_x : texldd _pp dest r# is allowed";
string PS_2701 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp r0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2702_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2702_Desc = "ps_2_x : texldd _sat dest r# is NOT allowed";
string PS_2702 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat r0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2703_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2703_Desc = "ps_2_x : texldd dest v# is NOT allowed";
string PS_2703 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd v0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2704_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2704_Desc = "ps_2_x : texldd _pp dest v# is NOT allowed";
string PS_2704 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp v0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2705_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2705_Desc = "ps_2_x : texldd _sat dest v# is NOT allowed";
string PS_2705 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat v0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2706_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2706_Desc = "ps_2_x : texldd dest c# is NOT allowed";
string PS_2706 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd c1, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2707_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2707_Desc = "ps_2_x : texldd _pp dest c# is NOT allowed";
string PS_2707 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp c1, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2708_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2708_Desc = "ps_2_x : texldd _sat dest c# is NOT allowed";
string PS_2708 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat c1, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2709_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2709_Desc = "ps_2_x : texldd dest b# is NOT allowed";
string PS_2709 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd b0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2710_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2710_Desc = "ps_2_x : texldd _pp dest b# is NOT allowed";
string PS_2710 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp b0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2711_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2711_Desc = "ps_2_x : texldd _sat dest b# is NOT allowed";
string PS_2711 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat b0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2712_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2712_Desc = "ps_2_x : texldd dest i# is NOT allowed";
string PS_2712 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd i0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2713_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2713_Desc = "ps_2_x : texldd _pp dest i# is NOT allowed";
string PS_2713 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp i0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2714_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2714_Desc = "ps_2_x : texldd _sat dest i# is NOT allowed";
string PS_2714 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat i0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2715_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_2715_Desc = "ps_2_x : texldd dest p0 is NOT allowed";
string PS_2715 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd p0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2716_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_2716_Desc = "ps_2_x : texldd _pp dest p0 is NOT allowed";
string PS_2716 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp p0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2717_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 9.109f };
string PS_2717_Desc = "ps_2_x : texldd _sat dest p0 is NOT allowed";
string PS_2717 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat p0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2718_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2718_Desc = "ps_2_x : texldd dest s# is NOT allowed";
string PS_2718 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd s0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2719_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2719_Desc = "ps_2_x : texldd _pp dest s# is NOT allowed";
string PS_2719 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp s0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2720_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2720_Desc = "ps_2_x : texldd _sat dest s# is NOT allowed";
string PS_2720 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat s0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2721_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2721_Desc = "ps_2_x : texldd dest t# is NOT allowed";
string PS_2721 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd t0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2722_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2722_Desc = "ps_2_x : texldd _pp dest t# is NOT allowed";
string PS_2722 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp t0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2723_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2723_Desc = "ps_2_x : texldd _sat dest t# is NOT allowed";
string PS_2723 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat t0, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2724_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2724_Desc = "ps_2_x : texldd dest oC# is NOT allowed";
string PS_2724 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd oC0, r0, s0, v0, r1 ";

PS_CRITERIA PS_2725_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2725_Desc = "ps_2_x : texldd _pp dest oC# is NOT allowed";
string PS_2725 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp oC0, r0, s0, v0, r1 ";

PS_CRITERIA PS_2726_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2726_Desc = "ps_2_x : texldd _sat dest oC# is NOT allowed";
string PS_2726 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat oC0, r0, s0, v0, r1 ";

PS_CRITERIA PS_2727_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2727_Desc = "ps_2_x : texldd dest oDepth is NOT allowed";
string PS_2727 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd oDepth, r0.x, s0.x, v0.x, r1.x "
	"mov oC0, c0 ";

PS_CRITERIA PS_2728_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2728_Desc = "ps_2_x : texldd _pp dest oDepth is NOT allowed";
string PS_2728 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_pp oDepth, r0.x, s0.x, v0.x, r1.x "
	"mov oC0, c0 ";

PS_CRITERIA PS_2729_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2729_Desc = "ps_2_x : texldd _sat dest oDepth is NOT allowed";
string PS_2729 = 
	"ps_2_x "
	"dcl_volume s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd_sat oDepth, r0.x, s0.x, v0.x, r1.x "
	"mov oC0, c0 ";

PS_CRITERIA PS_2730_Criteria = { true, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2730_Desc = "ps_2_x : texldd write mask .rgba is allowed";
string PS_2730 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.rgba, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2731_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2731_Desc = "ps_2_x : texldd write mask .r is NOT allowed";
string PS_2731 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.r, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2732_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2732_Desc = "ps_2_x : texldd write mask .g is NOT allowed";
string PS_2732 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.g, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2733_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2733_Desc = "ps_2_x : texldd write mask .b is NOT allowed";
string PS_2733 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.b, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2734_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2734_Desc = "ps_2_x : texldd write mask .a is NOT allowed";
string PS_2734 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.a, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2735_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2735_Desc = "ps_2_x : texldd write mask .rg is NOT allowed";
string PS_2735 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.rg, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2736_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2736_Desc = "ps_2_x : texldd write mask .gb is NOT allowed";
string PS_2736 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.gb, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2737_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2737_Desc = "ps_2_x : texldd write mask .ba is NOT allowed";
string PS_2737 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.ba, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2738_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2738_Desc = "ps_2_x : texldd write mask .rb is NOT allowed";
string PS_2738 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.rb, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2739_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2739_Desc = "ps_2_x : texldd write mask .ra is NOT allowed";
string PS_2739 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.ra, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2740_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2740_Desc = "ps_2_x : texldd write mask .ga is NOT allowed";
string PS_2740 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.ga, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2741_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2741_Desc = "ps_2_x : texldd write mask .rgb is NOT allowed";
string PS_2741 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.rgb, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2742_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2742_Desc = "ps_2_x : texldd write mask .gba is NOT allowed";
string PS_2742 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.gba, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2743_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2743_Desc = "ps_2_x : texldd write mask .rba is NOT allowed";
string PS_2743 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.rba, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2744_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2744_Desc = "ps_2_x : texldd write mask .yx is NOT allowed";
string PS_2744 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.yx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2745_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2745_Desc = "ps_2_x : texldd write mask .zx is NOT allowed";
string PS_2745 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.zx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2746_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2746_Desc = "ps_2_x : texldd write mask .zy is NOT allowed";
string PS_2746 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.zy, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2747_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2747_Desc = "ps_2_x : texldd write mask .wx is NOT allowed";
string PS_2747 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2748_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2748_Desc = "ps_2_x : texldd write mask .wz is NOT allowed";
string PS_2748 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wz, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2749_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2749_Desc = "ps_2_x : texldd write mask .wy is NOT allowed";
string PS_2749 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wy, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2750_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2750_Desc = "ps_2_x : texldd write mask .zyx is NOT allowed";
string PS_2750 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.zyx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2751_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2751_Desc = "ps_2_x : texldd write mask .wzy is NOT allowed";
string PS_2751 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wzy, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2752_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2752_Desc = "ps_2_x : texldd write mask .wzx is NOT allowed";
string PS_2752 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wzx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2753_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2753_Desc = "ps_2_x : texldd write mask .wyx is NOT allowed";
string PS_2753 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wyx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2754_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2754_Desc = "ps_2_x : texldd write mask .yxw is NOT allowed";
string PS_2754 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.yxw, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2755_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2755_Desc = "ps_2_x : texldd write mask .wzyx is NOT allowed";
string PS_2755 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.wzyx, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2756_Criteria = { false, false, 0x0200, -1, -1, -1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2756_Desc = "ps_2_x : texldd write mask .zxwy is NOT allowed";
string PS_2756 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"texldd r0.zxwy, r0, s0, v0, r1 "
	"mov oC0, c0 ";

PS_CRITERIA PS_2757_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2757_Desc = "ps_2_x : texldd is allowed in a 1 level if b0 block";
string PS_2757 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2758_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2758_Desc = "ps_2_x : texldd is allowed in a 8 level if b0 block";
string PS_2758 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 "
	"	if b0 "
	"		if b0 "
	"			if b0 "
	"				if b0 "
	"					if b0 "
	"						if b0 "
	"							if b0 "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2759_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2759_Desc = "ps_2_x : texldd is allowed in a 16 level if b0 block";
string PS_2759 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 "
	"	if b0 "
	"		if b0 "
	"			if b0 "
	"				if b0 "
	"					if b0 "
	"						if b0 "
	"							if b0 "
	"								if b0 "
	"									if b0 "
	"										if b0 "
	"											if b0 "
	"												if b0 "
	"													if b0 "
	"														if b0 "
	"															if b0 "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2760_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2760_Desc = "ps_2_x : texldd is allowed in a 24 level if b0 block";
string PS_2760 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 "
	"	if b0 "
	"		if b0 "
	"			if b0 "
	"				if b0 "
	"					if b0 "
	"						if b0 "
	"							if b0 "
	"								if b0 "
	"									if b0 "
	"										if b0 "
	"											if b0 "
	"												if b0 "
	"													if b0 "
	"														if b0 "
	"															if b0 "
	"																if b0 "
	"																	if b0 "
	"																		if b0 "
	"																			if b0 "
	"																				if b0 "
	"																					if b0 "
	"																						if b0 "
	"																							if b0 "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2761_Criteria = { false, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2761_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if b0 block";
string PS_2761 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 "
	"	if b0 "
	"		if b0 "
	"			if b0 "
	"				if b0 "
	"					if b0 "
	"						if b0 "
	"							if b0 "
	"								if b0 "
	"									if b0 "
	"										if b0 "
	"											if b0 "
	"												if b0 "
	"													if b0 "
	"														if b0 "
	"															if b0 "
	"																if b0 "
	"																	if b0 "
	"																		if b0 "
	"																			if b0 "
	"																				if b0 "
	"																					if b0 "
	"																						if b0 "
	"																							if b0 "
	"																								if b0 "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2762_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2762_Desc = "ps_2_x : texldd is allowed in a 1 level if b0 nop else block";
string PS_2762 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 nop else "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2763_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2763_Desc = "ps_2_x : texldd is allowed in a 8 level if b0 nop else block";
string PS_2763 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 nop else "
	"	if b0 nop else "
	"		if b0 nop else "
	"			if b0 nop else "
	"				if b0 nop else "
	"					if b0 nop else "
	"						if b0 nop else "
	"							if b0 nop else "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2764_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2764_Desc = "ps_2_x : texldd is allowed in a 16 level if b0 nop else block";
string PS_2764 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 nop else "
	"	if b0 nop else "
	"		if b0 nop else "
	"			if b0 nop else "
	"				if b0 nop else "
	"					if b0 nop else "
	"						if b0 nop else "
	"							if b0 nop else "
	"								if b0 nop else "
	"									if b0 nop else "
	"										if b0 nop else "
	"											if b0 nop else "
	"												if b0 nop else "
	"													if b0 nop else "
	"														if b0 nop else "
	"															if b0 nop else "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2765_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2765_Desc = "ps_2_x : texldd is allowed in a 24 level if b0 nop else block";
string PS_2765 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 nop else "
	"	if b0 nop else "
	"		if b0 nop else "
	"			if b0 nop else "
	"				if b0 nop else "
	"					if b0 nop else "
	"						if b0 nop else "
	"							if b0 nop else "
	"								if b0 nop else "
	"									if b0 nop else "
	"										if b0 nop else "
	"											if b0 nop else "
	"												if b0 nop else "
	"													if b0 nop else "
	"														if b0 nop else "
	"															if b0 nop else "
	"																if b0 nop else "
	"																	if b0 nop else "
	"																		if b0 nop else "
	"																			if b0 nop else "
	"																				if b0 nop else "
	"																					if b0 nop else "
	"																						if b0 nop else "
	"																							if b0 nop else "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2766_Criteria = { false, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2766_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if b0 nop else block";
string PS_2766 = 
	"ps_2_x "
	"defb b0, true "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if b0 nop else "
	"	if b0 nop else "
	"		if b0 nop else "
	"			if b0 nop else "
	"				if b0 nop else "
	"					if b0 nop else "
	"						if b0 nop else "
	"							if b0 nop else "
	"								if b0 nop else "
	"									if b0 nop else "
	"										if b0 nop else "
	"											if b0 nop else "
	"												if b0 nop else "
	"													if b0 nop else "
	"														if b0 nop else "
	"															if b0 nop else "
	"																if b0 nop else "
	"																	if b0 nop else "
	"																		if b0 nop else "
	"																			if b0 nop else "
	"																				if b0 nop else "
	"																					if b0 nop else "
	"																						if b0 nop else "
	"																							if b0 nop else "
	"																								if b0 nop else "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2767_Criteria = { false, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2767_Desc = "ps_2_x : texldd is NOT allowed in a 1 level loop aL, i0 block";
string PS_2767 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 "
	"	texldd r0, r0, s0, v0, r1 "
	"endloop ";

PS_CRITERIA PS_2768_Criteria = { false, false, 0x0200, 0, -1, 2, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2768_Desc = "ps_2_x : texldd is NOT allowed in a 2 level loop aL, i0 block";
string PS_2768 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 "
	"	loop aL, i0 "
	"		texldd r0, r0, s0, v0, r1 "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2769_Criteria = { false, false, 0x0200, 0, -1, 3, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2769_Desc = "ps_2_x : texldd is NOT allowed in a 3 level loop aL, i0 block";
string PS_2769 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 "
	"	loop aL, i0 "
	"		loop aL, i0 "
	"			texldd r0, r0, s0, v0, r1 "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2770_Criteria = { false, false, 0x0200, 0, -1, 4, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2770_Desc = "ps_2_x : texldd is NOT allowed in a 4 level loop aL, i0 block";
string PS_2770 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 "
	"	loop aL, i0 "
	"		loop aL, i0 "
	"			loop aL, i0 "
	"				texldd r0, r0, s0, v0, r1 "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2771_Criteria = { false, false, 0x0200, 0, -1, 5, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2771_Desc = "ps_2_x : texldd is NOT allowed in a 5 level loop aL, i0 block";
string PS_2771 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 "
	"	loop aL, i0 "
	"		loop aL, i0 "
	"			loop aL, i0 "
	"				loop aL, i0 "
	"					texldd r0, r0, s0, v0, r1 "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2772_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2772_Desc = "ps_2_x : texldd is allowed in a 1 level rep i0 block";
string PS_2772 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 "
	"	texldd r0, r0, s0, v0, r1 "
	"endrep ";

PS_CRITERIA PS_2773_Criteria = { true, false, 0x0200, 0, -1, 2, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2773_Desc = "ps_2_x : texldd is allowed in a 2 level rep i0 block";
string PS_2773 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 "
	"	rep i0 "
	"		texldd r0, r0, s0, v0, r1 "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2774_Criteria = { true, false, 0x0200, 0, -1, 3, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2774_Desc = "ps_2_x : texldd is allowed in a 3 level rep i0 block";
string PS_2774 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 "
	"	rep i0 "
	"		rep i0 "
	"			texldd r0, r0, s0, v0, r1 "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2775_Criteria = { true, false, 0x0200, 0, -1, 4, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2775_Desc = "ps_2_x : texldd is allowed in a 4 level rep i0 block";
string PS_2775 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 "
	"	rep i0 "
	"		rep i0 "
	"			rep i0 "
	"				texldd r0, r0, s0, v0, r1 "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2776_Criteria = { false, false, 0x0200, 0, -1, 5, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2776_Desc = "ps_2_x : texldd is NOT allowed in a 5 level rep i0 block";
string PS_2776 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 "
	"	rep i0 "
	"		rep i0 "
	"			rep i0 "
	"				rep i0 "
	"					texldd r0, r0, s0, v0, r1 "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2777_Criteria = { true, false, 0x0200, 1, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2777_Desc = "ps_2_x : texldd is allowed in a 1 level if_gt c0.x, c0.y block";
string PS_2777 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2778_Criteria = { true, false, 0x0200, 8, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2778_Desc = "ps_2_x : texldd is allowed in a 8 level if_gt c0.x, c0.y block";
string PS_2778 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y "
	"	if_gt c0.x, c0.y "
	"		if_gt c0.x, c0.y "
	"			if_gt c0.x, c0.y "
	"				if_gt c0.x, c0.y "
	"					if_gt c0.x, c0.y "
	"						if_gt c0.x, c0.y "
	"							if_gt c0.x, c0.y "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2779_Criteria = { true, false, 0x0200, 16, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2779_Desc = "ps_2_x : texldd is allowed in a 16 level if_gt c0.x, c0.y block";
string PS_2779 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y "
	"	if_gt c0.x, c0.y "
	"		if_gt c0.x, c0.y "
	"			if_gt c0.x, c0.y "
	"				if_gt c0.x, c0.y "
	"					if_gt c0.x, c0.y "
	"						if_gt c0.x, c0.y "
	"							if_gt c0.x, c0.y "
	"								if_gt c0.x, c0.y "
	"									if_gt c0.x, c0.y "
	"										if_gt c0.x, c0.y "
	"											if_gt c0.x, c0.y "
	"												if_gt c0.x, c0.y "
	"													if_gt c0.x, c0.y "
	"														if_gt c0.x, c0.y "
	"															if_gt c0.x, c0.y "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2780_Criteria = { true, false, 0x0200, 24, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2780_Desc = "ps_2_x : texldd is allowed in a 24 level if_gt c0.x, c0.y block";
string PS_2780 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y "
	"	if_gt c0.x, c0.y "
	"		if_gt c0.x, c0.y "
	"			if_gt c0.x, c0.y "
	"				if_gt c0.x, c0.y "
	"					if_gt c0.x, c0.y "
	"						if_gt c0.x, c0.y "
	"							if_gt c0.x, c0.y "
	"								if_gt c0.x, c0.y "
	"									if_gt c0.x, c0.y "
	"										if_gt c0.x, c0.y "
	"											if_gt c0.x, c0.y "
	"												if_gt c0.x, c0.y "
	"													if_gt c0.x, c0.y "
	"														if_gt c0.x, c0.y "
	"															if_gt c0.x, c0.y "
	"																if_gt c0.x, c0.y "
	"																	if_gt c0.x, c0.y "
	"																		if_gt c0.x, c0.y "
	"																			if_gt c0.x, c0.y "
	"																				if_gt c0.x, c0.y "
	"																					if_gt c0.x, c0.y "
	"																						if_gt c0.x, c0.y "
	"																							if_gt c0.x, c0.y "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2781_Criteria = { false, false, 0x0200, 25, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2781_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if_gt c0.x, c0.y block";
string PS_2781 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y "
	"	if_gt c0.x, c0.y "
	"		if_gt c0.x, c0.y "
	"			if_gt c0.x, c0.y "
	"				if_gt c0.x, c0.y "
	"					if_gt c0.x, c0.y "
	"						if_gt c0.x, c0.y "
	"							if_gt c0.x, c0.y "
	"								if_gt c0.x, c0.y "
	"									if_gt c0.x, c0.y "
	"										if_gt c0.x, c0.y "
	"											if_gt c0.x, c0.y "
	"												if_gt c0.x, c0.y "
	"													if_gt c0.x, c0.y "
	"														if_gt c0.x, c0.y "
	"															if_gt c0.x, c0.y "
	"																if_gt c0.x, c0.y "
	"																	if_gt c0.x, c0.y "
	"																		if_gt c0.x, c0.y "
	"																			if_gt c0.x, c0.y "
	"																				if_gt c0.x, c0.y "
	"																					if_gt c0.x, c0.y "
	"																						if_gt c0.x, c0.y "
	"																							if_gt c0.x, c0.y "
	"																								if_gt c0.x, c0.y "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2782_Criteria = { true, false, 0x0200, 1, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2782_Desc = "ps_2_x : texldd is allowed in a 1 level if_gt c0.x, c0.y nop else block";
string PS_2782 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y nop else "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2783_Criteria = { true, false, 0x0200, 8, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2783_Desc = "ps_2_x : texldd is allowed in a 8 level if_gt c0.x, c0.y nop else block";
string PS_2783 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y nop else "
	"	if_gt c0.x, c0.y nop else "
	"		if_gt c0.x, c0.y nop else "
	"			if_gt c0.x, c0.y nop else "
	"				if_gt c0.x, c0.y nop else "
	"					if_gt c0.x, c0.y nop else "
	"						if_gt c0.x, c0.y nop else "
	"							if_gt c0.x, c0.y nop else "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2784_Criteria = { true, false, 0x0200, 16, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2784_Desc = "ps_2_x : texldd is allowed in a 16 level if_gt c0.x, c0.y nop else block";
string PS_2784 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y nop else "
	"	if_gt c0.x, c0.y nop else "
	"		if_gt c0.x, c0.y nop else "
	"			if_gt c0.x, c0.y nop else "
	"				if_gt c0.x, c0.y nop else "
	"					if_gt c0.x, c0.y nop else "
	"						if_gt c0.x, c0.y nop else "
	"							if_gt c0.x, c0.y nop else "
	"								if_gt c0.x, c0.y nop else "
	"									if_gt c0.x, c0.y nop else "
	"										if_gt c0.x, c0.y nop else "
	"											if_gt c0.x, c0.y nop else "
	"												if_gt c0.x, c0.y nop else "
	"													if_gt c0.x, c0.y nop else "
	"														if_gt c0.x, c0.y nop else "
	"															if_gt c0.x, c0.y nop else "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2785_Criteria = { true, false, 0x0200, 24, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2785_Desc = "ps_2_x : texldd is allowed in a 24 level if_gt c0.x, c0.y nop else block";
string PS_2785 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y nop else "
	"	if_gt c0.x, c0.y nop else "
	"		if_gt c0.x, c0.y nop else "
	"			if_gt c0.x, c0.y nop else "
	"				if_gt c0.x, c0.y nop else "
	"					if_gt c0.x, c0.y nop else "
	"						if_gt c0.x, c0.y nop else "
	"							if_gt c0.x, c0.y nop else "
	"								if_gt c0.x, c0.y nop else "
	"									if_gt c0.x, c0.y nop else "
	"										if_gt c0.x, c0.y nop else "
	"											if_gt c0.x, c0.y nop else "
	"												if_gt c0.x, c0.y nop else "
	"													if_gt c0.x, c0.y nop else "
	"														if_gt c0.x, c0.y nop else "
	"															if_gt c0.x, c0.y nop else "
	"																if_gt c0.x, c0.y nop else "
	"																	if_gt c0.x, c0.y nop else "
	"																		if_gt c0.x, c0.y nop else "
	"																			if_gt c0.x, c0.y nop else "
	"																				if_gt c0.x, c0.y nop else "
	"																					if_gt c0.x, c0.y nop else "
	"																						if_gt c0.x, c0.y nop else "
	"																							if_gt c0.x, c0.y nop else "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2786_Criteria = { false, false, 0x0200, 25, -1, 0, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2786_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if_gt c0.x, c0.y nop else block";
string PS_2786 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if_gt c0.x, c0.y nop else "
	"	if_gt c0.x, c0.y nop else "
	"		if_gt c0.x, c0.y nop else "
	"			if_gt c0.x, c0.y nop else "
	"				if_gt c0.x, c0.y nop else "
	"					if_gt c0.x, c0.y nop else "
	"						if_gt c0.x, c0.y nop else "
	"							if_gt c0.x, c0.y nop else "
	"								if_gt c0.x, c0.y nop else "
	"									if_gt c0.x, c0.y nop else "
	"										if_gt c0.x, c0.y nop else "
	"											if_gt c0.x, c0.y nop else "
	"												if_gt c0.x, c0.y nop else "
	"													if_gt c0.x, c0.y nop else "
	"														if_gt c0.x, c0.y nop else "
	"															if_gt c0.x, c0.y nop else "
	"																if_gt c0.x, c0.y nop else "
	"																	if_gt c0.x, c0.y nop else "
	"																		if_gt c0.x, c0.y nop else "
	"																			if_gt c0.x, c0.y nop else "
	"																				if_gt c0.x, c0.y nop else "
	"																					if_gt c0.x, c0.y nop else "
	"																						if_gt c0.x, c0.y nop else "
	"																							if_gt c0.x, c0.y nop else "
	"																								if_gt c0.x, c0.y nop else "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2787_Criteria = { true, false, 0x0200, 1, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2787_Desc = "ps_2_x : texldd is allowed in a 1 level if p0.x block";
string PS_2787 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2788_Criteria = { true, false, 0x0200, 8, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2788_Desc = "ps_2_x : texldd is allowed in a 8 level if p0.x block";
string PS_2788 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x "
	"	if p0.x "
	"		if p0.x "
	"			if p0.x "
	"				if p0.x "
	"					if p0.x "
	"						if p0.x "
	"							if p0.x "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2789_Criteria = { true, false, 0x0200, 16, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2789_Desc = "ps_2_x : texldd is allowed in a 16 level if p0.x block";
string PS_2789 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x "
	"	if p0.x "
	"		if p0.x "
	"			if p0.x "
	"				if p0.x "
	"					if p0.x "
	"						if p0.x "
	"							if p0.x "
	"								if p0.x "
	"									if p0.x "
	"										if p0.x "
	"											if p0.x "
	"												if p0.x "
	"													if p0.x "
	"														if p0.x "
	"															if p0.x "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2790_Criteria = { true, false, 0x0200, 24, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2790_Desc = "ps_2_x : texldd is allowed in a 24 level if p0.x block";
string PS_2790 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x "
	"	if p0.x "
	"		if p0.x "
	"			if p0.x "
	"				if p0.x "
	"					if p0.x "
	"						if p0.x "
	"							if p0.x "
	"								if p0.x "
	"									if p0.x "
	"										if p0.x "
	"											if p0.x "
	"												if p0.x "
	"													if p0.x "
	"														if p0.x "
	"															if p0.x "
	"																if p0.x "
	"																	if p0.x "
	"																		if p0.x "
	"																			if p0.x "
	"																				if p0.x "
	"																					if p0.x "
	"																						if p0.x "
	"																							if p0.x "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2791_Criteria = { false, false, 0x0200, 25, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2791_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if p0.x block";
string PS_2791 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x "
	"	if p0.x "
	"		if p0.x "
	"			if p0.x "
	"				if p0.x "
	"					if p0.x "
	"						if p0.x "
	"							if p0.x "
	"								if p0.x "
	"									if p0.x "
	"										if p0.x "
	"											if p0.x "
	"												if p0.x "
	"													if p0.x "
	"														if p0.x "
	"															if p0.x "
	"																if p0.x "
	"																	if p0.x "
	"																		if p0.x "
	"																			if p0.x "
	"																				if p0.x "
	"																					if p0.x "
	"																						if p0.x "
	"																							if p0.x "
	"																								if p0.x "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2792_Criteria = { true, false, 0x0200, 1, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2792_Desc = "ps_2_x : texldd is allowed in a 1 level if p0.x nop else block";
string PS_2792 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x nop else "
	"	texldd r0, r0, s0, v0, r1 "
	"endif ";

PS_CRITERIA PS_2793_Criteria = { true, false, 0x0200, 8, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2793_Desc = "ps_2_x : texldd is allowed in a 8 level if p0.x nop else block";
string PS_2793 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x nop else "
	"	if p0.x nop else "
	"		if p0.x nop else "
	"			if p0.x nop else "
	"				if p0.x nop else "
	"					if p0.x nop else "
	"						if p0.x nop else "
	"							if p0.x nop else "
	"								texldd r0, r0, s0, v0, r1 "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2794_Criteria = { true, false, 0x0200, 16, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2794_Desc = "ps_2_x : texldd is allowed in a 16 level if p0.x nop else block";
string PS_2794 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x nop else "
	"	if p0.x nop else "
	"		if p0.x nop else "
	"			if p0.x nop else "
	"				if p0.x nop else "
	"					if p0.x nop else "
	"						if p0.x nop else "
	"							if p0.x nop else "
	"								if p0.x nop else "
	"									if p0.x nop else "
	"										if p0.x nop else "
	"											if p0.x nop else "
	"												if p0.x nop else "
	"													if p0.x nop else "
	"														if p0.x nop else "
	"															if p0.x nop else "
	"																texldd r0, r0, s0, v0, r1 "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2795_Criteria = { true, false, 0x0200, 24, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2795_Desc = "ps_2_x : texldd is allowed in a 24 level if p0.x nop else block";
string PS_2795 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x nop else "
	"	if p0.x nop else "
	"		if p0.x nop else "
	"			if p0.x nop else "
	"				if p0.x nop else "
	"					if p0.x nop else "
	"						if p0.x nop else "
	"							if p0.x nop else "
	"								if p0.x nop else "
	"									if p0.x nop else "
	"										if p0.x nop else "
	"											if p0.x nop else "
	"												if p0.x nop else "
	"													if p0.x nop else "
	"														if p0.x nop else "
	"															if p0.x nop else "
	"																if p0.x nop else "
	"																	if p0.x nop else "
	"																		if p0.x nop else "
	"																			if p0.x nop else "
	"																				if p0.x nop else "
	"																					if p0.x nop else "
	"																						if p0.x nop else "
	"																							if p0.x nop else "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2796_Criteria = { false, false, 0x0200, 25, -1, 0, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2796_Desc = "ps_2_x : texldd is NOT allowed in a 25 level if p0.x nop else block";
string PS_2796 = 
	"ps_2_x "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"if p0.x nop else "
	"	if p0.x nop else "
	"		if p0.x nop else "
	"			if p0.x nop else "
	"				if p0.x nop else "
	"					if p0.x nop else "
	"						if p0.x nop else "
	"							if p0.x nop else "
	"								if p0.x nop else "
	"									if p0.x nop else "
	"										if p0.x nop else "
	"											if p0.x nop else "
	"												if p0.x nop else "
	"													if p0.x nop else "
	"														if p0.x nop else "
	"															if p0.x nop else "
	"																if p0.x nop else "
	"																	if p0.x nop else "
	"																		if p0.x nop else "
	"																			if p0.x nop else "
	"																				if p0.x nop else "
	"																					if p0.x nop else "
	"																						if p0.x nop else "
	"																							if p0.x nop else "
	"																								if p0.x nop else "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endif "
	"																							endif "
	"																						endif "
	"																					endif "
	"																				endif "
	"																			endif "
	"																		endif "
	"																	endif "
	"																endif "
	"															endif "
	"														endif "
	"													endif "
	"												endif "
	"											endif "
	"										endif "
	"									endif "
	"								endif "
	"							endif "
	"						endif "
	"					endif "
	"				endif "
	"			endif "
	"		endif "
	"	endif "
	"endif ";

PS_CRITERIA PS_2797_Criteria = { false, false, 0x0200, 1, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2797_Desc = "ps_2_x : texldd is NOT allowed in a 1 level loop aL, i0 break block";
string PS_2797 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break "
	"	texldd r0, r0, s0, v0, r1 "
	"endloop ";

PS_CRITERIA PS_2798_Criteria = { false, false, 0x0200, 8, -1, 8, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2798_Desc = "ps_2_x : texldd is NOT allowed in a 8 level loop aL, i0 break block";
string PS_2798 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break "
	"	loop aL, i0 break "
	"		loop aL, i0 break "
	"			loop aL, i0 break "
	"				loop aL, i0 break "
	"					loop aL, i0 break "
	"						loop aL, i0 break "
	"							loop aL, i0 break "
	"								texldd r0, r0, s0, v0, r1 "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2799_Criteria = { false, false, 0x0200, 16, -1, 16, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2799_Desc = "ps_2_x : texldd is NOT allowed in a 16 level loop aL, i0 break block";
string PS_2799 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break "
	"	loop aL, i0 break "
	"		loop aL, i0 break "
	"			loop aL, i0 break "
	"				loop aL, i0 break "
	"					loop aL, i0 break "
	"						loop aL, i0 break "
	"							loop aL, i0 break "
	"								loop aL, i0 break "
	"									loop aL, i0 break "
	"										loop aL, i0 break "
	"											loop aL, i0 break "
	"												loop aL, i0 break "
	"													loop aL, i0 break "
	"														loop aL, i0 break "
	"															loop aL, i0 break "
	"																texldd r0, r0, s0, v0, r1 "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2800_Criteria = { false, false, 0x0200, 24, -1, 24, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2800_Desc = "ps_2_x : texldd is NOT allowed in a 24 level loop aL, i0 break block";
string PS_2800 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break "
	"	loop aL, i0 break "
	"		loop aL, i0 break "
	"			loop aL, i0 break "
	"				loop aL, i0 break "
	"					loop aL, i0 break "
	"						loop aL, i0 break "
	"							loop aL, i0 break "
	"								loop aL, i0 break "
	"									loop aL, i0 break "
	"										loop aL, i0 break "
	"											loop aL, i0 break "
	"												loop aL, i0 break "
	"													loop aL, i0 break "
	"														loop aL, i0 break "
	"															loop aL, i0 break "
	"																loop aL, i0 break "
	"																	loop aL, i0 break "
	"																		loop aL, i0 break "
	"																			loop aL, i0 break "
	"																				loop aL, i0 break "
	"																					loop aL, i0 break "
	"																						loop aL, i0 break "
	"																							loop aL, i0 break "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endloop "
	"																						endloop "
	"																					endloop "
	"																				endloop "
	"																			endloop "
	"																		endloop "
	"																	endloop "
	"																endloop "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2801_Criteria = { false, false, 0x0200, 25, -1, 25, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2801_Desc = "ps_2_x : texldd is NOT allowed in a 25 level loop aL, i0 break block";
string PS_2801 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break "
	"	loop aL, i0 break "
	"		loop aL, i0 break "
	"			loop aL, i0 break "
	"				loop aL, i0 break "
	"					loop aL, i0 break "
	"						loop aL, i0 break "
	"							loop aL, i0 break "
	"								loop aL, i0 break "
	"									loop aL, i0 break "
	"										loop aL, i0 break "
	"											loop aL, i0 break "
	"												loop aL, i0 break "
	"													loop aL, i0 break "
	"														loop aL, i0 break "
	"															loop aL, i0 break "
	"																loop aL, i0 break "
	"																	loop aL, i0 break "
	"																		loop aL, i0 break "
	"																			loop aL, i0 break "
	"																				loop aL, i0 break "
	"																					loop aL, i0 break "
	"																						loop aL, i0 break "
	"																							loop aL, i0 break "
	"																								loop aL, i0 break "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endloop "
	"																							endloop "
	"																						endloop "
	"																					endloop "
	"																				endloop "
	"																			endloop "
	"																		endloop "
	"																	endloop "
	"																endloop "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2802_Criteria = { true, false, 0x0200, 1, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2802_Desc = "ps_2_x : texldd is allowed in a 1 level rep i0 break block";
string PS_2802 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 break "
	"	texldd r0, r0, s0, v0, r1 "
	"endrep ";

PS_CRITERIA PS_2803_Criteria = { false, false, 0x0200, 8, -1, 8, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2803_Desc = "ps_2_x : texldd is NOT allowed in a 8 level rep i0 break block";
string PS_2803 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 break "
	"	rep i0 break "
	"		rep i0 break "
	"			rep i0 break "
	"				rep i0 break "
	"					rep i0 break "
	"						rep i0 break "
	"							rep i0 break "
	"								texldd r0, r0, s0, v0, r1 "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2804_Criteria = { false, false, 0x0200, 16, -1, 16, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2804_Desc = "ps_2_x : texldd is NOT allowed in a 16 level rep i0 break block";
string PS_2804 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 break "
	"	rep i0 break "
	"		rep i0 break "
	"			rep i0 break "
	"				rep i0 break "
	"					rep i0 break "
	"						rep i0 break "
	"							rep i0 break "
	"								rep i0 break "
	"									rep i0 break "
	"										rep i0 break "
	"											rep i0 break "
	"												rep i0 break "
	"													rep i0 break "
	"														rep i0 break "
	"															rep i0 break "
	"																texldd r0, r0, s0, v0, r1 "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2805_Criteria = { false, false, 0x0200, 24, -1, 24, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2805_Desc = "ps_2_x : texldd is NOT allowed in a 24 level rep i0 break block";
string PS_2805 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 break "
	"	rep i0 break "
	"		rep i0 break "
	"			rep i0 break "
	"				rep i0 break "
	"					rep i0 break "
	"						rep i0 break "
	"							rep i0 break "
	"								rep i0 break "
	"									rep i0 break "
	"										rep i0 break "
	"											rep i0 break "
	"												rep i0 break "
	"													rep i0 break "
	"														rep i0 break "
	"															rep i0 break "
	"																rep i0 break "
	"																	rep i0 break "
	"																		rep i0 break "
	"																			rep i0 break "
	"																				rep i0 break "
	"																					rep i0 break "
	"																						rep i0 break "
	"																							rep i0 break "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endrep "
	"																						endrep "
	"																					endrep "
	"																				endrep "
	"																			endrep "
	"																		endrep "
	"																	endrep "
	"																endrep "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2806_Criteria = { false, false, 0x0200, 25, -1, 25, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2806_Desc = "ps_2_x : texldd is NOT allowed in a 25 level rep i0 break block";
string PS_2806 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 break "
	"	rep i0 break "
	"		rep i0 break "
	"			rep i0 break "
	"				rep i0 break "
	"					rep i0 break "
	"						rep i0 break "
	"							rep i0 break "
	"								rep i0 break "
	"									rep i0 break "
	"										rep i0 break "
	"											rep i0 break "
	"												rep i0 break "
	"													rep i0 break "
	"														rep i0 break "
	"															rep i0 break "
	"																rep i0 break "
	"																	rep i0 break "
	"																		rep i0 break "
	"																			rep i0 break "
	"																				rep i0 break "
	"																					rep i0 break "
	"																						rep i0 break "
	"																							rep i0 break "
	"																								rep i0 break "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endrep "
	"																							endrep "
	"																						endrep "
	"																					endrep "
	"																				endrep "
	"																			endrep "
	"																		endrep "
	"																	endrep "
	"																endrep "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2807_Criteria = { false, false, 0x0200, 0, -1, 1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2807_Desc = "ps_2_x : texldd is NOT allowed in a 1 level loop aL, i0 breakp p0.x block";
string PS_2807 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 breakp p0.x "
	"	texldd r0, r0, s0, v0, r1 "
	"endloop ";

PS_CRITERIA PS_2808_Criteria = { false, false, 0x0200, 0, -1, 8, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2808_Desc = "ps_2_x : texldd is NOT allowed in a 8 level loop aL, i0 breakp p0.x block";
string PS_2808 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 breakp p0.x "
	"	loop aL, i0 breakp p0.x "
	"		loop aL, i0 breakp p0.x "
	"			loop aL, i0 breakp p0.x "
	"				loop aL, i0 breakp p0.x "
	"					loop aL, i0 breakp p0.x "
	"						loop aL, i0 breakp p0.x "
	"							loop aL, i0 breakp p0.x "
	"								texldd r0, r0, s0, v0, r1 "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2809_Criteria = { false, false, 0x0200, 0, -1, 16, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2809_Desc = "ps_2_x : texldd is NOT allowed in a 16 level loop aL, i0 breakp p0.x block";
string PS_2809 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 breakp p0.x "
	"	loop aL, i0 breakp p0.x "
	"		loop aL, i0 breakp p0.x "
	"			loop aL, i0 breakp p0.x "
	"				loop aL, i0 breakp p0.x "
	"					loop aL, i0 breakp p0.x "
	"						loop aL, i0 breakp p0.x "
	"							loop aL, i0 breakp p0.x "
	"								loop aL, i0 breakp p0.x "
	"									loop aL, i0 breakp p0.x "
	"										loop aL, i0 breakp p0.x "
	"											loop aL, i0 breakp p0.x "
	"												loop aL, i0 breakp p0.x "
	"													loop aL, i0 breakp p0.x "
	"														loop aL, i0 breakp p0.x "
	"															loop aL, i0 breakp p0.x "
	"																texldd r0, r0, s0, v0, r1 "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2810_Criteria = { false, false, 0x0200, 0, -1, 24, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2810_Desc = "ps_2_x : texldd is NOT allowed in a 24 level loop aL, i0 breakp p0.x block";
string PS_2810 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 breakp p0.x "
	"	loop aL, i0 breakp p0.x "
	"		loop aL, i0 breakp p0.x "
	"			loop aL, i0 breakp p0.x "
	"				loop aL, i0 breakp p0.x "
	"					loop aL, i0 breakp p0.x "
	"						loop aL, i0 breakp p0.x "
	"							loop aL, i0 breakp p0.x "
	"								loop aL, i0 breakp p0.x "
	"									loop aL, i0 breakp p0.x "
	"										loop aL, i0 breakp p0.x "
	"											loop aL, i0 breakp p0.x "
	"												loop aL, i0 breakp p0.x "
	"													loop aL, i0 breakp p0.x "
	"														loop aL, i0 breakp p0.x "
	"															loop aL, i0 breakp p0.x "
	"																loop aL, i0 breakp p0.x "
	"																	loop aL, i0 breakp p0.x "
	"																		loop aL, i0 breakp p0.x "
	"																			loop aL, i0 breakp p0.x "
	"																				loop aL, i0 breakp p0.x "
	"																					loop aL, i0 breakp p0.x "
	"																						loop aL, i0 breakp p0.x "
	"																							loop aL, i0 breakp p0.x "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endloop "
	"																						endloop "
	"																					endloop "
	"																				endloop "
	"																			endloop "
	"																		endloop "
	"																	endloop "
	"																endloop "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2811_Criteria = { false, false, 0x0200, 0, -1, 25, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2811_Desc = "ps_2_x : texldd is NOT allowed in a 25 level loop aL, i0 breakp p0.x block";
string PS_2811 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 breakp p0.x "
	"	loop aL, i0 breakp p0.x "
	"		loop aL, i0 breakp p0.x "
	"			loop aL, i0 breakp p0.x "
	"				loop aL, i0 breakp p0.x "
	"					loop aL, i0 breakp p0.x "
	"						loop aL, i0 breakp p0.x "
	"							loop aL, i0 breakp p0.x "
	"								loop aL, i0 breakp p0.x "
	"									loop aL, i0 breakp p0.x "
	"										loop aL, i0 breakp p0.x "
	"											loop aL, i0 breakp p0.x "
	"												loop aL, i0 breakp p0.x "
	"													loop aL, i0 breakp p0.x "
	"														loop aL, i0 breakp p0.x "
	"															loop aL, i0 breakp p0.x "
	"																loop aL, i0 breakp p0.x "
	"																	loop aL, i0 breakp p0.x "
	"																		loop aL, i0 breakp p0.x "
	"																			loop aL, i0 breakp p0.x "
	"																				loop aL, i0 breakp p0.x "
	"																					loop aL, i0 breakp p0.x "
	"																						loop aL, i0 breakp p0.x "
	"																							loop aL, i0 breakp p0.x "
	"																								loop aL, i0 breakp p0.x "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endloop "
	"																							endloop "
	"																						endloop "
	"																					endloop "
	"																				endloop "
	"																			endloop "
	"																		endloop "
	"																	endloop "
	"																endloop "
	"															endloop "
	"														endloop "
	"													endloop "
	"												endloop "
	"											endloop "
	"										endloop "
	"									endloop "
	"								endloop "
	"							endloop "
	"						endloop "
	"					endloop "
	"				endloop "
	"			endloop "
	"		endloop "
	"	endloop "
	"endloop ";

PS_CRITERIA PS_2812_Criteria = { true, false, 0x0200, 0, -1, 1, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2812_Desc = "ps_2_x : texldd is allowed in a 1 level rep i0 breakp p0.x block";
string PS_2812 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 breakp p0.x "
	"	texldd r0, r0, s0, v0, r1 "
	"endrep ";

PS_CRITERIA PS_2813_Criteria = { false, false, 0x0200, 0, -1, 8, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2813_Desc = "ps_2_x : texldd is NOT allowed in a 8 level rep i0 breakp p0.x block";
string PS_2813 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 breakp p0.x "
	"	rep i0 breakp p0.x "
	"		rep i0 breakp p0.x "
	"			rep i0 breakp p0.x "
	"				rep i0 breakp p0.x "
	"					rep i0 breakp p0.x "
	"						rep i0 breakp p0.x "
	"							rep i0 breakp p0.x "
	"								texldd r0, r0, s0, v0, r1 "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2814_Criteria = { false, false, 0x0200, 0, -1, 16, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2814_Desc = "ps_2_x : texldd is NOT allowed in a 16 level rep i0 breakp p0.x block";
string PS_2814 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 breakp p0.x "
	"	rep i0 breakp p0.x "
	"		rep i0 breakp p0.x "
	"			rep i0 breakp p0.x "
	"				rep i0 breakp p0.x "
	"					rep i0 breakp p0.x "
	"						rep i0 breakp p0.x "
	"							rep i0 breakp p0.x "
	"								rep i0 breakp p0.x "
	"									rep i0 breakp p0.x "
	"										rep i0 breakp p0.x "
	"											rep i0 breakp p0.x "
	"												rep i0 breakp p0.x "
	"													rep i0 breakp p0.x "
	"														rep i0 breakp p0.x "
	"															rep i0 breakp p0.x "
	"																texldd r0, r0, s0, v0, r1 "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2815_Criteria = { false, false, 0x0200, 0, -1, 24, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2815_Desc = "ps_2_x : texldd is NOT allowed in a 24 level rep i0 breakp p0.x block";
string PS_2815 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 breakp p0.x "
	"	rep i0 breakp p0.x "
	"		rep i0 breakp p0.x "
	"			rep i0 breakp p0.x "
	"				rep i0 breakp p0.x "
	"					rep i0 breakp p0.x "
	"						rep i0 breakp p0.x "
	"							rep i0 breakp p0.x "
	"								rep i0 breakp p0.x "
	"									rep i0 breakp p0.x "
	"										rep i0 breakp p0.x "
	"											rep i0 breakp p0.x "
	"												rep i0 breakp p0.x "
	"													rep i0 breakp p0.x "
	"														rep i0 breakp p0.x "
	"															rep i0 breakp p0.x "
	"																rep i0 breakp p0.x "
	"																	rep i0 breakp p0.x "
	"																		rep i0 breakp p0.x "
	"																			rep i0 breakp p0.x "
	"																				rep i0 breakp p0.x "
	"																					rep i0 breakp p0.x "
	"																						rep i0 breakp p0.x "
	"																							rep i0 breakp p0.x "
	"																								texldd r0, r0, s0, v0, r1 "
	"																							endrep "
	"																						endrep "
	"																					endrep "
	"																				endrep "
	"																			endrep "
	"																		endrep "
	"																	endrep "
	"																endrep "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2816_Criteria = { false, false, 0x0200, 0, -1, 25, -1, CAPS_PREDICATION + CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2816_Desc = "ps_2_x : texldd is NOT allowed in a 25 level rep i0 breakp p0.x block";
string PS_2816 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"rep i0 breakp p0.x "
	"	rep i0 breakp p0.x "
	"		rep i0 breakp p0.x "
	"			rep i0 breakp p0.x "
	"				rep i0 breakp p0.x "
	"					rep i0 breakp p0.x "
	"						rep i0 breakp p0.x "
	"							rep i0 breakp p0.x "
	"								rep i0 breakp p0.x "
	"									rep i0 breakp p0.x "
	"										rep i0 breakp p0.x "
	"											rep i0 breakp p0.x "
	"												rep i0 breakp p0.x "
	"													rep i0 breakp p0.x "
	"														rep i0 breakp p0.x "
	"															rep i0 breakp p0.x "
	"																rep i0 breakp p0.x "
	"																	rep i0 breakp p0.x "
	"																		rep i0 breakp p0.x "
	"																			rep i0 breakp p0.x "
	"																				rep i0 breakp p0.x "
	"																					rep i0 breakp p0.x "
	"																						rep i0 breakp p0.x "
	"																							rep i0 breakp p0.x "
	"																								rep i0 breakp p0.x "
	"																									texldd r0, r0, s0, v0, r1 "
	"																								endrep "
	"																							endrep "
	"																						endrep "
	"																					endrep "
	"																				endrep "
	"																			endrep "
	"																		endrep "
	"																	endrep "
	"																endrep "
	"															endrep "
	"														endrep "
	"													endrep "
	"												endrep "
	"											endrep "
	"										endrep "
	"									endrep "
	"								endrep "
	"							endrep "
	"						endrep "
	"					endrep "
	"				endrep "
	"			endrep "
	"		endrep "
	"	endrep "
	"endrep ";

PS_CRITERIA PS_2817_Criteria = { false, false, 0x0200, 1, -1, 1, -1, CAPS_GRADIENTINSTRUCTIONS, -1, 0.f };
string PS_2817_Desc = "ps_2_x : texldd is NOT allowed in a 1 level loop aL, i0 break_gt c0.x, c0.y block";
string PS_2817 = 
	"ps_2_x "
	"defi i0, 0, 0, 0, 0 "
	"dcl_2d s0 "
	"dcl v0 "
	"mov r0, c0 "
	"mov r1, c0 "
	"loop aL, i0 break