/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    clientinvk.c

Abstract:

    This file contains routines used by the class extension to invoke client
    driver supplied interfaces.


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"

#if defined(EVENT_TRACING)
#include "clientinvk.tmh"         // auto-generated by WPP
#endif

#include "clientinvk.h"

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(PAGE, GpioClnInvokePrepareController)
#pragma alloc_text(PAGE, GpioClnInvokeQueryControllerBasicInformation)
#pragma alloc_text(PAGE, GpioClnInvokeEnableInterrupt)
#pragma alloc_text(PAGE, GpioClnInvokeControllerSpecificFunctionHandler)
#pragma alloc_text(PAGE, GpioClnInvokeQueryControllerFunctionBankMapping)
#pragma alloc_text(PAGE, GpioClnInvokeControllerCleanupCallback)
#pragma alloc_text(PAGE, GpioClnInvokeConnectFunctionConfigPins)
#pragma alloc_text(PAGE, GpioClnInvokeDisconnectFunctionConfigPins)

//
// ------------------------------------------------------------------ Functions
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokePrepareController (
    __in WDFDEVICE Device,
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesRaw,
    __in WDFCMRESLIST ResourcesTranslated
    )

/*++

Routine Description:

    This routine invokes the client driver's device initialization callback
    supplied at registration time. This is called when the driver framework
    requests the GPIO class extension to prepare the hardware device (to be
    started).

Arguments:

    Device - Supplies a pointer to the framework device object.

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    NTSTATUS Status;

    PAGED_CODE();

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT(ClientPacket->CLIENT_PrepareController != NULL);

    Status = ClientPacket->CLIENT_PrepareController(Device,
                                                    Extension,
                                                    ResourcesRaw,
                                                    ResourcesTranslated);

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeReleaseController (
    __in WDFDEVICE Device,
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to uninitialize the
    controller. This is called when the driver framework requests the GPIO class
    extension to release the hardware device (prior to it being stopped).

Arguments:

    Device - Supplies a pointer to the framework device object.

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    NTSTATUS Status;

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT(ClientPacket->CLIENT_ReleaseController != NULL);

    Status = ClientPacket->CLIENT_ReleaseController(Device, Extension);
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeStartController (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN RestoreContext,
    __in WDF_POWER_DEVICE_STATE PreviousPowerState
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to start the controller.
    This is called when the driver framework requests the GPIO class extension
    to start the device.

    N.B. This function is not marked pageable because this function is in
         the device power up path. When a function is marked pagable and the
         code section is paged out, it will generate a page fault which could
         impact the fast resume behavior because the client driver will have
         to wait until the system drivers can service this page fault.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    RestoreContext - Supplies a flag indicating whether the client driver
        is supposed to restore previously saved state or not.

    PreviousPowerState - Supplies the device power state that the device was in
        before this transition to D0.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    NTSTATUS Status;

    GPIO_ASSERT(KeGetCurrentIrql() < DISPATCH_LEVEL);

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT(ClientPacket->CLIENT_StartController != NULL);

    Status = ClientPacket->CLIENT_StartController(Extension,
                                                  RestoreContext,
                                                  PreviousPowerState);

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeStopController (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN SaveContext,
    __in WDF_POWER_DEVICE_STATE TargetState
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to stop the controller.
    This is called when the driver framework requests the GPIO class extension
    to stop the device.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    SaveContext - Supplies a flag indicating whether the client driver
        is supposed to save current state or not.

    TargetState - Supplies the device power state which the device will be put
        in once the callback is complete.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    NTSTATUS Status;

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT(ClientPacket->CLIENT_StopController != NULL);

    Status = ClientPacket->CLIENT_StopController(Extension,
                                                 SaveContext,
                                                 TargetState);

    return Status;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeQueryBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank,
    __out PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT PowerInformation
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to query the GPIO
    controller's bank information.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    BankInformation - Supplies a pointer to a buffer that receives the
        bank information provided by the client driver.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT Input;
    USHORT InputSize;
    PGPIO_CLIENT_QUERY_SET_CONTROLLER_INFORMATION QuerySetControllerInformation;
    NTSTATUS Status;

    PAGED_CODE();

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    QuerySetControllerInformation =
        ClientPacket->CLIENT_QuerySetControllerInformation;

    //
    // The query bank information callback is optional. If a callback is not
    // supplied, return default information. Note by default a bank is
    // assumed to be not power managed.
    //

    Status = STATUS_NOT_IMPLEMENTED;
    if (QuerySetControllerInformation != NULL) {
        InputSize = sizeof(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT);
        RtlZeroMemory(&Input, sizeof(Input));
        Input.RequestType = QueryBankPowerInformation;
        Input.Size = InputSize;
        Input.BankPowerInformation.BankId = GpioBank->BankId;
        Status = QuerySetControllerInformation(Extension,
                                               &Input,
                                               PowerInformation);
    }

    //
    // If the query routine is not supplied or the query routine failed with
    // not_implemented or not_supported, then assume the bank does not
    // support any power management.
    //

    if ((Status == STATUS_NOT_IMPLEMENTED) ||
        (Status == STATUS_NOT_SUPPORTED)) {

        //
        // N.B. Output buffer size if already correct.
        //

        PowerInformation->Version = GPIO_BANK_POWER_INFORMATION_OUTPUT_VERSION;
        PowerInformation->BankPowerInformation.F1StateSupported = FALSE;
        Status = STATUS_SUCCESS;
    }

    return Status;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeQueryInterruptBinding (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw,
    __in ULONG InterruptCount,
    __out_xcount(BindingOutput->Size)
        PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT BindingOutput
   )

/*++

Routine Description:

    This routine invokes the client driver's callback to query how the
    GPIO controller's interrupt resources map to banks.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

    BindingOutput - Supplies a buffer that receives the mappings filled
        by the client driver.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankIndex;
    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR Descriptor;
    PVOID Extension;
    ULONG Index;
    CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT Input;
    USHORT InputSize;
    PGPIO_CLIENT_QUERY_SET_CONTROLLER_INFORMATION QuerySetControllerInformation;
    PULONG ResourceMapping;
    NTSTATUS Status;
    ULONG TotalCount;

    PAGED_CODE();

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    QuerySetControllerInformation =
        ClientPacket->CLIENT_QuerySetControllerInformation;

    //
    // The query bank interrupt callback is optional.
    //

    Status = STATUS_NOT_IMPLEMENTED;
    if (QuerySetControllerInformation != NULL) {
        InputSize = sizeof(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT);
        RtlZeroMemory(&Input, InputSize);
        Input.RequestType = QueryBankInterruptBindingInformation;
        Input.Size = InputSize;
        Input.BankInterruptBinding.ResourcesTranslated = ResourcesTranslated;
        Input.BankInterruptBinding.ResourcesRaw = ResourcesRaw;
        Input.BankInterruptBinding.TotalBanks = GpioExtension->TotalBanks;
        Status = QuerySetControllerInformation(Extension,
                                               &Input,
                                               BindingOutput);
    }

    //
    // If the callback is not implemented or the specific query is not
    // supported, then assume the interrupts are described in the order of the
    // banks provided interrupt count is greater than or equal to banks.
    //

    if (((Status == STATUS_NOT_IMPLEMENTED) ||
         (Status == STATUS_NOT_SUPPORTED)) &&
        (InterruptCount >= GpioExtension->TotalBanks)) {

        //
        // Iterate over the resource list until interrupt resource for
        // each bank is found.
        //
        // N.B. Output buffer size if already correct.
        //

        BankIndex = 0;
        BindingOutput->Version =
            GPIO_BANK_INTERRUPT_BINDING_INFORMATION_OUTPUT_VERSION;

        ResourceMapping =
            (PULONG)&BindingOutput->BankInterruptBinding.ResourceMapping;

        TotalCount = WdfCmResourceListGetCount(ResourcesTranslated);
        for (Index = 0;
             BankIndex < GpioExtension->TotalBanks; // N.B. check for bank count
             Index += 1) {

            GPIO_ASSERT(Index < TotalCount);

            Descriptor = WdfCmResourceListGetDescriptor(ResourcesTranslated,
                                                        Index);

            if (Descriptor->Type == CmResourceTypeInterrupt) {
                ResourceMapping[BankIndex] = Index;
                BankIndex += 1;
            }
        }

        Status = STATUS_SUCCESS;
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeQueryControllerBasicInformation (
    __in PDEVICE_EXTENSION GpioExtension,
    __out PCLIENT_CONTROLLER_BASIC_INFORMATION ControllerInformation
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to query the GPIO
    controller's attributes.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ControllerInformation - Supplies a pointer to a buffer that receives the
        information provided by the controller.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PGPIO_CLIENT_QUERY_CONTROLLER_BASIC_INFORMATION QueryBasicInformation;
    NTSTATUS Status;

    PAGED_CODE();

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    QueryBasicInformation =
        ClientPacket->CLIENT_QueryControllerBasicInformation;

    GPIO_ASSERT(QueryBasicInformation != NULL);

    Status = QueryBasicInformation(Extension, ControllerInformation);
    return Status;
}

//
// ------------------------------------------------------- Interrupt interfaces
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeEnableInterrupt (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PIN_NUMBER RelativePin,
    __in __drv_strictTypeMatch(__drv_typeConst) KINTERRUPT_MODE InterruptMode,
    __in __drv_strictTypeMatch(__drv_typeConst) KINTERRUPT_POLARITY Polarity,
    __in_opt PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR Descriptor
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to enable the pin
    associated with the supplied bank for interrupt.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) that should be
        enabled for interrupt.

    InterruptMode - Supplies the trigger mode (edge or level) associated with
        this interrupt.

    Polarity - Supplies the polarity (active low or active high) associated with
        this interrupt.

    Descriptor - Supplies a pointer to the BIOS GPIO descriptor. This value is
        only present if the descriptor conforms to the new GPIO descriptor
        format.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_ENABLE_INTERRUPT_PARAMETERS Parameters;
    NTSTATUS Status;

    PAGED_CODE();

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    if (ClientPacket->CLIENT_EnableInterrupt == NULL) {
        GPIO_ASSERT(FALSE);

        Status = STATUS_NOT_SUPPORTED;
        goto InvokeEnableInterruptEnd;
    }

    GPIO_ASSERT(RelativePin < GpioBank->PinCount);

    RtlZeroMemory(&Parameters, sizeof(GPIO_ENABLE_INTERRUPT_PARAMETERS));
    Parameters.BankId = GpioBank->BankId;
    Parameters.PinNumber = RelativePin;
    Parameters.InterruptMode = InterruptMode;
    Parameters.Polarity = Polarity;
    if (ARGUMENT_PRESENT(Descriptor) != FALSE) {

        //
        // If deboucing is emulated in GPIO class extension, then do not
        // supply the debounce timeout to the client driver.
        //

        if (GPIO_IS_DEBOUNCING_EMULATED(GpioExtension) == FALSE) {
            Parameters.DebounceTimeout = Descriptor->DebounceTimeout;
        }

        Parameters.PullConfiguration =
            GPIO_BIOS_DESCRIPTOR_PULL_CONFIGURATION(Descriptor);

        if (Descriptor->VendorDataLength > 0) {
            Parameters.VendorData =
                Add2Ptr(Descriptor, Descriptor->VendorDataOffset);

            Parameters.VendorDataLength = Descriptor->VendorDataLength;
        }
    }

    //
    // N.B. If this is a memory-mapped controller, the client driver will
    //      acquire the interrupt lock.
    //

    EventWrite_ENABLE_INTERRUPT_START(&GpioBank->ActivityId,
                                      GpioExtension->BiosName.Buffer,
                                      GpioBank->BankId,
                                      RelativePin);

    Status = ClientPacket->CLIENT_EnableInterrupt(Extension, &Parameters);
    EventWrite_ENABLE_INTERRUPT_COMPLETE(&GpioBank->ActivityId, RelativePin);

InvokeEnableInterruptEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeDisableInterrupt (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PIN_NUMBER RelativePin,
    __in BOOLEAN RetryAttempt
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to disable the pin
    associated with the supplied VIRQ from interrupting.

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) that should be
        disabled for interrupt.

    RetryAttempt - Supplies a flag indicating whether this is a first disable
        attempt on this pin or a retry attempt (due to a previous failure).

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PDEVICE_EXTENSION GpioExtension;
    PVOID Extension;
    GPIO_DISABLE_INTERRUPT_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT((ClientPacket->CLIENT_DisableInterrupt != NULL) &&
                (RelativePin < GpioBank->PinCount));

    RtlZeroMemory(&Parameters, sizeof(GPIO_DISABLE_INTERRUPT_PARAMETERS));
    Parameters.BankId = GpioBank->BankId;
    Parameters.PinNumber = RelativePin;
    if (RetryAttempt != FALSE) {
        Parameters.Flags.RetryDisableOnFailure = 1;
    }

    //
    // N.B. If this is a memory-mapped controller, the client driver will
    //      acquire the interrupt lock.
    //

    EventWrite_DISABLE_INTERRUPT_START(&GpioBank->ActivityId,
                                       GpioExtension->BiosName.Buffer,
                                       GpioBank->BankId,
                                       RelativePin);

    Status = ClientPacket->CLIENT_DisableInterrupt(Extension, &Parameters);
    EventWrite_DISABLE_INTERRUPT_COMPLETE(&GpioBank->ActivityId, RelativePin);

    //
    // On-Soc GPIO disable interrupt should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeQueryEnabledInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank,
    __out PULONG64 EnabledMask
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to read the controller's
    current interrupt enable register value for the specified bank.

    N.B. Caller requirements:

           Off-SoC: The bank lock must be held (i.e. PASSIVE_LEVEL)
           On-SoC: The interrupt lock must be held (i.e. >= DISPATCH_LEVEL)

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    EnabledMask - Supplies a bitmask that receives the enabled interrupt mask
        of the specified bank.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_INTERRUPT_DATA InterruptData;
    GPIO_QUERY_ENABLED_INTERRUPTS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    InterruptData = &GpioBank->InterruptData;

    GPIO_ASSERT((ClientPacket->CLIENT_QueryEnabledInterrupts != NULL) &&
                (GpioBank->BankId < GpioExtension->TotalBanks) &&
                (KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql));

    Parameters.BankId = GpioBank->BankId;
    Parameters.EnabledMask = 0;

    //
    // N.B. This does not race with CLIENT_EnableInterrupt and
    //      CLIENT_DisableInterrupt:
    //
    //        Off-SoC:
    //          CLIENT_QueryEnabledInterrupts and those 2 functions are all
    //          called with the bank lock held.
    //
    //        On-SoC:
    //          The client driver must implement those 2 functions to acquire
    //          the interrupt lock.
    //

    EventWrite_QUERY_ENABLED_INTERRUPTS_START(&GpioBank->ActivityId,
                                              GpioExtension->BiosName.Buffer,
                                              GpioBank->BankId);

    Status = ClientPacket->CLIENT_QueryEnabledInterrupts(Extension, &Parameters);
    EventWrite_QUERY_ENABLED_INTERRUPTS_COMPLETE(&GpioBank->ActivityId);

    //
    // On-Soc GPIO query enabled interrupts should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    *EnabledMask = Parameters.EnabledMask;
    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeMaskInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in ULONG64 InterruptMask,
    __out PULONG64 FailedMask
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to mask some interrupt
    pin preventing them from interrupting the CPU.

    N.B. This routine can be called at DIRQL for memory-mapped controllers.

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    InterruptMask - Supplies a bitmap of pins to be masked. A bit is set if a
        pin is to be masked.

    FailedMask - Supplies a bitmap of pins that could not be masked. A bit is
        set if a pin couldn't be masked.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_INTERRUPT_DATA InterruptData;
    GPIO_MASK_INTERRUPT_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    InterruptData = &GpioBank->InterruptData;

    GPIO_ASSERT((ClientPacket->CLIENT_MaskInterrupts != NULL) &&
                (InterruptMask != 0x0) &&
                (KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql));

    Parameters.BankId = GpioBank->BankId;
    Parameters.PinMask = InterruptMask;
    Parameters.FailedMask = 0x0;
    EventWrite_MASK_INTERRUPTS_START(&GpioBank->ActivityId,
                                     GpioExtension->BiosName.Buffer,
                                     GpioBank->BankId,
                                     InterruptMask);

    Status = ClientPacket->CLIENT_MaskInterrupts(Extension, &Parameters);
    EventWrite_MASK_INTERRUPTS_COMPLETE(&GpioBank->ActivityId, InterruptMask);
    *FailedMask = Parameters.FailedMask;

    //
    // On-Soc GPIO mask operation should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    return Status;
}

NTSTATUS
GpioClnInvokeUnmaskInterrupt (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PIN_NUMBER RelativePin,
    __in __drv_strictTypeMatch(__drv_typeConst) KINTERRUPT_MODE InterruptMode,
    __in __drv_strictTypeMatch(__drv_typeConst) KINTERRUPT_POLARITY Polarity,
    __in UCHAR PullConfiguration,
    __in USHORT DebounceTimeout
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to unmask the pin
    associated with the supplied VIRQ thereby re-enabling it to interrupt.

    N.B. This routine can be called at DIRQL for memory-mapped controllers.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) that should be
        disabled for interrupt.

    InterruptMode - Supplies the trigger mode (edge or level) configured for
        this interrupt when it was enabled.

    Polarity - Supplies the polarity (active low or active high) configured for
        this interrupt when it was enabled.

    PullConfiguration - Supplies the pin pull configuration (pull-up, pull-down,
        etc.) configured for this interrupt when it was enabled.

    DebounceTimeout - Supplies the debounce timeout configured for this
        interrupt when it was enabled.


Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_ENABLE_INTERRUPT_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT((ClientPacket->CLIENT_UnmaskInterrupt != NULL) &&
                (RelativePin < GpioBank->PinCount) &&
                (KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql));

    RtlZeroMemory(&Parameters, sizeof(GPIO_ENABLE_INTERRUPT_PARAMETERS));
    Parameters.BankId = GpioBank->BankId;
    Parameters.PinNumber = RelativePin;
    Parameters.InterruptMode = InterruptMode;
    Parameters.Polarity = Polarity;
    Parameters.DebounceTimeout = DebounceTimeout;
    Parameters.PullConfiguration = PullConfiguration;
    EventWrite_UNMASK_INTERRUPT_START(&GpioBank->ActivityId,
                                      GpioExtension->BiosName.Buffer,
                                      GpioBank->BankId,
                                      RelativePin);

    Status = ClientPacket->CLIENT_UnmaskInterrupt(Extension, &Parameters);
    EventWrite_UNMASK_INTERRUPT_COMPLETE(&GpioBank->ActivityId, RelativePin);

    //
    // On-Soc GPIO unmask operation should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeQueryActiveInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to read the controller's
    current interrupt status register value for the specified bank.

    N.B. 1. This routine can be called at DIRQL (if device is not passive-only).
         2. This routine updates the StatusRegister field.

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_INTERRUPT_DATA InterruptData;
    GPIO_QUERY_ACTIVE_INTERRUPTS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    InterruptData = &GpioBank->InterruptData;

    GPIO_ASSERT((ClientPacket->CLIENT_QueryActiveInterrupts != NULL) &&
                (GpioBank->BankId < GpioExtension->TotalBanks) &&
                (KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql));

    Parameters.BankId = GpioBank->BankId;
    Parameters.EnabledMask = InterruptData->EnableRegister;
    Parameters.ActiveMask = 0;
    EventWrite_QUERY_ACTIVE_INTERRUPTS_START(&GpioBank->ActivityId,
                                             GpioExtension->BiosName.Buffer,
                                             GpioBank->BankId);

    Status = ClientPacket->CLIENT_QueryActiveInterrupts(Extension, &Parameters);
    EventWrite_QUERY_ACTIVE_INTERRUPTS_COMPLETE(&GpioBank->ActivityId);

    //
    // On-Soc GPIO query active interrupts should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    InterruptData->StatusRegister = Parameters.ActiveMask;
    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeClearActiveInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in ULONG64 ClearMask,
    __out PULONG64 FailedClearMask
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to clear the controller's
    interrupt status register bit for the specified pin.

    N.B. 1. This routine can be called at DIRQL (if device is not passive-only).
         2. This routine assumes the ClearMask field has been initialized with
            pins to be cleared.

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_INTERRUPT_DATA InterruptData;
    GPIO_CLEAR_ACTIVE_INTERRUPTS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    InterruptData = &GpioBank->InterruptData;

    GPIO_ASSERT((ClientPacket->CLIENT_ClearActiveInterrupts != NULL) &&
                (GpioExtension->ClientInformation.
                    Flags.ActiveInterruptsAutoClearOnRead == FALSE) &&
                (GpioBank->BankId < GpioExtension->TotalBanks) &&
                (ClearMask != 0x0) &&
                (KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql));

    Parameters.BankId = GpioBank->BankId;
    Parameters.ClearActiveMask = ClearMask;
    Parameters.FailedClearMask = 0;
    EventWrite_CLEAR_ACTIVE_INTERRUPTS_START(
        &GpioBank->ActivityId,
        GpioExtension->BiosName.Buffer,
        GpioBank->BankId,
        ClearMask);

    Status = ClientPacket->CLIENT_ClearActiveInterrupts(Extension, &Parameters);
    EventWrite_CLEAR_ACTIVE_INTERRUPTS_COMPLETE(&GpioBank->ActivityId, ClearMask);

    //
    // On-Soc GPIO clear active interrupts should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     INTERRUPT_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     (ULONG_PTR)&Parameters);
    }

    *FailedClearMask = Parameters.FailedClearMask;
    return Status;
}

NTSTATUS
GpioClnInvokeReconfigureInterrupt (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PIN_NUMBER PinNumber,
    __in KINTERRUPT_MODE InterruptMode,
    __in KINTERRUPT_POLARITY Polarity,
    __in BOOLEAN TolerateFailure
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to reconfigure the
    specified interrupt pin's mode and polarity.

    N.B. This routine can be called at DIRQL (if device is memory-mapped).

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    PinNumber - Supplies the bank-relative pin number to be reconfigured.

    InterruptMode - Supplies the new mode for the pin.

    Polarity - Supplies the new polarity for the pin.

    TolerateFailure - Supplies whether a reconfigure failure should be tolerated
        or not on this call. This flag is only applicable to memory-mapped
        GPIO controllers. It is always tolerated for off-SoC GPIO controllers.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_RECONFIGURE_INTERRUPTS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    GPIO_ASSERT(KeGetCurrentIrql() <= GpioExtension->SynchronizationIrql);

    if (ClientPacket->CLIENT_ReconfigureInterrupt != NULL) {
        Parameters.BankId = GpioBank->BankId;
        Parameters.PinNumber = PinNumber;
        Parameters.InterruptMode = InterruptMode;
        Parameters.Polarity = Polarity;
        Parameters.Flags.AsULONG = 0x0;
        EventWrite_RECONFIGURE_INTERRUPT_START(&GpioBank->ActivityId,
                                               GpioExtension->BiosName.Buffer,
                                               GpioBank->BankId,
                                               PinNumber);

        Status = ClientPacket->CLIENT_ReconfigureInterrupt(Extension,
                                                           &Parameters);

        EventWrite_RECONFIGURE_INTERRUPT_COMPLETE(&GpioBank->ActivityId,
                                                  PinNumber);

        //
        // On-Soc GPIO reconfigure operation should not fail unless it is
        // temporarily permitted by the debounce engine.
        //

        if (!NT_SUCCESS(Status) &&
            (TolerateFailure == FALSE) &&
            (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

            KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                         INTERRUPT_OPERATION_FAILURE,
                         (ULONG_PTR)Status,
                         (ULONG_PTR)GpioExtension,
                         (ULONG_PTR)&Parameters);
        }

    } else {
        Status = STATUS_NOT_SUPPORTED;
    }

    return Status;
}

VOID
GpioClnInvokePreProcessControllerInterrupt (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any
    pre-processing required prior to handling the GPIO interrupt.

    Some GPIO controller implementations may require their status bits
    to be snapshotted as soon as the interrupt fires (at DIRQL level)
    even though the device is itself behind serial. If the snapshot is
    delayed until the IRQL drops to passive, the status bits may get lost.
    The interrupt pre-processing callback enables such handling.

    The status bits are copied to some memory location in this
    case by some hardware entity (otherwise the client driver will not
    be able to access it at DIRQL).

    N.B. This routine is called at DIRQL level (for both memory-mapped and
         serially-accesssible GPIO controller cases).

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_CLIENT_PRE_PROCESS_CONTROLLER_INTERRUPT PreProcessInterrupt;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    PreProcessInterrupt = ClientPacket->CLIENT_PreProcessControllerInterrupt;
    if (PreProcessInterrupt != NULL) {
        PreProcessInterrupt(Extension,
                            GpioBank->BankId,
                            GpioBank->InterruptData.EnableRegister);
    }

    return;
}

//
// ------------------------------------------------Function config interfaces
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeConnectFunctionConfigPins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in PPNP_FUNCTION_CONFIG_DESCRIPTOR Descriptor
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to configure a set of
    lines for function config.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    Descriptor - Supplies a pointer to the BIOS function config descriptor.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PGPIO_CLIENT_CONNECT_FUNCTION_CONFIG_PINS ConnectFunctionConfigPins;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_CONNECT_FUNCTION_CONFIG_PINS_PARAMETERS Parameters;
    NTSTATUS Status;

    PAGED_CODE();

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    ConnectFunctionConfigPins = ClientPacket->CLIENT_ConnectFunctionConfigPins;
    if (ConnectFunctionConfigPins == NULL) {

        GPIO_ASSERT(FALSE);

        Status = STATUS_NOT_SUPPORTED;
        goto InvokeConnectFunctionConfigPinsEnd;
    }

    GPIO_ASSERT((KeGetCurrentIrql() == PASSIVE_LEVEL) &&
                (PinCount <= GpioBank->PinCount));

    RtlZeroMemory(&Parameters, 
                  sizeof(GPIO_CONNECT_FUNCTION_CONFIG_PINS_PARAMETERS));

    Parameters.BankId = GpioBank->BankId;
    Parameters.PinCount = (USHORT)PinCount;
    Parameters.PinNumberTable = PinNumberTable;
    Parameters.FunctionNumber = 
        FUNCTION_CONFIG_BIOS_DESCRIPTOR_FUNCTION(Descriptor);

    Parameters.PullConfiguration =
        FUNCTION_CONFIG_BIOS_DESCRIPTOR_PULL_CONFIGURATION(Descriptor);

    //
    // If vendor is supplied in the FW descriptor, then pass that
    // information.
    //

    if (Descriptor->VendorDataLength > 0) {
        Parameters.VendorData =
            Add2Ptr(Descriptor, Descriptor->VendorDataOffset);

        Parameters.VendorDataLength = Descriptor->VendorDataLength;
    }

    Status = ConnectFunctionConfigPins(Extension, &Parameters);

InvokeConnectFunctionConfigPinsEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeDisconnectFunctionConfigPins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in ULONG DisconnectFlags
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to disconnect a previously
    connected set of lines for function config.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    DisconnectFlags - Supplies a flag value controlling the connection behavior.
        Currently no flags are defined.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PGPIO_CLIENT_DISCONNECT_FUNCTION_CONFIG_PINS DisconnectFunctionConfigPins;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_DISCONNECT_FUNCTION_CONFIG_PINS_PARAMETERS Parameters;
    NTSTATUS Status;

    PAGED_CODE();

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    DisconnectFunctionConfigPins = 
        ClientPacket->CLIENT_DisconnectFunctionConfigPins;

    if (DisconnectFunctionConfigPins != NULL) {
        RtlZeroMemory(&Parameters,
                      sizeof(GPIO_DISCONNECT_FUNCTION_CONFIG_PINS_PARAMETERS));

        Parameters.BankId = GpioBank->BankId;
        Parameters.PinNumberTable = PinNumberTable;
        Parameters.PinCount = PinCount;
        Parameters.DisconnectFlags.AsULONG = DisconnectFlags;
        Status = DisconnectFunctionConfigPins(Extension, &Parameters);

    } else {

        GPIO_ASSERT(FALSE);

        Status = STATUS_NOT_SUPPORTED;
    }

    return Status;
}

//
// -------------------------------------------------------------- IO interfaces
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeConnectPins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in GPIO_CONNECT_IO_PINS_MODE ConnectMode,
    __in_opt PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR Descriptor
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to configure a set of
    lines for input/output.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    ConnectMode - Supplies the mode in which the pins should be configured
        (input or output).

    Descriptor - Supplies a pointer to the BIOS GPIO descriptor. This value is
        only present if the descriptor conforms to the new GPIO descriptor
        format.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PGPIO_CLIENT_CONNECT_IO_PINS ConnectIoPins;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_CONNECT_IO_PINS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    ConnectIoPins = ClientPacket->CLIENT_ConnectIoPins;
    if (ConnectIoPins == NULL) {
        GPIO_ASSERT(FALSE);

        Status = STATUS_NOT_SUPPORTED;
        goto InvokeConnectPinsEnd;
    }

    GPIO_ASSERT((KeGetCurrentIrql() == PASSIVE_LEVEL) &&
                (PinCount <= GpioBank->PinCount) &&
                ((ConnectMode == ConnectModeInput) ||
                    (ConnectMode == ConnectModeOutput)));

    RtlZeroMemory(&Parameters, sizeof(GPIO_CONNECT_IO_PINS_PARAMETERS));
    Parameters.BankId = GpioBank->BankId;
    Parameters.PinCount = (USHORT)PinCount;
    Parameters.PinNumberTable = PinNumberTable;
    Parameters.ConnectMode = ConnectMode;
    if (ARGUMENT_PRESENT(Descriptor) != FALSE) {
        Parameters.DebounceTimeout = Descriptor->DebounceTimeout;
        Parameters.PullConfiguration =
            GPIO_BIOS_DESCRIPTOR_PULL_CONFIGURATION(Descriptor);
        Parameters.DriveStrength = Descriptor->DriveStrength;

        //
        // If vendor is supplied in the FW descriptor, then pass that
        // information.
        //

        if (Descriptor->VendorDataLength > 0) {
            Parameters.VendorData =
                Add2Ptr(Descriptor, Descriptor->VendorDataOffset);

            Parameters.VendorDataLength = Descriptor->VendorDataLength;
        }
    }

    Status = ConnectIoPins(Extension, &Parameters);

InvokeConnectPinsEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeDisconnectPins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in ULONG DisconnectMode,
    __in ULONG DisconnectFlags
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to disconnect a previously
    connected set of lines.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    DisconnectMode - Supplies the mode in which the pins are currently
        configured (input or output).

    DisconnectFlags - Supplies a flag value controlling the connection behavior.
        Currently no flags are defined.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PGPIO_CLIENT_DISCONNECT_IO_PINS DisconnectIoPins;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_DISCONNECT_IO_PINS_PARAMETERS Parameters;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    DisconnectIoPins = ClientPacket->CLIENT_DisconnectIoPins;
    if (DisconnectIoPins != NULL) {
        RtlZeroMemory(&Parameters, sizeof(GPIO_DISCONNECT_IO_PINS_PARAMETERS));
        Parameters.BankId = GpioBank->BankId;
        Parameters.PinNumberTable = PinNumberTable;
        Parameters.PinCount = PinCount;
        Parameters.DisconnectMode = DisconnectMode;
        Parameters.DisconnectFlags.AsULONG = DisconnectFlags;
        Status = DisconnectIoPins(Extension, &Parameters);

    } else {
        GPIO_ASSERT(FALSE);

        Status = STATUS_NOT_SUPPORTED;
    }

    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeReadPins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __out_bcount((PinCount + 7) / 8) PVOID Buffer,
    __out PULONG64 PinValues,
    __in_opt PVOID ClientIoContext,
    __in BOOLEAN PinsWriteConfigured
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to read to a set of lines
    (i.e. input).

    N.B. This routine is called at DIRQL for memory-mapped GPIOs and thus not
         marked as PAGED.

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    Buffer - Supplies a buffer that will receive the read values.

    PinValues - Supplies a pointer that receives the bitmask of pin values.

    ClientIoContext - Supplies a pointer that contains the context the
        client driver created in its CreateIoContext callback.

    PinsWriteConfigured - Supplies a flag that indicates if the read is
        being performed on pins opened for write (output).

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_CLIENT_READ_PINS ReadGpioPins;
    PGPIO_CLIENT_READ_PINS_MASK ReadGpioPinsUsingMask;
    ULONG_PTR ParametersBuffer;
    GPIO_READ_PINS_PARAMETERS ParametersRead;
    GPIO_READ_PINS_MASK_PARAMETERS ParametersMask;
    NTSTATUS Status;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    ReadGpioPins = NULL;
    ReadGpioPinsUsingMask = NULL;
    Status = STATUS_SUCCESS;
    ParametersBuffer = 0;

    //
    // Determine the client driver preference and use the appropriate interface
    // to read from those pins.
    //
    // If the appropriate interface is not specified, then bail out.
    //

    if (GpioExtension->ClientInformation.Flags.FormatIoRequestsAsMasks == 0) {
        ReadGpioPins = ClientPacket->CLIENT_ReadGpioPins;
        if (ReadGpioPins == NULL) {

            GPIO_ASSERT(FALSE);

            Status = STATUS_NOT_SUPPORTED;
            goto InvokeReadPinsEnd;
        }

        GPIO_ASSERT(PinCount <= GpioBank->PinCount);

        ParametersBuffer = (ULONG_PTR)&ParametersRead;
        RtlZeroMemory(&ParametersRead, sizeof(GPIO_READ_PINS_PARAMETERS));
        ParametersRead.BankId = GpioBank->BankId;
        ParametersRead.PinNumberTable = PinNumberTable;
        ParametersRead.PinCount = PinCount;
        ParametersRead.Buffer = Buffer;
        ParametersRead.Reserved = ClientIoContext;
        ParametersRead.Flags.WriteConfiguredPins = PinsWriteConfigured;
        Status = ReadGpioPins(Extension, &ParametersRead);

    } else {
        ReadGpioPinsUsingMask = ClientPacket->CLIENT_ReadGpioPinsUsingMask;
        if (ReadGpioPinsUsingMask == NULL) {
            GPIO_ASSERT(FALSE);

            Status = STATUS_NOT_SUPPORTED;
            goto InvokeReadPinsEnd;
        }

        ParametersBuffer = (ULONG_PTR)&ParametersMask;
        RtlZeroMemory(&ParametersMask, sizeof(GPIO_READ_PINS_MASK_PARAMETERS));
        ParametersMask.BankId = GpioBank->BankId;
        ParametersMask.PinValues = PinValues;
        ParametersMask.Reserved = ClientIoContext;
        ParametersMask.Flags.WriteConfiguredPins = PinsWriteConfigured;
        Status = ReadGpioPinsUsingMask(Extension, &ParametersMask);
    }

InvokeReadPinsEnd:

    //
    // On-Soc GPIO read pin operation should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     IO_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     ParametersBuffer);
    }

    return Status;
}

_Must_inspect_result_
NTSTATUS
GpioClnInvokeWritePins (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in_bcount((PinCount + 7) / 8) PVOID Buffer,
    __in ULONG64 SetMask,
    __in ULONG64 ClearMask,
    __in_opt PVOID ClientIoContext
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to write to a set of lines
    (i.e. output).

    N.B. This routine is called at DIRQL for memory-mapped GPIOs and thus not
         marked as PAGED.

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    Buffer - Supplies a buffer containing the values to be written.

    SetMask - Supplies a bitmask of pins to be set. This value is used only
        if FormatIoRequestsAsMasks attribute is set.

    ClearMask - Supplies a bitmask of pins to be cleared. This value is used
        only if FormatIoRequestsAsMasks attribute is set.

    ClientIoContext - Supplies a pointer that contains the context the
        client driver created in its CreateIoContext callback.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    ULONG_PTR ParametersBuffer;
    GPIO_WRITE_PINS_PARAMETERS ParametersWrite;
    GPIO_WRITE_PINS_MASK_PARAMETERS ParametersMask;
    NTSTATUS Status;
    PGPIO_CLIENT_WRITE_PINS WriteGpioPins;
    PGPIO_CLIENT_WRITE_PINS_MASK WriteGpioPinsUsingMask;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    Status = STATUS_SUCCESS;
    ParametersBuffer = 0;

    //
    // Determine the client driver preference and use the appropriate interface
    // to write to those pins.
    //
    // If the appropriate interface is not specified, then bail out.
    //

    if (GpioExtension->ClientInformation.Flags.FormatIoRequestsAsMasks == 0) {
        WriteGpioPins = ClientPacket->CLIENT_WriteGpioPins;
        if (WriteGpioPins == NULL) {

            GPIO_ASSERT(FALSE);

            Status = STATUS_NOT_SUPPORTED;
            goto InvokeWritePinsEnd;
        }

        ParametersBuffer = (ULONG_PTR)&ParametersWrite;
        RtlZeroMemory(&ParametersWrite, sizeof(GPIO_WRITE_PINS_PARAMETERS));
        ParametersWrite.BankId = GpioBank->BankId;
        ParametersWrite.PinNumberTable = PinNumberTable;
        ParametersWrite.PinCount = PinCount;
        ParametersWrite.Buffer = Buffer;
        ParametersWrite.Reserved = ClientIoContext;
        Status = WriteGpioPins(Extension, &ParametersWrite);

    } else {
        WriteGpioPinsUsingMask = ClientPacket->CLIENT_WriteGpioPinsUsingMask;
        if (WriteGpioPinsUsingMask == NULL) {

            GPIO_ASSERT(FALSE);

            Status = STATUS_NOT_SUPPORTED;
            goto InvokeWritePinsEnd;
        }

        ParametersBuffer = (ULONG_PTR)&ParametersMask;
        RtlZeroMemory(&ParametersMask, sizeof(GPIO_WRITE_PINS_MASK_PARAMETERS));
        ParametersMask.BankId  = GpioBank->BankId;
        ParametersMask.SetMask = SetMask;
        ParametersMask.ClearMask = ClearMask;
        ParametersMask.Reserved = ClientIoContext;
        Status = WriteGpioPinsUsingMask(Extension, &ParametersMask);
    }

InvokeWritePinsEnd:

    //
    // On-Soc GPIO write pin operation should not fail.
    //

    if (!NT_SUCCESS(Status) &&
        (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE)) {

        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     IO_OPERATION_FAILURE,
                     (ULONG_PTR)Status,
                     (ULONG_PTR)GpioExtension,
                     ParametersBuffer);
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeCreateIoContext (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in ULONG ConnectMode,
    __deref_out_opt PVOID *ClientIoContext
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to prepare for IO on the
    specified set of pins.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    ConnectMode - Supplies the mode in which the pins should be configured
        (input or output).

    ClientIoContext - Supplies a pointer that receives the context the
        client driver created.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PGPIO_CLIENT_CREATE_IO_CONTEXT CreateIoContext;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    GPIO_ASSERT((KeGetCurrentIrql() == PASSIVE_LEVEL) &&
                (PinCount > 0) &&
                (PinCount <= GpioBank->PinCount));

    GpioExtension = GpioBank->GpioExtension;
    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        CreateIoContext = ClientPacket->CLIENT_CreateIoContext;

    } else {
        CreateIoContext = NULL;
    }

    if (CreateIoContext != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Status = CreateIoContext(Extension,
                                 GpioBank->BankId,
                                 PinNumberTable,
                                 PinCount,
                                 ConnectMode,
                                 ClientIoContext);

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeDeleteIoContext (
    __in PDEVICE_EXTENSION GpioExtension,
    __in_opt PVOID ClientIoContext
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to delete a previously
    created IO context.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ClientIoContext - Supplies a pointer that contains the context the
        client driver created in its CreateIoContext callback.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PGPIO_CLIENT_CREATE_IO_CONTEXT CreateIoContext;
    PGPIO_CLIENT_DELETE_IO_CONTEXT DeleteIoContext;
    PVOID Extension;
    NTSTATUS Status;

    GPIO_ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        CreateIoContext = ClientPacket->CLIENT_CreateIoContext;
        DeleteIoContext = ClientPacket->CLIENT_DeleteIoContext;

    } else {
        CreateIoContext = NULL;
        DeleteIoContext = NULL;
    }

    //
    // Only call the delete callback interface, if the create was ever called.
    //

    if ((CreateIoContext != NULL) && (DeleteIoContext != NULL)) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Status = DeleteIoContext(Extension, ClientIoContext);

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

//
// --------------------------------------------------- Bank Pwr mgmt interfaces
//

VOID
GpioClnInvokeSaveBankHardwareContext (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in BOOLEAN CriticalTransition
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to save all the hardware
    context associated with the specified bank.

    This routine is called at DIRQL level associated with the GPIO bank for
    memory-mapped GPIOs. For non memory-mapped GPIOs (off-SoC GPIOs), this
    routine will be called at PASSIVE_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    CriticalTransition - Supplies whether this is a critical F-state transition
        (TRUE) or not (FALSE).

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_SAVE_RESTORE_BANK_HARDWARE_CONTEXT_PARAMETERS Parameters;
    PGPIO_CLIENT_SAVE_BANK_HARDWARE_CONTEXT SaveHardwareContext;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);

    //
    // The save bank hardware context callback is optional.
    //

    SaveHardwareContext = ClientPacket->CLIENT_SaveBankHardwareContext;
    if (SaveHardwareContext != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Parameters.BankId = GpioBank->BankId;
        Parameters.State  = FStateF1;
        Parameters.Flags.AsULONG = 0;
        Parameters.Flags.CriticalTransition = CriticalTransition;
        SaveHardwareContext(Extension, &Parameters);
    }

    return;
}

VOID
GpioClnInvokeRestoreBankHardwareContext (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in BOOLEAN CriticalTransition
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to restore all the
    hardware context associated with the specified bank.

    This routine is called at DIRQL level associated with the GPIO bank for
    memory-mapped GPIOs. For non memory-mapped GPIOs (off-SoC GPIOs), this
    routine will be called at PASSIVE_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    CriticalTransition - Supplies whether this is a critical F-state transition
        (TRUE) or not (FALSE).

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    GPIO_SAVE_RESTORE_BANK_HARDWARE_CONTEXT_PARAMETERS Parameters;
    PGPIO_CLIENT_RESTORE_BANK_HARDWARE_CONTEXT RestoreHardwareContext;

    GpioExtension = GpioBank->GpioExtension;
    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);

    //
    // The restore bank hardware context callback is optional.
    //

    RestoreHardwareContext = ClientPacket->CLIENT_RestoreBankHardwareContext;
    if (RestoreHardwareContext != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Parameters.BankId = GpioBank->BankId;
        Parameters.State  = FStateF0;
        Parameters.Flags.AsULONG = 0;
        Parameters.Flags.CriticalTransition = CriticalTransition;
        RestoreHardwareContext(Extension, &Parameters);
    }

    return;
}

//
// ------------------------------------------------------------ Misc interfaces
//

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeControllerSpecificFunctionHandler (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PVOID InputBuffer,
    __in SIZE_T InputBufferSize,
    __inout PVOID OutputBuffer,
    __in SIZE_T OutputBufferSize,
    __out PULONG BytesReadWritten
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to service a
    controller-specific IOCTL request.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InputBuffer - Supplies a pointer to the input buffer supplied by the
        IOCTL caller.

    InputBufferSize - Supplies the size of the input buffer.

    OutputBuffer - Supplies a pointer to the output buffer.

    OutputBufferSize - Supplies the size of the output buffer.

    BytesReadWritten - Supplies a pointer to a variable that receives the total
        number of bytes read/written.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    GPIO_CLIENT_CONTROLLER_SPECIFIC_FUNCTION_PARAMETERS Parameters;
    NTSTATUS Status;

    PAGED_CODE();

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
    if (ClientPacket->CLIENT_ControllerSpecificFunction != NULL) {
        RtlZeroMemory(&Parameters, sizeof(Parameters));
        Parameters.InputBuffer = InputBuffer;
        Parameters.InputBufferLength  = InputBufferSize;
        Parameters.OutputBuffer = OutputBuffer;
        Parameters.OutputBufferLength = OutputBufferSize;
        Status = ClientPacket->CLIENT_ControllerSpecificFunction(Extension,
                                                                 &Parameters);

        *BytesReadWritten = Parameters.BytesWritten;

    } else {
        Status = STATUS_NOT_SUPPORTED;
    }

    return Status;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokeQueryControllerFunctionBankMapping (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PVOID InputBuffer,
    __in SIZE_T InputBufferSize,
    __in SIZE_T OutputBufferSize,
    __out_xcount(MappingOutput->Size)
        PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT MappingOutput
   )

/*++

Routine Description:

    This routine invokes the client driver's callback to query IOCTL to
    GPIO bank mappings (i.e. which banks must be powered ON for servicing the
    given IOCTL).

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InputBuffer - Supplies a pointer to the input buffer supplied by the
        IOCTL caller.

    InputBufferSize - Supplies the size of the input buffer.

    OutputBufferSize - Supplies the size of the output buffer.

    MappingOutput - Supplies a buffer that receives the IOCTL <-> bank mappings
        filled by the client driver.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    PVOID Extension;
    CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT LocalInput;
    USHORT LocalInputSize;
    PGPIO_CLIENT_QUERY_SET_CONTROLLER_INFORMATION QuerySetControllerInformation;
    NTSTATUS Status;

    PAGED_CODE();

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);

    QuerySetControllerInformation =
        ClientPacket->CLIENT_QuerySetControllerInformation;

    //
    // The query bank interrupt callback is optional.
    //

    Status = STATUS_NOT_IMPLEMENTED;
    if (QuerySetControllerInformation != NULL) {
        LocalInputSize = sizeof(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_INPUT);
        RtlZeroMemory(&LocalInput, LocalInputSize);
        LocalInput.RequestType = QueryControllerFunctionBankMappingInformation;
        LocalInput.Size = LocalInputSize;
        LocalInput.ControllerFunctionBankMapping.InputBuffer = InputBuffer;
        LocalInput.ControllerFunctionBankMapping.InputBufferSize = InputBufferSize;
        LocalInput.ControllerFunctionBankMapping.OutputBufferSize = OutputBufferSize;
        LocalInput.ControllerFunctionBankMapping.TotalBanks =
            GpioExtension->TotalBanks;

        Status = QuerySetControllerInformation(Extension,
                                               &LocalInput,
                                               MappingOutput);
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokePreControllerInterruptDisabledCallback (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any actions
    prior to the controller interrupt being disabled.

    Note this routine may be called before/after the boot device is in D0/D3 if
    boot device has GPIO dependencies.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PVOID Extension;
    PGPIO_CLIENT_PRE_CONTROLLER_INTERRUPTS_DISABLED PreInterruptDisabled;
    NTSTATUS Status;

    //
    // The pre-interrupt disable callback is optional.
    //

    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        PreInterruptDisabled = ClientPacket->CLIENT_PreControllerInterruptDisabled;

    } else {
        PreInterruptDisabled = NULL;
    }

    if (PreInterruptDisabled != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Status = PreInterruptDisabled(Extension);

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClnInvokePostControllerInterruptEnabledCallback (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any actions
    prior to the controller interrupt being disabled.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PVOID Extension;
    PGPIO_CLIENT_POST_CONTROLLER_INTERRUPTS_ENABLED PostInterruptEnabled;
    NTSTATUS Status;

    PAGED_CODE();

    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        PostInterruptEnabled = ClientPacket->CLIENT_PostControllerInterruptEnabled;

    } else {
        PostInterruptEnabled = NULL;
    }

    //
    // The post-interrupt enable callback is optional.
    //

    if (PostInterruptEnabled != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        Status = PostInterruptEnabled(Extension);

    } else {
        Status = STATUS_SUCCESS;
    }

    return Status;
}

VOID
GpioClnInvokePreBankFStateTransitionCallback (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in FSTATE FState
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any actions
    prior to the bank's Fx state transition.

    N.B. Could be called at DISPATCH_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    FState - Supplies the target FState for the f-state transition. Note
        currently only F1State is possible.

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_CLIENT_PRE_BANK_FSTATE_TRANSITION PreFStateTransition;

    //
    // The pre-fstate transition callback is optional.
    //

    GpioExtension = GpioBank->GpioExtension;
    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        PreFStateTransition = ClientPacket->CLIENT_PreFStateTransition;

    } else {
        PreFStateTransition = NULL;
    }

    if (PreFStateTransition != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        PreFStateTransition(Extension, GpioBank->BankId, FState);
    }

    return;
}

VOID
GpioClnInvokePostBankFStateTransitionCallback (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in FSTATE FState
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any actions
    after the bank's Fx state transition.

    N.B. Could be called at DISPATCH_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PVOID Extension;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_CLIENT_POST_BANK_FSTATE_TRANSITION PostFStateTransition;

    //
    // The post-fstate transition callback is optional.
    //

    GpioExtension = GpioBank->GpioExtension;
    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        PostFStateTransition = ClientPacket->CLIENT_PostFStateTransition;

    } else {
        PostFStateTransition = NULL;
    }

    if (PostFStateTransition != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        PostFStateTransition(Extension, GpioBank->BankId, FState);
    }

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpioClnInvokeQueryDebounceNoiseIntervals (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PIN_NUMBER AbsolutePinNumber,
    __inout PULONG DebounceTimeout,
    __inout PULONG64 NoiseFilterTimeout,
    __out PGPIO_DEBOUNCE_MODEL DebounceModelOverride
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to query for debounce
    and noise filter timer interval overrides. This callback is only supported
    for internal test clients.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    AbsolutePinNumber - Supplies the absolute pin number for the GPIO pin.

    DebounceTimeout - Supplies a pointer that receives the override debounce
        timeout value. On input, supplies the ACPI firmware specified value.

    NoiseFilterTimeout - Supplies a pointer that receives the override noise
        filter interval value. On input, supplies the ACPI firmware specified
        value.

    DebounceModelOverride - Supplies a pointer that receives the debounce
        model override, if any. DebounceModelNone implies no override.

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PVOID Extension;
    TEST_GPIO_DEBOUNCE_MODEL TestDebounceOverride;
    PGPIO_CLIENT_QUERY_DEBOUNCE_NOISE_INTERVALS QueryDebounceNoiseIntervals;

    PAGED_CODE();

    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        QueryDebounceNoiseIntervals = ClientPacket->CLIENT_QueryDebounceNoiseIntervals;

    } else {
        QueryDebounceNoiseIntervals = NULL;
    }

    //
    // The post-interrupt enable callback is optional.
    //

    if (QueryDebounceNoiseIntervals != NULL) {
        Extension = GPIO_GET_CLIENT_DEVICE_EXTENSION(GpioExtension);
        TestDebounceOverride = TestDebounceModelNoOverride;
        QueryDebounceNoiseIntervals(Extension,
                                    AbsolutePinNumber,
                                    DebounceTimeout,
                                    NoiseFilterTimeout,
                                    &TestDebounceOverride);

        //
        // Ensure that the debounce timeout is at least 5 times the noise filter
        // interval when overriden by test GPIO clients.
        //
        // Debounce timeout is in hundredths of milli-seconds. Noise filter
        // timeout is in micro-seconds.
        //

        GPIO_ASSERT((*DebounceTimeout * 10) >= 5 * (*NoiseFilterTimeout));

        //
        // Convert test debounce override to debounce engine override values.
        //

        switch (TestDebounceOverride) {

        case TestDebounceModelForceLegacy:
            *DebounceModelOverride = DebounceModelLegacy;
            break;

        case TestDebounceModelForceEnhanced:
            *DebounceModelOverride = DebounceModelEnhanced;
            break;

        case TestDebounceModelForcePossiblyEnhanced:
            *DebounceModelOverride = DebounceModelPossiblyEnhanced;
            break;

        //
        // If the test supplies no override, unknown or invalid debounce model,
        // then fallback to no override.
        //

        default:

            GPIO_ASSERT(TestDebounceOverride == TestDebounceModelNoOverride);

            *DebounceModelOverride = DebounceModelNone;
        }
    }

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpioClnInvokeControllerCleanupCallback (
    _In_ PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine invokes the client driver's callback to perform any actions
    at the end of EvtCleanupCallback.  The device extension has already been
    mostly uninitialized.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED ClientPacket;
    PGPIO_CLIENT_CONTROLLER_CLEANUP_CALLBACK CleanupCallback;

    PAGED_CODE();

    //
    // The cleanup callback is optional.
    //

    if (GpioExtension->IsInternalClient != FALSE) {
        ClientPacket = GPIO_GET_CLIENT_COMBINED_REGISTRATION_PACKET(GpioExtension);
        CleanupCallback = ClientPacket->CLIENT_ControllerCleanupCallback;

    } else {
        CleanupCallback = NULL;
    }

    if (CleanupCallback != NULL) {
        CleanupCallback(GpioExtension->Device);
    }

    return;
}


