/*
	File:		LH_StdConversionLuts.c

	Contains:	conversion lut generation

	Written by:	H.Siegeritz

	Copyright:	 1993-1997 by Heidelberger Druckmaschinen AG, all rights reserved.

*/
#ifndef LHGeneralIncs_h
#include "General.h"
#endif

#ifndef __LHStdConversionLuts__
#include "StdConv.h"
#endif

#if ! realThing
#ifdef DEBUG_OUTPUT
#define kThisFile kLHStdConversionLutsID
#define __TYPES__
#endif
#endif


/* ______________________________________________________________________

	void
		Lab2XYZ_forCube16(	unsigned short	*theCube,
							long			count )
	
	Abstract:
		converts Lab data (16 bit) to linear XYZ (16 bit, 0 to 2.0)

	Params:
		theCube			(in/out)	cube grid data
		count			(in)		number of grid points

	Return:
		noErr			successful
   _____________________________________________________________________ */

void  Lab2XYZ_forCube16(unsigned short *theCube, long count)
{
	register unsigned short *v;
	long			i;
	long			lon_aOff, lon_bOff, lon_L, Xl, Yl, Zl;
	long			fract, index, leftVal, rightVal, lAux;

#ifdef DEBUG_OUTPUT
	OSErr err=noErr;
#endif

#ifdef SYM_C
	unsigned short	L2Y[432];
	v = L2Y;
	*v++=    0; *v++=    7; *v++=   14; *v++=   21; *v++=   28; *v++=   35; *v++=   43; *v++=   50;
	*v++=   57; *v++=   64; *v++=   71; *v++=   78; *v++=   85; *v++=   92; *v++=   99; *v++=  106;
	*v++=  113; *v++=  120; *v++=  128; *v++=  135; *v++=  142; *v++=  149; *v++=  156; *v++=  164;
	*v++=  171; *v++=  180; *v++=  188; *v++=  196; *v++=  205; *v++=  214; *v++=  224; *v++=  233;
	*v++=  243; *v++=  253; *v++=  263; *v++=  274; *v++=  285; *v++=  296; *v++=  308; *v++=  320;
	*v++=  332; *v++=  344; *v++=  357; *v++=  370; *v++=  384; *v++=  397; *v++=  411; *v++=  426;
	*v++=  440; *v++=  455; *v++=  471; *v++=  487; *v++=  503; *v++=  519; *v++=  536; *v++=  553;
	*v++=  570; *v++=  588; *v++=  606; *v++=  625; *v++=  644; *v++=  663; *v++=  683; *v++=  703;
	*v++=  723; *v++=  744; *v++=  766; *v++=  787; *v++=  809; *v++=  832; *v++=  855; *v++=  878;
	*v++=  902; *v++=  926; *v++=  950; *v++=  975; *v++= 1001; *v++= 1027; *v++= 1053; *v++= 1080;
	*v++= 1107; *v++= 1135; *v++= 1163; *v++= 1192; *v++= 1221; *v++= 1250; *v++= 1280; *v++= 1311;
	*v++= 1342; *v++= 1373; *v++= 1405; *v++= 1438; *v++= 1470; *v++= 1504; *v++= 1538; *v++= 1572;
	*v++= 1607; *v++= 1643; *v++= 1679; *v++= 1715; *v++= 1752; *v++= 1790; *v++= 1828; *v++= 1866;
	*v++= 1906; *v++= 1945; *v++= 1986; *v++= 2026; *v++= 2068; *v++= 2110; *v++= 2152; *v++= 2195;
	*v++= 2239; *v++= 2283; *v++= 2328; *v++= 2373; *v++= 2419; *v++= 2466; *v++= 2513; *v++= 2561;
	*v++= 2609; *v++= 2658; *v++= 2707; *v++= 2757; *v++= 2808; *v++= 2860; *v++= 2912; *v++= 2964;
	*v++= 3018; *v++= 3071; *v++= 3126; *v++= 3181; *v++= 3237; *v++= 3293; *v++= 3351; *v++= 3408;
	*v++= 3467; *v++= 3526; *v++= 3586; *v++= 3646; *v++= 3707; *v++= 3769; *v++= 3831; *v++= 3895;
	*v++= 3959; *v++= 4023; *v++= 4088; *v++= 4154; *v++= 4221; *v++= 4288; *v++= 4356; *v++= 4425;
	*v++= 4495; *v++= 4565; *v++= 4636; *v++= 4708; *v++= 4780; *v++= 4853; *v++= 4927; *v++= 5002;
	*v++= 5077; *v++= 5153; *v++= 5230; *v++= 5308; *v++= 5386; *v++= 5466; *v++= 5546; *v++= 5626;
	*v++= 5708; *v++= 5790; *v++= 5874; *v++= 5957; *v++= 6042; *v++= 6128; *v++= 6214; *v++= 6301;
	*v++= 6389; *v++= 6478; *v++= 6567; *v++= 6658; *v++= 6749; *v++= 6841; *v++= 6934; *v++= 7028;
	*v++= 7122; *v++= 7218; *v++= 7314; *v++= 7411; *v++= 7509; *v++= 7608; *v++= 7707; *v++= 7808;
	*v++= 7909; *v++= 8012; *v++= 8115; *v++= 8219; *v++= 8324; *v++= 8430; *v++= 8536; *v++= 8644;
	*v++= 8753; *v++= 8862; *v++= 8972; *v++= 9084; *v++= 9196; *v++= 9309; *v++= 9423; *v++= 9538;
	*v++= 9654; *v++= 9770; *v++= 9888; *v++=10007; *v++=10126; *v++=10247; *v++=10368; *v++=10491;
	*v++=10614; *v++=10739; *v++=10864; *v++=10991; *v++=11118; *v++=11246; *v++=11375; *v++=11506;
	*v++=11637; *v++=11769; *v++=11902; *v++=12037; *v++=12172; *v++=12308; *v++=12446; *v++=12584;
	*v++=12723; *v++=12864; *v++=13005; *v++=13147; *v++=13291; *v++=13435; *v++=13581; *v++=13727;
	*v++=13875; *v++=14024; *v++=14173; *v++=14324; *v++=14476; *v++=14629; *v++=14783; *v++=14938;
	*v++=15094; *v++=15252; *v++=15410; *v++=15569; *v++=15730; *v++=15891; *v++=16054; *v++=16218;
	*v++=16383; *v++=16549; *v++=16716; *v++=16885; *v++=17054; *v++=17225; *v++=17396; *v++=17569;
	*v++=17743; *v++=17918; *v++=18094; *v++=18272; *v++=18450; *v++=18630; *v++=18811; *v++=18993;
	*v++=19176; *v++=19361; *v++=19546; *v++=19733; *v++=19921; *v++=20110; *v++=20301; *v++=20492;
	*v++=20685; *v++=20879; *v++=21074; *v++=21270; *v++=21468; *v++=21667; *v++=21867; *v++=22068;
	*v++=22270; *v++=22474; *v++=22679; *v++=22885; *v++=23093; *v++=23301; *v++=23511; *v++=23723;
	*v++=23935; *v++=24149; *v++=24364; *v++=24580; *v++=24798; *v++=25016; *v++=25237; *v++=25458;
	*v++=25681; *v++=25905; *v++=26130; *v++=26356; *v++=26584; *v++=26814; *v++=27044; *v++=27276;
	*v++=27509; *v++=27744; *v++=27979; *v++=28216; *v++=28455; *v++=28695; *v++=28936; *v++=29178;
	*v++=29422; *v++=29668; *v++=29914; *v++=30162; *v++=30411; *v++=30662; *v++=30914; *v++=31167;
	*v++=31422; *v++=31678; *v++=31936; *v++=32195; *v++=32455; *v++=32717; *v++=32980; *v++=33245;
	*v++=33511; *v++=33778; *v++=34047; *v++=34317; *v++=34589; *v++=34862; *v++=35137; *v++=35413;
	*v++=35690; *v++=35969; *v++=36249; *v++=36531; *v++=36814; *v++=37099; *v++=37385; *v++=37673;
	*v++=37962; *v++=38252; *v++=38544; *v++=38838; *v++=39133; *v++=39429; *v++=39727; *v++=40027;
	*v++=40328; *v++=40630; *v++=40934; *v++=41240; *v++=41547; *v++=41855; *v++=42165; *v++=42477;
	*v++=42790; *v++=43105; *v++=43421; *v++=43739; *v++=44058; *v++=44379; *v++=44701; *v++=45025;
	*v++=45351; *v++=45678; *v++=46006; *v++=46337; *v++=46668; *v++=47002; *v++=47337; *v++=47673;
	*v++=48011; *v++=48351; *v++=48692; *v++=49035; *v++=49380; *v++=49726; *v++=50074; *v++=50423;
	*v++=50774; *v++=51127; *v++=51481; *v++=51837; *v++=52194; *v++=52554; *v++=52914; *v++=53277;
	*v++=53641; *v++=54007; *v++=54374; *v++=54743; *v++=55114; *v++=55486; *v++=55861; *v++=56236;
	*v++=56614; *v++=56993; *v++=57374; *v++=57756; *v++=58141; *v++=58526; *v++=58914; *v++=59303;
	*v++=59694; *v++=60087; *v++=60482; *v++=60878; *v++=61276; *v++=61675; *v++=62077; *v++=62480;
	*v++=62885; *v++=63292; *v++=63700; *v++=64110; *v++=64522; *v++=64935; *v++=65351; *v++=65535;

#else
	static unsigned short	L2Y[432] = {		/* L==100 (65535 resp 256) -> 2 ^ 14 - 1  Static OK? */
						    0,     7,    14,    21,    28,    35,    43,    50,
						   57,    64,    71,    78,    85,    92,    99,   106,
						  113,   120,   128,   135,   142,   149,   156,   164,
						  171,   180,   188,   196,   205,   214,   224,   233,
						  243,   253,   263,   274,   285,   296,   308,   320,
						  332,   344,   357,   370,   384,   397,   411,   426,
						  440,   455,   471,   487,   503,   519,   536,   553,
						  570,   588,   606,   625,   644,   663,   683,   703,
						  723,   744,   766,   787,   809,   832,   855,   878,
						  902,   926,   950,   975,  1001,  1027,  1053,  1080,
						 1107,  1135,  1163,  1192,  1221,  1250,  1280,  1311,
						 1342,  1373,  1405,  1438,  1470,  1504,  1538,  1572,
						 1607,  1643,  1679,  1715,  1752,  1790,  1828,  1866,
						 1906,  1945,  1986,  2026,  2068,  2110,  2152,  2195,
						 2239,  2283,  2328,  2373,  2419,  2466,  2513,  2561,
						 2609,  2658,  2707,  2757,  2808,  2860,  2912,  2964,
						 3018,  3071,  3126,  3181,  3237,  3293,  3351,  3408,
						 3467,  3526,  3586,  3646,  3707,  3769,  3831,  3895,
						 3959,  4023,  4088,  4154,  4221,  4288,  4356,  4425,
						 4495,  4565,  4636,  4708,  4780,  4853,  4927,  5002,
						 5077,  5153,  5230,  5308,  5386,  5466,  5546,  5626,
						 5708,  5790,  5874,  5957,  6042,  6128,  6214,  6301,
						 6389,  6478,  6567,  6658,  6749,  6841,  6934,  7028,
						 7122,  7218,  7314,  7411,  7509,  7608,  7707,  7808,
						 7909,  8012,  8115,  8219,  8324,  8430,  8536,  8644,
						 8753,  8862,  8972,  9084,  9196,  9309,  9423,  9538,
						 9654,  9770,  9888, 10007, 10126, 10247, 10368, 10491,
						10614, 10739, 10864, 10991, 11118, 11246, 11375, 11506,
						11637, 11769, 11902, 12037, 12172, 12308, 12446, 12584,
						12723, 12864, 13005, 13147, 13291, 13435, 13581, 13727,
						13875, 14024, 14173, 14324, 14476, 14629, 14783, 14938,
						15094, 15252, 15410, 15569, 15730, 15891, 16054, 16218,
						16383, 16549, 16716, 16885, 17054, 17225, 17396, 17569,
						17743, 17918, 18094, 18272, 18450, 18630, 18811, 18993,
						19176, 19361, 19546, 19733, 19921, 20110, 20301, 20492,
						20685, 20879, 21074, 21270, 21468, 21667, 21867, 22068,
						22270, 22474, 22679, 22885, 23093, 23301, 23511, 23723,
						23935, 24149, 24364, 24580, 24798, 25016, 25237, 25458,
						25681, 25905, 26130, 26356, 26584, 26814, 27044, 27276,
						27509, 27744, 27979, 28216, 28455, 28695, 28936, 29178,
						29422, 29668, 29914, 30162, 30411, 30662, 30914, 31167,
						31422, 31678, 31936, 32195, 32455, 32717, 32980, 33245,
						33511, 33778, 34047, 34317, 34589, 34862, 35137, 35413,
						35690, 35969, 36249, 36531, 36814, 37099, 37385, 37673,
						37962, 38252, 38544, 38838, 39133, 39429, 39727, 40027,
						40328, 40630, 40934, 41240, 41547, 41855, 42165, 42477,
						42790, 43105, 43421, 43739, 44058, 44379, 44701, 45025,
						45351, 45678, 46006, 46337, 46668, 47002, 47337, 47673,
						48011, 48351, 48692, 49035, 49380, 49726, 50074, 50423,
						50774, 51127, 51481, 51837, 52194, 52554, 52914, 53277,
						53641, 54007, 54374, 54743, 55114, 55486, 55861, 56236,
						56614, 56993, 57374, 57756, 58141, 58526, 58914, 59303,
						59694, 60087, 60482, 60878, 61276, 61675, 62077, 62480,
						62885, 63292, 63700, 64110, 64522, 64935, 65351, 65535 };
#endif

	LH_START_PROC("Lab2XYZ_forCube16")

	v = theCube;											/* pos. of first color */
	
	for(i=0; i<count; i++)
	{
		lon_L    = (long)v[0];
		lon_aOff = (long)v[1];
		lon_bOff = (long)v[2];
		
	/* If Xl is an L*-distorted X value: Xl = 116/500 * a + L (from CIELAB formula).  */
		
		Xl = ((lon_aOff * 19 + 15) >> 5) - 19456 + lon_L;	/* -19456 for -128 */
		if(Xl < 0)											/* Xl=65535 with white */
			Xl = 0;
		else
		{
			index = Xl >> 8;
			if(index > 431)						/* end of table */
				Xl = (long)L2Y[431] << 1;
			else											/* ordinary case */
			{
				fract    = Xl & 0x0FF;						/* 8 bits for interpolation */
				leftVal  = (long)L2Y[index];
				rightVal = (long)L2Y[index + 1];
				lAux     = (fract * (rightVal - leftVal) + 63) >> 7;	/* one bit more */
				Xl       = (leftVal << 1) + lAux;
			}
		}
		
	/* If Zl is an L*-distorted Z value: Zl = L - 116/200 * b (from CIELAB formula).  */

		Zl = lon_L - ((lon_bOff * 95 + 31) >> 6) + 48640;	/* +48640 for -128 */
		if(Zl < 0)											/* Zl=65535 with white */
			Zl = 0;
		else
		{
			index = Zl >> 8;
			if(index > 431)						/* end of table */
				Zl = (long)L2Y[431] << 1;
			else											/* ordinary case */
			{
				fract    = Zl & 0x0FF;						/* 8 bits for interpolation */
				leftVal  = (long)L2Y[index];
				rightVal = (long)L2Y[index + 1];
				lAux     = (fract * (rightVal - leftVal) + 63) >> 7;	/* one bit more */
				Zl       = (leftVal << 1) + lAux;
			}
		}
				
		Yl = lon_L;
		{
			index = Yl >> 8;								/* Y value; L<=100 -> Y index < 431 */
			{												/* ordinary case */
				fract    = Yl & 0x0FF;						/* 8 bits for interpolation */
				leftVal  = (long)L2Y[index];
				rightVal = (long)L2Y[index + 1];
				lAux     = (fract * (rightVal - leftVal) + 63) >> 7;	/* one bit more */
				Yl       = (leftVal << 1) + lAux;
			}	
		}
		
		
			/* D50 adjustment for X and Z, which are in 0 ... 1.623 range */
		Xl = (Xl * 31595 + 16384) >> 15;		/* factor 0.9642 */
		Zl = (Zl * 13515 +  8192) >> 14;		/* factor 0.8249 */

		if (Xl > 65535) Xl = 65535;
		if (Zl > 65535) Zl = 65535;
		
		*v++ = (unsigned short)Xl;							/* X value */
		*v++ = (unsigned short)Yl;							/* Y value */
		*v++ = (unsigned short)Zl;							/* Z value */
	}
	LH_END_PROC("Lab2XYZ_forCube16")
}
/* ______________________________________________________________________

	void
		XYZ2Lab_forCube16(	unsigned short	*theCube,
							long			count )
	
	Abstract:
		converts cube grid from XYZ (16 bit linear, 0 to 2.0) to Lab (16 bit)

	Params:
		theCube			(in/out)	cube grid data
		count			(in)		number of grid points

	Return:
		noErr			successful
   _____________________________________________________________________ */
void  XYZ2Lab_forCube16(unsigned short *theCube, long count)
{
	register unsigned short *v;
	long			i;
	long			lon_aOff, lon_bOff, lon_X, lon_Y, lon_Z;
	long			fract, index, leftVal, rightVal;

#ifdef DEBUG_OUTPUT
	OSErr err=noErr;
#endif

#ifdef SYM_C
	unsigned short	Y2L[432];
	v = Y2L;
	*v++=    0; *v++= 1156; *v++= 2312; *v++= 3391; *v++= 4260; *v++= 4993; *v++= 5635; *v++= 6208;
	*v++= 6730; *v++= 7209; *v++= 7654; *v++= 8070; *v++= 8462; *v++= 8833; *v++= 9185; *v++= 9520;
	*v++= 9841; *v++=10149; *v++=10445; *v++=10731; *v++=11006; *v++=11273; *v++=11531; *v++=11781;
	*v++=12024; *v++=12261; *v++=12491; *v++=12716; *v++=12935; *v++=13149; *v++=13358; *v++=13562;
	*v++=13762; *v++=13958; *v++=14150; *v++=14338; *v++=14523; *v++=14704; *v++=14883; *v++=15058;
	*v++=15230; *v++=15399; *v++=15565; *v++=15729; *v++=15890; *v++=16049; *v++=16206; *v++=16360;
	*v++=16512; *v++=16662; *v++=16810; *v++=16957; *v++=17101; *v++=17243; *v++=17384; *v++=17522;
	*v++=17659; *v++=17795; *v++=17929; *v++=18061; *v++=18192; *v++=18322; *v++=18450; *v++=18577;
	*v++=18702; *v++=18826; *v++=18949; *v++=19070; *v++=19191; *v++=19310; *v++=19428; *v++=19545;
	*v++=19661; *v++=19775; *v++=19889; *v++=20002; *v++=20114; *v++=20224; *v++=20334; *v++=20443;
	*v++=20551; *v++=20658; *v++=20764; *v++=20869; *v++=20974; *v++=21077; *v++=21180; *v++=21282;
	*v++=21383; *v++=21484; *v++=21584; *v++=21683; *v++=21781; *v++=21878; *v++=21975; *v++=22072;
	*v++=22167; *v++=22262; *v++=22356; *v++=22450; *v++=22543; *v++=22635; *v++=22727; *v++=22818;
	*v++=22908; *v++=22998; *v++=23087; *v++=23176; *v++=23265; *v++=23352; *v++=23439; *v++=23526;
	*v++=23612; *v++=23698; *v++=23783; *v++=23868; *v++=23952; *v++=24035; *v++=24119; *v++=24201;
	*v++=24284; *v++=24365; *v++=24447; *v++=24528; *v++=24608; *v++=24688; *v++=24768; *v++=24847;
	*v++=24926; *v++=25004; *v++=25082; *v++=25159; *v++=25237; *v++=25313; *v++=25390; *v++=25466;
	*v++=25541; *v++=25617; *v++=25692; *v++=25766; *v++=25840; *v++=25914; *v++=25988; *v++=26061;
	*v++=26134; *v++=26206; *v++=26278; *v++=26350; *v++=26421; *v++=26493; *v++=26563; *v++=26634;
	*v++=26704; *v++=26774; *v++=26844; *v++=26913; *v++=26982; *v++=27051; *v++=27119; *v++=27187;
	*v++=27255; *v++=27323; *v++=27390; *v++=27457; *v++=27524; *v++=27590; *v++=27656; *v++=27722;
	*v++=27788; *v++=27853; *v++=27919; *v++=27983; *v++=28048; *v++=28112; *v++=28177; *v++=28241;
	*v++=28304; *v++=28368; *v++=28431; *v++=28494; *v++=28556; *v++=28619; *v++=28681; *v++=28743;
	*v++=28805; *v++=28867; *v++=28928; *v++=28989; *v++=29050; *v++=29111; *v++=29171; *v++=29231;
	*v++=29291; *v++=29351; *v++=29411; *v++=29470; *v++=29530; *v++=29589; *v++=29647; *v++=29706;
	*v++=29765; *v++=29823; *v++=29881; *v++=29939; *v++=29996; *v++=30054; *v++=30111; *v++=30168;
	*v++=30225; *v++=30282; *v++=30339; *v++=30395; *v++=30451; *v++=30507; *v++=30563; *v++=30619;
	*v++=30674; *v++=30730; *v++=30785; *v++=30840; *v++=30895; *v++=30949; *v++=31004; *v++=31058;
	*v++=31112; *v++=31166; *v++=31220; *v++=31274; *v++=31327; *v++=31381; *v++=31434; *v++=31487;
	*v++=31540; *v++=31593; *v++=31645; *v++=31698; *v++=31750; *v++=31802; *v++=31854; *v++=31906;
	*v++=31958; *v++=32010; *v++=32061; *v++=32112; *v++=32164; *v++=32215; *v++=32265; *v++=32316;
	*v++=32367; *v++=32417; *v++=32468; *v++=32518; *v++=32568; *v++=32618; *v++=32668; *v++=32717;
	*v++=32767; *v++=32816; *v++=32866; *v++=32915; *v++=32964; *v++=33013; *v++=33062; *v++=33110;
	*v++=33159; *v++=33207; *v++=33256; *v++=33304; *v++=33352; *v++=33400; *v++=33448; *v++=33495;
	*v++=33543; *v++=33590; *v++=33638; *v++=33685; *v++=33732; *v++=33779; *v++=33826; *v++=33873;
	*v++=33919; *v++=33966; *v++=34013; *v++=34059; *v++=34105; *v++=34151; *v++=34197; *v++=34243;
	*v++=34289; *v++=34335; *v++=34380; *v++=34426; *v++=34471; *v++=34516; *v++=34562; *v++=34607;
	*v++=34652; *v++=34697; *v++=34741; *v++=34786; *v++=34831; *v++=34875; *v++=34919; *v++=34964;
	*v++=35008; *v++=35052; *v++=35096; *v++=35140; *v++=35184; *v++=35227; *v++=35271; *v++=35314;
	*v++=35358; *v++=35401; *v++=35444; *v++=35488; *v++=35531; *v++=35574; *v++=35617; *v++=35659;
	*v++=35702; *v++=35745; *v++=35787; *v++=35830; *v++=35872; *v++=35914; *v++=35956; *v++=35998;
	*v++=36040; *v++=36082; *v++=36124; *v++=36166; *v++=36208; *v++=36249; *v++=36291; *v++=36332;
	*v++=36373; *v++=36415; *v++=36456; *v++=36497; *v++=36538; *v++=36579; *v++=36620; *v++=36660;
	*v++=36701; *v++=36742; *v++=36782; *v++=36823; *v++=36863; *v++=36903; *v++=36943; *v++=36984;
	*v++=37024; *v++=37064; *v++=37104; *v++=37143; *v++=37183; *v++=37223; *v++=37262; *v++=37302;
	*v++=37342; *v++=37381; *v++=37420; *v++=37459; *v++=37499; *v++=37538; *v++=37577; *v++=37616;
	*v++=37655; *v++=37693; *v++=37732; *v++=37771; *v++=37810; *v++=37848; *v++=37887; *v++=37925;
	*v++=37963; *v++=38002; *v++=38040; *v++=38078; *v++=38116; *v++=38154; *v++=38192; *v++=38230;
	*v++=38268; *v++=38305; *v++=38343; *v++=38381; *v++=38418; *v++=38456; *v++=38493; *v++=38530;
	*v++=38568; *v++=38605; *v++=38642; *v++=38679; *v++=38716; *v++=38753; *v++=38790; *v++=38827;
	*v++=38864; *v++=38900; *v++=38937; *v++=38974; *v++=39010; *v++=39047; *v++=39083; *v++=39119;
	*v++=39156; *v++=39192; *v++=39228; *v++=39264; *v++=39300; *v++=39336; *v++=39372; *v++=39408;
	*v++=39444; *v++=39480; *v++=39516; *v++=39551; *v++=39587; *v++=39622; *v++=39658; *v++=39693;
	*v++=39729; *v++=39764; *v++=39799; *v++=39835; *v++=39870; *v++=39905; *v++=39940; *v++=39975;
#else
	static unsigned short	Y2L[439] = {			/* Y==1.0 is at 256 (not 255) and -> 2^15-1 Static OK? */
						    0,  1156,  2312,  3391,  4260,  4993,  5635,  6208,
						 6730,  7209,  7654,  8070,  8462,  8833,  9185,  9520,
						 9841, 10149, 10445, 10731, 11006, 11273, 11531, 11781,
						12024, 12261, 12491, 12716, 12935, 13149, 13358, 13562,
						13762, 13958, 14150, 14338, 14523, 14704, 14883, 15058,
						15230, 15399, 15565, 15729, 15890, 16049, 16206, 16360,
						16512, 16662, 16810, 16957, 17101, 17243, 17384, 17522,
						17659, 17795, 17929, 18061, 18192, 18322, 18450, 18577,
						18702, 18826, 18949, 19070, 19191, 19310, 19428, 19545,
						19661, 19775, 19889, 20002, 20114, 20224, 20334, 20443,
						20551, 20658, 20764, 20869, 20974, 21077, 21180, 21282,
						21383, 21484, 21584, 21683, 21781, 21878, 21975, 22072,
						22167, 22262, 22356, 22450, 22543, 22635, 22727, 22818,
						22908, 22998, 23087, 23176, 23265, 23352, 23439, 23526,
						23612, 23698, 23783, 23868, 23952, 24035, 24119, 24201,
						24284, 24365, 24447, 24528, 24608, 24688, 24768, 24847,
						24926, 25004, 25082, 25159, 25237, 25313, 25390, 25466,
						25541, 25617, 25692, 25766, 25840, 25914, 25988, 26061,
						26134, 26206, 26278, 26350, 26421, 26493, 26563, 26634,
						26704, 26774, 26844, 26913, 26982, 27051, 27119, 27187,
						27255, 27323, 27390, 27457, 27524, 27590, 27656, 27722,
						27788, 27853, 27919, 27983, 28048, 28112, 28177, 28241,
						28304, 28368, 28431, 28494, 28556, 28619, 28681, 28743,
						28805, 28867, 28928, 28989, 29050, 29111, 29171, 29231,
						29291, 29351, 29411, 29470, 29530, 29589, 29647, 29706,
						29765, 29823, 29881, 29939, 29996, 30054, 30111, 30168,
						30225, 30282, 30339, 30395, 30451, 30507, 30563, 30619,
						30674, 30730, 30785, 30840, 30895, 30949, 31004, 31058,
						31112, 31166, 31220, 31274, 31327, 31381, 31434, 31487,
						31540, 31593, 31645, 31698, 31750, 31802, 31854, 31906,
						31958, 32010, 32061, 32112, 32164, 32215, 32265, 32316,
						32367, 32417, 32468, 32518, 32568, 32618, 32668, 32717,
						32767, 32816, 32866, 32915, 32964, 33013, 33062, 33110,
						33159, 33207, 33256, 33304, 33352, 33400, 33448, 33495,
						33543, 33590, 33638, 33685, 33732, 33779, 33826, 33873,
						33919, 33966, 34013, 34059, 34105, 34151, 34197, 34243,
						34289, 34335, 34380, 34426, 34471, 34516, 34562, 34607,
						34652, 34697, 34741, 34786, 34831, 34875, 34919, 34964,
						35008, 35052, 35096, 35140, 35184, 35227, 35271, 35314,
						35358, 35401, 35444, 35488, 35531, 35574, 35617, 35659,
						35702, 35745, 35787, 35830, 35872, 35914, 35956, 35998,
						36040, 36082, 36124, 36166, 36208, 36249, 36291, 36332,
						36373, 36415, 36456, 36497, 36538, 36579, 36620, 36660,
						36701, 36742, 36782, 36823, 36863, 36903, 36943, 36984,
						37024, 37064, 37104, 37143, 37183, 37223, 37262, 37302,
						37342, 37381, 37420, 37459, 37499, 37538, 37577, 37616,
						37655, 37693, 37732, 37771, 37810, 37848, 37887, 37925,
						37963, 38002, 38040, 38078, 38116, 38154, 38192, 38230,
						38268, 38305, 38343, 38381, 38418, 38456, 38493, 38530,
						38568, 38605, 38642, 38679, 38716, 38753, 38790, 38827,
						38864, 38900, 38937, 38974, 39010, 39047, 39083, 39119,
						39156, 39192, 39228, 39264, 39300, 39336, 39372, 39409,
						39444, 39480, 39516, 39551, 39588, 39622, 39658, 39693,
						39729, 39764, 39799, 39835, 39870, 39905, 39940, 39975,
						27253, 29259, 25185, 25954, 26734, 25967, 29798 };
#endif
	
	LH_START_PROC("XYZ2Lab_forCube16")

	v = theCube;							/* pos. of first color */

	for(i=0; i<count; i++)
	{
		lon_X = (long)v[0];					/* lin. 16 bit XYZ values */
		lon_Y = (long)v[1];
		lon_Z = (long)v[2];
		
			/*------transform to L* - distorted values------*/
			
		index = lon_Y >> 7;		/* adjust for table lookup, lin. XYZ: 1.0 -> 32767 */
		fract = lon_Y & 0x7F;	/* 7 bit for interpolation */
		if(index < 431)			/* within table range ? */
		{
			leftVal  = Y2L[index];
			rightVal = Y2L[index + 1];
			lon_Y    = leftVal + ((fract * (rightVal - leftVal) + 63) >> 7);
		}
		else
			lon_Y = Y2L[431];				/* clip to the end of the table */
		
			/* D50 to white adjustment for X and Z:*/
		
		lon_X = 531 * lon_X >> 4;	/* factor 1./0.9642 with (* 531 >> 9) */
		index = lon_X >> 12;
		fract = lon_X & 0x0FFF;		/* (9 - 4) + 7 = 12 bit for interpolation */
		if(index < 431)				/* within table range ? */
		{
			leftVal  = Y2L[index];
			rightVal = Y2L[index + 1];
			lon_X    = leftVal + ((fract * (rightVal - leftVal) + 2047) >> 12);
		}
		else
			lon_X = Y2L[431];				/* clip to the end of the table */
		
		lon_Z = lon_Z * 9931 >> 8; /* factor 1./0.8249 with (* 9931 >> 13) */
		index = lon_Z >> 12;
		fract = lon_Z & 0x0FFF;		/* (13 - 8) + 7 = 12 bit for interpolation */
		if(index < 431)				/* within table range ? */
		{
			leftVal  = Y2L[index];
			rightVal = Y2L[index + 1];
			lon_Z    = leftVal + ((fract * (rightVal - leftVal) + 2047) >> 12);
		}
		else
			lon_Z = Y2L[431];				/* clip to the end of the table */
		
	/* lon_Z, lon_Z, lon_Z are L* -distorted values now with 100 <-> 32767 */
		
				/* 431 >> 7 : factor 3.3676 = 500. / 116. * 100 / 32767 * 256 */
		lon_aOff = ((lon_X - lon_Y) * 431 + 4194368) >> 7; /* 4194368 = (32768 + 0.5) * 2^7 */
		if(lon_aOff & 0xFFFF0000)
		{							/* clip to range 0...65535 (a = -128 ... 127.996) */
			if(lon_aOff < 0)
				lon_aOff = 0;
			else if(lon_aOff > 65535)
				lon_aOff = 65535;
		}
		
				/* 11035 >> 13 : factor 1.437 = 200. / 116. * 100 / 32767 * 256 */
		lon_bOff = ((lon_Y - lon_Z) * 11035 + 268439552) >> 13; /* 268439552 = (32768 + 0.5) * 2^13 */
		if(lon_bOff & 0xFFFF0000)
		{							/* clip to range 0...65535 (b = -128 ... 127.996) */
			if(lon_bOff < 0)
				lon_bOff = 0;
			else if(lon_bOff > 65535)
				lon_bOff = 65535;
		}
		
		lon_Y <<= 1;			/* see range of table above */
		if(lon_Y > 65535)
			lon_Y = 65535;		/* clip L if > 100 */

		*v++ = (unsigned short)lon_Y;		/* L */
		*v++ = (unsigned short)lon_aOff;	/* a */
		*v++ = (unsigned short)lon_bOff;	/* b */
	}
	LH_END_PROC("XYZ2Lab_forCube16")
}
