/***
 **
 **   Module: Encoding
 **
 **   Description:
 **      This is a module of the T1 to TT font converter. The module
 **      contains interface functions for the global encoding table,
 **      i.e. this is an abstract data type.
 **
 **   Author: Michael Jansson
 **
 **   Created: 6/13/93
 **
 ***/


/**** INCLUDES */
#include <string.h>
#include "types.h"
#include "encoding.h"
#include "safemem.h"



/***** LOCAL TYPES */
struct encoding {
   const char *name;            /* Postscript name of a glyph. */
   USHORT codes[ENC_MAXCODES];  /* Character codes for each encoding scheme. */
};


/***** CONSTANTS */
static const struct encoding StdEncoding[] = {
   {".notdef",       {0xffff, 0xffff, 0xffff, 0xffff}},
   {"A",             {0x0041, 0x0041, 0x0041, 0x0041}},
   {"AE",            {0x00c6, 0x00c6, 0x00e1, 174   }},
   {"Aacute",        {0x00c1, 0x00c1, 0xffff, 231   }},
   {"Abreve",        {0x0102, 0xffff, 0xffff, 0xffff}},
   {"Acircumflex",   {0x00c2, 0x00c2, 0xffff, 229   }},
   {"Adieresis",     {0x00c4, 0x00c4, 0xffff, 128   }},
   {"Agrave",        {0x00c0, 0x00c0, 0xffff, 203   }},
   {"Aogonek",       {0x0104, 0xffff, 0xffff, 0xffff}},
   {"Aring",         {0x00c5, 0x00c5, 0xffff, 129   }},
   {"Atilde",        {0x00c3, 0x00c3, 0xffff, 204   }},
   {"B",             {0x0042, 0x0042, 0x0042, 0x0042}},
   {"C",             {0x0043, 0x0043, 0x0043, 0x0043}},
   {"Cacute",        {0x0106, 0xffff, 0xffff, 0xffff}},
   {"Ccaron",        {0x010c, 0xffff, 0xffff, 0xffff}},
   {"Ccedilla",      {0x00c7, 0x00c7, 0xffff, 130   }},
   {"D",             {0x0044, 0x0044, 0x0044, 0x0044}},
   {"Dcaron",        {0x010e, 0xffff, 0xffff, 0xffff}},
   {"Delta",         {0x2206, 0xffff, 0xffff, 198   }},
   {"Dslash",        {0x0110, 0xffff, 0xffff, 0xffff}},
   {"E",             {0x0045, 0x0045, 0x0045, 0x0045}},
   {"Eacute",        {0x00c9, 0x00c9, 0xffff, 131   }},
   {"Ecaron",        {0x011a, 0xffff, 0xffff, 0xffff}},
   {"Ecircumflex",   {0x00ca, 0x00ca, 0xffff, 230   }},
   {"Edieresis",     {0x00cb, 0x00cb, 0xffff, 232   }},
   {"Egrave",        {0x00c8, 0x00c8, 0xffff, 233   }},
   {"Eogonek",       {0x0118, 0xffff, 0xffff, 0xffff}},
   {"Eth",           {0x00d0, 0x00d0, 0xffff, 0xffff}},
   {"F",             {0x0046, 0x0046, 0x0046, 0x0046}},
   {"G",             {0x0047, 0x0047, 0x0047, 0x0047}},
   {"Gamma",         {0x0393, 0xffff, 0xffff, 0xffff}},
   {"Gbreve",        {0x011e, 0xffff, 0xffff, 0xffff}},
   {"H",             {0x0048, 0x0048, 0x0048, 0x0048}},
   {"I",             {0x0049, 0x0049, 0x0049, 0x0049}},
   {"Iacute",        {0x00cd, 0x00cd, 0xffff, 234   }},
   {"Icircumflex",   {0x00ce, 0x00ce, 0xffff, 235   }},
   {"Idieresis",     {0x00cf, 0x00cf, 0xffff, 236   }},
   {"Idot",          {0x0130, 0xffff, 0xffff, 0xffff}},
   {"Igrave",        {0x00cc, 0x00cc, 0xffff, 237   }},
   {"J",             {0x004a, 0x004a, 0x004a, 0x004a}},
   {"K",             {0x004b, 0x004b, 0x004b, 0x004b}},
   {"L",             {0x004c, 0x004c, 0x004c, 0x004c}},
   {"Lacute",        {0x0139, 0xffff, 0xffff, 0xffff}},
   {"Lcaron",        {0x013d, 0xffff, 0xffff, 0xffff}},
   {"Ldot",          {0x013f, 0xffff, 0xffff, 0xffff}},
   {"Lslash",        {0x0141, 0xffff, 0x00e8, 0xffff}},
   {"M",             {0x004d, 0x004d, 0x004d, 0x004d}},
   {"N",             {0x004e, 0x004e, 0x004e, 0x004e}},
   {"Nacute",        {0x0143, 0xffff, 0xffff, 0xffff}},
   {"Ncaron",        {0x0147, 0xffff, 0xffff, 0xffff}},
   {"Ntilde",        {0x00d1, 0x00d1, 0xffff, 132   }},
   {"O",             {0x004f, 0x004f, 0x004f, 0x004f}},
   {"OE",            {0x0152, 0x008c, 0x00ea, 206   }},
   {"Oacute",        {0x00d3, 0x00d3, 0xffff, 238   }},
   {"Ocircumflex",   {0x00d4, 0x00d4, 0xffff, 239   }},
   {"Odblacute",     {0x0150, 0xffff, 0xffff, 0xffff}},
   {"Odieresis",     {0x00d6, 0x00d6, 0xffff, 133   }},
   {"Ograve",        {0x00d2, 0x00d2, 0xffff, 241   }},
   {"Oslash",        {0x00d8, 0x00d8, 0x00e9, 175   }},
   {"Otilde",        {0x00d5, 0x00d5, 0xffff, 205   }},
   {"P",             {0x0050, 0x0050, 0x0050, 0x0050}},
   {"Phi",           {0x03a6, 0xffff, 0xffff, 0xffff}},
   {"Q",             {0x0051, 0x0051, 0x0051, 0x0051}},
   {"R",             {0x0052, 0x0052, 0x0052, 0x0052}},
   {"Racute",        {0x0154, 0xffff, 0xffff, 0xffff}},
   {"Rcaron",        {0x0158, 0xffff, 0xffff, 0xffff}},
   {"S",             {0x0053, 0x0053, 0x0053, 0x0053}},
   {"Sacute",        {0x015a, 0xffff, 0xffff, 0xffff}},
   {"Scaron",        {0x0160, 0x008A, 0xffff, 0xffff}},
   {"Scedilla",      {0x015e, 0xffff, 0xffff, 0xffff}},
   {"T",             {0x0054, 0x0054, 0x0054, 0x0054}},
   {"Tcaron",        {0x0164, 0xffff, 0xffff, 0xffff}},
   {"Tcedilla",      {0x0162, 0xffff, 0xffff, 0xffff}},
   {"Theta",         {0x0398, 0xffff, 0xffff, 0xffff}},
   {"Thorn",         {0x00de, 0x00de, 0xffff, 0xffff}},
   {"U",             {0x0055, 0x0055, 0x0055, 0x0055}},
   {"Uacute",        {0x00da, 0x00da, 0xffff, 242   }},
   {"Ucircumflex",   {0x00db, 0x00db, 0xffff, 243   }},
   {"Udblacute",     {0x0170, 0xffff, 0xffff, 0xffff}},
   {"Udieresis",     {0x00dc, 0x00dc, 0xffff, 134   }},
   {"Ugrave",        {0x00d9, 0x00d9, 0xffff, 244   }},
   {"Uring",         {0x016e, 0xffff, 0xffff, 0xffff}},
   {"V",             {0x0056, 0x0056, 0x0056, 0x0056}},
   {"W",             {0x0057, 0x0057, 0x0057, 0x0057}},
   {"X",             {0x0058, 0x0058, 0x0058, 0x0058}},
   {"Y",             {0x0059, 0x0059, 0x0059, 0x0059}},
   {"Yacute",        {0x00dd, 0x00dd, 0xffff, 0xffff}},
   {"Ydieresis",     {0x0178, 0x009f, 0xffff, 217   }},
   {"Z",             {0x005a, 0x005a, 0x005a, 0x005a}},
   {"Zacute",        {0x0179, 0xffff, 0xffff, 0xffff}},
   {"Zcaron",        {0x017d, 0xffff, 0xffff, 0xffff}},
   {"Zdot",          {0x017b, 0xffff, 0xffff, 0xffff}},
   {"a",             {0x0061, 0x0061, 0x0061, 0x0061}},
   {"aacute",        {0x00e1, 0x00e1, 0xffff, 135   }},
   {"abreve",        {0x0103, 0xffff, 0xffff, 0xffff}},
   {"acircumflex",   {0x00e2, 0x00e2, 0xffff, 137   }},
   {"acute",         {0x00b4, 0x00b4, 0x00c2, 171   }},
   {"adieresis",     {0x00e4, 0x00e4, 0xffff, 138   }},
   {"ae",            {0x00e6, 0x00e6, 0x00f1, 190   }},
   {"agrave",        {0x00e0, 0x00e0, 0xffff, 136   }},
   {"alpha",         {0x03b1, 0xffff, 0xffff, 0xffff}},
   {"ampersand",     {0x0026, 0x0026, 0x0026, 0x0026}},
   {"aogonek",       {0x0105, 0xffff, 0xffff, 0xffff}},
   {"apple",         {0xf000, 0x000, 0xffff, 240   }},
   {"approxequal",   {0x2248, 0xffff, 0xffff, 197   }},
   {"aring",         {0x00e5, 0x00e5, 0xffff, 140   }},
   {"arrowboth",     {0x2194, 0xffff, 0xffff, 0xffff}},
   {"arrowdown",     {0x2193, 0xffff, 0xffff, 0xffff}},
   {"aroowleft",     {0x2190, 0xffff, 0xffff, 0xffff}},
   {"arrowright",    {0x2192, 0xffff, 0xffff, 0xffff}},
   {"arrowup",       {0x2191, 0xffff, 0xffff, 0xffff}},
   {"arrowupdn",     {0x2195, 0xffff, 0xffff, 0xffff}},
   {"arrowupdnbse",  {0x21a8, 0xffff, 0xffff, 0xffff}},
   {"asciicircum",   {0x005e, 0x005e, 0x005e, 0x005e}},
   {"asciitilde",    {0x007e, 0x007e, 0x007e, 0x007e}},
   {"asterisk",      {0x002a, 0x002a, 0x002a, 0x002a}},
   {"at",            {0x0040, 0x0040, 0x0040, 0x0040}},
   {"atilde",        {0x00e3, 0x00e3, 0xffff, 139   }},
   {"b",             {0x0062, 0x0062, 0x0062, 0x0062}},
   {"backslash",     {0x005c, 0x005c, 0x005c, 0x005c}},
   {"bar",           {0x007c, 0x007c, 0x007c, 0x007c}},
   {"block",         {0x2588, 0xffff, 0xffff, 0xffff}},
   {"braceleft",     {0x007b, 0x007b, 0x007b, 0x007b}},
   {"braceright",    {0x007d, 0x007d, 0x007d, 0x007d}},
   {"bracketleft",   {0x005b, 0x005b, 0x005b, 0x005b}},
   {"bracketright",  {0x005d, 0x005d, 0x005d, 0x005d}},
   {"breve",         {0x02d8, 0xffff, 0x00c6, 249   }},
   {"brokenbar",     {0x00a6, 0x00a6, 0xffff, 0xffff}},
   {"bullet",        {0x2022, 0x0095, 0x00b7, 165   }},
   {"c",             {0x0063, 0x0063, 0x0063, 0x0063}},
   {"cacute",        {0x0107, 0xffff, 0xffff, 0xffff}},
   {"caron",         {0x02c7, 0xffff, 0x00cf, 255   }},
   {"ccaron",        {0x010d, 0xffff, 0xffff, 0xffff}},
   {"ccedilla",      {0x00e7, 0x00e7, 0xffff, 141   }},
   {"cedilla",       {0x00b8, 0x00b8, 0x00cb, 252   }},
   {"cent",          {0x00a2, 0x00a2, 0x00a2, 162   }},
   {"circle",        {0x25cb, 0xffff, 0xffff, 0xffff}},
   {"circumflex",    {0x02c6, 0x0088, 0x00c3, 246   }},
   {"club",          {0x2663, 0xffff, 0xffff, 0xffff}},
   {"colon",         {0x003a, 0x003a, 0x003a, 0x003a}},
   {"comma",         {0x002c, 0x002c, 0x002c, 0x002c}},
   {"copyright",     {0x00a9, 0x00a9, 0xffff, 169   }},
   {"currency",      {0x00a4, 0x00a4, 0x00a8, 219   }},
   {"d",             {0x0064, 0x0064, 0x0064, 0x0064}},
   {"dagger",        {0x2020, 0x0086, 0x00b2, 160   }},
   {"daggerdbl",     {0x2021, 0x0087, 0x00b3, 224   }},
   {"dcaron",        {0x010f, 0xffff, 0xffff, 0xffff}},
   {"degree",        {0x00b0, 0x00b0, 0xffff, 161   }},
   {"delta",         {0x03b4, 0xffff, 0xffff, 0xffff}},
   {"diamond",       {0x2666, 0xffff, 0xffff, 0xffff}},
   {"dieresis",      {0x00a8, 0x00a8, 0x00c8, 172   }},
   {"divide",        {0x00f7, 0x00f7, 0xffff, 214   }},
   {"dkshade",       {0x2593, 0xffff, 0xffff, 0xffff}},
   {"dmacron",       {0x0111, 0xffff, 0xffff, 0xffff}},
   {"dnblock",       {0x2584, 0xffff, 0xffff, 0xffff}},
   {"dollar",        {0x0024, 0x0024, 0x0024, 0x0024}},
   {"dotaccent",     {0x02d9, 0xffff, 0x00c7, 250   }},
   {"dotlessi",      {0x0131, 0xffff, 0x00f5, 245   }},
   {"e",             {0x0065, 0x0065, 0x0065, 0x0065}},
   {"eacute",        {0x00e9, 0x00e9, 0xffff, 142   }},
   {"ecaron",        {0x011b, 0xffff, 0xffff, 0xffff}},
   {"ecircumflex",   {0x00ea, 0x00ea, 0xffff, 144   }},
   {"edieresis",     {0x00eb, 0x00eb, 0xffff, 145   }},
   {"egrave",        {0x00e8, 0x00e8, 0xffff, 143   }},
   {"eight",         {0x0038, 0x0038, 0x0038, 0x0038}},
   {"ellipsis",      {0x2026, 0x0085, 0x00bc, 201   }},
   {"emdash",        {0x2014, 0x0097, 0x00d0, 209   }},
   {"endash",        {0x2013, 0x0096, 0x00b1, 208   }},
   {"eogonek",       {0x0119, 0xffff, 0xffff, 0xffff}},
   {"epsilon",       {0x03b4, 0xffff, 0xffff, 0xffff}},
   {"equal",         {0x003d, 0x003d, 0x003d, 0x003d}},
   {"equivalence",   {0x2261, 0xffff, 0xffff, 0xffff}},
   {"eth",           {0x00f0, 0x00f0, 0xffff, 0xffff}},
   {"exclam",        {0x0021, 0x0021, 0x0021, 0x0021}},
   {"exclamdbl",     {0x203c, 0xffff, 0xffff, 0xffff}},
   {"exclamdown",    {0x00a1, 0x00a1, 0x00a1, 193   }},
   {"f",             {0x0066, 0x0066, 0x0066, 0x0066}},
   {"female",        {0x2640, 0xffff, 0xffff, 0xffff}},
   {"fi",            {0xf001, 0xffff, 0x00ae, 222   }},
   {"filledbox",     {0x25a0, 0xffff, 0xffff, 0xffff}},
   {"filledrect",    {0x25ac, 0xffff, 0xffff, 0xffff}},
   {"five",          {0x0035, 0x0035, 0x0035, 0x0035}},
   {"fl",            {0xf002, 0xffff, 0x00af, 223   }},
   {"florin",        {0x0192, 0x0083, 0x00a6, 196   }},
   {"four",          {0x0034, 0x0034, 0x0034, 0x0034}},
   {"fraction",      {0x2215, 0xffff, 0x00a4, 218   }},
   {"franc",         {0x20a3, 0xffff, 0xffff, 0xffff}},
   {"g",             {0x0067, 0x0067, 0x0067, 0x0067}},
   {"gbreve",        {0x011f, 0xffff, 0xffff, 0xffff}},
   {"germandbls",    {0x00df, 0x00df, 0x00fb, 167   }},
   {"grave",         {0x0060, 0x0060, 0x00c1, 0x0060}},
   {"greater",       {0x003e, 0x003e, 0x003e, 0x003e}},
   {"greaterequal",  {0x2265, 0xffff, 0xffff, 179   }},
   {"guillemotleft", {0x00ab, 0x00ab, 0x00ab, 199   }},
   {"guillemotright",{0x00bb, 0x00bb, 0x00bb, 200   }},
   {"guilsinglleft", {0x2039, 0x008b, 0x00ac, 220   }},
   {"guilsinglright",{0x203a, 0x009b, 0x00ad, 221   }},
   {"h",             {0x0068, 0x0068, 0x0068, 0x0068}},
   {"heart",         {0x2665, 0xffff, 0xffff, 0xffff}},
   {"house",         {0x2302, 0xffff, 0xffff, 0xffff}},
   {"hungarumlaut",  {0x02dd, 0xffff, 0x00cd, 0xffff}},
   {"hyphen",        {0x002d, 0x002d, 0x002d, 0x002d}},
   {"i",             {0x0069, 0x0069, 0x0069, 0x0069}},
   {"iacute",        {0x00ed, 0x00ed, 0xffff, 146   }},
   {"icircumflex",   {0x00ee, 0x00ee, 0xffff, 148   }},
   {"idieresis",     {0x00ef, 0x00ef, 0xffff, 149   }},
   {"igrave",        {0x00ec, 0x00ec, 0xffff, 147   }},
   {"infinity",      {0x221e, 0xffff, 0xffff, 176   }},
   {"integral",      {0x222b, 0xffff, 0xffff, 186   }},
   {"integralbt",    {0x2321, 0xffff, 0xffff, 0xffff}},
   {"integraltp",    {0x2320, 0xffff, 0xffff, 0xffff}},
   {"intersection",  {0x2229, 0xffff, 0xffff, 0xffff}},
   {"invbullet",     {0x25d8, 0xffff, 0xffff, 0xffff}},
   {"invcircle",     {0x25d9, 0xffff, 0xffff, 0xffff}},
   {"invsmileface",  {0x263b, 0xffff, 0xffff, 0xffff}},
   {"j",             {0x006a, 0x006a, 0x006a, 0x006a}},
   {"k",             {0x006b, 0x006b, 0x006b, 0x006b}},
   {"l",             {0x006c, 0x006c, 0x006c, 0x006c}},
   {"lacute",        {0x013a, 0xffff, 0xffff, 0xffff}},
   {"lcaron",        {0x013e, 0xffff, 0xffff, 0xffff}},
   {"ldot",          {0x0140, 0xffff, 0xffff, 0xffff}},
   {"less",          {0x003c, 0x003c, 0x003c, 0x003c}},
   {"lessequal",     {0x2264, 0xffff, 0xffff, 178   }},
   {"lfblock",       {0x258c, 0xffff, 0xffff, 0xffff}},
   {"logicalnot",    {0x00ac, 0x00ac, 0xffff, 194   }},
   {"lozenge",       {0x25ca, 0xffff, 0xffff, 215   }},
   {"lslash",        {0x0142, 0xffff, 0x00f8, 0xffff}},
   {"ltshade",       {0x2591, 0xffff, 0xffff, 0xffff}},
   {"m",             {0x006d, 0x006d, 0x006d, 0x006d}},
   {"macron",        {0x00af, 0xffff, 0x00c5, 248   }},
   {"male",          {0x2642, 0xffff, 0xffff, 0xffff}},
   {"middot",        {0x00b7, 0x00b7, 0xffff, 0xffff}},
   {"minus",         {0x2212, 0xffff, 0xffff, 0xffff}},
   {"mu",            {0x00b5, 0x00b5, 0xffff, 181   }},
   {"multiply",      {0x00d7, 0x00d7, 0xffff, 0xffff}},
   {"musicalnote",   {0x266a, 0xffff, 0xffff, 0xffff}},
   {"musicalnotedbl",{0x266b, 0xffff, 0xffff, 0xffff}},
   {"n",             {0x006e, 0x006e, 0x006e, 0x006e}},
   {"nacute",        {0x0144, 0xffff, 0xffff, 0xffff}},
   {"nbspace",       {0x00a0, 0x00a0, 0xffff, 0xffff}},
   {"ncaron",        {0x0148, 0xffff, 0xffff, 0xffff}},
   {"nine",          {0x0039, 0x0039, 0x0039, 0x0039}},
   {"notequal",      {0x2260, 0xffff, 0xffff, 173   }},
   {"nsuperior",     {0x207f, 0xffff, 0xffff, 0xffff}},
   {"ntilde",        {0x00f1, 0x00f1, 0xffff, 150   }},
   {"numbersign",    {0x0023, 0x0023, 0x0023, 0x0023}},
   {"o",             {0x006f, 0x006f, 0x006f, 0x006f}},
   {"oacute",        {0x00f3, 0x00f3, 0xffff, 151   }},
   {"ocircumflex",   {0x00f4, 0x00f4, 0xffff, 153   }},
   {"odblacute",     {0x0151, 0xffff, 0xffff, 0xffff}},
   {"odieresis",     {0x00f6, 0x00f6, 0xffff, 154   }},
   {"oe",            {0x0153, 0x009c, 0x00fa, 207   }},
   {"ogonek",        {0x02db, 0xffff, 0x00ce, 254   }},
   {"ograve",        {0x00f2, 0x00f2, 0xffff, 152   }},
   {"ohm",           {0x2126, 0xffff, 0xffff, 189   }},
   {"one",           {0x0031, 0x0031, 0x0031, 0x0031}},
   {"onehalf",       {0x00bd, 0x00bd, 0xffff, 0xffff}},
   {"onequarter",    {0x00bc, 0x00bc, 0xffff, 0xffff}},
   {"onesuperior",   {0x00b9, 0x00b9, 0xffff, 0xffff}},
   {"ordfeminine",   {0x00aa, 0x00aa, 0x00e3, 187   }},
   {"ordmasculine",  {0x00ba, 0x00ba, 0x00eb, 188   }},
   {"orthogonal",    {0x221f, 0xffff, 0xffff, 0xffff}},
   {"oslash",        {0x00f8, 0x00f8, 0x00f9, 191   }},
   {"otilde",        {0x00f5, 0x00f5, 0xffff, 155   }},
   {"overscore",     {0x00af, 0x00af, 0xffff, 0xffff}},
   {"p",             {0x0070, 0x0070, 0x0070, 0x0070}},
   {"paragraph",     {0x00b6, 0x00b6, 0x00b6, 166   }},
   {"parenleft",     {0x0028, 0x0028, 0x0028, 0x0028}},
   {"parenright",    {0x0029, 0x0029, 0x0029, 0x0029}},
   {"partialdiff",   {0x2202, 0xffff, 0xffff, 182   }},
   {"percent",       {0x0025, 0x0025, 0x0025, 0x0025}},
   {"period",        {0x002e, 0x002e, 0x002e, 0x002e}},
   {"periodcentered",{0x2219, 0xffff, 0x00b4, 225   }},
   {"perthousand",   {0x2030, 0x0089, 0x00bd, 228   }},
   {"peseta",        {0x20a7, 0xffff, 0xffff, 0xffff}},
   {"phi",           {0x03c6, 0xffff, 0xffff, 0xffff}},
   {"pi",            {0x03c0, 0xffff, 0xffff, 185   }},
   {"plus",          {0x002b, 0x002b, 0x002b, 0x002b}},
   {"plusminus",     {0x00b1, 0x00b1, 0xffff, 177   }},
   {"product",       {0x220f, 0xffff, 0xffff, 184   }},
   {"q",             {0x0071, 0x0071, 0x0071, 0x0071}},
   {"question",      {0x003f, 0x003f, 0x003f, 0x003f}},
   {"questiondown",  {0x00bf, 0x00bf, 0x00bf, 192   }},
   {"quotedbl",      {0x0022, 0x0022, 0x0022, 0x0022}},
   {"quotedblbase",  {0x201e, 0x0084, 0x00b9, 227   }},
   {"quotedblleft",  {0x201c, 0x0093, 0x00aa, 210   }},
   {"quotedblright", {0x201d, 0x0094, 0x00ba, 211   }},
   {"quoteleft",     {0x2018, 0x0091, 0x0060, 212   }},
   {"quoteright",    {0x2019, 0x0092, 0x0027, 213   }},
   {"quotesinglbase",{0x201a, 0x0082, 0x00b8, 226   }},
   {"quotesingle",   {0x0027, 0x0027, 0x00a9, 0x0027}},
   {"r",             {0x0072, 0x0072, 0x0072, 0x0072}},
   {"racute",        {0x0155, 0xffff, 0xffff, 0xffff}},
   {"radical",       {0x221a, 0xffff, 0xffff, 195   }},
   {"rcaron",        {0x0159, 0xffff, 0xffff, 0xffff}},
   {"registered",    {0x00ae, 0x00ae, 0xffff, 168   }},
   {"revlogicalnot", {0x2310, 0xffff, 0xffff, 0xffff}},
   {"ring",          {0x02da, 0xffff, 0x00ca, 251   }},
   {"rtblock",       {0x2590, 0xffff, 0xffff, 0xffff}},
   {"s",             {0x0073, 0x0073, 0x0073, 0x0073}},
   {"sacute",        {0x015b, 0xffff, 0xffff, 0xffff}},
   {"scaron",        {0x0161, 0x009a, 0xffff, 0xffff}},
   {"scedilla",      {0x015f, 0xffff, 0xffff, 0xffff}},
   {"section",       {0x00a7, 0x00a7, 0x00a7, 164   }},
   {"semicolon",     {0x003b, 0x003b, 0x003b, 0x003b}},
   {"seven",         {0x0037, 0x0037, 0x0037, 0x0037}},
   {"sfthyphen",     {0x00ad, 0x00ad, 0xffff, 0xffff}},
   {"shade",         {0x2592, 0xffff, 0xffff, 0xffff}},
   {"sigma",         {0x03c3, 0xffff, 0xffff, 0xffff}},
   {"six",           {0x0036, 0x0036, 0x0036, 0x0036}},
   {"slash",         {0x002f, 0x002f, 0x002f, 0x002f}},
   {"smileface",     {0x263a, 0xffff, 0xffff, 0xffff}},
   {"space",         {0x0020, 0x0020, 0x0020, 0x0020}},
   {"spade",         {0x2660, 0xffff, 0xffff, 0xffff}},
   {"sterling",      {0x00a3, 0x00a3, 0x00a3, 163   }},
   {"summation",     {0x2211, 0xffff, 0xffff, 183   }},
   {"sun",           {0x263c, 0xffff, 0xffff, 0xffff}},
   {"t",             {0x0074, 0x0074, 0x0074, 0x0074}},
   {"tau",           {0x03c4, 0xffff, 0xffff, 0xffff}},
   {"tcaron",        {0x0165, 0xffff, 0xffff, 0xffff}},
   {"tcedilla",      {0x0163, 0xffff, 0xffff, 0xffff}},
   {"thorn",         {0x00fe, 0x00fe, 0xffff, 0xffff}},
   {"three",         {0x0033, 0x0033, 0x0033, 0x0033}},
   {"threequarters", {0x00be, 0x00be, 0xffff, 0xffff}},
   {"threesuperior", {0x00b3, 0x00b3, 0xffff, 0xffff}},
   {"tilde",         {0x02dc, 0x0098, 0x00c4, 245   }},
   {"trademark",     {0x2122, 0x0099, 0xffff, 170   }},
   {"triagdn",       {0x25bc, 0xffff, 0xffff, 0xffff}},
   {"triagrt",       {0x25ba, 0xffff, 0xffff, 0xffff}},
   {"triagup",       {0x25b2, 0xffff, 0xffff, 0xffff}},
   {"traglf",        {0x25c4, 0xffff, 0xffff, 0xffff}},
   {"two",           {0x0032, 0x0032, 0x0032, 0x0032}},
   {"twosuperior",   {0x00b2, 0x00b2, 0xffff, 0xffff}},
   {"u",             {0x0075, 0x0075, 0x0075, 0x0075}},
   {"uacute",        {0x00fa, 0x00fa, 0xffff, 156   }},
   {"ucircumflex",   {0x00fb, 0x00fb, 0xffff, 158   }},
   {"udblacute",     {0x0171, 0xffff, 0xffff, 0xffff}},
   {"udieresis",     {0x00fc, 0x00fc, 0xffff, 159   }},
   {"ugrave",        {0x00f9, 0x00f9, 0xffff, 157   }},
   {"underscore",    {0x005f, 0x005f, 0x005f, 0x005f}},
   {"underscoredbl", {0x2017, 0xffff, 0xffff, 0xffff}},
   {"upblock",       {0x2580, 0xffff, 0xffff, 0xffff}},
   {"uring",         {0x016f, 0xffff, 0xffff, 0xffff}},
   {"v",             {0x0076, 0x0076, 0x0076, 0x0076}},
   {"w",             {0x0077, 0x0077, 0x0077, 0x0077}},
   {"x",             {0x0078, 0x0078, 0x0078, 0x0078}},
   {"y",             {0x0079, 0x0079, 0x0079, 0x0079}},
   {"yacute",        {0x00fd, 0x00fd, 0xffff, 0xffff}},
   {"ydieresis",     {0x00ff, 0x00ff, 0xffff, 216   }},
   {"yen",           {0x00a5, 0x00a5, 0x00a5, 180   }},
   {"z",             {0x007a, 0x007a, 0x007a, 0x007a}},
   {"zacute",        {0x017a, 0xffff, 0xffff, 0xffff}},
   {"zcaron",        {0x017e, 0xffff, 0xffff, 0xffff}},
   {"zdot",          {0x017c, 0xffff, 0xffff, 0xffff}},
   {"zero",          {0x0030, 0x0030, 0x0030, 0x0030}},
};


static const struct encoding SeacEncoding[] = {
   {"A",             {0xf041, 0x0041, 0x0041, 0x0041}},
   {"AE",            {0xf0c6, 0x00c6, 0x00e1, 174   }},
   {"B",             {0xf042, 0x0042, 0x0042, 0x0042}},
   {"C",             {0xf043, 0x0043, 0x0043, 0x0043}},
   {"D",             {0xf044, 0x0044, 0x0044, 0x0044}},
   {"E",             {0xf045, 0x0045, 0x0045, 0x0045}},
   {"F",             {0xf046, 0x0046, 0x0046, 0x0046}},
   {"G",             {0xf047, 0x0047, 0x0047, 0x0047}},
   {"H",             {0xf048, 0x0048, 0x0048, 0x0048}},
   {"I",             {0xf049, 0x0049, 0x0049, 0x0049}},
   {"J",             {0xf04a, 0x004a, 0x004a, 0x004a}},
   {"K",             {0xf04b, 0x004b, 0x004b, 0x004b}},
   {"L",             {0xf04c, 0x004c, 0x004c, 0x004c}},
   {"Lslash",        {0xf141, 0xffff, 0x00e8, 0xffff}},
   {"M",             {0xf04d, 0x004d, 0x004d, 0x004d}},
   {"N",             {0xf04e, 0x004e, 0x004e, 0x004e}},
   {"O",             {0xf04f, 0x004f, 0x004f, 0x004f}},
   {"OE",            {0xf152, 0x008c, 0x00ea, 206   }},
   {"Oslash",        {0xf0d8, 0x00d8, 0x00e9, 175   }},
   {"P",             {0xf050, 0x0050, 0x0050, 0x0050}},
   {"Q",             {0xf051, 0x0051, 0x0051, 0x0051}},
   {"R",             {0xf052, 0x0052, 0x0052, 0x0052}},
   {"S",             {0xf053, 0x0053, 0x0053, 0x0053}},
   {"T",             {0xf054, 0x0054, 0x0054, 0x0054}},
   {"U",             {0xf055, 0x0055, 0x0055, 0x0055}},
   {"V",             {0xf056, 0x0056, 0x0056, 0x0056}},
   {"W",             {0xf057, 0x0057, 0x0057, 0x0057}},
   {"X",             {0xf058, 0x0058, 0x0058, 0x0058}},
   {"Y",             {0xf059, 0x0059, 0x0059, 0x0059}},
   {"Z",             {0xf05a, 0x005a, 0x005a, 0x005a}},
   {"a",             {0xf061, 0x0061, 0x0061, 0x0061}},
   {"acute",         {0xf0b4, 0x00b4, 0x00c2, 171   }},
   {"ae",            {0xf0e6, 0x00e6, 0x00f1, 190   }},
   {"ampersand",     {0xf026, 0x0026, 0x0026, 0x0026}},
   {"asciicircum",   {0xf05e, 0x005e, 0x005e, 0x005e}},
   {"asciitilde",    {0xf07e, 0x007e, 0x007e, 0x007e}},
   {"asterisk",      {0xf02a, 0x002a, 0x002a, 0x002a}},
   {"at",            {0xf040, 0x0040, 0x0040, 0x0040}},
   {"b",             {0xf062, 0x0062, 0x0062, 0x0062}},
   {"backslash",     {0xf05c, 0x005c, 0x005c, 0x005c}},
   {"bar",           {0xf07c, 0x007c, 0x007c, 0x007c}},
   {"braceleft",     {0xf07b, 0x007b, 0x007b, 0x007b}},
   {"braceright",    {0xf07d, 0x007d, 0x007d, 0x007d}},
   {"bracketleft",   {0xf05b, 0x005b, 0x005b, 0x005b}},
   {"bracketright",  {0xf05d, 0x005d, 0x005d, 0x005d}},
   {"breve",         {0xf2d8, 0xffff, 0x00c6, 249   }},
   {"bullet",        {0x2022, 0x0095, 0x00b7, 165   }},
   {"c",             {0xf063, 0x0063, 0x0063, 0x0063}},
   {"caron",         {0xf2c7, 0xffff, 0x00cf, 255   }},
   {"cedilla",       {0xf0b8, 0x00b8, 0x00cb, 252   }},
   {"cent",          {0xf0a2, 0x00a2, 0x00a2, 162   }},
   {"circumflex",    {0xf2c6, 0x0088, 0x00c3, 246   }},
   {"colon",         {0xf03a, 0x003a, 0x003a, 0x003a}},
   {"comma",         {0xf02c, 0x002c, 0x002c, 0x002c}},
   {"currency",      {0xf0a4, 0x00a4, 0x00a8, 219   }},
   {"d",             {0xf064, 0x0064, 0x0064, 0x0064}},
   {"dagger",        {0x2020, 0x0086, 0x00b2, 160   }},
   {"daggerdbl",     {0x2021, 0x0087, 0x00b3, 224   }},
   {"dieresis",      {0xf0a8, 0x00a8, 0x00c8, 172   }},
   {"dollar",        {0xf024, 0x0024, 0x0024, 0x0024}},
   {"dotaccent",     {0xf2d9, 0xffff, 0x00c7, 250   }},
   {"dotlessi",      {0xf131, 0xffff, 0x00f5, 245   }},
   {"e",             {0xf065, 0x0065, 0x0065, 0x0065}},
   {"eight",         {0xf038, 0x0038, 0x0038, 0x0038}},
   {"ellipsis",      {0x2026, 0x0085, 0x00bc, 201   }},
   {"emdash",        {0x2014, 0x0097, 0x00d0, 209   }},
   {"endash",        {0x2013, 0x0096, 0x00b1, 208   }},
   {"equal",         {0xf03d, 0x003d, 0x003d, 0x003d}},
   {"exclam",        {0xf021, 0x0021, 0x0021, 0x0021}},
   {"exclamdown",    {0xf0a1, 0x00a1, 0x00a1, 193   }},
   {"f",             {0xf066, 0x0066, 0x0066, 0x0066}},
   {"fi",            {0xf001, 0xffff, 0x00ae, 222   }},
   {"five",          {0xf035, 0x0035, 0x0035, 0x0035}},
   {"fl",            {0xf002, 0xffff, 0x00af, 223   }},
   {"florin",        {0xf192, 0x0083, 0x00a6, 196   }},
   {"four",          {0xf034, 0x0034, 0x0034, 0x0034}},
   {"fraction",      {0x2215, 0xffff, 0x00a4, 218   }},
   {"g",             {0xf067, 0x0067, 0x0067, 0x0067}},
   {"germandbls",    {0xf0df, 0x00df, 0x00fb, 167   }},
   {"grave",         {0xf060, 0x0060, 0x00c1, 0x0060}},
   {"greater",       {0xf03e, 0x003e, 0x003e, 0x003e}},
   {"guillemotleft", {0xf0ab, 0x00ab, 0x00ab, 199   }},
   {"guillemotright",{0xf0bb, 0x00bb, 0x00bb, 200   }},
   {"guilsinglleft", {0x2039, 0x008b, 0x00ac, 220   }},
   {"guilsinglright",{0x203a, 0x009b, 0x00ad, 221   }},
   {"h",             {0xf068, 0x0068, 0x0068, 0x0068}},
   {"hungarumlaut",  {0xf2dd, 0xffff, 0x00cd, 0xffff}},
   {"hyphen",        {0xf02d, 0x002d, 0x002d, 0x002d}},
   {"i",             {0xf069, 0x0069, 0x0069, 0x0069}},
   {"j",             {0xf06a, 0x006a, 0x006a, 0x006a}},
   {"k",             {0xf06b, 0x006b, 0x006b, 0x006b}},
   {"l",             {0xf06c, 0x006c, 0x006c, 0x006c}},
   {"less",          {0xf03c, 0x003c, 0x003c, 0x003c}},
   {"lslash",        {0xf142, 0xffff, 0x00f8, 0xffff}},
   {"m",             {0xf06d, 0x006d, 0x006d, 0x006d}},
   {"macron",        {0xf0af, 0xffff, 0x00c5, 248   }},
   {"n",             {0xf06e, 0x006e, 0x006e, 0x006e}},
   {"nine",          {0xf039, 0x0039, 0x0039, 0x0039}},
   {"numbersign",    {0xf023, 0x0023, 0x0023, 0x0023}},
   {"o",             {0xf06f, 0x006f, 0x006f, 0x006f}},
   {"oe",            {0xf153, 0x009c, 0x00fa, 207   }},
   {"ogonek",        {0xf2db, 0xffff, 0x00ce, 254   }},
   {"one",           {0xf031, 0x0031, 0x0031, 0x0031}},
   {"ordfeminine",   {0xf0aa, 0x00aa, 0x00e3, 187   }},
   {"ordmasculine",  {0xf0ba, 0x00ba, 0x00eb, 188   }},
   {"oslash",        {0xf0f8, 0x00f8, 0x00f9, 191   }},
   {"p",             {0xf070, 0x0070, 0x0070, 0x0070}},
   {"paragraph",     {0xf0b6, 0x00b6, 0x00b6, 166   }},
   {"parenleft",     {0xf028, 0x0028, 0x0028, 0x0028}},
   {"parenright",    {0xf029, 0x0029, 0x0029, 0x0029}},
   {"percent",       {0xf025, 0x0025, 0x0025, 0x0025}},
   {"period",        {0xf02e, 0x002e, 0x002e, 0x002e}},
   {"periodcentered",{0x2219, 0xffff, 0x00b4, 225   }},
   {"perthousand",   {0x2030, 0x0089, 0x00bd, 228   }},
   {"plus",          {0xf02b, 0x002b, 0x002b, 0x002b}},
   {"q",             {0xf071, 0x0071, 0x0071, 0x0071}},
   {"question",      {0xf03f, 0x003f, 0x003f, 0x003f}},
   {"questiondown",  {0xf0bf, 0x00bf, 0x00bf, 192   }},
   {"quotedbl",      {0xf022, 0x0022, 0x0022, 0x0022}},
   {"quotedblbase",  {0x201e, 0x0084, 0x00b9, 227   }},
   {"quotedblleft",  {0x201c, 0x0093, 0x00aa, 210   }},
   {"quotedblright", {0x201d, 0x0094, 0x00ba, 211   }},
   {"quoteleft",     {0x2018, 0x0091, 0x0060, 212   }},
   {"quoteright",    {0x2019, 0x0092, 0x0027, 213   }},
   {"quotesinglbase",{0x201a, 0x0082, 0x00b8, 226   }},
   {"quotesingle",   {0xf027, 0x0027, 0x00a9, 0x0027}},
   {"r",             {0xf072, 0x0072, 0x0072, 0x0072}},
   {"ring",          {0xf2da, 0xffff, 0x00ca, 251   }},
   {"s",             {0xf073, 0x0073, 0x0073, 0x0073}},
   {"section",       {0xf0a7, 0x00a7, 0x00a7, 164   }},
   {"semicolon",     {0xf03b, 0x003b, 0x003b, 0x003b}},
   {"seven",         {0xf037, 0x0037, 0x0037, 0x0037}},
   {"six",           {0xf036, 0x0036, 0x0036, 0x0036}},
   {"slash",         {0xf02f, 0x002f, 0x002f, 0x002f}},
   {"space",         {0xf020, 0x0020, 0x0020, 0x0020}},
   {"sterling",      {0xf0a3, 0x00a3, 0x00a3, 163   }},
   {"t",             {0xf074, 0x0074, 0x0074, 0x0074}},
   {"three",         {0xf033, 0x0033, 0x0033, 0x0033}},
   {"tilde",         {0xf2dc, 0x0098, 0x00c4, 245   }},
   {"two",           {0xf032, 0x0032, 0x0032, 0x0032}},
   {"u",             {0xf075, 0x0075, 0x0075, 0x0075}},
   {"underscore",    {0xf05f, 0x005f, 0x005f, 0x005f}},
   {"v",             {0xf076, 0x0076, 0x0076, 0x0076}},
   {"w",             {0xf077, 0x0077, 0x0077, 0x0077}},
   {"x",             {0xf078, 0x0078, 0x0078, 0x0078}},
   {"y",             {0xf079, 0x0079, 0x0079, 0x0079}},
   {"yen",           {0xf0a5, 0x00a5, 0x00a5, 180   }},
   {"z",             {0xf07a, 0x007a, 0x007a, 0x007a}},
   {"zero",          {0xf030, 0x0030, 0x0030, 0x0030}},
};


/***** MACROS */
/*-none-*/


/***** STATIC FUNCTIONS */
/*-none-*/


/***** FUNCTIONS */

/***
** Function: AllocEncodingTable
**
** Description:
**   Create a new encoding ADT.
***/
struct encoding *AllocEncodingTable(const USHORT num)
{
   struct encoding *enc;
   if ((enc = Malloc(sizeof(struct encoding)*num))!=NULL)
      memset(enc, NOTDEFINIT, sizeof(struct encoding)*num);
   return enc;
}


/***
** Function: RehashEncodingTable
**
** Description:
**   Prepare an encoding ADT so that entries can be
**   located in it.
***/
void RehashEncodingTable(struct encoding *Encoding, const USHORT num)
{
   USHORT i;
   short j;
   struct encoding entry;

   for (i=1; i<num; i++) {
      j=(short)(i-1);
      entry = Encoding[i];
      while(j>=0 && strcmp(Encoding[j].name, entry.name)>0) {
         memcpy(&Encoding[j+1], &Encoding[j], sizeof(struct encoding));
         j--;
      }
      Encoding[j+1] = entry;
   }
}


/***
** Function: SetEncodingEntry
**
** Description:
**   Set the mapping from a glyph name to character
**   codes for various platforms.
***/
void SetEncodingEntry(struct encoding *Encoding,
                      const USHORT entry,
                      const char *name,
                      const USHORT num,
                      const USHORT *codes)
{
   USHORT i;

   Encoding[entry].name = name;
   for (i=0; i<num; i++) {
      Encoding[entry].codes[i] = codes[i];
   }
}


static int CDECL compare(const void *arg1, const void *arg2)
{
   return strcmp( *((const char **)arg1),
                  ((const struct encoding *)arg2)->name );
}


/***
** Function: LookupPSName
**
** Description:
**   Do a binary search for a postscript name, and return
**   a handle that can be used to look up a the character
**   code for a specific encoding schema.
**
**   If a custom encoding array is used and a glyph is not
**	 encoded by that array, though it is recognized by the
**   StandardEncoding array, then use the SeacEncoding.
**   This is needed in order to support bogus "seac" calls.
***/
struct encoding *LookupPSName(const struct encoding *Encoding,
                              USHORT size,
                              const char *name)
{
#ifndef BSEARCH
   struct encoding *result;
#else
   USHORT low, mid, high;
   short diff;

   /* Initiate. */
   low = 0;
   high = size;
#endif

   /* Use default encoding? */
   if (Encoding==NULL) {
      Encoding = StdEncoding;
      size = sizeof(StdEncoding)/sizeof(StdEncoding[0]);
   }

#ifndef BSEARCH
   result = (struct encoding *)bsearch((char *)&name,
                                       (char *)Encoding, size,
                                       sizeof(struct encoding),
                                       compare);
#else
   do {

	   mid = (low+high)>>1;
	   diff = (short)strcmp(Encoding[mid].name, name);

	   if (diff>0) {
		   high = (low+high)>>1;
	   } else if (diff<0) {
		   low = (short)((low+high+1)>>1);
	   } else if (diff==0) {
		   result =  &Encoding[mid];
	   }

   } while (low<high && result==NULL);
#endif


   if ((result==NULL) &&
	   (Encoding!=SeacEncoding) &&
	   (Encoding!=StdEncoding))
	   result = LookupPSName(SeacEncoding,
							 sizeof(SeacEncoding)/sizeof(SeacEncoding[0]),
							 name);

   return result;
}


/***
** Function: LookupCharCode
**
** Description:
**   look up a the character code for a
**   specific encoding scheme.
***/
USHORT LookupCharCode(const struct encoding *enc, const USHORT type)
{
   USHORT code = 0;

   if (enc)
      code = enc->codes[type];
   return code;
}


/***
** Function: LookupCharName
**
** Description:
**   look up a the character name for a
**   specific encoding scheme.
***/
const char *LookupCharName(const struct encoding *enc)
{
   return enc->name;
}


/***
** Function: LookupNotDef
**
** Description:
**   look up a the .notdef character
***/
const struct encoding *LookupNotDef(void)
{
   return &StdEncoding[0];
}



/***
** Function: DecodeChar
**
** Description:
**   look up an encoding record for a character code in some
**   known encoding.
***/
const struct encoding *DecodeChar(const struct encoding *Encoding,
                                  const USHORT max,
                                  const USHORT type,
                                  const USHORT code)
{
   const struct encoding *enc = &StdEncoding[0];
   USHORT i, high;

   /* Use default encoding? */
   if (Encoding==NULL) {
      Encoding = StdEncoding;
      high = sizeof(StdEncoding)/sizeof(StdEncoding[0]);
   } else {
      high = max;
   }

   for (i=0; i<high; i++) {
      if (Encoding[i].codes[type]==code) {
         enc = &Encoding[i];
         break;
      }
   }

   return enc;
}


/***
** Function: FreeEncoding
**
** Description:
**   Deallocate memory associated to the encoding array.
***/
void FreeEncoding(struct encoding *enc, const USHORT size)
{
   USHORT i;

   if (enc) {
      for (i=0; i<size; i++) {
         if (enc[i].name)
            Free((char *)enc[i].name);
      }
      Free(enc);
   }
}

/***
** Function: LookupFirstEnc
**
** Description:
**   Locate the first encoding for a given glyph.
***/
const struct encoding *LookupFirstEnc(
    const struct encoding *encRoot,
    const int              encSize,
    const struct encoding *encItem
)
{
	/* Back up to the first item with the same glyph name. */
	while (encItem>encRoot && !strcmp(encItem[-1].name, encItem[0].name))
		encItem--;

	return encItem;
}

/***
** Function: LookupNextEnc
**
** Description:
**   Locate the first encoding for a given glyph.
***/
const struct encoding *LookupNextEnc(
    const struct encoding *encRoot,
    const int              encSize,
    const struct encoding *encItem
)
{
	/* Back up to the first item with the same glyph name. */
	if (encItem<(&encRoot[encSize-1]) &&
		 !strcmp(encItem[0].name, encItem[1].name))
		return ++encItem;

	return NULL;
}
