/*++ BUILD Version: 0001    // Increment this if a change has global effects

Copyright (c) 1992  Microsoft Corporation

Module Name:

    port.c

Abstract:

    Contains functions responsible for data collection from the RAS ports.

Created:

    Patrick Y. Ng               12 Aug 93

Revision History

--*/

//
//  Include Files
//

#include <nt.h>
#include <ntrtl.h>
#include <nturtl.h>
#include <ntddser.h>

#include <raserror.h>
#include <malloc.h>
#include <windows.h>
#include <string.h>
#include <wcstr.h>

#include "rasctrs.h" // error message definition
#include "perfmsg.h"
#include "perfutil.h"
#include "dataras.h"
#include "globals.h"
#include "port.h"

#include <rasman.h>
#include <serial.h>
#include <isdn.h>


static HANDLE   ghRasmanLib;             // Handle of RASMAN.DLL

extern VOID SsPrintf ( char *Format, ... );

#define RASMAN_DLL              "rasman.dll"


//
// Function types for the functions in RASMAN.DLL
//

typedef DWORD ( WINAPI *FPRASPORTENUM ) ( LPBYTE, LPWORD, LPWORD );
typedef DWORD ( WINAPI *FPRASGETINFO ) ( HPORT, RASMAN_INFO* );
typedef DWORD ( WINAPI *FPRASPORTGETSTATISTICS ) ( HPORT, LPBYTE, LPWORD );
typedef DWORD ( WINAPI *FPRASINITIALIZE) ();

FPRASPORTENUM                   lpRasPortEnum;
FPRASGETINFO                    lpRasGetInfo;
FPRASPORTGETSTATISTICS          lpRasPortGetStatistics;
FPRASINITIALIZE                 lpRasInitialize;

//
// Pointer to the port table array.
//

static PRAS_PORT_DATA          gpPortDataArray;
static RAS_PORT_STAT           gTotalStat;

static WORD	               gcPorts;
static RASMAN_PORT             *gpPorts = NULL;
static WORD	               gPortEnumSize;

static DWORD                   gTotalConnections;
		
//***
//
// Routine Description:
//
//      It will load rasman.dll and call GetProcAddress to obtain all the
//      necessary RAS functions.
//
// Arguments:
//
//      None.
//
// Return Value:
//
//      ERROR_SUCCESS - Successful.
//      ERROR_CAN_NOT_COMPLETE - Otherwise.
//
//***

LONG InitRasFunctions()
{
    ghRasmanLib = LoadLibrary( RASMAN_DLL );

    // log error if unsuccessful

    if( !ghRasmanLib )
    {
        REPORT_ERROR (RASPERF_OPEN_FILE_DRIVER_ERROR, LOG_USER);

        // this is fatal, if we can't get data then there's no
        // point in continuing.

        return ERROR_CAN_NOT_COMPLETE;

    }

    lpRasInitialize =
	(FPRASPORTENUM) GetProcAddress( ghRasmanLib, "RasInitialize" );

    lpRasPortEnum =
	(FPRASPORTENUM) GetProcAddress( ghRasmanLib, "RasPortEnum" );

    lpRasGetInfo =
	(FPRASGETINFO) GetProcAddress( ghRasmanLib, "RasGetInfo" );

    lpRasPortGetStatistics =
	(FPRASPORTGETSTATISTICS) GetProcAddress( ghRasmanLib, "RasPortGetStatistics" );


    if( !lpRasInitialize || !lpRasPortEnum || !lpRasGetInfo
	        || !lpRasPortGetStatistics || lpRasInitialize() )
    {
        return ERROR_CAN_NOT_COMPLETE;
    }
    else
    {
        return ERROR_SUCCESS;
    }
}


//***
//
// Routine Description:
//
//      This routine will call lpRasPortEnum() and generate an array of port
//      tables which contains all the information for all the ports such as
//      number of bytes transferred, and number of errors, etc.
//
//      The remaining initialization work of gRasPortDataDefinition is also
//      finished here.
//
// Arguments:
//
//      None.
//
// Return Value:
//
//      ERROR_SUCCESS - Successful.
//      ERROR_CAN_NOT_COMPLETE - Otherwise.
//
//***

LONG InitPortInfo()
{
    WORD        Size;
    INT         i;


    if( lpRasPortEnum(NULL, &gPortEnumSize, &gcPorts) != ERROR_BUFFER_TOO_SMALL )
    {
        return ERROR_CAN_NOT_COMPLETE;
    }


    gpPorts = (RASMAN_PORT *) malloc( gPortEnumSize );

    if (!gpPorts)
    {
        return ERROR_CAN_NOT_COMPLETE;
    }


    if (lpRasPortEnum((LPBYTE) gpPorts, &gPortEnumSize, &gcPorts))
    {
        return ERROR_CAN_NOT_COMPLETE;
    }



    //
    // Generate the array of data tables for all the ports, and fill up the
    // name of each port.
    //

    Size = gcPorts * sizeof( RAS_PORT_DATA );

    gpPortDataArray = ( PRAS_PORT_DATA ) malloc( Size );

    if( gpPortDataArray == NULL )
    {
        return ERROR_CAN_NOT_COMPLETE;
    }

    memset( gpPortDataArray, 0, Size );



    //
    // Fill up the names.
    //

    for( i = 0; i < gcPorts; i++ )
    {
        //
        // Note that the names passed to perfmon are in Unicodes.
        //

        MultiByteToWideChar( CP_ACP, MB_PRECOMPOSED, 
                             gpPorts[i].P_PortName, 
                             MAX_PORT_NAME,
                             gpPortDataArray[i].PortName,
                             sizeof(WCHAR) * MAX_PORT_NAME);
    }


    //
    // Finish the initialization of gRasPortDataDefinition.
    //

    gRasPortDataDefinition.RasObjectType.TotalByteLength =
                sizeof( RAS_PORT_DATA_DEFINITION ) +
                gcPorts * ( sizeof( RAS_PORT_INSTANCE_DEFINITION ) +
                           SIZE_OF_RAS_PORT_PERFORMANCE_DATA );

    gRasPortDataDefinition.RasObjectType.NumInstances = gcPorts;

    return ERROR_SUCCESS;
}


VOID ClosePortInfo()
{
    free( gpPortDataArray );

    free( gpPorts );
}


WORD GetNumOfPorts()
{
    return gcPorts;
}


LPWSTR GetInstanceName( INT i )
{
    return (LPWSTR) gpPortDataArray[i].PortName;
}


VOID GetInstanceData( INT Port, PVOID *lppData )
{
    PPERF_COUNTER_BLOCK pPerfCounterBlock;
    PDWORD              pdwCounter;
    PRAS_PORT_STAT      pRasPortStat;


    pPerfCounterBlock = (PERF_COUNTER_BLOCK *) *lppData;

    pPerfCounterBlock->ByteLength = SIZE_OF_RAS_PORT_PERFORMANCE_DATA;

    pRasPortStat = &gpPortDataArray[Port].RasPortStat;


    //
    // Go to end of PerfCounterBlock to get of array of counters
    //

    pdwCounter = (PDWORD) (&pPerfCounterBlock[1]);

    *pdwCounter++ = pRasPortStat->BytesTransmitted;
    *pdwCounter++ = pRasPortStat->BytesReceived;

    *pdwCounter++ = pRasPortStat->FramesTransmitted;
    *pdwCounter++ = pRasPortStat->FramesReceived;

    if (pRasPortStat->BytesTransmittedUncompressed > 0 &&
        pRasPortStat->BytesTransmittedUncompressed > pRasPortStat->BytesTransmittedCompressed)
    {
	*pdwCounter++ = ((pRasPortStat->BytesTransmittedUncompressed - pRasPortStat->BytesTransmittedCompressed) * 100) /
			pRasPortStat->BytesTransmittedUncompressed;
    }
    else
    {
	*pdwCounter++ = (DWORD)0;
    }

    if (pRasPortStat->BytesReceivedUncompressed > 0 &&
        pRasPortStat->BytesReceivedUncompressed > pRasPortStat->BytesReceivedCompressed)
    {
	*pdwCounter++ = ((pRasPortStat->BytesReceivedUncompressed - pRasPortStat->BytesReceivedCompressed) * 100) /
			pRasPortStat->BytesReceivedUncompressed;
    }
    else
    {
	*pdwCounter++ = (DWORD)0;
    }

    *pdwCounter++ = pRasPortStat->CRCErrors;
    *pdwCounter++ = pRasPortStat->TimeoutErrors;
    *pdwCounter++ = pRasPortStat->SerialOverrunErrors;
    *pdwCounter++ = pRasPortStat->AlignmentErrors;
    *pdwCounter++ = pRasPortStat->BufferOverrunErrors;

    *pdwCounter++ = pRasPortStat->TotalErrors;

    *pdwCounter++ = pRasPortStat->BytesTransmitted;
    *pdwCounter++ = pRasPortStat->BytesReceived;

    *pdwCounter++ = pRasPortStat->FramesTransmitted;
    *pdwCounter++ = pRasPortStat->FramesReceived;

    *pdwCounter++ = pRasPortStat->TotalErrors;


    //
    // Update *lppData to the next available byte.
    //

    *lppData = (PVOID) pdwCounter;

}


VOID GetTotalData( PVOID *lppData )
{
    PPERF_COUNTER_BLOCK pPerfCounterBlock;
    PDWORD              pdwCounter;


    pPerfCounterBlock = (PERF_COUNTER_BLOCK *) *lppData;

    pPerfCounterBlock->ByteLength = SIZE_OF_RAS_TOTAL_PERFORMANCE_DATA;


    //
    // Go to end of PerfCounterBlock to get of array of counters
    //

    pdwCounter = (PDWORD) (&pPerfCounterBlock[1]);

    *pdwCounter++ = gTotalStat.BytesTransmitted;
    *pdwCounter++ = gTotalStat.BytesReceived;

    *pdwCounter++ = gTotalStat.FramesTransmitted;
    *pdwCounter++ = gTotalStat.FramesReceived;

    if (gTotalStat.BytesTransmittedUncompressed > 0 &&
        gTotalStat.BytesTransmittedUncompressed > gTotalStat.BytesTransmittedCompressed)
    {
	*pdwCounter++ = ((gTotalStat.BytesTransmittedUncompressed - gTotalStat.BytesTransmittedCompressed) * 100) /
			gTotalStat.BytesTransmittedUncompressed;

    }
    else
    {
	*pdwCounter++ = (DWORD)0;
    }

    if (gTotalStat.BytesReceivedUncompressed > 0 &&
        gTotalStat.BytesReceivedUncompressed > gTotalStat.BytesReceivedCompressed)
    {
	*pdwCounter++ = ((gTotalStat.BytesReceivedUncompressed - gTotalStat.BytesReceivedCompressed) * 100) /
			gTotalStat.BytesReceivedUncompressed;
    }
    else
    {
	*pdwCounter++ = (DWORD)0;
    }

    *pdwCounter++ = gTotalStat.CRCErrors;
    *pdwCounter++ = gTotalStat.TimeoutErrors;
    *pdwCounter++ = gTotalStat.SerialOverrunErrors;
    *pdwCounter++ = gTotalStat.AlignmentErrors;
    *pdwCounter++ = gTotalStat.BufferOverrunErrors;

    *pdwCounter++ = gTotalStat.TotalErrors;

    *pdwCounter++ = gTotalStat.BytesTransmitted;
    *pdwCounter++ = gTotalStat.BytesReceived;

    *pdwCounter++ = gTotalStat.FramesTransmitted;
    *pdwCounter++ = gTotalStat.FramesReceived;

    *pdwCounter++ = gTotalStat.TotalErrors;

    *pdwCounter++ = gTotalConnections;

    //
    // Update *lppData to the next available byte.
    //

    *lppData = (PVOID) pdwCounter;

}


//***
//
// Routine Description:
//
//      This routine will return the number of gTotalStat.Bytes needed for all the
//      objects requested.
//
// Arguments:
//
//      None.
//
// Return Value:
//
//      The number of gTotalStat.Bytes.
//
//***

ULONG GetSpaceNeeded( BOOL IsRasPortObject, BOOL IsRasTotalObject )
{
    ULONG       Space = 0;

    if( IsRasPortObject )
    {
        Space += gRasPortDataDefinition.RasObjectType.TotalByteLength;
    }

    if( IsRasTotalObject )
    {
        Space += gRasTotalDataDefinition.RasObjectType.TotalByteLength;
    }

    return Space;
}


//***
//
// Routine Description:
//
//      This routine will return the number of bytes needed for all the
//      objects requested.
//
// Arguments:
//
//      None.
//
// Return Value:
//
//      The number of bytes.
//
//***

NTSTATUS CollectRasStatistics()
{
    NTSTATUS    status;
    INT         i;

    gTotalConnections = 0;

    //
    // We also initialize the data structure for the total.
    //

    memset( &gTotalStat, 0, sizeof( gTotalStat ) );

    //
    // First we do a lpRasPortEnum to obtain the port connection info.
    //

    status = lpRasPortEnum((LPBYTE) gpPorts, &gPortEnumSize, &gcPorts);

    if( status != ERROR_SUCCESS )
    {
        REPORT_ERROR_DATA (RASPERF_RASPORTENUM_FAILED, LOG_USER,
                &status, sizeof(status));

        return ERROR_CAN_NOT_COMPLETE;
    }


    for( i = 0; i < gcPorts; i++ )
    {
        RASMAN_INFO	RasmanInfo;
        HPORT           hPort;
        WORD            wSize;
        BOOL            IsISDN;
        RAS_STATISTICS  *pStats;
        PRAS_PORT_STAT  pData;


        //
        // First we want to know if the port is open.
        //

	if( gpPorts[i].P_Status != OPEN )
        {
            //
            // Reset the port data and continue with next port.
            //

            memset( &gpPortDataArray[i].RasPortStat,0, sizeof(RAS_PORT_STAT));

            continue;
        }

        hPort = gpPorts[i].P_Handle;


        //
        // Check if the port is connected.
        //

        lpRasGetInfo( hPort, &RasmanInfo );

        if( RasmanInfo.RI_ConnState != CONNECTED )
        {
            //
            // Reset the port data and continue with next port.
            //

            memset( &gpPortDataArray[i].RasPortStat,0, sizeof(RAS_PORT_STAT));

            continue;
        }

        gTotalConnections++;


        //
        // See if it's a ISDN connection.
        //

	if( stricmp( gpPorts[i].P_DeviceType, ISDN_TXT ) == 0 )
        {
            IsISDN = TRUE;
        }
        else
        {
            IsISDN = FALSE;
        }


        //
        // Obtain the statistics for the port.
        //

        if( IsISDN )
        {	
            wSize = sizeof(RAS_STATISTICS) + (NUM_ISDN_STATS * sizeof(ULONG));
        }
        else
        {
            wSize = sizeof(RAS_STATISTICS) +
                        (NUM_RAS_SERIAL_STATS * sizeof(ULONG));
        }

        pStats = (RAS_STATISTICS* )malloc( wSize );

        if (!pStats)
        {
            //
            // If it fails then we should return error.
            //

            status = ERROR_NOT_ENOUGH_MEMORY;

            REPORT_ERROR_DATA (RASPERF_NOT_ENOUGH_MEMORY, LOG_USER,
                &status, sizeof(status));

            return status;
        }

        lpRasPortGetStatistics( hPort, (PVOID)pStats, &wSize );


        //
        // Now store the data in the data array.
        //

        pData = &(gpPortDataArray[i].RasPortStat);


        pData->BytesTransmitted =     pStats->S_Statistics[ BYTES_XMITED ];
        pData->BytesReceived =        pStats->S_Statistics[ BYTES_RCVED ];
        pData->FramesTransmitted =    pStats->S_Statistics[ FRAMES_XMITED ];
        pData->FramesReceived =       pStats->S_Statistics[ FRAMES_RCVED ];
	
	
	if( !IsISDN )
        {
            pData->CRCErrors =            pStats->S_Statistics[ CRC_ERR ];
            pData->TimeoutErrors =        pStats->S_Statistics[ TIMEOUT_ERR ];
            pData->SerialOverrunErrors =  pStats->S_Statistics[ SERIAL_OVERRUN_ERR ];
            pData->AlignmentErrors =      pStats->S_Statistics[ ALIGNMENT_ERR ];
            pData->BufferOverrunErrors =  pStats->S_Statistics[ BUFFER_OVERRUN_ERR ];

            pData->TotalErrors =   pStats->S_Statistics[ CRC_ERR ] +
                                   pStats->S_Statistics[ TIMEOUT_ERR ] +
                                   pStats->S_Statistics[ SERIAL_OVERRUN_ERR ] +
                                   pStats->S_Statistics[ ALIGNMENT_ERR ] +
                                   pStats->S_Statistics[ BUFFER_OVERRUN_ERR ];
        }
        	
			
        pData->BytesTransmittedUncompressed = pStats->S_Statistics[ BYTES_XMITED_UNCOMP ];

        pData->BytesReceivedUncompressed = pStats->S_Statistics[ BYTES_RCVED_UNCOMP ];

        pData->BytesTransmittedCompressed = pStats->S_Statistics[ BYTES_XMITED_COMP ];

        pData->BytesReceivedCompressed = pStats->S_Statistics[ BYTES_RCVED_COMP ];


        //
        // Also update the total data structure
        //

        gTotalStat.BytesTransmitted +=  pData->BytesTransmitted;
        gTotalStat.BytesReceived +=     pData->BytesReceived;
        gTotalStat.FramesTransmitted += pData->FramesTransmitted;
        gTotalStat.FramesReceived +=    pData->FramesReceived;

        gTotalStat.CRCErrors +=           pData->CRCErrors;
        gTotalStat.TimeoutErrors +=       pData->TimeoutErrors;
        gTotalStat.SerialOverrunErrors += pData->SerialOverrunErrors;
        gTotalStat.AlignmentErrors +=     pData->AlignmentErrors;
        gTotalStat.BufferOverrunErrors += pData->BufferOverrunErrors;

        gTotalStat.BytesTransmittedUncompressed += pData->BytesTransmittedUncompressed;
        gTotalStat.BytesReceivedUncompressed +=    pData->BytesReceivedUncompressed;
        gTotalStat.BytesTransmittedCompressed +=   pData->BytesTransmittedCompressed;
        gTotalStat.BytesReceivedCompressed +=      pData->BytesReceivedCompressed;

        gTotalStat.TotalErrors +=      pData->TotalErrors;

        free( pStats );
    }

    return ERROR_SUCCESS;
}


