/****************************************************************************
           Microsoft RPC Version 1`1
         Copyright Microsoft Corp. 1992
                Hello Example

    FILE:   helloc.c
    USAGE:   client    -n network_address
          -p protocol_sequence
          -e endpoint
          -o options
          -u uuid

    PURPOSE:   Client side of RPC distributed application
    FUNCTIONS:   main() - binds to server and calls remote procedure
    COMMENTS:
    This distributed application prints a string such as "hello, world"
    on the server. The client manages its connection to the server.
    The client uses the binding handle hello_IfHandle defined in the
    file hello.h.

****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <rpc.h>       // RPC API functions, types
#include "hello.h"     // header file generated by MIDL compiler

void Usage(char * pszProgramName)
{
    fprintf(stderr, "Usage:  %s\n", pszProgramName);
    fprintf(stderr, " -p protocol_sequence\n");
    fprintf(stderr, " -n network_address\n");
    fprintf(stderr, " -e endpoint\n");
    fprintf(stderr, " -o options\n");
    fprintf(stderr, " -u uuid\n");
    fprintf(stderr, " -s string\n");
    exit(1);
}

int _CRTAPI1
main (argc, argv)
    int argc;
    char *argv[];
{
    RPC_STATUS status;             // returned by RPC API function
    unsigned char * pszUuid = "12345678-1234-1234-1234-123456789ABC";
    unsigned char * pszProtocolSequence = "ncacn_np";
    unsigned char * pszNetworkAddress   = NULL;
    unsigned char * pszEndpoint    = "\\pipe\\hello";
    unsigned char * pszOptions          = NULL;
    unsigned char * pszStringBinding   = NULL;
    unsigned char * pszString      = "hello, world";
    int i;

    // allow the user to override settings with command line switches
    for (i = 1; i < argc; i++) {
        if ((*argv[i] == '-') || (*argv[i] == '/')) {
            switch (tolower(*(argv[i]+1))) {
            case 'p':  // protocol sequence
                pszProtocolSequence = argv[++i];
                break;
            case 'n':  // network address
                pszNetworkAddress = argv[++i];
                break;
            case 'e':
                pszEndpoint = argv[++i];
                break;
            case 'o':
                pszOptions = argv[++i];
                break;
            case 'u':
                pszUuid = argv[++i];
                break;
            case 's':
                pszString = argv[++i];
                break;
            case 'h':
            case '?':
            default:
                Usage(argv[0]);
            }
        } else {
            Usage(argv[0]);
        }
    }


    // Use a convenience function to concatenate the elements of
    // the string binding into the proper sequence.

    status = RpcStringBindingCompose(pszUuid,
                                     pszProtocolSequence,
                                     pszNetworkAddress,
                                     pszEndpoint,
                                     pszOptions,
                                     &pszStringBinding);

    if (status) {
        printf("RpcStringBindingCompose returned 0x%x\n", status);
        exit(2);
    }
    printf("pszStringBinding = %s\n", pszStringBinding);



    //
    // Set the binding handle that will be used to bind to the server.
    //

    status = RpcBindingFromStringBinding(pszStringBinding,
                &hello_IfHandle);
    if (status) {
        printf("RpcBindingFromStringBinding returned 0x%x\n", status);
        exit(2);
    }


    //
    // Tell RPC to do the security thing.
    //

    status = RpcBindingSetAuthInfo(
                    hello_IfHandle,
                    "frank",
                    RPC_C_AUTHN_LEVEL_CONNECT,
                    10,
                    NULL,
                    RPC_C_AUTHZ_NAME );

    if ( status ) {
        printf("RpcBindingSetAuthInfo returned %ld\n", status);
        exit(2);
    }


    //
    // Do the actual RPC calls to the server.
    //

    printf("  print the string '%s' on the server\n", pszString);

    RpcTryExcept {
        int i;
        for ( i=0; i<100 ; i++ ) {
            HelloProc(pszString);    // make call with user message
        }
        Shutdown();              // shut down the server side
    } RpcExcept(1) {
        printf("Runtime library reported an exception 0x%lx\n",
               RpcExceptionCode());
    } RpcEndExcept



    // The calls to the remote procedures are complete.
    // Free the string and the binding handle

    status = RpcStringFree(&pszStringBinding);   // remote calls done; unbind
    if (status) {
        printf("RpcStringFree returned 0x%x\n", status);
        exit(2);
    }

    status = RpcBindingFree(&hello_IfHandle);  // remote calls done; unbind
    if (status) {
        printf("RpcBindingFree returned 0x%x\n", status);
        exit(2);
    }


    return 0;

}

// ====================================================================
//                MIDL allocate and free
// ====================================================================


void __RPC_FAR * __RPC_API MIDL_user_allocate(size_t len)
{
    return(malloc(len));
}

void __RPC_API MIDL_user_free(void __RPC_FAR * ptr)
{
    free(ptr);
}

/* end file helloc.c */
