/*++

Copyright (c) 1992-1993  Microsoft Corporation

Module Name:

    ExpDir.h

Abstract:

    This file contains structures, function prototypes, and definitions
    for the replicator export directory worker routines.

Author:

    John Rogers (JohnRo) 08-Jan-1992

Environment:

    User Mode - Win32
    Portable to any flat, 32-bit environment.  (Uses Win32 typedefs.)
    Requires ANSI C extensions: slash-slash comments, long external names.

Notes:

    You must include LmCons.h before this file.

Revision History:

    08-Jan-1992 JohnRo
        Created.
    09-Jan-1992 JohnRo
        Added EXPORT_DIR_SECTION_NAME equate.
        Added ExportDir{Read,Write}ConfigData.
    20-Jan-1992 JohnRo
        Netr prototypes are now generated by MIDL and put in repl.h.
    23-Jan-1992 JohnRo
        Clarify units for time parameters.
        Changed EXPORT_DIR_SECTION_NAME.
        Changed ExportDirBuildApiRecord's interface.
        Added ExportDirIsApiRecordValid().
        Added ExportDirAllocApiRecord().
    28-Jan-1992 JohnRo
        Changed ExportDirAllocApiRecords() to allow arrays.
        Added ExportDirConfigDataExists() and ExportDirDeleteConfigData().
        Changed to use LPTSTR etc.
    09-Feb-1992 JohnRo
        Added ExportDir{Start,Stop}Repl routines.
    13-Feb-1992 JohnRo
        Moved section name equates to ConfName.h.
    15-Mar-1992 JohnRo
        Update registry with new values.
        Improve support for setinfo info levels.
    23-Mar-1992 JohnRo
        Added ExportDirReadMasterList().
    30-Jul-1992 JohnRo
        Help PC-LINT understand ExportDirIsLevelValid().
    29-Sep-1992 JohnRo
        RAID 7962: Repl APIs in wrong role kill svc.
        Also fix remote repl admin.
    01-Dec-1992 JohnRo
        RAID 3844: remote NetReplSetInfo uses local machine type.
    13-Jan-1993 JohnRo
        RAID 7053: locked trees added to pulse msg.  (Actually fix all
        kinds of remote lock handling.)
    13-Apr-1993 JohnRo
        RAID 3107: locking directory over the net gives network path not found.

--*/


#ifndef _EXPDIR_
#define _EXPDIR_


#include <netlib.h>             // IN_RANGE().


//
// Export dir helper routines and macros:
//

// Allocate one or more API records for an export directory.  Callable whether
// or not the replicator service is started.  (Used in getinfo stub, getinfo
// worker, and enum stub.)
NET_API_STATUS
ExportDirAllocApiRecords (
    IN DWORD Level,
    IN DWORD EntryCount,
    OUT LPBYTE * BufPtr,
    IN OUT LPBYTE *StringLocation       // Points just past top of data.
    );

// Build API record for an export directory.  Callable whether or not
// the replicator service is started.  (Used in getinfo and enum.)
NET_API_STATUS
ExportDirBuildApiRecord (
    IN DWORD Level,
    IN LPTSTR DirName,
    IN DWORD Integrity,
    IN DWORD Extent,
    IN DWORD LockCount,
    IN DWORD TimeOfFirstLock,           // Seconds since 1970.
    OUT LPVOID Buffer,
    IN OUT LPBYTE *StringLocation       // Points just past top of data.
    );

// Tells whether or not config data for this directory exists.
// Callable even if the replicator service is not started.
BOOL
ExportDirConfigDataExists (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName                   // Caller must check dir name syntax.
    );

NET_API_STATUS
ExportDirConfigSetInfo (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName,
    IN DWORD Level,
    IN LPVOID Buf,
    OUT LPDWORD ParmError OPTIONAL
    );

// Delete config data for this directory.
// Returns NERR_UnknownDevDir if config data doesn't exist for this dir.
// Callable even if the replicator service is not started.
NET_API_STATUS
ExportDirDeleteConfigData (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName                   // Caller must check dir name syntax.
    );

// Callable even if the replicator service is not started.
NET_API_STATUS
ExportDirEnumApiRecords(
    IN LPTSTR UncServerName OPTIONAL,
    IN DWORD Level,
    OUT LPBYTE * BufPtr,
    IN DWORD PrefMaxSize,
    OUT LPDWORD EntriesRead,
    OUT LPDWORD TotalEntries
    );

// Callable even if the replicator service is not started.
NET_API_STATUS
ExportDirFixUserLockFiles(
    IN LPCTSTR ExportPath,  // Must include drive letter.
    IN LPCTSTR DirName,
    IN DWORD   LockCount
    );

// Callable even if the replicator service is not started.
NET_API_STATUS
ExportDirGetApiRecord (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName,
    IN DWORD Level,
    OUT LPBYTE * BufPtr
    );

BOOL
ExportDirIsApiRecordValid (
    IN DWORD Level,
    IN LPVOID ApiRecord,
    OUT LPDWORD ParmError OPTIONAL
    );

// Callable whether or not service is started.
// If service is running, assume caller has lock (any kind) on RMGlobalListLock.
NET_API_STATUS
ExportDirLockInRegistry(
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName
    );

// Parse config data for a single export directory.  Callable whether or not
// the replicator service is started.  (This function is used by routines
// in Repl/Common/ExpConf.c and by the NetReplExportDirEnum routine.)
NET_API_STATUS
ExportDirParseConfigData (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR ValueString,
    OUT LPDWORD IntegrityPtr,
    OUT LPDWORD ExtentPtr,
    OUT LPDWORD LockCountPtr,
    OUT LPDWORD LockTimePtr             // Seconds since 1970.
    );

// Read config data for a single export directory.  Callable whether or not
// the replicator service is started.
NET_API_STATUS
ExportDirReadConfigData (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName,
    OUT LPDWORD IntegrityPtr,
    OUT LPDWORD ExtentPtr,
    OUT LPDWORD LockCountPtr,
    OUT LPDWORD LockTimePtr             // Seconds since 1970.
    );

// Read export dirs into service's master list.
// Only callable locally, when service is started.
// This also fixes the USERLOCK.* file(s) to match the lock count in registry.
NET_API_STATUS
ExportDirReadMasterList(
    VOID
    );

// Read specified export dir into service's master list.
NET_API_STATUS
ExportDirGetRegistryValues(
    IN LPTSTR ServiceRegPath OPTIONAL,
    IN LPTSTR TargetName
    );

// Start replicating (exporting).
// Called when service starts or user does NetReplSetInfo() and changes role.
NET_API_STATUS
ExportDirStartRepl (
    IN BOOL ServiceIsStarting
    );

// Stop replicating (exporting).
// Called when service stops or user does NetReplSetInfo() and changes role.
NET_API_STATUS
ExportDirStopRepl (
    VOID
    );

// Callable whether or not service is started.
// If service is running, assume caller has lock (any kind) on RMGlobalListLock.
NET_API_STATUS
ExportDirUnlockInRegistry(
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName,
    IN DWORD UnlockForce
    );

// Write config data for a single export directory.  Callable whether or not
// the replicator service is started.
NET_API_STATUS
ExportDirWriteConfigData (
    IN LPTSTR UncServerName OPTIONAL,
    IN LPTSTR DirName,
    IN DWORD Integrity,
    IN DWORD Extent,
    IN DWORD LockCount,
    IN DWORD LockTime                   // Seconds since 1970.
    );

// BOOL
// ExportDirIsLevelValid(
//     IN DWORD Level,          // Info level
//     IN BOOL SetInfo          // Are setinfo levels allowed?
//     );
//
#define ExportDirIsLevelValid(Level,SetInfo) \
    /*lint -e506 */  /* don't complain about constant values here */ \
    ( ( (Level) <= 2 ) \
      || ( (SetInfo) && (IN_RANGE((Level), 1000, 1001)) ) ) \
    /*lint +e506 */ \


#endif // _EXPDIR_
