/*++

Copyright (c) 1992 Microsoft Corporation

Module Name:

    registry.c

Abstract:

    This module contains registry access routines for the NT server
    service.

Author:

    Chuck Lenzmeier (chuckl) 19-Mar-1992

Revision History:

--*/

#include "srvsvcp.h"
#include "ssdata.h"
#include "ssreg.h"
#include "srvconfg.h"

#include <tstr.h>

#include <netevent.h>

//
// Simple MIN and MAX macros.  Watch out for side effects!
//

#define MIN(a,b) ( ((a) < (b)) ? (a) : (b) )
#define MAX(a,b) ( ((a) < (b)) ? (b) : (a) )

#define MAX_INTEGER_STRING 32

#define MB * 1024 * 1024
#define INF 0xffffffff

//
// ( u, n )
// u is units to allocate for every n megabytes on a medium server.
//

#define CONFIG_TUPLE_SIZE 2
typedef struct {
    DWORD initworkitems[CONFIG_TUPLE_SIZE];
    DWORD maxworkitems[CONFIG_TUPLE_SIZE];
    DWORD rawworkitems[CONFIG_TUPLE_SIZE];
    DWORD maxrawworkitems[CONFIG_TUPLE_SIZE];
    DWORD maxpagedmemoryusage[CONFIG_TUPLE_SIZE];
    DWORD maxnonpagedmemoryusage[CONFIG_TUPLE_SIZE];
    DWORD nonblockingthreads[CONFIG_TUPLE_SIZE];
    DWORD blockingthreads[CONFIG_TUPLE_SIZE];
} CONFIG_SERVER_TABLE;

CONFIG_SERVER_TABLE MedSrvCfgTbl = {

//
// ** NOTE ** : If the second column is greater than 4, then
// you will need to add a check to make sure the statistic
// did not drop to zero.
//
//                          Units / MB
// Parameter
// ---------
//
/* initworkitems          */ { 1    , 4  },
/* maxworkitems           */ { 4    , 1  },
/* rawworkitems           */ { 1    , 4  },
/* maxrawworkitems        */ { 4    , 1  },
/* maxpagedmemoryusage    */ { 1    , 1  },
/* maxnonpagedmemoryusage */ { 1    , 8  },
/* nonblockingthreads     */ { 1    , 8  },
/* blockingthreads        */ { 1    , 8  }

};

//
// Minimum configuration system size is 8MB.  Anything lower treated
// as if 8 MB.
//

#define MIN_SYSTEM_SIZE                 8

//
// A medium server reaches its max at 32M.  A small server at 16M.
//

#define MAX_SMALL_SIZE                  16
#define MAX_MEDIUM_SIZE                 32

//
// Min and Max for number of threads.
//

#define SRV_MIN_NONBLOCKING_COUNT        2
#define SRV_MAX_NONBLOCKING_COUNT       16

#define SRV_MIN_BLOCKING_COUNT           2
#define SRV_MAX_BLOCKING_COUNT           8

//
// Note that the user limit is always -1 (unlimited).  Autodisconnect
// always defaults to 15 minutes.
//

//
// Forward declarations
//

NTSTATUS
BindToTransport (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    );

NTSTATUS
EnumerateStickyShare (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    );

NET_API_STATUS
FillStickyShareInfo(
        IN PSRVSVC_SHARE_ENUM_INFO ShareEnumInfo,
        IN PSHARE_INFO_502 Shi502
        );

NTSTATUS
GetSdFromRegistry(
        IN PWSTR ValueName,
        IN ULONG ValueType,
        IN PVOID ValueData,
        IN ULONG ValueLength,
        IN PVOID Context,
        IN PVOID EntryContext
        );

BOOLEAN
GetStickyShareInfo (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN PUNICODE_STRING RemarkString,
    IN PUNICODE_STRING PathString,
    OUT PSHARE_INFO_502 shi502
    );

LONG
LoadParameters (
    PWCH Path
    );

LONG
LoadSizeParameter (
    VOID
    );

NTSTATUS
RecreateStickyShare (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    );

NTSTATUS
SaveSdToRegistry(
    IN PSECURITY_DESCRIPTOR SecurityDescriptor,
    IN PWSTR ShareName
    );

NTSTATUS
SetSizeParameters (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    );

NTSTATUS
SetStickyParameter (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    );


ULONG
SsRtlQueryEnvironmentLength (
    IN PVOID Environment
    )
{
    PWCH p;
    ULONG length;

    p = Environment;
    ASSERT( p != NULL );

    //
    // The environment variable block consists of zero or more null
    // terminated ASCII strings.  Each string is of the form:
    //
    //      name=value
    //
    // where the null termination is after the value.
    //

    while ( *p ) {
        while ( *p ) {
            p++;
        }
        p++;
    }
    p++;
    length = (PCHAR)p - (PCHAR)Environment;

    //
    // Return accumulated length.
    //

    return length;
}


VOID
SsAddParameterToRegistry (
    PFIELD_DESCRIPTOR Field,
    PVOID Value
    )
{
    NTSTATUS status;
    PWCH valueName;
    DWORD valueType;
    LPBYTE valuePtr;
    DWORD valueDataLength;

    //
    // The value name is the parameter name and the value data is the
    // parameter value.
    //

    valueName = Field->FieldName;

    switch ( Field->FieldType ) {

    case BOOLEAN_FIELD:
    case DWORD_FIELD:
        valueType = REG_DWORD;
        valuePtr = Value;
        valueDataLength = sizeof(DWORD);
        break;

    case LPSTR_FIELD:
        valueType = REG_SZ;
        valuePtr = *(LPBYTE *)Value;
        if ( valuePtr != NULL ) {
            valueDataLength = SIZE_WSTR( (PWCH)valuePtr );
        } else {
            valueDataLength = 0;
        }
        break;

    }

    //
    // Set the value into the Parameters key.
    //

    status = RtlWriteRegistryValue(
                RTL_REGISTRY_SERVICES,
                PARAMETERS_REGISTRY_PATH,
                valueName,
                valueType,
                valuePtr,
                valueDataLength
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddParameterToRegistry: SetValue failed: %lx; "
                        "parameter %ws won't stick\n", status, valueName ));
        }
    }

    return;

} // SsAddParameterToRegistry


VOID
SsAddShareToRegistry (
    IN PSHARE_INFO_2 ShareInfo2,
    IN PSECURITY_DESCRIPTOR SecurityDescriptor OPTIONAL
    )
{
    NTSTATUS status;
    PWCH valueName;
    PVOID environment;
    UNICODE_STRING nameString;
    UNICODE_STRING valueString;
    WCHAR integerString[MAX_INTEGER_STRING + 1];
    ULONG environmentLength;

    //
    // Build the value name and data strings.  The value name is the
    // share name (netname), while the value data is share information
    // in REG_MULTI_SZ format.  To build the value data, we use the
    // RTL environment routines.
    //

    valueName = ShareInfo2->shi2_netname;

    status = RtlCreateEnvironment( FALSE, &environment );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: CreateEnvironment failed: %lx; "
                        "share %ws won't stick\n", status, valueName ));
        }
        goto exit1;
    }

    RtlInitUnicodeString( &nameString, PATH_VARIABLE_NAME );
    RtlInitUnicodeString( &valueString, ShareInfo2->shi2_path );

    status = RtlSetEnvironmentVariable(
                &environment,
                &nameString,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: SetEnvironment failed: %lx; "
                        "share %s won't stick\n", status, valueName ));
        }
        goto exit2;
    }

    if ( ShareInfo2->shi2_remark != NULL ) {

        RtlInitUnicodeString( &nameString, REMARK_VARIABLE_NAME );
        RtlInitUnicodeString( &valueString, ShareInfo2->shi2_remark );

        status = RtlSetEnvironmentVariable(
                    &environment,
                    &nameString,
                    &valueString
                    );
        if ( !NT_SUCCESS(status) ) {
            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "SsAddShareToRegistry: SetEnvironment failed: %lx; "
                            "share %s won't stick\n", status, valueName ));
            }
            goto exit2;
        }

    }

    RtlInitUnicodeString( &nameString, TYPE_VARIABLE_NAME );
    valueString.Buffer = integerString;
    valueString.MaximumLength = (MAX_INTEGER_STRING + 1) * sizeof(WCHAR);
    status = RtlIntegerToUnicodeString(
                ShareInfo2->shi2_type,
                10,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: IntegerToUnicode failed: %lx; "
                        "share %ws won't stick\n", status, valueName ));
        }
        goto exit2;
    }
    status = RtlSetEnvironmentVariable(
                &environment,
                &nameString,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: SetEnvironment failed: %lx; "
                        "share %s won't stick\n", status, valueName ));
        }
        goto exit2;
    }

    RtlInitUnicodeString( &nameString, PERMISSIONS_VARIABLE_NAME );
    valueString.Buffer = integerString;
    valueString.MaximumLength = (MAX_INTEGER_STRING + 1) * sizeof(WCHAR);
    status = RtlIntegerToUnicodeString(
                ShareInfo2->shi2_permissions,
                10,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: IntegerToUnicode failed: %lx; "
                        "share %ws won't stick\n", status, valueName ));
        }
        goto exit2;
    }
    status = RtlSetEnvironmentVariable(
                &environment,
                &nameString,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: SetEnvironment failed: %lx; "
                        "share %s won't stick\n", status, valueName ));
        }
        goto exit2;
    }

    RtlInitUnicodeString( &nameString, MAXUSES_VARIABLE_NAME );
    valueString.Buffer = integerString;
    valueString.MaximumLength = (MAX_INTEGER_STRING + 1) * sizeof(WCHAR);
    status = RtlIntegerToUnicodeString(
                ShareInfo2->shi2_max_uses,
                10,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: IntegerToUnicode failed: %lx; "
                        "share %ws won't stick\n", status, valueName ));
        }
        goto exit2;
    }
    status = RtlSetEnvironmentVariable(
                &environment,
                &nameString,
                &valueString
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: SetEnvironment failed: %lx; "
                        "share %s won't stick\n", status, valueName ));
        }
        goto exit2;
    }

    //
    // Set the value into the Shares key.
    //

    environmentLength = SsRtlQueryEnvironmentLength( environment );
    status = RtlWriteRegistryValue(
                RTL_REGISTRY_SERVICES,
                SHARES_REGISTRY_PATH,
                valueName,
                REG_MULTI_SZ,
                (LPBYTE)environment,
                environmentLength
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsAddShareToRegistry: SetValue failed: %lx; share %ws "
                        "won't stick\n", status, valueName ));
        }
    }

    //
    // Save the file security descriptor
    //

    if ( ARGUMENT_PRESENT( SecurityDescriptor ) ) {

        status = SaveSdToRegistry(
                        SecurityDescriptor,
                        valueName
                        );

        if ( !NT_SUCCESS(status) ) {
            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "SsAddShareToRegistry: SaveSd failed: %lx; share %ws\n"
                            , status, valueName ));
            }
        }
    }

exit2:
    RtlDestroyEnvironment( environment );

exit1:

    return;

} // SsAddShareToRegistry


NET_API_STATUS
SsBindToTransports (
    VOID
    )

/*++

Routine Description:

    Reads the registry to bind to specified transports.

Arguments:

    None.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/

{
    NTSTATUS status;
    PRTL_QUERY_REGISTRY_TABLE queryTable;
    ULONG numberOfBindings = 0;

    //
    // Ask the RTL to call us back for each subvalue in the MULTI_SZ
    // value \LanmanServer\Linkage\Bind.
    //

    queryTable = MIDL_user_allocate( sizeof(RTL_QUERY_REGISTRY_TABLE) * 2 );

    if ( queryTable != NULL ) {

        queryTable[0].QueryRoutine = BindToTransport;
        queryTable[0].Flags = 0;
        queryTable[0].Name = BIND_VALUE_NAME;
        queryTable[0].EntryContext = NULL;
        queryTable[0].DefaultType = REG_NONE;
        queryTable[0].DefaultData = NULL;
        queryTable[0].DefaultLength = 0;

        queryTable[1].QueryRoutine = NULL;
        queryTable[1].Flags = 0;
        queryTable[1].Name = NULL;

        status = RtlQueryRegistryValues(
                    RTL_REGISTRY_SERVICES,
                    LINKAGE_REGISTRY_PATH,
                    queryTable,
                    &numberOfBindings,
                    NULL
                    );

        MIDL_user_free( queryTable );

    }

    //
    // If the above failed to bind to any transports, fail to start the
    // server.
    //

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsBindToTransports: RtlQueryRegistryValues failed: "
                        "%lx\n", status ));
        }
        return RtlNtStatusToDosError( status );
    }

    if ( numberOfBindings == 0 ) {

        SsLogEvent(
            EVENT_SRV_NO_TRANSPORTS_BOUND,
            0,
            NULL,
            NO_ERROR
            );

        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsBindToTransports: no bindings created\n" ));
        }
        return ERROR_INVALID_PARAMETER;    // !!! Need better error
    }

    return NO_ERROR;

} // SsBindToTransports

NET_API_STATUS
SsCheckRegistry (
    VOID
    )

/*++

Routine Description:

    This function verifies that the keys used by the server exist.

Arguments:

    None.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/

{
    NTSTATUS status;
    LPWSTR subStrings[1];

    //
    // Verify the existence of the main server service key.  If this
    // fails, the server service fails to start.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                SERVER_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        subStrings[0] = SERVER_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_KEY_NOT_FOUND,
            1,
            subStrings,
            RtlNtStatusToDosError( status )
            );

        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsCheckRegistry: main key doesn't exist\n" ));
        }
        return ERROR_INVALID_PARAMETER; // !!! Need better error

    }

    //
    // Verify the existence of the Linkage subkey.  If this fails, the
    // server service fails to start.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                LINKAGE_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        subStrings[0] = LINKAGE_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_KEY_NOT_FOUND,
            1,
            subStrings,
            RtlNtStatusToDosError( status )
            );

        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsCheckRegistry: Linkage subkey doesn't exist\n" ));
        }
        return ERROR_INVALID_PARAMETER; // !!! Need better error

    }

    //
    // If the Parameters subkey doesn't exist, create it.  If it can't
    // be created, fail to start the server.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                PARAMETERS_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        status = RtlCreateRegistryKey(
                    RTL_REGISTRY_SERVICES,
                    PARAMETERS_REGISTRY_PATH
                    );

        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = PARAMETERS_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_KEY_NOT_CREATED,
                1,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(INITIALIZATION) {
                SS_PRINT(( "SsCheckRegistry: Can't create Parameters subkey: "
                            "%lx\n", status ));
            }
            return RtlNtStatusToDosError( status );

        }

    }

    //
    // If the AutotunedParameters subkey doesn't exist, create it.  If
    // it can't be created, fail to start the server.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                AUTOTUNED_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        status = RtlCreateRegistryKey(
                    RTL_REGISTRY_SERVICES,
                    AUTOTUNED_REGISTRY_PATH
                    );

        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = AUTOTUNED_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_KEY_NOT_CREATED,
                1,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(INITIALIZATION) {
                SS_PRINT(( "SsCheckRegistry: Can't create AutotunedParameters "
                            "subkey: %lx\n", status ));
            }
            return RtlNtStatusToDosError( status );

        }

    }

    //
    // If the Shares subkey doesn't exist, create it.  If it can't be
    // created, fail to start the server.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                SHARES_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        status = RtlCreateRegistryKey(
                    RTL_REGISTRY_SERVICES,
                    SHARES_REGISTRY_PATH
                    );

        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_KEY_NOT_CREATED,
                1,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(INITIALIZATION) {
                SS_PRINT(( "SsCheckRegistry: Can't create Shares subkey: "
                            "%lx\n", status ));
            }
            return RtlNtStatusToDosError( status );

        }

    }

    //
    // If the Shares Security  subkey doesn't exist, create it.  If it
    // can't be created, fail to start the server.
    //

    status = RtlCheckRegistryKey(
                RTL_REGISTRY_SERVICES,
                SHARES_SECURITY_REGISTRY_PATH
                );

    if ( !NT_SUCCESS(status) ) {

        status = RtlCreateRegistryKey(
                    RTL_REGISTRY_SERVICES,
                    SHARES_SECURITY_REGISTRY_PATH
                    );

        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = SHARES_SECURITY_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_KEY_NOT_CREATED,
                1,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(INITIALIZATION) {
                SS_PRINT(( "SsCheckRegistry: Can't create Shares Security subkey: "
                            "%lx\n", status ));
            }
            return RtlNtStatusToDosError( status );

        }

    }

    //
    // All keys successfully checked.
    //

    return NO_ERROR;

} // SsCheckRegistry


NET_API_STATUS
SsEnumerateStickyShares (
    IN OUT PSRVSVC_SHARE_ENUM_INFO ShareEnumInfo
    )

/*++

Routine Description:

    Reads the registry to find and return sticky shares.

Arguments:

    ShareEnumInfo - points to a structure that contains the parameters
        to the NetShareEnumSticky call.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/

{
    NTSTATUS status;
    PRTL_QUERY_REGISTRY_TABLE queryTable;

    ShareEnumInfo->TotalBytesNeeded = 0;
    ShareEnumInfo->TotalEntries = 0;
    ShareEnumInfo->EntriesRead = 0;

    //
    // Initialize the reserve fields.  This tells the callback routine,
    // how many times it has been called.
    //

    ShareEnumInfo->ShareEnumIndex = 0;
    ShareEnumInfo->StartOfFixedData = (PCHAR)ShareEnumInfo->OutputBuffer;
    ShareEnumInfo->EndOfVariableData = (PCHAR)ShareEnumInfo->OutputBuffer +
                            ShareEnumInfo->OutputBufferLength;

    //
    // We need to align it since we deal with unicode strings.
    //

    ShareEnumInfo->EndOfVariableData =
                    (PCHAR)((ULONG)ShareEnumInfo->EndOfVariableData & ~1);

    //
    // Ask the RTL to call us back for each value in the Shares key.
    //

    queryTable = MIDL_user_allocate( sizeof(RTL_QUERY_REGISTRY_TABLE) * 2 );

    if ( queryTable != NULL ) {

        queryTable[0].QueryRoutine = EnumerateStickyShare;
        queryTable[0].Flags = RTL_QUERY_REGISTRY_NOEXPAND;
        queryTable[0].Name = NULL;
        queryTable[0].EntryContext = NULL;
        queryTable[0].DefaultType = REG_NONE;
        queryTable[0].DefaultData = NULL;
        queryTable[0].DefaultLength = 0;

        queryTable[1].QueryRoutine = NULL;
        queryTable[1].Flags = 0;
        queryTable[1].Name = NULL;

        status = RtlQueryRegistryValues(
                    RTL_REGISTRY_SERVICES | RTL_REGISTRY_OPTIONAL,
                    SHARES_REGISTRY_PATH,
                    queryTable,
                    ShareEnumInfo,
                    NULL
                    );

        MIDL_user_free( queryTable );

    }

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsEnumerateStickyShares: RtlQueryRegistryValues "
                        "failed: %lx\n", status ));
        }
        return RtlNtStatusToDosError( status );
    }

    return NO_ERROR;

} // SsEnumerateStickyShares


NET_API_STATUS
SsLoadConfigurationParameters (
    VOID
    )

/*++

Routine Description:

    Reads the registry to get server configuration parameters.  These
    server parameters must be set before the server FSP has been
    started.

Arguments:

    None.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/

{
    LONG error;

    //
    // Get the basic Size parameter, then load autotuned parameters,
    // then load manually set parameters.  This ordering allows manual
    // settings to override autotuning.
    //

    error = LoadSizeParameter( );

    if ( error == NO_ERROR ) {

        error = LoadParameters( AUTOTUNED_REGISTRY_PATH );

        if ( error == NO_ERROR ) {

            error = LoadParameters( PARAMETERS_REGISTRY_PATH );

        }

    }

    //
    // The copy read to MDL read switchover must occur at or below the
    // SMB buffer size.
    //

    SsData.ServerInfo598.sv598_mdlreadswitchover =
        MIN(
            SsData.ServerInfo598.sv598_mdlreadswitchover,
            SsData.ServerInfo599.sv599_sizreqbuf);

    //
    // Override parameters that cannot be set on WinNT (vs. NTAS).
    //
    // The server itself also performs most of these overrides, in case
    // somebody figures out the FSCTL that changes parameters.  We also
    // override in the service in order to keep the service's view
    // consistent with the server's.  If you make any changes here, also
    // make them in srv\svcsrv.c.
    //
    if ( SsData.ServerInfo598.sv598_producttype == NtProductWinNt ) {

        //
        // On WinNT, the maximum value of certain parameters is fixed at
        // build time.  These include: concurrent users, SMB buffers,
        // and threads.
        //

#define MINIMIZE(_param,_max) _param = MIN( _param, _max );

        MINIMIZE( SsData.ServerInfo102.sv102_users, MAX_USERS_WKSTA );
        MINIMIZE( SsData.ServerInfo599.sv599_maxworkitems, MAX_MAXWORKITEMS_WKSTA );
        MINIMIZE( SsData.ServerInfo598.sv598_nonblockingthreads, MAX_NONBLOCKINGTHREADS_WKSTA );
        MINIMIZE( SsData.ServerInfo598.sv598_blockingthreads, MAX_BLOCKINGTHREADS_WKSTA );
        MINIMIZE( SsData.ServerInfo598.sv598_criticalthreads, MAX_CRITICALTHREADS_WKSTA );

        //
        // On WinNT, we do not cache closed RFCBs.
        //

        SsData.ServerInfo598.sv598_cachedopenlimit = 0;

        //
        // Sharing of redirected drives is not allowed on WinNT.
        //

        SsData.ServerInfo599.sv599_enablesharednetdrives = FALSE;

    }

    return error;

} // SsLoadConfigurationParameters


NET_API_STATUS
SsRecreateStickyShares (
    VOID
    )

/*++

Routine Description:

    Reads the registry to find and create sticky shares.

Arguments:

    None.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/

{
    NTSTATUS status;
    PRTL_QUERY_REGISTRY_TABLE queryTable;

    //
    // Ask the RTL to call us back for each value in the Shares key.
    //

    queryTable = MIDL_user_allocate( sizeof(RTL_QUERY_REGISTRY_TABLE) * 2 );

    if ( queryTable != NULL ) {

        queryTable[0].QueryRoutine = RecreateStickyShare;
        queryTable[0].Flags = RTL_QUERY_REGISTRY_NOEXPAND;
        queryTable[0].Name = NULL;
        queryTable[0].EntryContext = NULL;
        queryTable[0].DefaultType = REG_NONE;
        queryTable[0].DefaultData = NULL;
        queryTable[0].DefaultLength = 0;

        queryTable[1].QueryRoutine = NULL;
        queryTable[1].Flags = 0;
        queryTable[1].Name = NULL;

        status = RtlQueryRegistryValues(
                    RTL_REGISTRY_SERVICES | RTL_REGISTRY_OPTIONAL,
                    SHARES_REGISTRY_PATH,
                    queryTable,
                    NULL,
                    NULL
                    );

        MIDL_user_free( queryTable );

    }

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "SsRecreateStickyShares: RtlQueryRegistryValues "
                        "failed: %lx\n", status ));
        }
        return RtlNtStatusToDosError( status );
    }

    return NO_ERROR;

} // SsRecreateStickyShares


NET_API_STATUS
SsRemoveShareFromRegistry (
    LPWSTR NetName
    )
{
    NET_API_STATUS error = NO_ERROR;
    NTSTATUS status;
    PWCH valueName;

    //
    // The value name is the share name.  Remove that value from the
    // Shares key.
    //

    valueName = NetName;

    //
    // Delete the share security
    //

    status = RtlDeleteRegistryValue(
                RTL_REGISTRY_SERVICES,
                SHARES_SECURITY_REGISTRY_PATH,
                valueName
                );

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsRemoveShareFromRegistry: Delete Security value failed: %lx; "
                        "share %ws will return\n", status, valueName ));
        }
    }

    //
    // Delete the share
    //

    status = RtlDeleteRegistryValue(
                RTL_REGISTRY_SERVICES,
                SHARES_REGISTRY_PATH,
                valueName
                );
    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SsRemoveShareFromRegistry: DeleteValue failed: %lx; "
                        "share %ws will return\n", status, valueName ));
        }

        error = RtlNtStatusToDosError( status );
    }

    return error;

} // SsRemoveShareFromRegistry


NTSTATUS
BindToTransport (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    )
{
    NTSTATUS status;
    NET_API_STATUS error;
    PULONG numberOfBindings = Context;
    SERVER_TRANSPORT_INFO_0 svti0;
    LPWSTR subStrings[2];

    ValueName, ValueLength, EntryContext;

    //
    // The value type must be REG_SZ (translated from REG_MULTI_SZ by
    // the RTL).
    //

    if ( ValueType != REG_SZ ) {

        subStrings[0] = ValueName;
        subStrings[1] = LINKAGE_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            NO_ERROR
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "BindToTransports: skipping invalid value %ws\n",
                        ValueName ));
        }
        return STATUS_SUCCESS;

    }

    //
    // The value data is the name of the transport device object.
    //

    svti0.svti0_transportname = ValueData;
    svti0.svti0_transportaddress = NULL;
    svti0.svti0_transportaddresslength = 0;

    //
    // Bind to the transport.
    //

    IF_DEBUG(INITIALIZATION) {
        SS_PRINT(( "SsBindToTransports: binding to transport %ws\n",
                    ValueData ));
    }

    error = NetrServerTransportAdd( NULL, 0, &svti0 );

    if ( error != NO_ERROR ) {

        DWORD eventId;

        //
        // In general, we do not fail server startup just because we
        // failed to bind to a single transport.  Instead, we keep a
        // count of the number of bindings we have, and in
        // SsBindToTransports, after attempting all transports listed in
        // the registry, only if no bindings were made do we fail server
        // startup.
        //
        // The exception to the rule is that we do not start the server
        // service if we get a duplicate name error for any transport.
        // This is considered to be a "user error".
        //

        IF_DEBUG(INITIALIZATION_ERRORS) {
            SS_PRINT(( "SsBindToTransports: failed to bind to %ws: "
                        "%ld\n", ValueData, error ));
        }

        eventId = EVENT_SRV_CANT_BIND_TO_TRANSPORT;
        status = STATUS_SUCCESS;    // assume not duplicate name

        if ( error == ERROR_DUP_NAME ) {
            eventId = EVENT_SRV_CANT_BIND_DUP_NAME;
            status = STATUS_DUPLICATE_NAME;
            *numberOfBindings = 0;
        }

        subStrings[0] = ValueData;
        SsLogEvent(
            eventId,
            1,
            subStrings,
            error
            );

        return status;

    }

    (*numberOfBindings)++;
    return STATUS_SUCCESS;

} // BindToTransport


NTSTATUS
EnumerateStickyShare (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    )

/*++

Routine Description:

    Callback routine for SsEnumerateStickyShare.  Routine will get information
    on share and fill in the output buffer.

Arguments:

    ValueName - Name of the share
    ValueType - Value type of the share name.
    ValueData - Data associated with the ValueName.
    Context - Pointer to our enum information structure.

Return Value:

    NET_API_STATUS - success/failure of the operation.

--*/
{

    NET_API_STATUS error;
    SHARE_INFO_502 shi502;
    UNICODE_STRING pathString;
    UNICODE_STRING remarkString;
    PSRVSVC_SHARE_ENUM_INFO enumInfo = (PSRVSVC_SHARE_ENUM_INFO) Context;

    ValueLength, EntryContext;

    remarkString.Buffer = NULL;
    pathString.Buffer = NULL;

    if ( GetStickyShareInfo(
                        ValueName,
                        ValueType,
                        ValueData,
                        &remarkString,
                        &pathString,
                        &shi502
                        ) ) {

        //
        // Do the actual add of the share.
        //

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "EnumerateStickyShares: adding share %ws\n", ValueName ));
        }

        shi502.shi502_remark = remarkString.Buffer;
        shi502.shi502_path = pathString.Buffer;

        //
        // Skip until we have the right share to resume from
        //

        if ( (enumInfo->TotalEntries == 0) &&
             (enumInfo->ShareEnumIndex < enumInfo->ResumeHandle) ) {

            enumInfo->ShareEnumIndex++;

        } else {

            enumInfo->TotalEntries++;
            error = FillStickyShareInfo( enumInfo, &shi502 );

            if ( error != NO_ERROR ) {

                IF_DEBUG(REGISTRY) {
                    SS_PRINT(( "EnumerateStickyShares: failed to add share "
                                "%ws = %wZ: %ld\n", ValueName, &pathString, error ));
                }
            } else {
                enumInfo->EntriesRead++;
                enumInfo->ResumeHandle++;
            }
        }

        //
        // free buffers allocated by GetStickyShareInfo
        //

        if ( remarkString.Buffer != NULL ) {
            RtlFreeUnicodeString( &remarkString );
        }

        if ( pathString.Buffer != NULL ) {
            RtlFreeUnicodeString( &pathString );
        }

        if ( shi502.shi502_security_descriptor != NULL ) {
            MIDL_user_free( shi502.shi502_security_descriptor );
        }

    }

    return STATUS_SUCCESS;

} // EnumerateStickyShare


NTSTATUS
GetSdFromRegistry(
        IN PWSTR ValueName,
        IN ULONG ValueType,
        IN PVOID ValueData,
        IN ULONG ValueLength,
        IN PVOID Context,
        IN PVOID EntryContext
        )

{
    NTSTATUS status = STATUS_SUCCESS;
    PSECURITY_DESCRIPTOR fileSD = NULL;
    PSHARE_INFO_502 shi502 = (PSHARE_INFO_502) Context;
    LPWSTR subStrings[1];

    EntryContext, ValueName, ValueType;

    if ( ValueLength > 0 ) {

        fileSD = MIDL_user_allocate( ValueLength );

        if ( fileSD == NULL) {
            status = STATUS_INSUFFICIENT_RESOURCES;
        } else {

            RtlCopyMemory(
                    fileSD,
                    ValueData,
                    ValueLength
                    );

            if ( !RtlValidSecurityDescriptor( fileSD ) ) {

                subStrings[0] = ValueName;
                SsLogEvent(
                    EVENT_SRV_INVALID_SD,
                    1,
                    subStrings,
                    RtlNtStatusToDosError( status )
                    );

                MIDL_user_free( fileSD );
                status = STATUS_INVALID_SECURITY_DESCR;
            }
        }
    }

    shi502->shi502_security_descriptor = fileSD;
    return(status);

} // GetSdFromRegistry


BOOLEAN
GetStickyShareInfo (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    OUT PUNICODE_STRING RemarkString,
    OUT PUNICODE_STRING PathString,
    OUT PSHARE_INFO_502 shi502
    )

/*++

Routine Description:

    Gets share information from the registry.

Arguments:

    ValueName - Name of the share
    ValueType - Value type of the share name.
    ValueData - Data associated with the ValueName.
    RemarkString - Upon return, points to a unicode string containing the
        user remark for this share.
    PathString - Upon return, points to a unicode string containing the
        path for this share.
    shi502 - Upon return, points to a unicode string containing a
        SHARE_INFO_502 structure.

Return Value:

    TRUE, if share information successfully retrieved.
    FALSE, otherwise.

--*/

{

    NTSTATUS status;
    UNICODE_STRING variableNameString;
    WCHAR integerStringBuffer[35];
    UNICODE_STRING unicodeString;
    LPWSTR subStrings[2];

    PathString->Buffer = NULL;
    RemarkString->Buffer = NULL;

    shi502->shi502_security_descriptor = NULL;
    shi502->shi502_path = NULL;
    shi502->shi502_remark = NULL;
    shi502->shi502_reserved = 0;

    //
    // Because the NT server doesn't support share-level security, the
    // password is always NULL.
    //

    shi502->shi502_passwd = NULL;

    //
    // The value type must be REG_MULTI_SZ, and the value name must not
    // be null.
    //

    if ( (ValueType != REG_MULTI_SZ) ||
         (wcslen(ValueName) == 0) ) {

        subStrings[0] = ValueName;
        subStrings[1] = SHARES_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            NO_ERROR
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "GetStickyShareInfo: skipping invalid value %ws\n",
                        ValueName ));
        }
        goto errorexit;

    }

    //
    // The share name is the value name.  The value data describes the
    // rest of the information about the share.
    //

    shi502->shi502_netname = ValueName;

    //
    // The REG_MULTI_SZ format is the same as that used for storing
    // environment variables.  Find known share parameters in the data.
    //
    // Get the share path.  It must be present.
    //

    RtlInitUnicodeString( &variableNameString, PATH_VARIABLE_NAME );

    PathString->MaximumLength = 0;
    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                PathString
                );
    if ( status != STATUS_BUFFER_TOO_SMALL ) {

        //
        // The path is not specified.  Ignore this share.
        //

        subStrings[0] = ValueName;
        subStrings[1] = SHARES_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            RtlNtStatusToDosError( status )
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "GetStickyShareInfo: No path; ignoring share.\n" ));
        }
        goto errorexit;

    }

    PathString->MaximumLength = (USHORT)(PathString->Length + sizeof(WCHAR));
    PathString->Buffer = MIDL_user_allocate( PathString->MaximumLength );

    if ( PathString->Buffer == NULL ) {

        //
        // No space for path.  Ignore this share.
        //

        subStrings[0] = ValueName;
        subStrings[1] = SHARES_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            ERROR_NOT_ENOUGH_MEMORY
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "GetStickyShareInfo: MIDL_user_allocate failed; ignoring "
                        "share.\n" ));
        }
        goto errorexit;

    }

    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                PathString
                );
    if ( !NT_SUCCESS(status) ) {

        //
        // Huh?  The second attempt failed.  Ignore this share.
        //

        subStrings[0] = ValueName;
        subStrings[1] = SHARES_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            RtlNtStatusToDosError( status )
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "GetStickyShareInfo: Second query failed!  Ignoring "
                        "share.\n" ));
        }
        goto errorexit;

    }

    //
    // Get the remark.  It may be omitted.
    //

    RtlInitUnicodeString( &variableNameString, REMARK_VARIABLE_NAME );

    RemarkString->MaximumLength = 0;
    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                RemarkString
                );
    if ( status == STATUS_BUFFER_TOO_SMALL ) {

        RemarkString->MaximumLength =
                    (USHORT)(RemarkString->Length + sizeof(WCHAR));
        RemarkString->Buffer =
                    MIDL_user_allocate( RemarkString->MaximumLength );
        if ( RemarkString->Buffer == NULL ) {

            //
            // No space for remark.  Ignore this share.
            //

            subStrings[0] = ValueName;
            subStrings[1] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                ERROR_NOT_ENOUGH_MEMORY
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "GetStickyShareInfo: MIDL_user_allocate failed; ignoring "
                            "share.\n" ));
            }
            goto errorexit;

        }

        status = RtlQueryEnvironmentVariable_U(
                    ValueData,
                    &variableNameString,
                    RemarkString
                    );
        if ( !NT_SUCCESS(status) ) {

            //
            // Huh?  The second attempt failed.  Ignore this share.
            //

            subStrings[0] = ValueName;
            subStrings[1] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "GetStickyShareInfo: Second query failed!  "
                            "Ignoring share.\n" ));
            }
            goto errorexit;

        }

    }

    //
    // Get the share type.  It may be omitted.
    //

    RtlInitUnicodeString( &variableNameString, TYPE_VARIABLE_NAME );

    unicodeString.Buffer = integerStringBuffer;
    unicodeString.MaximumLength = 35;
    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                &unicodeString
                );
    if ( !NT_SUCCESS(status) ) {

        shi502->shi502_type = STYPE_DISKTREE;

    } else {

        status = RtlUnicodeStringToInteger(
                    &unicodeString,
                    0,
                    &shi502->shi502_type
                    );
        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = ValueName;
            subStrings[1] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "GetStickyShareInfo: UnicodeToInteger failed: "
                            "%lx\n", status ));
            }
            goto errorexit;

        }

    }

    //
    // Get the share permissions.  It may be omitted.
    //

    RtlInitUnicodeString( &variableNameString, PERMISSIONS_VARIABLE_NAME );

    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                &unicodeString
                );
    if ( !NT_SUCCESS(status) ) {

        shi502->shi502_permissions = 0;

    } else {

        status = RtlUnicodeStringToInteger(
                    &unicodeString,
                    0,
                    &shi502->shi502_permissions
                    );
        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = ValueName;
            subStrings[1] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "GetStickyShareInfo: UnicodeToInteger failed: "
                            "%lx\n", status ));
            }
            goto errorexit;

        }

    }

    //
    // Get the maximum number of uses allowed.  It may be omitted.
    //

    RtlInitUnicodeString( &variableNameString, MAXUSES_VARIABLE_NAME );

    status = RtlQueryEnvironmentVariable_U(
                ValueData,
                &variableNameString,
                &unicodeString
                );
    if ( !NT_SUCCESS(status) ) {

        shi502->shi502_max_uses = (DWORD)SHI_USES_UNLIMITED;

    } else {

        status = RtlUnicodeStringToInteger(
                    &unicodeString,
                    0,
                    &shi502->shi502_max_uses
                    );
        if ( !NT_SUCCESS(status) ) {

            subStrings[0] = ValueName;
            subStrings[1] = SHARES_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                RtlNtStatusToDosError( status )
                );

            goto errorexit;

        }

    }

    {
        //
        // Get the Share file security descriptor
        //

        RTL_QUERY_REGISTRY_TABLE shareQueryTable[2];

        //
        // Fill up the query table
        //

        shareQueryTable[0].QueryRoutine = GetSdFromRegistry;
        shareQueryTable[0].Flags = RTL_QUERY_REGISTRY_REQUIRED;
        shareQueryTable[0].Name = shi502->shi502_netname;
        shareQueryTable[0].EntryContext = NULL;
        shareQueryTable[0].DefaultType = REG_NONE;

        shareQueryTable[1].QueryRoutine = NULL;
        shareQueryTable[1].Flags = 0;
        shareQueryTable[1].Name = NULL;


        status = RtlQueryRegistryValues(
                                RTL_REGISTRY_SERVICES | RTL_REGISTRY_OPTIONAL,
                                SHARES_SECURITY_REGISTRY_PATH,
                                shareQueryTable,
                                shi502,
                                NULL
                                );

        if ( !NT_SUCCESS( status) &&
             ( status != STATUS_OBJECT_NAME_NOT_FOUND ) ) {
            ASSERT(0);
            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "GetStickyShareInfo: Get file SD: "
                            "%lx\n", status ));
            }
            goto errorexit;
        }
    }

    return TRUE;

errorexit:

    if ( RemarkString->Buffer != NULL ) {
        RtlFreeUnicodeString( RemarkString );
    }

    if ( PathString->Buffer != NULL ) {
        RtlFreeUnicodeString( PathString );
    }

    if ( shi502->shi502_security_descriptor != NULL ) {
        MIDL_user_free( shi502->shi502_security_descriptor );
    }

    return FALSE;

} // GetStickyShareInfo


LONG
LoadParameters (
    PWCH Path
    )

/*++

Routine Description:

    Reads the registry to get server parameters.

Arguments:

    Path - PARAMETERS_REGISTRY_PATH or AUTOTUNED_REGISTRY_PATH

Return Value:

    LONG - success/failure of the operation.

--*/

{
    NTSTATUS status;
    PRTL_QUERY_REGISTRY_TABLE queryTable;
    ULONG numberOfBindings = 0;

    //
    // Ask the RTL to call us back for each value in the appropriate
    // key.
    //

    queryTable = MIDL_user_allocate( sizeof(RTL_QUERY_REGISTRY_TABLE) * 2 );

    if ( queryTable != NULL ) {

        queryTable[0].QueryRoutine = SetStickyParameter;
        queryTable[0].Flags = RTL_QUERY_REGISTRY_NOEXPAND;
        queryTable[0].Name = NULL;
        queryTable[0].EntryContext = NULL;
        queryTable[0].DefaultType = REG_NONE;
        queryTable[0].DefaultData = NULL;
        queryTable[0].DefaultLength = 0;

        queryTable[1].QueryRoutine = NULL;
        queryTable[1].Flags = 0;
        queryTable[1].Name = NULL;

        status = RtlQueryRegistryValues(
                    RTL_REGISTRY_SERVICES | RTL_REGISTRY_OPTIONAL,
                    Path,
                    queryTable,
                    Path,               // Context for SetStickyParameter
                    NULL
                    );

        MIDL_user_free( queryTable );

    }

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "LoadParameters: RtlQueryRegistryValues failed: "
                        "%lx\n", status ));
        }
        return RtlNtStatusToDosError( status );
    }

    return NO_ERROR;

} // LoadParameters


LONG
LoadSizeParameter (
    VOID
    )

/*++

Routine Description:

    Reads the registry to get the basic server Size parameter.

Arguments:

    None.

Return Value:

    LONG - success/failure of the operation.

--*/

{
    NTSTATUS status;
    PRTL_QUERY_REGISTRY_TABLE queryTable;
    ULONG numberOfBindings = 0;

    //
    // Ask the RTL to call us back if the Size parameter exists.
    //

    queryTable = MIDL_user_allocate( sizeof(RTL_QUERY_REGISTRY_TABLE) * 2 );

    if ( queryTable != NULL ) {

        queryTable[0].QueryRoutine = SetSizeParameters;
        queryTable[0].Flags = RTL_QUERY_REGISTRY_NOEXPAND;
        queryTable[0].Name = SIZE_VALUE_NAME;
        queryTable[0].EntryContext = NULL;
        queryTable[0].DefaultType = REG_NONE;
        queryTable[0].DefaultData = NULL;
        queryTable[0].DefaultLength = 0;

        queryTable[1].QueryRoutine = NULL;
        queryTable[1].Flags = 0;
        queryTable[1].Name = NULL;

        status = RtlQueryRegistryValues(
                    RTL_REGISTRY_SERVICES | RTL_REGISTRY_OPTIONAL,
                    PARAMETERS_REGISTRY_PATH,
                    queryTable,
                    NULL,
                    NULL
                    );

        MIDL_user_free( queryTable );

    }

    if ( !NT_SUCCESS(status) ) {
        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "LoadSizeParameter: RtlQueryRegistryValues failed: "
                        "%lx\n", status ));
        }
        return RtlNtStatusToDosError( status );
    }

    return NO_ERROR;

} // LoadSizeParameter


NTSTATUS
RecreateStickyShare (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    )
{

    NET_API_STATUS error;
    SHARE_INFO_502 shi502;
    SHARE_INFO shareInfo;
    UNICODE_STRING pathString;
    UNICODE_STRING remarkString;

    ValueLength, Context, EntryContext;

    remarkString.Buffer = NULL;
    pathString.Buffer = NULL;

    if ( GetStickyShareInfo(
                        ValueName,
                        ValueType,
                        ValueData,
                        &remarkString,
                        &pathString,
                        &shi502
                        ) ) {

        //
        // Do the actual add of the share.
        //

        IF_DEBUG(INITIALIZATION) {
            SS_PRINT(( "RecreateStickyShares: adding share %ws\n", ValueName ));
        }

        shi502.shi502_remark = remarkString.Buffer;
        shi502.shi502_path = pathString.Buffer;

        shareInfo.ShareInfo502 = (LPSHARE_INFO_502_I)&shi502;
        error = NetrShareAdd( NULL, 502, &shareInfo, NULL );

        if ( error != NO_ERROR ) {

            IF_DEBUG(INITIALIZATION_ERRORS) {
                SS_PRINT(( "RecreateStickyShares: failed to add share "
                            "%ws = %wZ: %ld\n", ValueName, &pathString, error ));
            }
        }

        //
        // free buffers allocated by GetStickyShareInfo
        //

        if ( remarkString.Buffer != NULL ) {
            RtlFreeUnicodeString( &remarkString );
        }

        if ( pathString.Buffer != NULL ) {
            RtlFreeUnicodeString( &pathString );
        }

        if ( shi502.shi502_security_descriptor != NULL ) {
            MIDL_user_free( shi502.shi502_security_descriptor );
        }

    }

    return NO_ERROR;

} // RecreateStickyShare


NTSTATUS
SaveSdToRegistry(
    IN PSECURITY_DESCRIPTOR SecurityDescriptor,
    IN PWSTR ShareName
    )
/*++

Routine Description:

    Stores the share file security descriptor in the registry.

Arguments:

    SecurityDescriptor - Points to a self-relative security descriptor
        describing the access rights for files under this share.

    ShareName - Points to a string containing the share name under
        which the SD is to be stored.

Return Value:

    Status of the operation.

--*/
{
    NTSTATUS status;

    //
    // Store the security descriptor
    //

    ULONG fileSDLength;

    if ( !RtlValidSecurityDescriptor( SecurityDescriptor ) ) {

        status = STATUS_INVALID_SECURITY_DESCR;

    } else {

        fileSDLength = RtlLengthSecurityDescriptor( SecurityDescriptor );

        status = RtlWriteRegistryValue(
                    RTL_REGISTRY_SERVICES,
                    SHARES_SECURITY_REGISTRY_PATH,
                    ShareName,
                    REG_BINARY,
                    (LPBYTE)SecurityDescriptor,
                    fileSDLength
                    );

    }

    return status;

} // SaveSdToRegistry


NTSTATUS
SetSizeParameters (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    )
{
    NT_PRODUCT_TYPE productType;
    DWORD size;

    LPWSTR subStrings[2];

    ValueLength, Context, EntryContext;

    //
    // Get the product type.
    //

    if ( !RtlGetNtProductType( &productType ) ) {
        productType = NtProductWinNt;
    }

    SsData.ServerInfo598.sv598_producttype = productType;

    //
    // Make sure that we got called for the right value.
    //

    ASSERT( _wcsicmp( ValueName, SIZE_VALUE_NAME ) == 0 );

    //
    // The Size value must be a DWORD, and must be in the following
    // range:
    //
    //      0 -> use defaults
    //      1 -> small server (minimize memory usage)
    //      2 -> medium server (balance)
    //      3 -> large server (maximize connections)
    //

    if ( ValueType == REG_DWORD ) {
        ASSERT( ValueLength == sizeof(DWORD) );
        size = *(LPDWORD)ValueData;
    }

    if ( (ValueType != REG_DWORD) || (size > 3) ) {

        subStrings[0] = ValueName;
        subStrings[1] = PARAMETERS_REGISTRY_PATH;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            NO_ERROR
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SetSizeParameters: skipping invalid value "
                        "%ws\n", ValueName ));
        }
        return STATUS_SUCCESS;

    }

    SsData.ServerInfo598.sv598_serversize = size;

    //
    // Set appropriate fields based on the product type (Windows NT or
    // Advanced Server) and the selected Size.  Note that a Size of 0
    // doesn't change any of the defaults.
    //
    // Note that the user limit is always -1 (unlimited).  Autodisconnect
    // always defaults to 15 minutes.
    //

    if ( size != 0 ) {

        SYSTEM_BASIC_INFORMATION basicInfo;
        NTSTATUS status;
        ULONG noOfMb;
        ULONG factor;
        ULONG asFactor;

        //
        // Get system memory size.
        //

        status = NtQuerySystemInformation(
                                    SystemBasicInformation,
                                    &basicInfo,
                                    sizeof( SYSTEM_BASIC_INFORMATION ),
                                    NULL
                                    );


        if ( status != STATUS_SUCCESS ) {

            subStrings[0] = ValueName;
            subStrings[1] = PARAMETERS_REGISTRY_PATH;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                NO_ERROR
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "SetSizeParameters: NtQuerySystemInfo failed %x\n",
                            status ));
            }
            return STATUS_SUCCESS;

        } else {

            //
            // Note that we first divide the page size by 512 in order to
            // allow for physical memory sizes above 2^32-1.  With this
            // calculation, we can handle up to two terabytes of physical
            // memory.  The calculation assumes that the page size is at
            // least 512, and is not very accurate if the page size is not
            // a power of 2 (very unlikely).
            //

            ASSERT( basicInfo.PageSize >= 512 );

            noOfMb = (((basicInfo.PageSize / 512) *
                      basicInfo.NumberOfPhysicalPages) +
                      (1 MB / 512 - 1)) / (1 MB / 512);

            //
            // Minimum is 8 MB
            //

            noOfMb = MAX( MIN_SYSTEM_SIZE, noOfMb );

            //
            // Set the maximum for the different sizes
            //

            if ( size == 1 ) {
                noOfMb = MIN( noOfMb, MAX_SMALL_SIZE );
            } else if ( size == 2 ) {
                noOfMb = MIN( noOfMb, MAX_MEDIUM_SIZE );
            }
        }

        //
        // If small, assume the system size is half of the real one.
        // This should give us half the paramater values of a medium server.
        // If large, double it.  Also set the free connection count.
        //

        if ( size == 1 ) {

            //
            // Small
            //

            factor = (noOfMb + 1) / 2;

            SsData.ServerInfo599.sv599_minfreeconnections = 2;
            SsData.ServerInfo599.sv599_maxfreeconnections = 2;

        } else if ( size == 2 ) {

            //
            // Balanced
            //

            factor = noOfMb;

            SsData.ServerInfo599.sv599_minfreeconnections = 2;
            SsData.ServerInfo599.sv599_maxfreeconnections = 4;

        } else {

            //
            // Large
            //

            factor = noOfMb * 2;

            SsData.ServerInfo599.sv599_minfreeconnections = 4;
            SsData.ServerInfo599.sv599_maxfreeconnections = 8;
        }

        //
        // If this is an Advanced Server with at least 24M, some
        // parameter will need to be even bigger.
        //

        asFactor = 1;
        if ( (productType != NtProductWinNt) && (noOfMb >= 24) ) asFactor = 2;

        //
        // Now set the values for a medium server with this much memory.
        //

        SsData.ServerInfo599.sv599_maxworkitems =
                        MedSrvCfgTbl.maxworkitems[0] * factor * asFactor /
                        MedSrvCfgTbl.maxworkitems[1];

        SsData.ServerInfo599.sv599_initworkitems =
                        MedSrvCfgTbl.initworkitems[0] * factor * asFactor /
                        MedSrvCfgTbl.initworkitems[1];

        SsData.ServerInfo599.sv599_rawworkitems =
                        MedSrvCfgTbl.rawworkitems[0] * factor /
                        MedSrvCfgTbl.rawworkitems[1];

        SsData.ServerInfo598.sv598_maxrawworkitems =
                        MedSrvCfgTbl.maxrawworkitems[0] * factor * asFactor /
                        MedSrvCfgTbl.maxrawworkitems[1];

        SsData.ServerInfo599.sv599_maxworkitems =
            MIN( SsData.ServerInfo599.sv599_maxworkitems, MAX_MAXWORKITEMS );
        SsData.ServerInfo599.sv599_initworkitems =
            MIN( SsData.ServerInfo599.sv599_initworkitems, MAX_INITWORKITEMS/4 );
        SsData.ServerInfo599.sv599_rawworkitems =
            MIN( SsData.ServerInfo599.sv599_rawworkitems, MAX_RAWWORKITEMS );
        SsData.ServerInfo598.sv598_maxrawworkitems =
            MIN( SsData.ServerInfo598.sv598_maxrawworkitems, MAX_MAXRAWWORKITEMS );

        if ( (productType != NtProductWinNt) || (size == 3) ) {
            SsData.ServerInfo599.sv599_maxpagedmemoryusage = INF;
            SsData.ServerInfo599.sv599_maxnonpagedmemoryusage = INF;
        } else {
            SsData.ServerInfo599.sv599_maxpagedmemoryusage =
                            MedSrvCfgTbl.maxpagedmemoryusage[0] * factor /
                            MedSrvCfgTbl.maxpagedmemoryusage[1] MB;

            SsData.ServerInfo599.sv599_maxpagedmemoryusage =
                MAX( SsData.ServerInfo599.sv599_maxpagedmemoryusage,
                     MIN_MAXPAGEDMEMORYUSAGE);

            SsData.ServerInfo599.sv599_maxnonpagedmemoryusage =
                            MedSrvCfgTbl.maxnonpagedmemoryusage[0] * factor /
                            MedSrvCfgTbl.maxnonpagedmemoryusage[1] MB;

            SsData.ServerInfo599.sv599_maxnonpagedmemoryusage =
                MAX( SsData.ServerInfo599.sv599_maxnonpagedmemoryusage,
                     MIN_MAXNONPAGEDMEMORYUSAGE);
        }

        //
        // Set the thread counts.
        //

        SsData.ServerInfo598.sv598_nonblockingthreads =
                    MedSrvCfgTbl.nonblockingthreads[0] * factor * asFactor /
                    MedSrvCfgTbl.nonblockingthreads[1];

        SsData.ServerInfo598.sv598_nonblockingthreads =
            MAX( SsData.ServerInfo598.sv598_nonblockingthreads,
                 SRV_MIN_NONBLOCKING_COUNT );
        SsData.ServerInfo598.sv598_nonblockingthreads =
            MIN( SsData.ServerInfo598.sv598_nonblockingthreads,
                 SRV_MAX_NONBLOCKING_COUNT );

        SsData.ServerInfo598.sv598_blockingthreads =
                    MedSrvCfgTbl.blockingthreads[0] * factor * asFactor /
                    MedSrvCfgTbl.blockingthreads[1];

        SsData.ServerInfo598.sv598_blockingthreads =
            MAX( SsData.ServerInfo598.sv598_blockingthreads,
                 SRV_MIN_BLOCKING_COUNT );
        SsData.ServerInfo598.sv598_blockingthreads =
            MIN( SsData.ServerInfo598.sv598_blockingthreads,
                 SRV_MAX_BLOCKING_COUNT );

    }

    return STATUS_SUCCESS;

} // SetSizeParameters


NTSTATUS
SetStickyParameter (
    IN PWSTR ValueName,
    IN ULONG ValueType,
    IN PVOID ValueData,
    IN ULONG ValueLength,
    IN PVOID Context,
    IN PVOID EntryContext
    )
{
    NET_API_STATUS error;
    DWORD i;
    PFIELD_DESCRIPTOR foundField = NULL;
    LPWSTR subStrings[2];

    ValueLength, EntryContext;

    //
    // Ignore the parameter if it is "Size".  This parameter was
    // already handled.
    //

    if ( _wcsicmp( ValueName, SIZE_VALUE_NAME ) == 0 ) {
        return STATUS_SUCCESS;
    }

    //
    // Ignore NullSessionPipes and NullSessionShares since they are
    // handled by the server driver.
    //

    if ( (_wcsicmp( ValueName, NULL_SESSION_SHARES_VALUE_NAME ) == 0) ||
         (_wcsicmp( ValueName, NULL_SESSION_PIPES_VALUE_NAME ) == 0) ) {
        return STATUS_SUCCESS;
    }

    //
    // Determine which field we need to set, based on the value
    // name.
    //
    // NOTE: For Daytona, disc and comment are now invalid registry names.
    //      We use their more famous aliases autodisconnect and srvcomment
    //      instead.  If we get more of these cases, we should consider adding
    //      a field to the FIELD_DESCRIPTOR structure that indicates whether
    //      the names are should appear on the registry or not.  Any change
    //      here should also be made to SsSetField().
    //

    if ( (_wcsicmp( ValueName, DISC_VALUE_NAME ) != 0) &&
         (_wcsicmp( ValueName, COMMENT_VALUE_NAME ) != 0) ) {

        for ( i = 0;
              SsServerInfoFields[i].FieldName != NULL;
              i++ ) {

            if ( _wcsicmp( ValueName, SsServerInfoFields[i].FieldName ) == 0 ) {
                foundField = &SsServerInfoFields[i];
                break;
            }
        }
    }

    if ( foundField == NULL || foundField->Settable == NOT_SETTABLE ) {

        subStrings[0] = ValueName;
        subStrings[1] = Context;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            NO_ERROR
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SetStickyParameter: ignoring %s \"%ws\"\n",
                        (foundField == NULL ? "unknown value name" :
                        "unsettable value"), ValueName ));
        }
        return STATUS_SUCCESS;

    }

    switch ( foundField->FieldType ) {

    case BOOLEAN_FIELD:
    case DWORD_FIELD:

        if ( ValueType != REG_DWORD ) {

            subStrings[0] = ValueName;
            subStrings[1] = Context;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                NO_ERROR
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "SetStickyParameter: skipping invalid value "
                            "%ws\n", ValueName ));
            }
            return STATUS_SUCCESS;

        }

        i = *(LPDWORD)ValueData;
        break;

    case LPSTR_FIELD:

        if ( ValueType != REG_SZ ) {

            subStrings[0] = ValueName;
            subStrings[1] = Context;
            SsLogEvent(
                EVENT_SRV_INVALID_REGISTRY_VALUE,
                2,
                subStrings,
                NO_ERROR
                );

            IF_DEBUG(REGISTRY) {
                SS_PRINT(( "SetStickyParameter: skipping invalid value "
                            "%ws\n", ValueName ));
            }
            return STATUS_SUCCESS;

        }

        i = (DWORD)ValueData;
        break;

    }

    //
    // Set the field.
    //

    error = SsSetField( foundField, &i, FALSE, NULL );

    if ( error != NO_ERROR ) {

        subStrings[0] = ValueName;
        subStrings[1] = Context;
        SsLogEvent(
            EVENT_SRV_INVALID_REGISTRY_VALUE,
            2,
            subStrings,
            error
            );

        IF_DEBUG(REGISTRY) {
            SS_PRINT(( "SetStickyParameter: error %ld ignored in setting "
                        "parameter \"%ws\"n", error, ValueName ));
        }
    }

    return STATUS_SUCCESS;

} // SetStickyParameter

