/**********************************************************************/
/**                       Microsoft Windows/NT                       **/
/**                Copyright(c) Microsoft Corp., 1992                **/
/**********************************************************************/

/*
    replmain.hxx
    Class declarations for Main Replicator Admin dialog.

    The REPL_MAIN_* classes implement the Main Replicator Admin dialog.
    This dialog is invoked from the Server Manager Main Property Sheet.

    The classes are structured as follows:

        LBI
            REPL_MAIN_LBI

        BLT_LISTBOX
            REPL_MAIN_LISTBOX

        DIALOG_WINDOW
            REPL_MAIN_DIALOG


    FILE HISTORY:
        KeithMo     06-Feb-1992     Created for the Server Manager.

*/


#ifndef _REPLMAIN_HXX_
#define _REPLMAIN_HXX_


#include <lmobj.hxx>
#include <lmosrv.hxx>
#include <strlst.hxx>
#include <lmorepl.hxx>


//
//  The number of columns in the REPL_MAIN_LISTBOX.
//

#define REPL_MAIN_LISTBOX_COLUMNS   1


//
//  REPL_MAIN classes.
//

/*************************************************************************

    NAME:       REPL_MAIN_LBI

    SYNOPSIS:   This class represents one item in the REPL_MAIN_LISTBOX

    INTERFACE:  REPL_MAIN_LBI           - Class constructor.

                ~REPL_MAIN_LBI          - Class destructor.

                Paint                   - Draw an item.

                QueryLeadingChar        - Query the first character for
                                          the keyboard interface.

                Compare                 - Compare two items.

    PARENT:     LBI

    USES:       NLS_STR

    HISTORY:
        KeithMo     06-Feb-1992     Created for the Server Manager.
        beng        22-Apr-1992     Change to LBI::Paint

**************************************************************************/

class REPL_MAIN_LBI : public LBI
{
private:
    //
    //  This string holds the name of the target server/domain.
    //

    NLS_STR _nlsTarget;

protected:

    //
    //  This method paints a single item into the listbox.
    //

    virtual VOID Paint( LISTBOX     * plb,
                        HDC           hdc,
                        const RECT  * prect,
                        GUILTT_INFO * pGUILTT ) const;

    //
    //  This method compares two listbox items.  This
    //  is used for sorting the listbox.
    //

    virtual INT Compare( const LBI * plbi ) const;

    //
    //  This method returns the first character in
    //  the displayed listbox item.  This is used for
    //  the keyboard interface.
    //

    virtual WCHAR QueryLeadingChar( VOID ) const;

public:

    //
    //  Usual constructor/destructor goodies.
    //

    REPL_MAIN_LBI( const TCHAR * pszTarget );

    virtual ~REPL_MAIN_LBI( VOID );

    //
    //  Accessors for the fields this LBI maintains.
    //

    const TCHAR * QueryTarget( VOID ) const
        { return _nlsTarget.QueryPch(); }

};  // class REPL_MAIN_LBI


/*************************************************************************

    NAME:       REPL_MAIN_LISTBOX

    SYNOPSIS:   This listbox displays the export/import targets.

    INTERFACE:  REPL_MAIN_LISTBOX       - Class constructor.

                ~REPL_MAIN_LISTBOX      - Class destructor.

                Fill                    - Fills the listbox with the
                                          export/import targets.

                QueryColumnWidths       - Called by REPL_MAIN_LBI::Paint(),
                                          this is the column width table
                                          used for painting the listbox
                                          items.

    PARENT:     BLT_LISTBOX

    USES:       SERVER_2
                NLS_STR

    HISTORY:
        KeithMo     06-Feb-1992     Created for the Server Manager.

**************************************************************************/

class REPL_MAIN_LISTBOX : public BLT_LISTBOX
{
private:

    //
    //  This array contains the column widths used
    //  while painting the listbox item.  This array
    //  is generated by the BuildColumnWidthTable()
    //  function.
    //

    UINT _adx[REPL_MAIN_LISTBOX_COLUMNS];

public:

    //
    //  Usual constructor/destructor goodies.
    //

    REPL_MAIN_LISTBOX( OWNER_WINDOW * powOwner,
                       CID            cid );

    ~REPL_MAIN_LISTBOX( VOID );

    //
    //  Methods to fill & refresh the listbox.
    //

    APIERR Fill( STRLIST * pstrlst );

    //
    //  This method is called by the REPL_MAIN_LBI::Paint()
    //  method for retrieving the column width table.
    //

    const UINT * QueryColumnWidths( VOID ) const
        { return _adx; }

    //
    //  The following macro will declare (& define) a new
    //  QueryItem() method which will return an REPL_MAIN_LBI *.
    //

    DECLARE_LB_QUERY_ITEM( REPL_MAIN_LBI )

};  // class REPL_MAIN_LISTBOX


/*************************************************************************

    NAME:       REPL_MAIN_DIALOG

    SYNOPSIS:   This class represents the Replicator Admin dialog which is
                invoked from the Server Manager main property sheet.

    INTERFACE:  REPL_MAIN_DIALOG        - Class constructor.

                ~REPL_MAIN_DIALOG       - Class destructor.

                OnCommand               - Called when the user presses one
                                          of the buttons or changes a selection
                                          in one of the listboxes.

                QueryHelpContext        - Called when the user presses "F1"
                                          or the "Help" button.  Used for
                                          selecting the appropriate help
                                          text for display.

    PARENT:     DIALOG_WINDOW

    USES:       REPL_MAIN_LISTBOX
                SERVER_2
                PUSH_BUTTON
                SLE
                RADIO_BUTTON
                MAGIC_GROUP

    HISTORY:
        KeithMo     06-Feb-1992     Created for the Server Manager.

**************************************************************************/
class REPL_MAIN_DIALOG : public DIALOG_WINDOW
{
private:

    //
    //  These controls are used to manipulate the
    //  export side of the replicator service.
    //

    MAGIC_GROUP * _pmgExport;

    PUSH_BUTTON * _ppbExportManage;
    PUSH_BUTTON * _ppbExportAdd;
    PUSH_BUTTON * _ppbExportRemove;
    REPL_MAIN_LISTBOX * _plbExportTargets;
    SLT * _psltExportPathLabel;
    SLT * _psltExportListLabel;
    SLE * _psleExportPath;

    //
    //  These strings contain the replicator's export
    //  and import paths on entry to this dialog.  We
    //  need to keep these around since changing the
    //  export & import paths on a running replicator
    //  is somewhat complex.
    //

    NLS_STR * _pnlsOldExportPath;
    NLS_STR   _nlsOldImportPath;

    //
    //  This SLE contains the logon script path.
    //

    SLE * _psleLogonScriptPath;

    //
    //  These controls are used to manipulate the
    //  import side of the replicator service.
    //

    MAGIC_GROUP _mgImport;

    PUSH_BUTTON _pbImportManage;
    PUSH_BUTTON _pbImportAdd;
    PUSH_BUTTON _pbImportRemove;
    REPL_MAIN_LISTBOX _lbImportTargets;
    SLT _sltImportPathLabel;
    SLT _sltImportListLabel;
    SLE _sleImportPath;

    PUSH_BUTTON _pbOK;

    //
    //  This points to an object representing the
    //  target server.
    //

    SERVER_2 * _pserver;

    //
    //  This points to an object representing the
    //  target replicator.
    //

    REPLICATOR_0 * _prepl;

    //
    //  This flag will be TRUE if the targetted server
    //  is running LanMan NT.
    //

    BOOL _fIsLanmanNT;

    //
    //  The %SystemRoot% value for the target server.
    //

    NLS_STR _nlsSystemRoot;

    //
    //  This method will read the replicator information
    //  for the main dialog.
    //

    APIERR ReadReplInfo( VOID );

    //
    //  This method will start/stop the replicator service.
    //

    APIERR ControlService( BOOL fStart );

    //
    //  These methods are responsible for ensuring that the
    //  REPL$ share points to the correct export path and
    //  have the appropriate ACL.
    //

    APIERR SetupReplShare( VOID );
    APIERR SetupReplACL( VOID );

    //
    //  These methods invoke the export/import management
    //  dialogs.
    //

    VOID ExportManageDialog( VOID );
    VOID ImportManageDialog( VOID );

    //
    //  This method will construct the semi-colon separated
    //  export/import list from a listbox contents.
    //

    APIERR BuildExportImportList( NLS_STR           * pnls,
                                  REPL_MAIN_LISTBOX * plb );

    //
    //  These methods add/remove export/import targets.
    //

    VOID AddNewTarget( REPL_MAIN_LISTBOX * plb,
                       PUSH_BUTTON       * ppb );

    VOID RemoveExistingTarget( REPL_MAIN_LISTBOX * plb,
                               PUSH_BUTTON       * ppb,
                               MSGID               idWarning );

    //
    //  This method will determine the current role of the
    //  replicator.
    //

    APIERR DetermineCurrentRole( ULONG * pRole );

    //
    //  These methods manipulate the logon script path.
    //

    APIERR ReadLogonScriptPath( NLS_STR * pnlsLogonScriptPath );

    APIERR WriteLogonScriptPath( NLS_STR * pnlsLogonScriptPath );

    //
    //  These methods deal with expanding/collapsing the %SystemRoot%
    //  environment string on remote machines.  These methods are
    //  called by Read/WriteLogonScriptPath.
    //

    APIERR ExpandSystemRoot( NLS_STR * pnlsPath );

    APIERR CollapseSystemRoot( NLS_STR * pnlsPath );

    //
    //  This method will retrieve the current %SystemRoot% value.
    //

    APIERR GetSystemRoot( VOID );

    //
    //  This worker method updates either the export or import
    //  path.
    //

    APIERR W_UpdatePath( ULONG * pnOldRole,
                         BOOL    fExport );

protected:

    //
    //  This method is called whenever a control
    //  is sending us a command.  This is where
    //  we handle the manipulation buttons.
    //

    virtual BOOL OnCommand( const CONTROL_EVENT & event );

    //
    //  Called when the user presses the "OK" button.
    //

    virtual BOOL OnOK( VOID );

    //
    //  Called during help processing to select the appropriate
    //  help text for display.
    //

    virtual ULONG QueryHelpContext( VOID );

public:

    //
    //  Usual constructor/destructor goodies.
    //

    REPL_MAIN_DIALOG( HWND       hwndOwner,
                      SERVER_2 * pserver,
                      BOOL       fIsLanmanNT );

    ~REPL_MAIN_DIALOG( VOID );

    //
    //  Provide easy access to the target server's name.
    //

    const TCHAR * QueryServerName( VOID ) const
        { return _pserver->QueryName(); }

};  // class REPL_MAIN_DIALOG


#endif  // _REPLMAIN_HXX_
