/*++

Copyright (c) 1991 Microsoft Corporation

Module Name:

    ApiShare.c

Abstract:

    This module contains individual API handlers for the NetShare APIs.

    SUPPORTED : NetShareAdd, NetShareCheck, NetShareDel, NetShareEnum,
                NetShareGetInfo, NetShareSetInfo.

Author:

    David Treadwell (davidtr)    07-Jan-1991
    Shanku Niyogi (w-shanku)

Revision History:

--*/

#include "XactSrvP.h"

//
// Declaration of descriptor strings.
//

STATIC const LPDESC Desc16_share_info_0 = REM16_share_info_0;
STATIC const LPDESC Desc32_share_info_0 = REM32_share_info_0;
STATIC const LPDESC Desc16_share_info_1 = REM16_share_info_1;
STATIC const LPDESC Desc32_share_info_1 = REM32_share_info_1;
STATIC const LPDESC Desc16_share_info_1_setinfo = REM16_share_info_1_setinfo;
STATIC const LPDESC Desc32_share_info_1_setinfo = REM32_share_info_1_setinfo;
STATIC const LPDESC Desc16_share_info_2 = REM16_share_info_2;
STATIC const LPDESC Desc32_share_info_2 = REM32_share_info_2;
STATIC const LPDESC Desc16_share_info_2_setinfo = REM16_share_info_2_setinfo;
STATIC const LPDESC Desc32_share_info_2_setinfo = REM32_share_info_2_setinfo;


NTSTATUS
XsNetShareAdd (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareAdd.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_ADD parameters = Parameters;
    LPVOID buffer = NULL;                   // Native parameters

    LPBYTE stringLocation = NULL;           // Conversion variables
    DWORD bytesRequired = 0;
    DWORD bufferSize;

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareAdd: header at %lx, params at %lx, "
                      "level %ld\n",
                      Header,
                      parameters,
                      SmbGetUshort( &parameters->Level ) );
    }

    //
    // Check for errors.
    //

    if ( SmbGetUshort( &parameters->Level ) != 2 ) {

        Header->Status = ERROR_INVALID_LEVEL;
        goto cleanup;
    }

    StructureDesc = Desc16_share_info_2;

    //
    // Figure out if there is enough room in the buffer for all the
    // data required. If not, return NERR_BufTooSmall.
    //

    if ( !XsCheckBufferSize(
             SmbGetUshort( &parameters->BufLen ),
             StructureDesc,
             FALSE  // not in native format
             )) {

        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareAdd: Buffer too small.\n" );
        }
        Header->Status = NERR_BufTooSmall;
        goto cleanup;
    }

    //
    // Find out how big a buffer we need to allocate to hold the native
    // 32-bit version of the input data structure.
    //

    bufferSize = XsBytesForConvertedStructure(
                     (LPBYTE)SmbGetUlong( &parameters->Buffer ),
                     StructureDesc,
                     Desc32_share_info_2,
                     RapToNative,
                     TRUE
                     );

    //
    // Allocate enough memory to hold the converted native buffer.
    //

    buffer = NetpMemoryAllocate( bufferSize );

    if ( buffer == NULL ) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareAdd: failed to create buffer" );
        }
        Header->Status = NERR_NoRoom;
        goto cleanup;

    }

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareAdd: buffer of %ld bytes at %lx\n",
                      bufferSize, buffer );
    }

    //
    // Convert the buffer from 16-bit to 32-bit.
    //

    stringLocation = (LPBYTE)buffer + bufferSize;
    bytesRequired = 0;

    status = RapConvertSingleEntry(
                 (LPBYTE)SmbGetUlong( &parameters->Buffer ),
                 StructureDesc,
                 TRUE,
                 buffer,
                 buffer,
                 Desc32_share_info_2,
                 FALSE,
                 &stringLocation,
                 &bytesRequired,
                 Response,
                 RapToNative
                 );


    if ( status != NERR_Success ) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareAdd: RapConvertSingleEntry failed: "
                          "%X\n", status );
        }

        Header->Status = NERR_InternalError;
        goto cleanup;
    }

    //
    // Make the local call.
    //

    status = NetShareAdd(
                 NULL,
                 (DWORD)SmbGetUshort( &parameters->Level ),
                 buffer,
                 NULL
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareAdd: NetShareAdd failed: %X\n", status );
        }

        if ( status == ERROR_DIRECTORY ) {
            Header->Status = NERR_UnknownDevDir;
        } else {
            Header->Status = (WORD)status;
        }
        goto cleanup;
    }

    //
    // There is no real return information for this API.
    //

cleanup:

    NetpMemoryFree( buffer );

    return STATUS_SUCCESS;

} // XsNetShareAdd


NTSTATUS
XsNetShareCheck (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareCheck.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_CHECK parameters = Parameters;
    LPTSTR nativeDeviceName = NULL;         // Native parameters
    DWORD shareType;

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    //
    // Translate parameters, check for errors.
    //

    XsConvertTextParameter(
        nativeDeviceName,
        (LPSTR)SmbGetUlong( &parameters->DeviceName )
        );

    //
    // Do the local call.
    //

    status = NetShareCheck(
                 NULL,
                 nativeDeviceName,
                 &shareType
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareCheck: NetShareCheck failed: "
                          "%X\n", status );
        }
    }

    //
    // Put type into return field.
    //

    SmbPutUshort( &parameters->Type, (WORD)shareType );

    Header->Status = (WORD)status;

cleanup:

    NetpMemoryFree( nativeDeviceName );
    return STATUS_SUCCESS;

} // XsNetShareCheck


NTSTATUS
XsNetShareDel (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareDel.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_DEL parameters = Parameters;
    LPTSTR nativeNetName = NULL;            // Native parameters

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareDel: header at %lx, params at %lx, name %s\n",
                      Header, parameters, SmbGetUlong( &parameters->NetName ));
    }

    //
    // Translate parameters, check for errors.
    //

    XsConvertTextParameter(
        nativeNetName,
        (LPSTR)SmbGetUlong( &parameters->NetName )
        );

    //
    // Make the local call.
    //

    status = NetShareDel(
                 NULL,
                 nativeNetName,
                 (DWORD)SmbGetUshort( &parameters->Reserved )
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareDel: NetShareDel failed: %X\n", status );
        }
    }

    //
    // Nothing to return.
    //

    Header->Status = (WORD)status;

cleanup:

    NetpMemoryFree( nativeNetName );
    return STATUS_SUCCESS;

} // XsNetShareDel


NTSTATUS
XsNetShareEnum (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareEnum.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_ENUM parameters = Parameters;
    LPVOID outBuffer= NULL;                 // Native parameters
    DWORD entriesRead;
    DWORD totalEntries;

    DWORD entriesFilled;                    // Conversion variables
    DWORD bytesRequired = 0;
    LPDESC nativeStructureDesc;

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareEnum: header at %lx, params at %lx, "
                      "level %ld, buf size %ld\n",
                      Header, parameters, SmbGetUshort( &parameters->Level ),
                      SmbGetUshort( &parameters->BufLen ));
    }

    //
    // Check for errors.
    //

    if ( XsWordParamOutOfRange( parameters->Level, 0, 2 )) {

        Header->Status = ERROR_INVALID_LEVEL;
        goto cleanup;
    }

    //
    // Make the local call.
    //

    status = NetShareEnum(
                 NULL,
                 (DWORD)SmbGetUshort( &parameters->Level ),
                 (LPBYTE *)&outBuffer,
                 XsNativeBufferSize( SmbGetUshort( &parameters->BufLen )),
                 &entriesRead,
                 &totalEntries,
                 NULL
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(API_ERRORS) {
            NetpDbgPrint( "XsNetShareEnum: NetShareEnum failed: "
                          "%X\n", status );
        }
        Header->Status = (WORD)status;
        goto cleanup;
    }

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareEnum: received %ld entries at %lx\n",
                      entriesRead, outBuffer );
    }

    //
    // Use the requested level to determine the format of the
    // data structure.
    //

    switch ( SmbGetUshort( &parameters->Level ) ) {

    case 0:

        nativeStructureDesc = Desc32_share_info_0;
        StructureDesc = Desc16_share_info_0;
        break;

    case 1:

        nativeStructureDesc = Desc32_share_info_1;
        StructureDesc = Desc16_share_info_1;
        break;

    case 2:

        nativeStructureDesc = Desc32_share_info_2;
        StructureDesc = Desc16_share_info_2;
        break;
    }

    //
    // Do the actual conversion from the 32-bit structures to 16-bit
    // structures.
    //

    XsFillEnumBuffer(
        outBuffer,
        entriesRead,
        nativeStructureDesc,
        (LPVOID)SmbGetUlong( &parameters->Buffer ),
        (LPVOID)SmbGetUlong( &parameters->Buffer ),
        SmbGetUshort( &parameters->BufLen ),
        StructureDesc,
        NULL,  // verify function
        &bytesRequired,
        &entriesFilled,
        NULL
        );

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "32-bit data at %lx, 16-bit data at %lx, %ld BR,"
                      " Entries %ld of %ld\n",
                      outBuffer, SmbGetUlong( &parameters->Buffer ),
                      bytesRequired, entriesFilled, totalEntries );
    }

    //
    // If all the entries could not be filled, return ERROR_MORE_DATA,
    // and return the buffer as is. Otherwise, the data needs to be
    // packed so that we don't send too much useless data.
    //

    if ( entriesFilled < totalEntries ) {

        Header->Status = ERROR_MORE_DATA;

    } else {

        Header->Converter = XsPackReturnData(
                                (LPVOID)SmbGetUlong( &parameters->Buffer ),
                                SmbGetUshort( &parameters->BufLen ),
                                StructureDesc,
                                entriesFilled
                                );

    }

    //
    // Set up the response parameters.
    //

    SmbPutUshort( &parameters->EntriesRead, (WORD)entriesFilled );
    SmbPutUshort( &parameters->TotalAvail, (WORD)totalEntries );

cleanup:

    NetApiBufferFree( outBuffer );

    //
    // Determine return buffer size.
    //

    XsSetDataCount(
        &parameters->BufLen,
        StructureDesc,
        Header->Converter,
        entriesFilled,
        Header->Status
        );

    return STATUS_SUCCESS;

} // XsNetShareEnum


NTSTATUS
XsNetShareGetInfo (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareGetInfo.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_GET_INFO parameters = Parameters;
    LPTSTR nativeNetName = NULL;            // Native parameters
    LPVOID outBuffer = NULL;

    LPBYTE stringLocation = NULL;           // Conversion variables
    DWORD bytesRequired = 0;
    LPDESC nativeStructureDesc;

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "XsNetShareGetInfo: header at %lx, "
                      "params at %lx, level %ld\n",
                      Header, parameters, SmbGetUshort( &parameters->Level ) );
    }

    //
    // Translate parameters, check for errors.
    //

    if ( XsWordParamOutOfRange( parameters->Level, 0, 2 )) {

        Header->Status = ERROR_INVALID_LEVEL;
        goto cleanup;
    }

    XsConvertTextParameter(
        nativeNetName,
        (LPSTR)SmbGetUlong( &parameters->NetName )
        );

    //
    // Make the local call.
    //

    status = NetShareGetInfo(
                 NULL,
                 nativeNetName,
                 (DWORD)SmbGetUshort( &parameters->Level ),
                 (LPBYTE *)&outBuffer
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(API_ERRORS) {
            NetpDbgPrint( "XsNetShareGetInfo: NetShareGetInfo failed: "
                          "%X\n", status );
        }
        Header->Status = (WORD)status;
        goto cleanup;

    }

    //
    // Use the requested level to determine the format of the
    // data structure.
    //

    switch ( SmbGetUshort( &parameters->Level ) ) {

    case 0:

        nativeStructureDesc = Desc32_share_info_0;
        StructureDesc = Desc16_share_info_0;
        break;

    case 1:

        nativeStructureDesc = Desc32_share_info_1;
        StructureDesc = Desc16_share_info_1;
        break;

    case 2:

        nativeStructureDesc = Desc32_share_info_2;
        StructureDesc = Desc16_share_info_2;
        break;
    }

    //
    // Convert the structure returned by the 32-bit call to a 16-bit
    // structure. The last possible location for variable data is
    // calculated from buffer location and length.
    //

    stringLocation = (LPBYTE)( SmbGetUlong( &parameters->Buffer )
                                  + SmbGetUshort( &parameters->BufLen ) );

    status = RapConvertSingleEntry(
                 outBuffer,
                 nativeStructureDesc,
                 FALSE,
                 (LPBYTE)SmbGetUlong( &parameters->Buffer ),
                 (LPBYTE)SmbGetUlong( &parameters->Buffer ),
                 StructureDesc,
                 TRUE,
                 &stringLocation,
                 &bytesRequired,
                 Response,
                 NativeToRap
                 );


    if ( status != NERR_Success ) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareGetInfo: RapConvertSingleEntry failed: "
                          "%X\n", status );
        }

        Header->Status = NERR_InternalError;
        goto cleanup;
    }

    IF_DEBUG(SHARE) {
        NetpDbgPrint( "32-bit data at %lx, 16-bit data at %lx, %ld BR\n",
                      outBuffer, SmbGetUlong( &parameters->Buffer ),
                      bytesRequired );
    }

    //
    // Determine return code based on the size of the buffer.
    //

    if ( !XsCheckBufferSize(
             SmbGetUshort( &parameters->BufLen ),
             StructureDesc,
             FALSE  // not in native format
             )) {

        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareGetInfo: Buffer too small.\n" );
        }
        Header->Status = NERR_BufTooSmall;

    } else if ( bytesRequired > (DWORD)SmbGetUshort( &parameters-> BufLen )) {

        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareGetInfo: More data available.\n" );
        }
        Header->Status = ERROR_MORE_DATA;

    } else {

        //
        // Pack the response data.
        //

        Header->Converter = XsPackReturnData(
                                (LPVOID)SmbGetUlong( &parameters->Buffer ),
                                SmbGetUshort( &parameters->BufLen ),
                                StructureDesc,
                                1
                                );
    }


    //
    // Set up the response parameters.
    //

    SmbPutUshort( &parameters->TotalAvail, (WORD)bytesRequired );

cleanup:

    NetApiBufferFree( outBuffer );
    NetpMemoryFree( nativeNetName );

    //
    // Determine return buffer size.
    //

    XsSetDataCount(
        &parameters->BufLen,
        StructureDesc,
        Header->Converter,
        1,
        Header->Status
        );

    return STATUS_SUCCESS;

} // XsNetShareGetInfo


NTSTATUS
XsNetShareSetInfo (
    API_HANDLER_PARAMETERS
    )

/*++

Routine Description:

    This routine handles a call to NetShareSetInfo.

Arguments:

    API_HANDLER_PARAMETERS - information about the API call. See
        XsTypes.h for details.

Return Value:

    NTSTATUS - STATUS_SUCCESS or reason for failure.

--*/

{
    NET_API_STATUS status;

    PXS_NET_SHARE_SET_INFO parameters = Parameters;
    LPTSTR nativeNetName = NULL;            // Native parameters
    LPVOID buffer = NULL;
    DWORD level;

    LPDESC setInfoDesc;                     // Conversion variables
    LPDESC nativeSetInfoDesc;
    LPDESC nativeStructureDesc;

    API_HANDLER_PARAMETERS_REFERENCE;       // Avoid warnings

    //
    // Translate parameters, check for errors.
    //

    XsConvertTextParameter(
        nativeNetName,
        (LPSTR)SmbGetUlong( &parameters->NetName )
        );

    //
    // Determine descriptor strings based on level.
    //

    switch ( SmbGetUshort( &parameters->Level )) {

    case 1:

        StructureDesc = Desc16_share_info_1;
        nativeStructureDesc = Desc32_share_info_1;
        setInfoDesc = Desc16_share_info_1_setinfo;
        nativeSetInfoDesc = Desc32_share_info_1_setinfo;

        break;

    case 2:

        StructureDesc = Desc16_share_info_2;
        nativeStructureDesc = Desc32_share_info_2;
        setInfoDesc = Desc16_share_info_2_setinfo;
        nativeSetInfoDesc = Desc32_share_info_2_setinfo;

        break;

    default:

        Header->Status = ERROR_INVALID_LEVEL;
        goto cleanup;
    }

    status = XsConvertSetInfoBuffer(
                 (LPBYTE)SmbGetUlong( &parameters->Buffer ),
                 SmbGetUshort( &parameters->BufLen ),
                 SmbGetUshort( &parameters->ParmNum ),
                 FALSE,
                 TRUE,
                 StructureDesc,
                 nativeStructureDesc,
                 setInfoDesc,
                 nativeSetInfoDesc,
                 (LPBYTE *)&buffer,
                 NULL
                 );

    if ( status != NERR_Success ) {

        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareSetInfo: Problem with conversion: %X\n",
                          status );
        }
        Header->Status = (WORD)status;
        goto cleanup;

    }

    //
    // Do the actual local call.
    //

    level = SmbGetUshort( &parameters->ParmNum );
    if ( level != 0 ) {
        level = level + PARMNUM_BASE_INFOLEVEL;
    } else {
        level = SmbGetUshort( &parameters->Level );
    }

    status = NetShareSetInfo(
                 NULL,
                 nativeNetName,
                 level,
                 (LPBYTE)buffer,
                 NULL
                 );

    if ( !XsApiSuccess( status )) {
        IF_DEBUG(ERRORS) {
            NetpDbgPrint( "XsNetShareSetInfo: NetShareSetInfo failed: %X\n",
                          status );
        }
        Header->Status = (WORD)status;
        goto cleanup;
    }

    //
    // No return information for this API.
    //

cleanup:

    //
    // If there is a native 32-bit buffer, free it.
    //

    NetpMemoryFree( buffer );
    NetpMemoryFree( nativeNetName );

    return STATUS_SUCCESS;

} // XsNetShareSetInfo

