/*++

Copyright (c) 1991  Microsoft Corporation

Module Name:

    cmapi.c

Abstract:

    This module contains CM level entry points for the registry.

Author:

    Bryan M. Willman (bryanwi) 30-Aug-1991

Revision History:

--*/

#include "cmp.h"



extern  BOOLEAN     CmpNoWrite;

extern  LIST_ENTRY  CmpHiveListHead;

extern  BOOLEAN CmpProfileLoaded;
extern  BOOLEAN CmpWasSetupBoot;

extern  PUCHAR  CmpStashBuffer;
extern  ULONG   CmpStashBufferSize;

//
// procedures private to this file
//
NTSTATUS
CmpSetValueKeyExisting(
    IN PHHIVE  Hive,
    IN HCELL_INDEX OldChild,
    IN ULONG TitleIndex OPTIONAL,
    IN ULONG Type,
    IN PVOID Data,
    IN ULONG DataSize,
    IN ULONG StorageType,
    IN ULONG TempData
    );


NTSTATUS
CmpSetValueKeyNew(
    IN PHHIVE  Hive,
    IN PCELL_DATA Parent,
    IN PUNICODE_STRING ValueName,
    IN ULONG TitleIndex OPTIONAL,
    IN ULONG Type,
    IN PVOID Data,
    IN ULONG DataSize,
    IN ULONG StorageType,
    IN ULONG TempData
    );

#ifdef ALLOC_PRAGMA
#pragma alloc_text(PAGE,CmDeleteValueKey)
#pragma alloc_text(PAGE,CmEnumerateKey)
#pragma alloc_text(PAGE,CmEnumerateValueKey)
#pragma alloc_text(PAGE,CmFlushKey)
#pragma alloc_text(PAGE,CmQueryKey)
#pragma alloc_text(PAGE,CmQueryValueKey)
#pragma alloc_text(PAGE,CmSetValueKey)
#pragma alloc_text(PAGE,CmpSetValueKeyExisting)
#pragma alloc_text(PAGE,CmpSetValueKeyNew)
#pragma alloc_text(PAGE,CmSetLastWriteTimeKey)
#pragma alloc_text(PAGE,CmLoadKey)
#pragma alloc_text(PAGE,CmUnloadKey)
#pragma alloc_text(PAGE,CmpDoFlushAll)
#pragma alloc_text(PAGE,CmReplaceKey)
#endif




NTSTATUS
CmDeleteValueKey(
    IN PCM_KEY_CONTROL_BLOCK    KeyControlBlock,
    IN UNICODE_STRING ValueName         // RAW
    )
/*++

Routine Description:

    One of the value entries of a registry key may be removed with this call.

    The value entry with ValueName matching ValueName is removed from the key.
    If no such entry exists, an error is returned.

Arguments:

    KeyControlBlock - pointer to kcb for key to operate on

    ValueName - The name of the value to be deleted.  NULL is a legal name.

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS status;
    PCELL_DATA  pcell;
    PCHILD_LIST plist;
    PCELL_DATA pvector;
    PCELL_DATA targetaddress;
    ULONG  targetindex;
    ULONG   newcount;
    HCELL_INDEX newcell;
    HCELL_INDEX ChildCell;
    PHHIVE      Hive;
    HCELL_INDEX Cell;
    ULONG realsize;
    LARGE_INTEGER systemtime;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmDeleteValueKey\n"));

    CmpLockRegistryExclusive();

    try {
        //
        // no edits, not even this one, on keys marked for deletion
        //
        if (KeyControlBlock->Delete) {
            return STATUS_KEY_DELETED;
        }

        Hive = KeyControlBlock->KeyHive;
        Cell = KeyControlBlock->KeyCell;

        status = STATUS_OBJECT_NAME_NOT_FOUND;

        pcell = HvGetCell(Hive, Cell);
        plist = &(pcell->u.KeyNode.ValueList);
        ChildCell = HCELL_NIL;

        if (plist->Count != 0) {

            //
            // The parent has at least one value, map in the list of
            // values and call CmpFindChildInList
            //

            //
            // plist -> the CHILD_LIST structure
            // pvector -> the actual list of cells (which the child_list refs)
            // pchild -> the child node structure being examined
            //
            pvector = HvGetCell(Hive, plist->List);

            status = CmpFindNameInList(
                        Hive,
                        pvector,
                        plist->Count,
                        ValueName,
                        &ChildCell,
                        &targetaddress,
                        &targetindex
                        );

            if (NT_SUCCESS(status)) {

                //
                // 1. the desired target was found
                // 2. ChildCell is it's HCELL_INDEX
                // 3. targetaddress points to it
                // 4. targetindex is it's index
                //

                //
                // attempt to mark all relevent cells dirty
                //
                if (!(HvMarkCellDirty(Hive, Cell) &&
                      HvMarkCellDirty(Hive, pcell->u.KeyNode.ValueList.List) &&
                      HvMarkCellDirty(Hive, ChildCell)))

                {
                    return STATUS_NO_LOG_SPACE;
                }

                if (!CmpIsHKeyValueSmall(realsize,targetaddress->u.KeyValue.DataLength)) {

                    if (!HvMarkCellDirty(Hive, targetaddress->u.KeyValue.Data)) {
                        return(STATUS_NO_LOG_SPACE);
                    }
                }

                newcount = plist->Count - 1;

                if (newcount > 0) {

                    //
                    // more than one entry list, squeeze
                    //
                    for ( ; targetindex < newcount; targetindex++) {
                        pvector->u.KeyList[ targetindex ] =
                            pvector->u.KeyList[ targetindex+1 ];
                    }

                    newcell = HvReallocateCell(
                                Hive,
                                plist->List,
                                newcount * sizeof(HCELL_INDEX)
                                );
                    ASSERT(newcell != HCELL_NIL);
                    plist->List = newcell;

                } else {

                    //
                    // list is empty, free it
                    //
                    HvFreeCell(Hive, plist->List);
                }
                plist->Count = newcount;
                CmpFreeValue(Hive, ChildCell);

                KeQuerySystemTime(&systemtime);
                pcell->u.KeyNode.LastWriteTime = systemtime;

                if (pcell->u.KeyNode.ValueList.Count == 0) {
                    pcell->u.KeyNode.MaxValueNameLen = 0;
                    pcell->u.KeyNode.MaxValueDataLen = 0;
                }

                CmpReportNotify(
                    KeyControlBlock->FullName,
                    KeyControlBlock->KeyHive,
                    KeyControlBlock->KeyCell,
                    REG_NOTIFY_CHANGE_LAST_SET
                    );
            }
        }
    } finally {
        CmpUnlockRegistry();
    }

    return status;
}


NTSTATUS
CmEnumerateKey(
    IN PCM_KEY_CONTROL_BLOCK    KeyControlBlock,
    IN ULONG Index,
    IN KEY_INFORMATION_CLASS KeyInformationClass,
    IN PVOID KeyInformation,
    IN ULONG Length,
    IN PULONG ResultLength
    )
/*++

Routine Description:

    Enumerate sub keys, return data on Index'th entry.

    CmEnumerateKey returns the name of the Index'th sub key of the open
    key specified.  The value STATUS_NO_MORE_ENTRIES will be
    returned if value of Index is larger than the number of sub keys.

    Note that Index is simply a way to select among child keys.  Two calls
    to CmEnumerateKey with the same Index are NOT guaranteed to return
    the same results.

    If KeyInformation is not long enough to hold all requested data,
    STATUS_BUFFER_OVERFLOW will be returned, and ResultLength will be
    set to the number of bytes actually required.

Arguments:

    KeyControlBlock - pointer to the KCB that describes the key

    Index - Specifies the (0-based) number of the sub key to be returned.

    KeyInformationClass - Specifies the type of information returned in
        Buffer.  One of the following types:

        KeyBasicInformation - return last write time, title index, and name.
            (see KEY_BASIC_INFORMATION structure)

        KeyNodeInformation - return last write time, title index, name, class.
            (see KEY_NODE_INFORMATION structure)

    KeyInformation -Supplies pointer to buffer to receive the data.

    Length - Length of KeyInformation in bytes.

    ResultLength - Number of bytes actually written into KeyInformation.

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    HCELL_INDEX childcell;
    PHHIVE Hive;
    HCELL_INDEX Cell;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmEnumerateKey\n"));

    CmpLockRegistry();

    if (KeyControlBlock->Delete) {
        CmpUnlockRegistry();
        return STATUS_KEY_DELETED;
    }
    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;

    //
    // fetch the child of interest
    //

    if (!NT_SUCCESS(status =
            CmpFindChildByNumber(
                Hive,
                Cell,
                Index,
                KeyBodyNode,
                &childcell
                )
       ))
    {
        //
        // no such child, clean up and return error
        //

        CmpUnlockRegistry();
        return status;
    }

    try {

        //
        // call a worker to perform data transfer
        //

        status = CmpQueryKeyData(
                    Hive,
                    childcell,
                    KeyInformationClass,
                    KeyInformation,
                    Length,
                    ResultLength
                    );

    } finally {
        CmpUnlockRegistry();
    }
    return status;
}



NTSTATUS
CmEnumerateValueKey(
    IN PCM_KEY_CONTROL_BLOCK    KeyControlBlock,
    IN ULONG Index,
    IN KEY_VALUE_INFORMATION_CLASS KeyValueInformationClass,
    IN PVOID KeyValueInformation,
    IN ULONG Length,
    IN PULONG ResultLength
    )
/*++

Routine Description:

    The value entries of an open key may be enumerated.

    CmEnumerateValueKey returns the name of the Index'th value
    entry of the open key specified by KeyHandle.  The value
    STATUS_NO_MORE_ENTRIES will be returned if value of Index is
    larger than the number of sub keys.

    Note that Index is simply a way to select among value
    entries.  Two calls to NtEnumerateValueKey with the same Index
    are NOT guaranteed to return the same results.

    If KeyValueInformation is not long enough to hold all requested data,
    STATUS_BUFFER_OVERFLOW will be returned, and ResultLength will be
    set to the number of bytes actually required.

Arguments:

    KeyControlBlock - pointer to the KCB that describes the key

    Index - Specifies the (0-based) number of the sub key to be returned.

    KeyValueInformationClass - Specifies the type of information returned
    in Buffer. One of the following types:

        KeyValueBasicInformation - return time of last write,
            title index, and name.  (See KEY_VALUE_BASIC_INFORMATION)

        KeyValueFullInformation - return time of last write,
            title index, name, class.  (See KEY_VALUE_FULL_INFORMATION)

    KeyValueInformation -Supplies pointer to buffer to receive the data.

    Length - Length of KeyValueInformation in bytes.

    ResultLength - Number of bytes actually written into KeyValueInformation.

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    HCELL_INDEX childcell;
    PHHIVE Hive;
    HCELL_INDEX Cell;


    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmEnumerateValueKey\n"));

    //
    // lock the parent cell
    //

    CmpLockRegistry();

    if (KeyControlBlock->Delete) {
        CmpUnlockRegistry();
        return STATUS_KEY_DELETED;
    }
    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;


    //
    // fetch the child of interest
    //

    if (!NT_SUCCESS(status =
            CmpFindChildByNumber(
                Hive,
                Cell,
                Index,
                KeyValueNode,
                &childcell
                )
       ))
    {
        //
        // no such child, clean up and return error
        //

        CmpUnlockRegistry();
        return status;
    }

    try {

        //
        // call a worker to perform data transfer
        //

        status = CmpQueryKeyValueData(
                    Hive,
                    childcell,
                    KeyValueInformationClass,
                    KeyValueInformation,
                    Length,
                    ResultLength
                    );

    } finally {
        CmpUnlockRegistry();
    }
    return status;
}



NTSTATUS
CmFlushKey(
    IN PHHIVE Hive,
    IN HCELL_INDEX Cell
    )
/*++

Routine Description:

    Forces changes made to a key to disk.

    CmFlushKey will not return to its caller until any changed data
    associated with the key has been written out.

    WARNING: CmFlushKey will flush the entire registry tree, and thus will
    burn cycles and I/O.

Arguments:

    Hive - supplies a pointer to the hive control structure for the hive

    Cell - supplies index of node to whose sub keys are to be found

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    PCMHIVE CmHive;
    NTSTATUS    status = STATUS_SUCCESS;
    extern PCMHIVE CmpMasterHive;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmFlushKey\n"));

    //
    // If writes are not working, lie and say we succeeded, will
    // clean up in a short time.  Only early system init code
    // will ever know the difference.
    //
    if (CmpNoWrite) {
        return STATUS_SUCCESS;
    }


    CmHive = CONTAINING_RECORD(Hive, CMHIVE, Hive);

    //
    // Don't flush the master hive.  If somebody asks for a flushkey on
    // the master hive, do a CmpDoFlushAll instead.  CmpDoFlushAll flushes
    // every hive except the master hive, which is what they REALLY want.
    //
    if (CmHive == CmpMasterHive) {
        CmpDoFlushAll();
    } else {
        DCmCheckRegistry(CONTAINING_RECORD(Hive, CMHIVE, Hive));

        if (! HvSyncHive(Hive)) {

            status = STATUS_REGISTRY_IO_FAILED;

            CMLOG(CML_MAJOR, CMS_IO_ERROR) {
                KdPrint(("CmFlushKey: HvSyncHive failed\n"));
            }
        }
    }


    return  status;
}


NTSTATUS
CmQueryKey(
    IN PCM_KEY_CONTROL_BLOCK    KeyControlBlock,
    IN KEY_INFORMATION_CLASS KeyInformationClass,
    IN PVOID KeyInformation,
    IN ULONG Length,
    IN PULONG ResultLength
    )
/*++

Routine Description:

    Data about the class of a key, and the numbers and sizes of its
    children and value entries may be queried with CmQueryKey.

    NOTE: The returned lengths are guaranteed to be at least as
          long as the described values, but may be longer in
          some circumstances.

Arguments:

    KeyControlBlock - pointer to the KCB that describes the key

    KeyInformationClass - Specifies the type of information
        returned in Buffer.  One of the following types:

        KeyBasicInformation - return last write time, title index, and name.
            (See KEY_BASIC_INFORMATION)

        KeyNodeInformation - return last write time, title index, name, class.
            (See KEY_NODE_INFORMATION)

        KeyFullInformation - return all data except for name and security.
            (See KEY_FULL_INFORMATION)

    KeyInformation -Supplies pointer to buffer to receive the data.

    Length - Length of KeyInformation in bytes.

    ResultLength - Number of bytes actually written into KeyInformation.

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    PHHIVE Hive;
    HCELL_INDEX Cell;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmQueryKey\n"));

    CmpLockRegistry();

    if (KeyControlBlock->Delete) {
        CmpUnlockRegistry();
        return STATUS_KEY_DELETED;
    }
    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;


    try {

        //
        // call a worker to perform data transfer
        //

        status = CmpQueryKeyData(
                    Hive,
                    Cell,
                    KeyInformationClass,
                    KeyInformation,
                    Length,
                    ResultLength
                    );

    } finally {
        CmpUnlockRegistry();
    }
    return status;
}


NTSTATUS
CmQueryValueKey(
    IN PCM_KEY_CONTROL_BLOCK    KeyControlBlock,
    IN UNICODE_STRING ValueName,
    IN KEY_VALUE_INFORMATION_CLASS KeyValueInformationClass,
    IN PVOID KeyValueInformation,
    IN ULONG Length,
    IN PULONG ResultLength
    )
/*++

Routine Description:

    The ValueName, TitleIndex, Type, and Data for any one of a key's
    value entries may be queried with CmQueryValueKey.

    If KeyValueInformation is not long enough to hold all requested data,
    STATUS_BUFFER_OVERFLOW will be returned, and ResultLength will be
    set to the number of bytes actually required.

Arguments:

    KeyControlBlock - pointer to the KCB that describes the key

    ValueName  - The name of the value entry to return data for.

    KeyValueInformationClass - Specifies the type of information
        returned in KeyValueInformation.  One of the following types:

        KeyValueBasicInformation - return time of last write, title
            index, and name.  (See KEY_VALUE_BASIC_INFORMATION)

        KeyValueFullInformation - return time of last write, title
            index, name, class.  (See KEY_VALUE_FULL_INFORMATION)

    KeyValueInformation -Supplies pointer to buffer to receive the data.

    Length - Length of KeyValueInformation in bytes.

    ResultLength - Number of bytes actually written into KeyValueInformation.

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    HCELL_INDEX childcell;
    PHCELL_INDEX  childindex;
    PHHIVE Hive;
    HCELL_INDEX Cell;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmQueryValueKey\n"));

    CmpLockRegistry();

    if (KeyControlBlock->Delete) {
        CmpUnlockRegistry();
        return STATUS_KEY_DELETED;
    }
    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;

    try {

        //
        // find the data
        //

        status = CmpFindChildByName(
                    Hive,
                    Cell,
                    ValueName,
                    KeyValueNode,
                    &childcell,
                    &childindex
                    );

        if (NT_SUCCESS(status)) {

            //
            // call a worker to perform data transfer
            //

            status = CmpQueryKeyValueData(
                        Hive,
                        childcell,
                        KeyValueInformationClass,
                        KeyValueInformation,
                        Length,
                        ResultLength
                        );
        }

    } finally {
        CmpUnlockRegistry();
    }
    return status;
}



NTSTATUS
CmSetValueKey(
    IN PCM_KEY_CONTROL_BLOCK KeyControlBlock,
    IN UNICODE_STRING ValueName,
    IN ULONG TitleIndex OPTIONAL,
    IN ULONG Type,
    IN PVOID Data,
    IN ULONG DataSize
    )
/*++

Routine Description:

    A value entry may be created or replaced with CmSetValueKey.

    If a value entry with a Value ID (i.e. name) matching the
    one specified by ValueName exists, it is deleted and replaced
    with the one specified.  If no such value entry exists, a new
    one is created.  NULL is a legal Value ID.  While Value IDs must
    be unique within any given key, the same Value ID may appear
    in many different keys.

Arguments:

    KeyControlBlock - pointer to kcb for the key to operate on

    ValueName - The unique (relative to the containing key) name
        of the value entry.  May be NULL.

    TitleIndex - Supplies the title index for ValueName.  The title
        index specifies the index of the localized alias for the ValueName.

    Type - The integer type number of the value entry.

    Data - Pointer to buffer with actual data for the value entry.

    DataSize - Size of Data buffer.


Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    PCELL_DATA parent;
    PCELL_DATA list;
    HCELL_INDEX oldchild;
    ULONG       oldindex;
    ULONG       count;
    PHHIVE      Hive;
    HCELL_INDEX Cell;
    ULONG       StorageType;
    ULONG       TempData;
    BOOLEAN     found;
    PCELL_DATA  pdata;
    LARGE_INTEGER systemtime;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmSetValueKey\n"));

    CmpLockRegistryExclusive();
    ASSERT(sizeof(ULONG) == CM_KEY_VALUE_SMALL);


    //
    // Check that we are not being asked to add a value to a key
    // that has been deleted
    //
    if (KeyControlBlock->Delete == TRUE) {
        status = STATUS_KEY_DELETED;
        goto Exit;
    }

    //
    // get reference to parent key, and mark it dirty, since we'll
    // at least set its time stamp
    //
    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;

    parent = HvGetCell(Hive, Cell);
    if (! HvMarkCellDirty(Hive, Cell)) {
        status = STATUS_NO_LOG_SPACE;
        goto Exit;
    }


    StorageType = HvGetCellType(Cell);

    //
    // stash small data if relevent
    //
    TempData = 0;
    if ((DataSize <= CM_KEY_VALUE_SMALL) &&
        (DataSize > 0))
    {
        try {
            RtlMoveMemory(          // yes, move memory, could be 1 byte
                &TempData,          // at the end of a page.
                Data,
                DataSize
                );
         } except (EXCEPTION_EXECUTE_HANDLER) {
            CMLOG(CML_API, CMS_EXCEPTION) {
                KdPrint(("!!CmSetValueKey: code:%08lx\n", GetExceptionCode()));
            }
            status = GetExceptionCode();
            goto Exit;
        }
    }


    //
    // try to find an existing value entry by the same name
    //
    count = parent->u.KeyNode.ValueList.Count;
    found = FALSE;

    if (count > 0) {
        list = HvGetCell(Hive, parent->u.KeyNode.ValueList.List);

        status = CmpFindNameInList(
                    Hive,
                    list,
                    count,
                    ValueName,
                    &oldchild,
                    &pdata,
                    &oldindex
                    );

        if (NT_SUCCESS(status)) {
            found = TRUE;
        }
    }

    if (found) {

        //
        // ----- Existing Value Entry Path -----
        //

        //
        // An existing value entry of the specified name exists,
        // set our data into it.
        //
        status = CmpSetValueKeyExisting(
                    Hive, oldchild, TitleIndex, Type,
                    Data, DataSize, StorageType, TempData
                    );

    } else {

        //
        // ----- New Value Entry Path -----
        //

        //
        // Either there are no existing value entries, or the one
        // specified is not in the list.  In either case, create and
        // fill a new one, and add it to the list
        //
        status = CmpSetValueKeyNew(
                    Hive, parent, &ValueName, TitleIndex,
                    Type, Data, DataSize,
                    StorageType, TempData
                    );
    }

    if (NT_SUCCESS(status)) {

        if (parent->u.KeyNode.MaxValueNameLen < ValueName.Length) {
            parent->u.KeyNode.MaxValueNameLen = ValueName.Length;
        }

        if (parent->u.KeyNode.MaxValueDataLen < DataSize) {
            parent->u.KeyNode.MaxValueDataLen = DataSize;
        }

        KeQuerySystemTime(&systemtime);
        parent->u.KeyNode.LastWriteTime = systemtime;

        CmpReportNotify(
            KeyControlBlock->FullName,
            KeyControlBlock->KeyHive,
            KeyControlBlock->KeyCell,
            REG_NOTIFY_CHANGE_LAST_SET
            );
    }

Exit:
    CmpUnlockRegistry();
    return status;
}


NTSTATUS
CmpSetValueKeyExisting(
    IN PHHIVE  Hive,
    IN HCELL_INDEX OldChild,
    IN ULONG TitleIndex OPTIONAL,
    IN ULONG Type,
    IN PVOID Data,
    IN ULONG DataSize,
    IN ULONG StorageType,
    IN ULONG TempData
    )
/*++

Routine Description:

    Helper for CmSetValueKey, implements the case where the value entry
    being set already exists.

Arguments:

    Hive - hive of interest

    OldChild - hcell_index of the value entry body to which we are to
                    set new data

    TitleIndex - Supplies the title index for ValueName.  The title
        index specifies the index of the localized alias for the ValueName.

    Type - The integer type number of the value entry.

    Data - Pointer to buffer with actual data for the value entry.

    DataSize - Size of Data buffer.

    StorageType - stable or volatile

    TempData - small values are passed here


Return Value:

    STATUS_SUCCESS if it worked, appropriate status code if it did not

--*/
{
    PCELL_DATA pvalue;
    HCELL_INDEX DataCell;
    PCELL_DATA pdata;
    HCELL_INDEX NewCell;
    ULONG realsize;
    BOOLEAN small;
    PUCHAR      StashBuffer;
    ULONG BufferSize;

    ASSERT_CM_LOCK_OWNED();

    //
    // value entry by the specified name already exists
    // oldchild is hcell_index of its value entry body
    //  which we will always edit, so mark it dirty
    //
    if (! HvMarkCellDirty(Hive, OldChild)) {
        return STATUS_NO_LOG_SPACE;
    }

    pvalue = HvGetCell(Hive, OldChild);

    small = CmpIsHKeyValueSmall(realsize, pvalue->u.KeyValue.DataLength);


    if (DataSize <= CM_KEY_VALUE_SMALL) {               // small

        //
        // We are storing a small datum, TempData has data.
        //
        if ((! small) && (realsize > 0))
        {
            //
            // value entry has existing external data to free
            //
            if (! HvMarkCellDirty(Hive, pvalue->u.KeyValue.Data)) {
                return STATUS_NO_LOG_SPACE;
            }
            HvFreeCell(Hive, pvalue->u.KeyValue.Data);
        }

        //
        // write our new small data into value entry body
        //
        pvalue->u.KeyValue.DataLength = DataSize + CM_KEY_VALUE_SPECIAL_SIZE;
        pvalue->u.KeyValue.Data = TempData;
        pvalue->u.KeyValue.Type = Type;

        return STATUS_SUCCESS;

    } else {                                            // large

        //
        // We are storing a "large" datum
        //

        //
        // See if we can write data on top of existing cell
        //
        if ((! small) && (realsize > 0)) {

            DataCell = pvalue->u.KeyValue.Data;
            ASSERT(DataCell != HCELL_NIL);
            pdata = HvGetCell(Hive, DataCell);

            ASSERT(HvGetCellSize(Hive, pdata) > 0);

            if (DataSize <= (ULONG)(HvGetCellSize(Hive, pdata))) {

                //
                // The existing data cell is big enough to hold the
                // new data.  Attempt to copy to stash buffer. if
                // we succeed we can copy directly onto the old cell.
                // if we fail, we must allocate and fill a new cell,
                // and replace the old one with it.
                //
                if (! HvMarkCellDirty(Hive, DataCell)) {
                    return STATUS_NO_LOG_SPACE;
                }

                StashBuffer = NULL;
                if (DataSize <= CmpStashBufferSize) {

                    StashBuffer = CmpStashBuffer;

                } else if (DataSize <= CM_MAX_STASH) {

                    //
                    // Try to allocate a bigger stash buffer.  If it works, keep it and
                    // free the old one.  This prevents pool from becoming too fragmented
                    // if somebody (like SAM) is repeatedly setting very large values
                    //
                    BufferSize = ((DataSize + PAGE_SIZE) & ~(PAGE_SIZE-1));

                    StashBuffer = ExAllocatePoolWithTag(PagedPool, BufferSize, 'bSmC');
                    if (StashBuffer != NULL) {
                        ExFreePool(CmpStashBuffer);
                        CmpStashBuffer = StashBuffer;
                        CmpStashBufferSize = BufferSize;
                    }
                }

                if (StashBuffer != NULL) {
                    //
                    // We have a stash buffer
                    //
                    try {

                        RtlCopyMemory(
                            StashBuffer,
                            Data,
                            DataSize
                            );

                    } except (EXCEPTION_EXECUTE_HANDLER) {
                        CMLOG(CML_API, CMS_EXCEPTION) {
                            KdPrint(("!!CmSetValueKey: code:%08lx\n", GetExceptionCode()));
                        }
                        return GetExceptionCode();
                    }

                    //
                    // We have filled the stash buffer, copy data and finish
                    //
                    RtlCopyMemory(
                        pdata,
                        StashBuffer,
                        DataSize
                        );

                    ASSERT(StashBuffer != NULL);

                    pvalue->u.KeyValue.DataLength = DataSize;
                    pvalue->u.KeyValue.Type = Type;

                    return STATUS_SUCCESS;

                 } // else stashbuffer == null
            } // else existing cell is too small
        } // else there is no existing cell
    } // new cell needed (always large)

    //
    // Either the existing cell is not large enough, or does
    // not exist, or we couldn't stash successfully.
    //
    // Allocate and fill a new cell.  Free the old one.  Store
    // the new's index into the value entry body.
    //
    NewCell = HvAllocateCell(Hive, DataSize, StorageType);

    if (NewCell == HCELL_NIL) {
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    //
    // fill the new data cell
    //
    pdata = HvGetCell(Hive, NewCell);
    try {

        RtlMoveMemory(
            pdata,
            Data,
            DataSize
            );

    } except (EXCEPTION_EXECUTE_HANDLER) {
        CMLOG(CML_API, CMS_EXCEPTION) {
            KdPrint(("!!CmSetValueKey: code:%08lx\n", GetExceptionCode()));
        }
        HvFreeCell(Hive, NewCell);
        return GetExceptionCode();
    }

    //
    // free the old data cell
    //
    if ((! small) && (realsize > 0)) {
        ASSERT(pvalue->u.KeyValue.Data != HCELL_NIL);
        if (! HvMarkCellDirty(Hive, pvalue->u.KeyValue.Data)) {
            HvFreeCell(Hive, NewCell);
            return STATUS_NO_LOG_SPACE;
        }
        HvFreeCell(Hive, pvalue->u.KeyValue.Data);
    }

    //
    // set body
    //
    pvalue->u.KeyValue.DataLength = DataSize;
    pvalue->u.KeyValue.Data = NewCell;
    pvalue->u.KeyValue.Type = Type;

    return STATUS_SUCCESS;
}

NTSTATUS
CmpSetValueKeyNew(
    IN PHHIVE  Hive,
    IN PCELL_DATA Parent,
    IN PUNICODE_STRING ValueName,
    IN ULONG TitleIndex OPTIONAL,
    IN ULONG Type,
    IN PVOID Data,
    IN ULONG DataSize,
    IN ULONG StorageType,
    IN ULONG TempData
    )
/*++

Routine Description:

    Helper for CmSetValueKey, implements the case where the value entry
    being set does not exist.  Will create new value entry and data,
    place in list (which may be created)

Arguments:

    Hive - hive of interest

    Parent - pointer to key node value entry is for

    ValueName - The unique (relative to the containing key) name
        of the value entry.  May be NULL.

    TitleIndex - Supplies the title index for ValueName.  The title
        index specifies the index of the localized alias for the ValueName.

    Type - The integer type number of the value entry.

    Data - Pointer to buffer with actual data for the value entry.

    DataSize - Size of Data buffer.

    StorageType - stable or volatile

    TempData - small data values passed here


Return Value:

    STATUS_SUCCESS if it worked, appropriate status code if it did not

--*/
{
    PCELL_DATA pvalue;
    HCELL_INDEX ValueCell;
    HCELL_INDEX DataCell;
    PCELL_DATA pdata;
    ULONG   count;
    HCELL_INDEX NewCell;

    //
    // Either Count == 0 (no list) or our entry is simply not in
    // the list.  Create a new value entry body, and data.  Add to list.
    // (May create the list.)
    //
    if (Parent->u.KeyNode.ValueList.Count != 0) {
        ASSERT(Parent->u.KeyNode.ValueList.List != HCELL_NIL);
        if (! HvMarkCellDirty(Hive, Parent->u.KeyNode.ValueList.List)) {
            return STATUS_NO_LOG_SPACE;
        }
    }

    //
    // allocate the body of the value entry, and the data
    //
    ValueCell = HvAllocateCell(
                    Hive,
                    CmpHKeyValueSize(Hive, ValueName),
                    StorageType
                    );

    if (ValueCell == HCELL_NIL) {
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    DataCell = HCELL_NIL;
    if (DataSize > CM_KEY_VALUE_SMALL) {
        DataCell = HvAllocateCell(Hive, DataSize, StorageType);
        if (DataCell == HCELL_NIL) {
            HvFreeCell(Hive, ValueCell);
            return STATUS_INSUFFICIENT_RESOURCES;
        }
    }

    //
    // map in the body, and fill in its fixed portion
    //
    pvalue = HvGetCell(Hive, ValueCell);
    pvalue->u.KeyValue.Signature = CM_KEY_VALUE_SIGNATURE;

    //
    // fill in the variable portions of the new value entry,  name and
    // and data are copied from caller space, could fault.
    //
    try {

        //
        // fill in the name
        //
        pvalue->u.KeyValue.NameLength = CmpCopyName(Hive,
                                                    pvalue->u.KeyValue.Name,
                                                    ValueName);
    } except (EXCEPTION_EXECUTE_HANDLER) {
        CMLOG(CML_API, CMS_EXCEPTION) {
            KdPrint(("!!CmSetValueKey: code:%08lx\n", GetExceptionCode()));
        }

        //
        // We have bombed out loading user data, clean up and exit.
        //
        if (DataCell != HCELL_NIL) {
            HvFreeCell(Hive, DataCell);
        }
        HvFreeCell(Hive, ValueCell);
        return GetExceptionCode();
    }

    if (pvalue->u.KeyValue.NameLength < ValueName->Length) {
        pvalue->u.KeyValue.Flags = VALUE_COMP_NAME;
    } else {
        pvalue->u.KeyValue.Flags = 0;
    }

    //
    // fill in the data
    //
    if (DataSize > CM_KEY_VALUE_SMALL) {
        pdata = HvGetCell(Hive, DataCell);

        try {

            RtlMoveMemory(pdata, Data, DataSize);

        } except (EXCEPTION_EXECUTE_HANDLER) {
            CMLOG(CML_API, CMS_EXCEPTION) {
                KdPrint(("!!CmSetValueKey: code:%08lx\n", GetExceptionCode()));
            }

            //
            // We have bombed out loading user data, clean up and exit.
            //
            if (DataCell != HCELL_NIL) {
                HvFreeCell(Hive, DataCell);
            }
            HvFreeCell(Hive, ValueCell);
            return GetExceptionCode();
        }

        pvalue->u.KeyValue.DataLength = DataSize;
        pvalue->u.KeyValue.Data = DataCell;

    } else {
        pvalue->u.KeyValue.DataLength = DataSize + CM_KEY_VALUE_SPECIAL_SIZE;
        pvalue->u.KeyValue.Data = TempData;
    }

    //
    // either add ourselves to list, or make new one with us in it.
    //
    count = Parent->u.KeyNode.ValueList.Count;
    count++;
    if (count > 1) {

        //
        // always grow by one.  if somebody wants some huge number
        // of value entries, they deserve the fragmentation they'll get
        //
        NewCell = HvReallocateCell(
                        Hive,
                        Parent->u.KeyNode.ValueList.List,
                        count * sizeof(HCELL_INDEX)
                        );
    } else {
        NewCell = HvAllocateCell(Hive, sizeof(HCELL_INDEX), StorageType);
    }

    //
    // put ourselves on the list
    //
    if (NewCell != HCELL_NIL) {
        Parent->u.KeyNode.ValueList.List = NewCell;
        pdata = HvGetCell(Hive, NewCell);

        pdata->u.KeyList[count-1] = ValueCell;
        Parent->u.KeyNode.ValueList.Count = count;
        pvalue->u.KeyValue.Type = Type;

        return STATUS_SUCCESS;

    } else {
        // out of space, free all allocated stuff
        if (DataCell != HCELL_NIL) {
            HvFreeCell(Hive, DataCell);
        }
        HvFreeCell(Hive, ValueCell);
        return STATUS_INSUFFICIENT_RESOURCES;
    }
}


NTSTATUS
CmSetLastWriteTimeKey(
    IN PCM_KEY_CONTROL_BLOCK KeyControlBlock,
    IN PLARGE_INTEGER LastWriteTime
    )
/*++

Routine Description:

    The LastWriteTime associated with a key node can be set with
    CmSetLastWriteTimeKey

Arguments:

    KeyControlBlock - pointer to kcb for the key to operate on

    LastWriteTime - new time for key

Return Value:

    NTSTATUS - Result code from call, among the following:

        <TBS>

--*/
{
    NTSTATUS    status;
    PCELL_DATA parent;
    PHHIVE      Hive;
    HCELL_INDEX Cell;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmSetLastWriteTimeKey\n"));

    CmpLockRegistryExclusive();

    //
    // Check that we are not being asked to modify a key
    // that has been deleted
    //
    if (KeyControlBlock->Delete == TRUE) {
        status = STATUS_KEY_DELETED;
        goto Exit;
    }

    Hive = KeyControlBlock->KeyHive;
    Cell = KeyControlBlock->KeyCell;

    parent = HvGetCell(Hive, Cell);
    if (! HvMarkCellDirty(Hive, Cell)) {
        status = STATUS_NO_LOG_SPACE;
        goto Exit;
    }

    parent->u.KeyNode.LastWriteTime = *LastWriteTime;

Exit:
    CmpUnlockRegistry();
    return status;
}





NTSTATUS
CmLoadKey(
    IN POBJECT_ATTRIBUTES TargetKey,
    IN POBJECT_ATTRIBUTES SourceFile
    )

/*++

Routine Description:

    A hive (file in the format created by NtSaveKey) may be linked
    into the active registry with this call.  UNLIKE NtRestoreKey,
    the file specified to NtLoadKey will become the actual backing
    store of part of the registry (that is, it will NOT be copied.)

    The file may have an associated .log file.

    If the hive file is marked as needing a .log file, and one is
    not present, the call will fail.

    The name specified by SourceFile must be such that ".log" can
    be appended to it to generate the name of the log file.  Thus,
    on FAT file systems, the hive file may not have an extension.

    This call is used by logon to make the user's profile available
    in the registry.  It is not intended for use doing backup,
    restore, etc.  Use NtRestoreKey for that.

    N.B.  This routine assumes that the object attributes for the file
          to be opened have been captured into kernel space so that
          they can safely be passed to the worker thread to open the file
          and do the actual I/O.

Arguments:

    TargetKey - specifies the path to a key to link the hive to.
                path must be of the form "\registry\user\<username>"

    SourceFile - specifies a file.  while file could be remote,
                that is strongly discouraged.

Return Value:

    NTSTATUS - values TBS.

--*/
{
    PCMHIVE NewHive;
    NTSTATUS Status;
    BOOLEAN Allocate;
    REGISTRY_COMMAND Command;
    SECURITY_QUALITY_OF_SERVICE ServiceQos;
    SECURITY_CLIENT_CONTEXT ClientSecurityContext;


    //
    // Obtain the security context here so we can use it
    // later to impersonate the user, which we will do
    // if we cannot access the file as SYSTEM.  This
    // usually occurs if the file is on a remote machine.
    //
    ServiceQos.Length = sizeof(SECURITY_QUALITY_OF_SERVICE);
    ServiceQos.ImpersonationLevel = SecurityImpersonation;
    ServiceQos.ContextTrackingMode = SECURITY_DYNAMIC_TRACKING;
    ServiceQos.EffectiveOnly = TRUE;
    Status = SeCreateClientSecurity(CONTAINING_RECORD(KeGetCurrentThread(),ETHREAD,Tcb),
                                    &ServiceQos,
                                    FALSE,
                                    &ClientSecurityContext);
    if (!NT_SUCCESS(Status)) {
        return(Status);
    }

    CmpLockRegistryExclusive();

    Command.Command = REG_CMD_HIVE_OPEN;
    Command.Allocate = TRUE;
    Command.FileAttributes = SourceFile;
    Command.ImpersonationContext = &ClientSecurityContext;

    Status = CmpWorkerCommand(&Command);

    SeDeleteClientSecurity( &ClientSecurityContext );

    NewHive = Command.CmHive;
    Allocate = Command.Allocate;

    if (!NT_SUCCESS(Status)) {
        CmpUnlockRegistry();
        return(Status);
    }

    //
    // We now have a succesfully loaded and initialized CmHive, so we
    // just need to link that into the appropriate spot in the master hive.
    //
    Status = CmpLinkHiveToMaster(TargetKey->ObjectName,
                                 TargetKey->RootDirectory,
                                 NewHive,
                                 Allocate,
                                 TargetKey->SecurityDescriptor);

    Command.CmHive = NewHive;
    if (NT_SUCCESS(Status)) {
        //
        // add new hive to hivelist
        //
        Command.Command = REG_CMD_ADD_HIVE_LIST;

    } else {
        //
        // Close the files we've opened.
        //
        Command.Command = REG_CMD_HIVE_CLOSE;
    }
    CmpWorkerCommand(&Command);

    //
    // We've given user chance to log on, so turn on quota
    //
    if ((CmpProfileLoaded == FALSE) &&
        (CmpWasSetupBoot == FALSE)) {
        CmpProfileLoaded = TRUE;
        CmpSetGlobalQuotaAllowed();
    }

    CmpUnlockRegistry();
    return(Status);
}

#if DBG
ULONG
CmpUnloadKeyWorker(
    PCM_KEY_CONTROL_BLOCK Current,
    PVOID                 Context1,
    PVOID                 Context2
    )
{
    if (Current->KeyHive == Context1) {
        KdPrint(("%wZ\n", &Current->FullName));
    }
    return KCB_WORKER_CONTINUE;   // always keep searching
}
#endif


NTSTATUS
CmUnloadKey(
    IN PHHIVE Hive,
    IN HCELL_INDEX Cell
    )

/*++

Routine Description:

    Unlinks a hive from its location in the registry, closes its file
    handles, and deallocates all its memory.

    There must be no key control blocks currently referencing the hive
    to be unloaded.

Arguments:

    Hive - Supplies a pointer to the hive control structure for the
           hive to be unloaded

    Cell - supplies the HCELL_INDEX for the root cell of the hive.

Return Value:

    NTSTATUS

--*/

{
    PCMHIVE CmHive;
    REGISTRY_COMMAND Command;

    CMLOG(CML_WORKER, CMS_CM) KdPrint(("CmUnloadKey\n"));

    CmpLockRegistryExclusive();

    //
    // Make sure the cell passed in is the root cell of the hive.
    //
    if (Cell != Hive->BaseBlock->RootCell) {
        CmpUnlockRegistry();
        return(STATUS_INVALID_PARAMETER);
    }

    //
    // Make sure there are no open references to key control blocks
    // for this hive.  If there are none, then we can unload the hive.
    //

    CmHive = CONTAINING_RECORD(Hive, CMHIVE, Hive);
    if (CmHive->KcbCount != 0) {
#if DBG
        KdPrint(("List of keys open against hive unload was attempted on:\n"));
        CmpSearchKeyControlBlockTree(
            CmpUnloadKeyWorker,
            Hive,
            NULL
            );
#endif
        CmpUnlockRegistry();
        return(STATUS_CANNOT_DELETE);
    }

    //
    // Remove the hive from the HiveFileList
    //
    Command.Command = REG_CMD_REMOVE_HIVE_LIST;
    Command.CmHive = (PCMHIVE)Hive;
    CmpWorkerCommand(&Command);

    //
    // Unlink from master hive, remove from list
    //
    CmpDestroyHive(Hive, Cell);

    CmpUnlockRegistry();

    HvFreeHive(Hive);

    //
    // Close the hive files
    //
    Command.Command = REG_CMD_HIVE_CLOSE;
    Command.CmHive = CmHive;
    CmpWorkerCommand(&Command);

    //
    // free the cm level structure
    //
    CmpFree(CmHive, sizeof(CMHIVE));

    return(STATUS_SUCCESS);

}



VOID
CmpDoFlushAll(
    VOID
    )
/*++

Routine Description:

    Flush all hives.

    Runs in the context of the CmpWorkerThread.

    Runs down list of Hives and applies HvSyncHive to them.

    NOTE: Hives which are marked as HV_NOLAZYFLUSH are *NOT* flushed
          by this call.  You must call HvSyncHive explicitly to flush
          a hive marked as HV_NOLAZYFLUSH.

Arguments:

Return Value:

    NONE

--*/
{
    NTSTATUS    Status;
    PLIST_ENTRY p;
    PCMHIVE     h;
/*
    ULONG rc;
*/
    extern PCMHIVE CmpMasterHive;

    //
    // If writes are not working, lie and say we succeeded, will
    // clean up in a short time.  Only early system init code
    // will ever know the difference.
    //
    if (CmpNoWrite) {
        return;
    }

    //
    // traverse list of hives, sync each one
    //
    p = CmpHiveListHead.Flink;
    while (p != &CmpHiveListHead) {

        h = CONTAINING_RECORD(p, CMHIVE, HiveList);

/*
#if DBG
        if (h!=CmpMasterHive) {
            rc = CmCheckRegistry(h, FALSE);

            if (rc!=0) {
                KdPrint(("CmpDoFlushAll: corrupt hive, rc = %08lx\n",rc));
                DbgBreakPoint();
            }
        }
#endif
*/
        if (!(h->Hive.HiveFlags & HIVE_NOLAZYFLUSH)) {

            Status = HvSyncHive((PHHIVE)h);

            //
            // WARNNOTE - the above means that a lazy flush or
            //            or shutdown flush did not work.  we don't
            //            know why.  there is noone to report an error
            //            to, so continue on and hope for the best.
            //            (in theory, worst that can happen is user changes
            //             are lost.)
            //
        }


        p = p->Flink;
    }
}


NTSTATUS
CmReplaceKey(
    IN PHHIVE Hive,
    IN HCELL_INDEX Cell,
    IN PUNICODE_STRING NewHiveName,
    IN PUNICODE_STRING OldFileName
    )

/*++

Routine Description:

    Renames the hive file for a running system and replaces it with a new
    file.  The new file is not actually used until the next boot.

Arguments:

    Hive - Supplies a hive control structure for the hive to be replaced.

    Cell - Supplies the HCELL_INDEX of the root cell of the hive to be
           replaced.

    NewHiveName - Supplies the name of the file which is to be installed
            as the new hive.

    OldFileName - Supplies the name of the file which the existing hive
            file is to be renamed to.

Return Value:

    NTSTATUS

--*/

{
    REGISTRY_COMMAND Command;
    CHAR ObjectInfoBuffer[512];
    NTSTATUS Status;
    OBJECT_ATTRIBUTES Attributes;
    PCMHIVE NewHive;
    POBJECT_NAME_INFORMATION NameInfo;

    CmpLockRegistryExclusive();

    if (Hive->HiveFlags & HIVE_HAS_BEEN_REPLACED) {
        CmpUnlockRegistry();
        return STATUS_FILE_RENAMED;
    }

    //
    // First open the new hive file and check to make sure it is valid.
    //
    InitializeObjectAttributes(&Attributes,
                               NewHiveName,
                               OBJ_CASE_INSENSITIVE,
                               NULL,
                               NULL);

    Command.Command = REG_CMD_HIVE_OPEN;
    Command.FileAttributes = &Attributes;
    Command.Allocate = FALSE;
    Command.ImpersonationContext = NULL;
    Status = CmpWorkerCommand(&Command);
    if (!NT_SUCCESS(Status)) {
        CmpUnlockRegistry();
        return(Status);
    }
    ASSERT(Command.Allocate == FALSE);

    NewHive = Command.CmHive;

    //
    // The new hive exists, and is consistent, and we have it open.
    // Now rename the current hive file.
    //
    Command.Command = REG_CMD_RENAME_HIVE;
    Command.NewName = OldFileName;
    Command.OldName = (POBJECT_NAME_INFORMATION)ObjectInfoBuffer;
    Command.NameInfoLength = sizeof(ObjectInfoBuffer);
    Command.CmHive = CONTAINING_RECORD(Hive, CMHIVE, Hive);
    Status = CmpWorkerCommand(&Command);
    if (!NT_SUCCESS(Status)) {
        //
        // rename failed, close the files associated with the new hive
        //
        Command.CmHive = NewHive;
        Command.Command = REG_CMD_HIVE_CLOSE;
        CmpWorkerCommand(&Command);
        CmpUnlockRegistry();
        return(Status);
    }

    //
    // The existing hive was successfully renamed, so try to rename the
    // new file to what the old hive file was named.  (which was returned
    // into ObjectInfoBuffer by the worker thread)
    //
    Hive->HiveFlags |= HIVE_HAS_BEEN_REPLACED;
    NameInfo = (POBJECT_NAME_INFORMATION)ObjectInfoBuffer;
    Command.Command = REG_CMD_RENAME_HIVE;
    Command.NewName = &NameInfo->Name;
    Command.OldName = NULL;
    Command.NameInfoLength = 0;
    Command.CmHive = NewHive;
    Status = CmpWorkerCommand(&Command);
    if (!NT_SUCCESS(Status)) {

        //
        // Close the handles to the new hive
        //
        Command.Command = REG_CMD_HIVE_CLOSE;
        Command.CmHive = NewHive;
        CmpWorkerCommand(&Command);

        //
        // We are in trouble now.  We have renamed the existing hive file,
        // but we couldn't rename the new hive file!  Try to rename the
        // existing hive file back to where it was.
        //
        Command.Command = REG_CMD_RENAME_HIVE;
        Command.NewName = &NameInfo->Name;
        Command.OldName = NULL;
        Command.NameInfoLength = 0;
        Command.CmHive = CONTAINING_RECORD(Hive, CMHIVE, Hive);
        CmpWorkerCommand(&Command);
        if (!NT_SUCCESS(Command.Status)) {
            CMLOG(CML_BUGCHECK, CMS_SAVRES) {
                KdPrint(("CmReplaceKey: renamed existing hive file, but couldn't\n"));
                KdPrint(("              rename new hive file (%08lx) ",Status));
                KdPrint((" or replace old hive file (%08lx)!\n",Command.Status));
            }

            //
            // WARNNOTE:
            //      To get into this state, the user must have relevent
            //      privileges, deliberately screw with system in an attempt
            //      to defeat it, AND get it done in a narrow timing window.
            //
            //      Further, if it's a user profile, the system will
            //      still come up.
            //
            //      Therefore, return an error code and go on.
            //

            Status = STATUS_REGISTRY_CORRUPT;

        }
    }

    //
    // All of the renaming is done.  However, we are holding an in-memory
    // image of the new hive.  Release it, since it will not actually
    // be used until next boot.
    //
    // Do not close the open file handles to the new hive, we need to
    // keep it locked exclusively until the system is rebooted to prevent
    // people from mucking with it.
    //
    RemoveEntryList(&(NewHive->HiveList));

    HvFreeHive((PHHIVE)NewHive);

    CmpFree(NewHive, sizeof(CMHIVE));

    CmpUnlockRegistry();
    return(Status);
}
