/*++

Copyright (c) 1992, 1993  Digital Equipment Corporation

Module Name:

    eisaprof.c

Abstract:

    This module handles the Profile Counter and all Profile counter functions 
    for the standard EISA interval timer.

Author:

    Jeff McLeman (mcleman) 05-June-1992

Environment:

    Kernel mode

Revision History:


    Rod Gamache	[DEC]	9-Mar-1993
			Fix profile clock.


--*/

#include "halp.h"
#include "eisa.h"
#include "halprof.h"

//
// Define global data.
//

//
// Values used for Profile Clock
//

//	Convert the interval to rollover count for 8254 timer. Since
//	the 8254 counts down a 16 bit value at the clock rate of 1.193 MHZ,
//	the computation is:
//
//	RolloverCount = (Interval * 0.0000001) * (1193 * 1000000)
//		      = Interval * .1193
//		      = Interval * 1193 / 10000

#define PROFILE_INTERVAL 1193
#define PROFILE_INTERVALS_PER_100NS 10000/1193
#define MIN_PROFILE_TICKS 4
#define MAX_PROFILE_TICKS 0x10000		// 16 bit counter (zero is max)

//
// Since the profile timer interrupts at a frequency of 1.193 MHZ, we
// have .1193 intervals each 100ns. So we need a more reasonable value.
// If we compute the timer based on 1600ns intervals, we get 16 * .1193 or
// about 1.9 ticks per 16 intervals.
//
// We round this to 2 ticks per 1600ns intervals.
//

#define PROFILE_TIMER_1600NS_TICKS 2

//
// Default Profile Interval to be about 1ms.
//

ULONG HalpProfileInterval = PROFILE_TIMER_1600NS_TICKS * PROFILE_INTERVALS_PER_100NS * 10000 / 16; // ~1ms

//
// Default Number of Profile Clock Ticks per sample
//

ULONG HalpNumberOfTicks = 1;


BOOLEAN
HalpProfileInterrupt(
    PKTRAP_FRAME TrapFrame
    )
/*++

Routine Description:

    This routine is entered as a result of an interrupt generated by
    the profile timer. Its function is to acknowlege the interrupt and
    transfer control to the standard system routine to update the
    system profile time.

Arguments:

    TrapFrame - Supplies a pointer to the trap frame for the profile interrupt.

Returned Value:

    None

--*/
{

    //
    // See if profiling is active
    //

    if ( PCR->ProfileCount ) {

        //
        // Check to see if the interval has expired
        // If it has then call the kernel routine for profile
        // and reset the count, else return.


        if ( !(--PCR->ProfileCount) ) {

            KeProfileInterrupt( TrapFrame );
            PCR->ProfileCount = HalpNumberOfTicks;

        }
    }

    return TRUE;

}

ULONG
HalSetProfileInterval (
    IN ULONG Interval
    )

/*++

Routine Description:

    This routine sets the profile interrupt interval.

Arguments:

    Interval - Supplies the desired profile interval in 100ns units.

Return Value:

    The actual profile interval.

--*/

{

    HalpProfileInterval = (Interval/16) * PROFILE_TIMER_1600NS_TICKS;

    HalpProfileInterval = ( HalpProfileInterval < MIN_PROFILE_TICKS ) ?
				MIN_PROFILE_TICKS : HalpProfileInterval;

    return HalpProfileInterval * PROFILE_INTERVALS_PER_100NS;
}


VOID
HalStartProfileInterrupt (
    ULONG Reserved
    )

/*++

Routine Description:

    This routine turns on the profile interrupt.

    N.B. This routine must be called at PROCLK_LEVEL while holding the
        profile lock.

Arguments:

    None.

Return Value:

    None.

--*/

{

    //
    // Assume that we only need 1 clock tick before we collect data
    //

    HalpNumberOfTicks = 1;

    if ( HalpProfileInterval > MAX_PROFILE_TICKS ) {

	HalpNumberOfTicks = HalpProfileInterval / (MAX_PROFILE_TICKS / 4);
	HalpNumberOfTicks = 4 * HalpNumberOfTicks;
	HalpProfileInterval = MAX_PROFILE_TICKS / 4;

    }

    //
    // Set current profile count and interval.
    //

    PCR->ProfileCount = HalpNumberOfTicks;
    PCR->ProfileInterval = HalpProfileInterval;

    PIC_PROFILER_ON(HalpProfileInterval);

    return;
}


VOID
HalStopProfileInterrupt (
    ULONG Reserved
    )

/*++

Routine Description:

    This routine turns off the profile interrupt.

    N.B. This routine must be called at PROCLK_LEVEL while holding the
        profile lock.

Arguments:

    None.

Return Value:

    None.

--*/

{

    //
    // Clear the current profile count and turn off the profiler timer.
    //

    PCR->ProfileCount = 0;

    PIC_PROFILER_OFF();

    return;
}


