/*++

   Copyright    (c)    1995    Microsoft Corporation

   Module  Name :

      inetlog.h

   Abstract:

      Declares manifests, functions and macros for Internet Serviecs Logging.

   Author:

       Murali R. Krishnan    ( MuraliK )    1-Feb-1995

   Environment:

      User Mode -- Win32

   Project:

      Internet Services Common DLL

   Revision History:
      MuraliK        15-May-1995    Extended INETLOG_INFORMATION to accomodate
                                      a few new fields.
--*/

# ifndef _INETLOG_H_
# define _INETLOG_H_

/*************************************************
 *
 *    Logging
 *
 *      Each service produces a log of requests processed.
 *      In our services we provide an option for multiple services running on
 *       multiple servers to log to the same place.
 *       This is especially true for logging to an SQL table.
 *       ( ==> requires writing information about service and server names)
 *
 *      The Log consists of a sequence of log records.
 *      Each log record includes following details:
 *          Security:  Host Name of Client ( or IP address),
 *                     User Name of Client ( used for secure access)
 *                     Password of Client ( for clear text case)
 *          TimeStamp: date and time of completion of processing request.
 *          Statistics: Count of bytes sent,
 *                      Count of bytes received,
 *                      Time Reqd. to process request.
 *          Service:   Service Name and Server Name.
 *          Service Specific Information ( generated by services).
 *             Status:  service specific status (DWORD)
 *                      win32 status (DWORD)
 *             Request: operation being performed.
 *                      target for the operation
 *                      parameters for the operation
 *
 **************************************************/



/************************************************************
 *     Include Headers
 ************************************************************/


# include <nt.h>
# include <ntrtl.h>
# include <nturtl.h>
# include <windows.h>
# include <lmcons.h>

#include <eventlog.hxx>

# define MAX_SERVER_NAME_LEN           ( MAX_COMPUTERNAME_LENGTH + 1)
# define MAX_SERVICE_NAME_LEN          ( SNLEN + 1)
# define MAX_DATABASE_NAME_LEN         (MAX_PATH)

# define MAX_LOG_RECORD_FORMAT_LEN     ( 120)
# define MAX_LOG_RECORD_LEN            ( 1200)

# define MAX_FILE_TRUNCATION_SIZE      ( 1024 * 1024)  // in bytes. set to 1MB



/************************************************************
 *   Type Definitions
 ************************************************************/


//
//  The following handle ( INETLOG_HANDLE) is to be used as
//     handle for processing log information.
//

typedef VOID * INETLOG_HANDLE;

# define INVALID_INETLOG_HANDLE_VALUE          ( NULL)



/*++

  struct   INETLOG_INFORMATION

     This structure specifies the information required to write
      one log record.

--*/

typedef struct  _INETLOG_INFORMATIONA {

    LPCSTR     pszClientHostName;
    LPCSTR     pszClientUserName;
    LPCSTR     pszClientPassword;     // useful for FTP style clear text logon

    LPCSTR     pszServerIpAddress;    // input ip address for connection

    DWORD      msTimeForProcessing;   // time required for processing
    LARGE_INTEGER liBytesSent;        // count of bytes sent
    LARGE_INTEGER liBytesRecvd;       // count of bytes recvd

    DWORD      dwWin32Status;         // Win32 error code. 0 for success
    DWORD      dwServiceSpecificStatus; // status: whatever service wants.

    //
    // Service Specific Message is broken down into 3 fields.
    //   Operation      Target      Parameters(if any)
    //

    LPCSTR     pszOperation;          //  eg: 'get'  in FTP
    LPCSTR     pszTarget;             // target path/machine name
    LPCSTR     pszParameters;         // string containing parameters.

} INETLOG_INFORMATIONA, * PINETLOG_INFORMATIONA;


typedef struct  _INETLOG_INFORMATIONW {

    LPCWSTR    pszClientHostName;
    LPCWSTR    pszClientUserName;
    LPCWSTR    pszClientPassword;     // useful for FTP style clear text logon

    LPCWSTR    pszServerIpAddress;    // input ip address for connection

    DWORD      msTimeForProcessing;   // time required for processing
    LARGE_INTEGER liBytesSent;        // count of bytes sent
    LARGE_INTEGER liBytesRecvd;       // count of bytes recvd


    DWORD      dwWin32Status;         // Win32 error code. 0 for success
    DWORD      dwServiceSpecificStatus; // status: whatever service wants.

    //
    // Service Specific Message is broken down into 3 fields.
    //   Operation      Target      Parameters(if any)
    //

    LPCWSTR    pszOperation;          //  eg: 'get'  in FTP
    LPCWSTR    pszTarget;             // target path/machine name
    LPCWSTR    pszParameters;         // string containing parameters.

} INETLOG_INFORMATIONW, * PINETLOG_INFORMATIONW;





typedef enum  _LOG_TYPE   {

    InetLogInvalidType = -1,
    InetNoLog = 0,
    InetLogToFile,
    InetLogToSql
}  INETLOG_TYPE;


typedef enum  _INETLOG_PERIOD {

    //
    // these options identify logging periods for InetLogToFile
    //
    InetLogInvalidPeriod = -1,
    InetLogNoPeriod = 0,           // same as INET_LOG_TO_FILE, with size
    InetLogDaily,                  // one file per day
    InetLogWeekly,                 // one log file per week
    InetLogMonthly,                // one log file per month
    InetLogYearly                  // one log file per year.

} INETLOG_PERIOD;


enum INETLOG_FORMAT {

    InternetStdLogFormat = 0,        // "Internet Standard Log"
    InetsvcsBinaryLogFormat,         // "Internet Services Binary Log"
    CustomLogFormat,
    NCSALogFormat

};  // enum INETLOG_FORMAT




/*++

  struct  INETLOG_CONFIGURATION

    This structure contains the configuration information used for logging.

    The configuration includes:
      Format of Log record -- specifies the order in which the log record
         is written. ( serialization of INETLOG_INFORMATION).
      Type of Logging.  ( LOG_TYPE)
      Parameters depending upon type of logging.

      Type                      Parameters:
      InetNoLog                None
      InetLogToFile            Directory containing file; truncation size +
                                period ( daily, weekly, monthly).
      INET_LOG_TO_SQL          Sql Server Name, Sql Database Name,
                               Sql Table Name.
                             ( the table has to be already created).

      for SQL and logging to remote files eg: \\logserver\logshare\logdir
      we also need information about UserName and Passwd ( LSA_SECRET)
         for logging.  NYI.

     We do not support the remote directory in present version ( 2/2/95)

--*/
typedef struct _INETLOG_CONFIGURATIONA {

    INETLOG_TYPE   inetLogType;

    union {

        struct {

        //
        //  Used for InetLogToFile and InetLogToPeriodicFile
        //
            CHAR       rgchLogFileDirectory[ MAX_PATH];
            DWORD      cbSizeForTruncate;
            INETLOG_PERIOD  ilPeriod;
            DWORD      cbBatchSize; // count of bytes to batch up per write.
            INETLOG_FORMAT  ilFormat;

        } logFile;

        struct {

            //
            // Used for InetLogToSql
            //  ODBC bundles DatabaseName and ServerName and ServerType
            //   using a logical name called DataSource.
            //
            CHAR       rgchDataSource[ MAX_DATABASE_NAME_LEN];
            CHAR       rgchTableName[ MAX_TABLE_NAME_LEN];
            CHAR       rgchUserName[ MAX_USER_NAME_LEN];
            CHAR       rgchPassword[ MAX_PASSWORD_LEN];
        } logSql;

    } u;

    CHAR      rgchLogRecordFormat[ MAX_LOG_RECORD_FORMAT_LEN];

}  INETLOG_CONFIGURATIONA, * PINETLOG_CONFIGURATIONA;



typedef struct _INETLOG_CONFIGURATIONW {

    INETLOG_TYPE   inetLogType;

    union {

        struct {

            WCHAR      rgchLogFileDirectory[ MAX_PATH];
            DWORD      cbSizeForTruncation;
            INETLOG_PERIOD  ilPeriod;
            WCHAR      rgchUserName[ MAX_USER_NAME_LEN];
            INETLOG_FORMAT  ilFormat;

        } logFile;

        struct {

            //
            // Used for InetLogToSql
            //  ODBC bundles DatabaseName and ServerName and ServerType
            //   using a logical name called DataSource.
            //
            WCHAR      rgchDataSource[ MAX_DATABASE_NAME_LEN];
            WCHAR      rgchTableName[ MAX_TABLE_NAME_LEN];
            WCHAR      rgchUserName[ MAX_USER_NAME_LEN];
            WCHAR      rgchPassword[ MAX_PASSWORD_LEN];
        } logSql;

    } u;


    WCHAR      rgchLogRecordFormat[ MAX_LOG_RECORD_FORMAT_LEN];

}  INETLOG_CONFIGURATIONW, * PINETLOG_CONFIGURATIONW;




/************************************************************
 *   Function Prototypes
 ************************************************************/


/*++
  TsInitializeInetLog()

  Description:

    This function initializes the InetLog module with the common parameters
        being read from the specified registry.
    It also initializes all other global information for this logging module

  Arguments:

     pszRegParamKey     pointer to Unicode string containing the
                            registry key name for common Parameters
                            for the logging module.

  Returns:
    Win32 Error code.  NO_ERROR on success.

--*/

dllexp
DWORD
TsInitializeInetLog%(
      IN    LPCTSTR%        pszSvcRegParamKey
      );


/*++
  TsCleanupInetLog()

  Description:

    This function cleans up common data structures.

  Arguments:

    None 

  Returns:
    Win32 Error code.  NO_ERROR on success.

--*/

dllexp
DWORD
TsCleanupInetLog(VOID);




/*++
  TsCreateInetLog()

  Description:

    This function creates a InetLog handle for a specified service.
    It loads parameters related to logging type, information and format
     from registry using the key provided.
    If the pszSvcRegParamKey is NULL,
     then this functions uses the registry entry provided by
       HKEY_LOCAL_MACHINE\System\CurrentControSet\pszServiceName\Parameters\...

    The logging related parameters in registry configure the type of log,
      structure of log files and records and fields to log.

  Arguments:

     pszServiceName        pointer to Unicode string containing service name.

     pszSvcRegParamKey     pointer to Unicode string containing the
                            registry key name for Parameters of the service.

  Returns:

    On success it returns valid INETLOG_HANDLE.
    On failure it returns INVALID_INETLOG_HANDLE_VALUE.
     Use GetLastError() for detailed error.

--*/

dllexp
INETLOG_HANDLE
TsCreateInetLog%(
      IN    LPCTSTR%        pszServiceName,
      IN    EVENT_LOG *     pEventLog,
      IN    LPCTSTR%        pszSvcRegParamKey           // Optional
      );



/*++
  TsLogInformation()

  Description:
     This function logs information provided to the destination as
      configured when the InetLog handle was created.

  Arguments:
     hInetLog     Handle for InetLog object. This contains the logging
                   configuration and state information.

     pInetLogInfo pointer to InetLogInformation object, that contains the
                  data to be written as a single log record.

     pszErrorMessage -  pointer to error buffer that on 
                  failure may contain the error message
     
     lpcchErrorMessage - pointer to DWORD containing the count of chars
                  that can be stored in the buffer pszErrorMessage.
                  On return contains the number of chars returned if there
                   is a failure and error message is generated.
          
  Returns:
     Win32 error code
--*/

dllexp
DWORD
TsLogInformation%(
      IN OUT INETLOG_HANDLE  hInetLog,
      IN const INETLOG_INFORMATION%  * pInetLogInfo,
      OUT LPTSTR%            pszErrorMessage,
      IN OUT LPDWORD         lpcchErrorMessage
      );


/*++
  TsCloseInetLog()

  Description:

     This functions closes the InetLog handle. It closes any open files or
       open handles maintained internally. It also cleans up other state
       information present.

  Arguments:

     hInetLog     handle for InetLog object, that needs to be closed.

  Returns:

     TRUE  on success and FALSE on failure.
     Detailed error message can be obtained using GetLastError().

--*/

dllexp
BOOL
TsCloseInetLog(
        IN OUT INETLOG_HANDLE  hInetLog
        );




typedef enum {

    IlIsLoggingOn = 1,   // value is BOOL
    IlLogType,           // value is DWORD containing INETLOG_TYPE
    IlLogFilePeriod,     // value is DWORD containing INETLOG_PERIOD
                         // only valid for file based logging
    IlConfigurationW
} INETLOG_INFO_OPTION;


/*++
  TsGetLogInformation()

  Description:

     This functions checks and reports information requested about logging
        handle in use. This function should be called with a valid
        InetLogHandle.

     Mainly this information can be used for optimizing the callers.
        For Example: If Logging is not turned on, then TsLogInformation need
          not be called.

  Arguments:

     hInetLog     handle for InetLog object, about which inquiry is made.
     IlOptionName Log Information Option indicating what info is required.
     pBuffer      pointer to buffer in which the data is copied into at return.
     lpdwBuffer   pointer to DWORD; contains size of buffer on call and
                    contains the count of bytes written on return.

  Returns:

     TRUE  on success and FALSE on failure.
     Detailed error message can be obtained using GetLastError().

--*/

dllexp
BOOL
TsGetLogInformation(
        IN INETLOG_HANDLE  hInetLog,
        IN INETLOG_INFO_OPTION  IlOptionName,
        IN PBYTE           pBuffer,
        IN LPDWORD         lpdwBuffer
        );



/*++
  TsModifyLogConfiguration()

  This function is used for dynamically changing the log configuration, given
   a new log configuration and inetlog handle.
  The function updates the registry if new logging object is
         successfully created.

  This function applies new configuration to dynamically change the manner
        logging is done or destination of logging.

  If the pszSvcRegParamKey is NULL, then this functions uses the
     registry entry provided by
       HKEY_LOCAL_MACHINE\System\CurrentControSet\pszServiceName\Parameters\...

  Arguments:

     hInetLog     handle for InetLog object, that needs to be modified.
     pszSvcRegParamKey     pointer to Unicode string containing the
                            registry key name for Parameters of the service.
     pInetLogConfig the new log configuration to be applied on this log handle.


     pszErrorMessage -  pointer to error buffer that on 
                  failure may contain the error message
     
     lpcchErrorMessage - pointer to DWORD containing the count of chars
                  that can be stored in the buffer pszErrorMessage.
                  On return contains the number of chars returned if there
                   is a failure and error message is generated.

  Returns:
     Win32 error code

  Atmost only one write is permitted to modify information associated with
    a logging handle. It is caller's responsibility to ensure the proper use.

--*/

dllexp
DWORD
TsModifyLogConfiguration%(
      IN OUT INETLOG_HANDLE  hInetLog,
      IN LPCTSTR%            pszSvcRegParamKey,      // Optional
      IN const INETLOG_CONFIGURATION%  * pInetLogConfig,
      OUT LPTSTR%            pszErrorMessage,
      IN OUT LPDWORD         lpcchErrorMessage
      );




//
//  The APIs TsReadInetLogConfiguration() and TsWriteInetLogConfiguration()
//    can be used by RPC functions for administering / changing parameters
//    from an admin UI
//



/*++

  TsReadInetLogConfiguration()

  Description:

     This function reads the INETLOG configuration information
      for given service name. The configuration information may be stored
      in registry.

    If the pszSvcRegParamKey is NULL, then this functions uses the
     registry entry provided by
       HKEY_LOCAL_MACHINE\System\CurrentControSet\pszServiceName\Parameters\...


  Arguments:

     pszServiceName        pointer to Unicode string containing service name.

     pszSvcRegParamKey     pointer to Unicode string containing the
                            registry key name for Parameters of the service.

     pInetLogConfiguration pointer to INETLOG_CONFIGURATION structure which
                            on successful read contains the data read.

  Returns:

     TRUE on success and FALSE if failure.
     Detailed error code can be obtained using GetLastError()

--*/

dllexp
BOOL
TsReadInetLogConfiguration%(
      IN LPCTSTR%                       pszServiceName,
      IN LPCTSTR%                       pszSvcRegParamKey,      // Optional
      OUT PINETLOG_CONFIGURATION%       pInetLogConfiguration
      );

/*++

  TsWriteInetLogConfiguration()

  Description:

     This function writes the INETLOG configuration information
      for given service name. The configuration information may be stored
      in registry.

    If the pszSvcRegParamKey is NULL, then this functions uses the
     registry entry provided by
       HKEY_LOCAL_MACHINE\System\CurrentControSet\pszServiceName\Parameters\...


  Arguments:

     pszServiceName        pointer to Unicode string containing service name.

     pszSvcRegParamKey     pointer to Unicode string containing the
                            registry key name for Parameters of the service.

     pEventLog             pointer to eventlog to write out any errors.

     pInetLogConfiguration pointer to INETLOG_CONFIGURATION structure that
                            contains the data to be written.

  Returns:

     TRUE on success and FALSE if failure.
     Detailed error code can be obtained using GetLastError()

--*/

dllexp
BOOL
TsWriteInetLogConfiguration%(
      IN LPCTSTR%                       pszServiceName,
      IN LPCTSTR%                       pszSvcRegParamKey,     // Optional
      IN EVENT_LOG *                    pEventLog,
      IN const INETLOG_CONFIGURATION% * pInetLogConfiguration
      );



# if DBG

VOID PrintInetLogInformationW( const INETLOG_INFORMATIONW * pinetLogInfo);

VOID PrintInetLogConfigurationW( const INETLOG_CONFIGURATIONW * pilConfig);


# endif // DBG

# endif // _INETLOG_H_

/************************ End of File ***********************/
