/**********************************************************************/
/**                       Microsoft Windows/NT                       **/
/**                Copyright(c) Microsoft Corp., 1991                **/
/**********************************************************************/

/*
    trust.hxx
    This file contains the class declarations for the TRUST_DIALOG
    class.  The TRUST_DIALOG class is used to manipulate the trusted
    domain list.


    FILE HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.

*/

#ifndef _TRUST_HXX_
#define _TRUST_HXX_


#include <lmoetd.hxx>
#include <lmoesu.hxx>


//
// These manifests define the level of access needed in this dialog.
// They are used in trust.cxx and also in usrmgr.cxx, which uses them
// to determine whether the user has privilege to enter TRUST_DIALOG.
//
#define UM_TRUST_ACCOUNT_DOMAIN   DEF_SAM_DOMAIN_ACCESS
#define UM_TRUST_BUILTIN_DOMAIN   DEF_SAM_DOMAIN_ACCESS
#define UM_TRUST_LSA_POLICY       POLICY_ALL_ACCESS
#define UM_TRUST_SAM_SERVER       DEF_SAM_SERVER_ACCESS


/*************************************************************************

    NAME:       TRUSTED_LBI

    SYNOPSIS:   This class represents one item in the TRUSTED_LISTBOX.

    INTERFACE:  TRUSTED_LBI               - Class constructor.

                ~TRUSTED_LBI              - Class destructor.

                Paint                   - Draw an item.

                QueryLeadingChar        - Query the first character for
                                          the keyboard interface.

                Compare                 - Compare two items.

                QueryDomainName         - Returns the trusted domain name
                                          associated with this item.

                QueryDomainPSID         - Returns the domain's PSID.

    PARENT:     LBI

    HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.
        beng        22-Apr-1992 Change to LBI::Paint

**************************************************************************/
class TRUSTED_LBI : public LBI
{
private:

    //
    //  These data members represent the various
    //  columns to be displayed in the listbox.
    //

    NLS_STR _nlsDomain;

    //
    //  The domain's SID.
    //

    PSID _psidDomain;

    //
    //  Storage for a copy of the domain's SID.
    //

    BUFFER _bufferSID;

    //
    //  Make a copy of a SID.
    //

    APIERR DuplicateSID( PSID psid );

protected:

    //
    //  This method paints a single item into the listbox.
    //

    virtual VOID Paint( LISTBOX     * plb,
                        HDC           hdc,
                        const RECT  * prect,
                        GUILTT_INFO * pGUILTT ) const;

    //
    //  This method returns the first character in the
    //  listbox item.  This is used for the listbox
    //  keyboard interface.
    //

    virtual WCHAR QueryLeadingChar( VOID ) const;

    //
    //  This method compares two listbox items.  This
    //  is used for sorting the listbox.
    //

    virtual INT Compare( const LBI * plbi ) const;

public:

    //
    //  Usual constructor/destructor goodies.
    //

    TRUSTED_LBI( const TRUSTED_DOMAIN_ENUM_OBJ * pobjDomain );
    TRUSTED_LBI( const TCHAR                    * pszDomain,
                 PSID                             psidDomain );
    virtual ~TRUSTED_LBI();

    //
    //  Retrieve the domain name associated with
    //  this listbox item.
    //

    const TCHAR * QueryDomainName( VOID ) const
        { return _nlsDomain.QueryPch(); }

    PSID QueryDomainPSID( VOID ) const
        { return _psidDomain; }

};  // class TRUSTED_LBI


/*************************************************************************

    NAME:       TRUSTED_LISTBOX

    SYNOPSIS:   This listbox displays the domains trusted by a
                specific domain.

    INTERFACE:  TRUSTED_LISTBOX           - Class constructor.

                ~TRUSTED_LISTBOX          - Class destructor.

                Fill                    - Fill the trust list.

                QueryColumnWidths       - Called by TRUSTED_LBI::Paint(),
                                          this is the column width table
                                          used for painting the listbox
                                          items.

    PARENT:     BLT_LISTBOX

    USES:       ??

    HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.

**************************************************************************/
class TRUSTED_LISTBOX : public BLT_LISTBOX
{
private:

    //
    //  This array contains the column widths used
    //  while painting the listbox item.  This array
    //  is generated by the BuildColumnWidthTable()
    //  function.
    //

    UINT _adx[1];

//-ckm    //
//-ckm    //  This is the cute little icon which is displayed
//-ckm    //  in each of the TRUSTED_LBI listbox items.
//-ckm    //
//-ckm
//-ckm    DMID_DTE _dteIcon;

    //
    //  This contains the name of the target "gullible" domain.
    //

    NLS_STR _nlsGullibleDomain;

    //
    //  We need to keep the trusted domain enumerator hanging
    //  around for the life of the listbox.
    //

    TRUSTED_DOMAIN_ENUM * _pTrustedDomainEnum;

public:

    //
    //  Usual constructor/destructor goodies.
    //

    TRUSTED_LISTBOX( OWNER_WINDOW   * powOwner,
                     CID              cid,
                     const TCHAR    * pszGullibleDomain );

    ~TRUSTED_LISTBOX();

    //
    //  This method fills the listbox with the trusted domains.
    //

    APIERR Fill( const LSA_POLICY * plsapol );


//-ckm    //
//-ckm    //  Return a pointer to the listbox icon.
//-ckm    //
//-ckm
//-ckm    DMID_DTE * QueryIcon( VOID ) const
//-ckm        { return &_dteIcon; }

    //
    //  This method is called by the TRUSTED_LBI::Paint()
    //  method for retrieving the column width table.
    //

    const UINT * QueryColumnWidths( VOID ) const
        { return _adx; }

    //
    //  The following macro will declare (& define) a new
    //  QueryItem() method which will return an TRUSTED_LBI *.
    //

    DECLARE_LB_QUERY_ITEM( TRUSTED_LBI )

};  // class TRUSTED_LISTBOX


/*************************************************************************

    NAME:       PERMITTED_LBI

    SYNOPSIS:   This class represents one item in the PERMITTED_LISTBOX.

    INTERFACE:  PERMITTED_LBI           - Class constructor.

                ~PERMITTED_LBI          - Class destructor.

                Paint                   - Draw an item.

                QueryLeadingChar        - Query the first character for
                                          the keyboard interface.

                Compare                 - Compare two items.

                QueryDomain             - Returns the permitted domain
                                          name associated with this item.

    PARENT:     LBI

    HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.
        beng        22-Apr-1992 Change to LBI::Paint

**************************************************************************/
class PERMITTED_LBI : public LBI
{
private:

    //
    //  These data members represent the various
    //  columns to be displayed in the listbox.
    //

    NLS_STR _nlsDomain;

protected:

    //
    //  This method paints a single item into the listbox.
    //

    virtual VOID Paint( LISTBOX     * plb,
                        HDC           hdc,
                        const RECT  * prect,
                        GUILTT_INFO * pGUILTT ) const;

    //
    //  This method returns the first character in the
    //  listbox item.  This is used for the listbox
    //  keyboard interface.
    //

    virtual WCHAR QueryLeadingChar( VOID ) const;

    //
    //  This method compares two listbox items.  This
    //  is used for sorting the listbox.
    //

    virtual INT Compare( const LBI * plbi ) const;

public:

    //
    //  Usual constructor/destructor goodies.
    //

    PERMITTED_LBI( const SAM_USER_ENUM_OBJ * pobjUser );
    PERMITTED_LBI( const TCHAR * pszDomain );
    virtual ~PERMITTED_LBI();

    //
    //  Retrieve the domain name associated with
    //  this listbox item.
    //

    const TCHAR * QueryDomainName( VOID ) const
        { return _nlsDomain.QueryPch(); }

};  // class PERMITTED_LBI


/*************************************************************************

    NAME:       PERMITTED_LISTBOX

    SYNOPSIS:   This listbox displays the domains permitted to trust a
                specific domain.

    INTERFACE:  PERMITTED_LISTBOX       - Class constructor.

                ~PERMITTED_LISTBOX      - Class destructor.

                Fill                    - Fill the trust list.

                QueryColumnWidths       - Called by PERMITTED_LBI::Paint(),
                                          this is the column width table
                                          used for painting the listbox
                                          items.

    PARENT:     BLT_LISTBOX

    USES:       ??

    HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.

**************************************************************************/
class PERMITTED_LISTBOX : public BLT_LISTBOX
{
private:

    //
    //  This array contains the column widths used
    //  while painting the listbox item.  This array
    //  is generated by the BuildColumnWidthTable()
    //  function.
    //

    UINT _adx[1];

//-ckm    //
//-ckm    //  This is the cute little icon which is displayed
//-ckm    //  in each of the PERMITTED_LBI listbox items.
//-ckm    //
//-ckm
//-ckm    DMID_DTE _dteIcon;

    //
    //  This contains the name of the "permitting" domain.
    //

    NLS_STR _nlsPermittingDomain;

public:

    //
    //  Usual constructor/destructor goodies.
    //

    PERMITTED_LISTBOX( OWNER_WINDOW   * powOwner,
                       CID              cid,
                       const TCHAR    * pszPermittingDomain );

    ~PERMITTED_LISTBOX();

    //
    //  This method fills the listbox with the permitted domains.
    //

    APIERR Fill( const SAM_DOMAIN * psamdom );


//-ckm    //
//-ckm    //  Return a pointer to the listbox icon.
//-ckm    //
//-ckm
//-ckm    DMID_DTE * QueryIcon( VOID ) const
//-ckm        { return &_dteIcon; }

    //
    //  This method is called by the PERMITTED_LBI::Paint()
    //  method for retrieving the column width table.
    //

    const UINT * QueryColumnWidths( VOID ) const
        { return _adx; }

    //
    //  The following macro will declare (& define) a new
    //  QueryItem() method which will return an PERMITTED_LBI *.
    //

    DECLARE_LB_QUERY_ITEM( PERMITTED_LBI )

};  // class PERMITTED_LISTBOX


/*************************************************************************

    NAME:       TRUST_DIALOG

    SYNOPSIS:   The TRUST_DIALOG class is used manipulate the trusted
                domain list.

    INTERFACE:  TRUST_DIALOG            - Class constructor.

                ~TRUST_DIALOG           - Class destructor.

    PARENT:     DIALOG_WINDOW

    USES:       ??

    HISTORY:
        KeithMo     08-Apr-1992 Created for the User Manager.

**************************************************************************/
class TRUST_DIALOG : public DIALOG_WINDOW
{
private:

    //
    //  The listboxen.
    //

    TRUSTED_LISTBOX   _lbTrustedDomains;
    PERMITTED_LISTBOX _lbPermittedDomains;

    //
    //  The various buttons we'll need to manipulate.
    //

    PUSH_BUTTON _pbCancel;
    PUSH_BUTTON _pbRemoveTrusted;
    PUSH_BUTTON _pbRemovePermitted;

    //
    //  The text we'll stuff into the "Cancel" button
    //  after a successful non undoable operation.
    //

    RESOURCE_STR _nlsCloseText;

    //
    //  ?
    //

    ADMIN_AUTHORITY * _padminauth;

    UM_ADMIN_APP * _pumadminapp;

    //
    //  This is used to display the current domain name.
    //

    SLT _sltDomain;

    //
    //  Storage for the domain name.
    //

    NLS_STR _nlsDomain;

    //
    //  This method will adjust the "Remove" buttons to
    //  reflect the current state of the listboxen.
    //

    VOID AdjustButtons( VOID );

    //
    //  Add new trusted/permitted domains.
    //

    APIERR GetNewTrustedDomain( VOID );
    APIERR GetNewPermittedDomain( VOID );

    APIERR ConfirmTrustRelationship( NLS_STR & nlsDomainName,
                                                 NLS_STR & nlsPassword,
                                                 BOOL * fTrustAccountOk );

    APIERR W_AddTrustedDomain( NLS_STR & nlsDomainName,
                               NLS_STR & nlsPassword,
                               LSA_PRIMARY_DOM_INFO_MEM * plsaprim );

    APIERR W_AddPermittedDomain( NLS_STR & nlsDomainName,
                                 NLS_STR & nlsPassword );

    //
    //  Remove the selected trusted/permitted domain.
    //

    APIERR RemoveTrustedDomain( VOID );
    APIERR RemovePermittedDomain( VOID );

    APIERR W_NukeTrustedDomain( const TRUSTED_LBI * plbi );
    APIERR W_NukePermittedDomain( const TCHAR * pszDomainName );

protected:

    //
    //  This virtual callback handles user commands.
    //

    virtual BOOL OnCommand( const CONTROL_EVENT & event );

public:

    //
    //  Usual constructor/destructor goodies.
    //

    TRUST_DIALOG( UM_ADMIN_APP * pumadminapp,
                  const TCHAR * pszDomainName,
                  ADMIN_AUTHORITY * padminauth );

    ~TRUST_DIALOG( VOID );

    virtual ULONG QueryHelpContext( VOID );

    UM_ADMIN_APP * QueryAdminApp()
        { return _pumadminapp; }


    ADMIN_AUTHORITY * QueryAdminAuthority()
        { return _padminauth; }


};  // class TRUST_DIALOG


/*************************************************************************

    NAME:       ADD_TRUST_DIALOG

    SYNOPSIS:   The ADD_TRUST_DIALOG class is used to add a new
                trusted domain to the trusted domain list.

    INTERFACE:  ADD_TRUST_DIALOG        - Class constructor.

                ~ADD_TRUST_DIALOG       - Class destructor.

    PARENT:     DIALOG_WINDOW

    USES:       ??

    HISTORY:
        KeithMo     16-Apr-1992 Created for the User Manager.

**************************************************************************/
class ADD_TRUST_DIALOG : public DIALOG_WINDOW
{
private:

    TRUST_DIALOG * _pdlgTrust;
    //
    //  The various controls on the dialog.
    //

    SLE              _sleDomainName;
    PASSWORD_CONTROL _password;

    //
    //  These pointers are used to return the
    //  domain name & password to the caller.
    //

    NLS_STR * _pnlsDomainName;
    NLS_STR * _pnlsPassword;

    //
    //  These strings will contain temporary copies of
    //  the domain name, password, and confirmed password.
    //  We preallocate these at dialog construct time to
    //  reduce the possibility of errors during OnOK processing.
    //

    NLS_STR _nlsTmpDomain;
    NLS_STR _nlsTmpPassword;

protected:

    //
    //  This virtual callback is invoked when the user
    //  presses the OK button.
    //

    virtual BOOL OnOK( VOID );

public:

    //
    //  Usual constructor/destructor goodies.
    //

    ADD_TRUST_DIALOG( TRUST_DIALOG * pdlgTrust,
                      NLS_STR * pnlsDomainName,
                      NLS_STR * pnlsPassword );

    ~ADD_TRUST_DIALOG( VOID );

    virtual ULONG QueryHelpContext( VOID );


};  // class ADD_TRUST_DIALOG


/*************************************************************************

    NAME:       ADD_PERMITTED_DIALOG

    SYNOPSIS:   The ADD_PERMITTED_DIALOG class is used to add a new
                domain that we will allow to trust us.

    INTERFACE:  ADD_PERMITTED_DIALOG    - Class constructor.

                ~ADD_PERMITTED_DIALOG   - Class destructor.

    PARENT:     DIALOG_WINDOW

    USES:       ??

    HISTORY:
        KeithMo     16-Apr-1992 Created for the User Manager.

**************************************************************************/
class ADD_PERMITTED_DIALOG : public DIALOG_WINDOW
{
private:

    TRUST_DIALOG * _pdlgTrust;
    //
    //  The various controls on the dialog.
    //

    SLE              _sleDomainName;
    PASSWORD_CONTROL _password;
    PASSWORD_CONTROL _passwordConfirm;

    //
    //  These pointers are used to return the
    //  domain name & password to the caller.
    //

    NLS_STR * _pnlsDomainName;
    NLS_STR * _pnlsPassword;

    //
    //  These strings will contain temporary copies of
    //  the domain name, password, and confirmed password.
    //  We preallocate these at dialog construct time to
    //  reduce the possibility of errors during OnOK processing.
    //

    NLS_STR _nlsTmpPassword;
    NLS_STR _nlsTmpConfirm;
    NLS_STR _nlsTmpDomain;

protected:

    //
    //  This virtual callback is invoked when the user
    //  presses the OK button.
    //

    virtual BOOL OnOK( VOID );

public:

    //
    //  Usual constructor/destructor goodies.
    //

    ADD_PERMITTED_DIALOG( TRUST_DIALOG * pdlgTrust,
                          NLS_STR * pnlsDomainName,
                          NLS_STR * pnlsPassword );

    ~ADD_PERMITTED_DIALOG( VOID );

    virtual ULONG QueryHelpContext( VOID );


};  // class ADD_PERMITTED_DIALOG


#endif  // _TRUST_HXX_
